/*
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.enquete;

import java.util.ArrayList;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnquete;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnqueteChoiceItem;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnqueteItem;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.enquete.util.EnqueteUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * Enqueteのフォームデータを管理するクラスです。 <BR>
 * 
 */
public class EnquetePracticeFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(EnqueteFormData.class.getName());

  /** アンケート名 */
  private ALStringField name;

  /** コメント */
  private ALStringField comment;

  /** アンケート項目・選択肢取得リスト */
  private List<EnqueteItemRecordData> itemresults;

  /**
   * init
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

    /**
     * 下記のコメントアウトのようにアクセスコントロールを追加する場合は
     * com.aimluck.eip.services.accessctl.ALAccessControlConstantsに
     * アクセスコントロール用の定数を追加してください。 (例) public static final String
     * POERTLET_FEATURE_ENQUETE_ENQUETE_OTHER = "enquete_enquete_other";
     */

    // aclPortletFeature =
    // ALAccessControlConstants.POERTLET_FEATURE_ENQUETE_ENQUETE_SELF;
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */

  public void initField() {

    // アンケート名
    name = new ALStringField();
    name.setFieldName("アンケート名");

    // メモ
    comment = new ALStringField();
    comment.setFieldName("コメント");
  }

  /**
   * アンケートの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {

  }

  /**
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    boolean res = super.setFormData(rundata, context, msgList);

    return res;
  }

  /**
   * アンケートのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {
    return (msgList.size() == 0);
  }

  /**
   * アンケートをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) {

    try {
      // // オブジェクトモデルを取得
      AvzTEnquete enquete = EnqueteUtils.getAvzTEnquete(rundata, context);
      if (enquete == null) {
        return false;
      }

      // アンケート名
      name.setValue(enquete.getName());
      // コメント
      comment.setValue(enquete.getComment());

      // アンケート項目入力フォーム内のデータの取得
      SelectQuery<AvzTEnqueteItem> itemquery =
        Database.query(AvzTEnqueteItem.class);
      Expression itemexp =
        ExpressionFactory
          .matchDbExp(AvzTEnqueteItem.AVZ_TENQUETE_PROPERTY
            + "."
            + AvzTEnquete.ENQUETE_ID_PK_COLUMN, enquete
            .getEnqueteId()
            .toString());
      itemquery.setQualifier(itemexp);
      itemquery.distinct(true);
      itemquery.orderAscending(AvzTEnqueteItem.ITEM_NUMBER_PROPERTY);

      ResultList<AvzTEnqueteItem> itemlist = itemquery.getResultList();
      // int loginuserid = Integer.valueOf(ALEipUtils.getUserId(rundata));

      // 回答者の一人を取得
      List getAnswer = enquete.getAvzTEnqueteChoiceItem();
      AvzTEnqueteChoiceItem cho = (AvzTEnqueteChoiceItem) getAnswer.get(0);
      int someAnswer = cho.getAnswerId();

      itemresults = new ArrayList<EnqueteItemRecordData>();

      for (int i = 0; i < itemlist.size(); i++) {
        AvzTEnqueteItem item = itemlist.get(i);
        EnqueteItemRecordData d = new EnqueteItemRecordData();
        ALStringField record = new ALStringField();

        d.setItemId(item.getItemId());
        int item_id = item.getItemId();
        d.setItemNumber(item.getItemNumber());
        record.setValue(item.getItemTitle());
        d.setItemName(record);
        d.setStatus(item.getStatus());
        d.setChoiceItemName(EnqueteUtils.getPracticePreview(
          rundata,
          context,
          item_id,
          someAnswer));

        itemresults.add(d);

      }

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * Enqueteをデータベースに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {

    return true;
  }

  /**
   * データベースに格納されているEnqueteを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      AvzTEnquete enquete = EnqueteUtils.getAvzTEnquete(rundata, context);
      if (enquete == null) {
        return false;
      }

      enquete.setPublicStatus(EnqueteUtils.PUBLIC_STATUS_PRACTICE);

      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        enquete.getEnqueteId(),
        ALEventlogConstants.PORTLET_TYPE_NONE,
        name.getValue());

    } catch (Exception ex) {
      Database.rollback();
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * Enqueteをデータベースから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {

    return true;
  }

  /**
   * アンケート名を取得します。 <BR>
   * 
   * @return
   */
  public ALStringField getName() {
    return name;
  }

  /**
   * @return
   */
  public String getComment() {
    return ALEipUtils.getMessageList(comment.getValue());
  }

  /**
   * アンケート項目・選択肢取得リストを取得します。 <BR>
   * 
   * @return
   */

  public List<EnqueteItemRecordData> getItemResults() {
    return itemresults;
  }

}
