/***************************************************************************
 *            cxp-handler.c
 *
 *  Mon Nov 29 23:50:12 2004
 *  Copyright  2004  Yasumichi Akahoshi
 *  yasumichi@users.sourceforge.jp
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include <string.h>
#include "cxp-profile.h"
#include "cxp-handler.h"
#include "cxp-entry-dialog.h"

/**
 * @brief private member of CxpHandler
 */

struct _CxpHandlerPrivate
{
	CxpProfile *profile;
	gboolean dispose_has_run;	/**< Is dispose funciton executed */
};

/*
 * privete methods
 */
static void cxp_handler_init (GTypeInstance * instance, gpointer g_class);
static void cxp_handler_class_init (gpointer g_class, gpointer g_class_data);
static void cxp_handler_dispose (GObject * obj);
static void cxp_handler_finalize (GObject * obj);

/**
 * regist CxpHandler to GTypeTable.
 * @return GType
 */
GType cxp_handler_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpHandlerClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			cxp_handler_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpHandler),
			0,	/* n_preallocs */
			cxp_handler_init	/* instance_init */
		};
		type = g_type_register_static (G_TYPE_OBJECT, "CxpHandlerType",
					       &info, 0);
	}

	return type;
}

/**
 * base init CxpHandler
 *
 */
static void cxp_handler_init (GTypeInstance * instance, gpointer g_class)
{
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");

	CxpHandler *self = CXP_HANDLER (instance);

	self->priv = g_new (CxpHandlerPrivate, 1);
	self->priv->profile = cxp_profile_new ("common", "handler");
	self->priv->dispose_has_run = FALSE;
}

/**
 * class init CxpHandlerClass
 *
 */
static void cxp_handler_class_init (gpointer g_class, gpointer g_class_data)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (g_class);

	gobject_class->dispose = cxp_handler_dispose;
	gobject_class->finalize = cxp_handler_finalize;
}

static void cxp_handler_dispose (GObject * obj)
{
	CxpHandler *self = CXP_HANDLER (obj);

	if (self->priv->dispose_has_run)
	{
		return;
	}

	self->priv->dispose_has_run = TRUE;
	g_object_unref (self->priv->profile);
}

static void cxp_handler_finalize (GObject * obj)
{
	CxpHandler *self = CXP_HANDLER (obj);

	g_free (self->priv);
}

CxpHandler *cxp_handler_new (void)
{
	CxpHandler *handler;

	handler = CXP_HANDLER (g_object_new (CXP_TYPE_HANDLER, NULL));

	return handler;
}

void cxp_handler_launch (CxpHandler * handler, const gchar * fullpath)
{
	gchar *filename;
	gchar *suffix = NULL;
	gchar *cmd = NULL;
	gchar *fullcmd;
	gchar *message;
	gint index;
	GtkWidget *dialog;

	g_return_if_fail (fullpath != NULL);

	filename = g_path_get_basename (fullpath);
	if ((suffix = g_strrstr (filename, ".")) != NULL)
	{
		cmd = cxp_profile_get_string (handler->priv->profile, suffix);
		if (cmd == NULL)
		{
			message =
				g_strdup_printf (_
						 ("Please input the command that opens the file whose extension is %s"),
						 suffix);
			dialog = cxp_entry_dialog_new (_("Undefined extension"),
						       message, "");
			if (gtk_dialog_run (GTK_DIALOG (dialog)) ==
			    GTK_RESPONSE_OK)
			{
				cmd = cxp_entry_dialog_get_entry_text
					(CXP_ENTRY_DIALOG (dialog));
				if (cmd != NULL)
				{
					cxp_profile_set_string (handler->priv->
								profile, suffix,
								cmd);
				}
			}
			gtk_widget_destroy (dialog);
			g_free (message);
		}
	}
	else
	{
		message =
			g_strdup_printf (_
					 ("Please input the command that opens file  %s"),
					 filename);
		dialog = cxp_entry_dialog_new (_("File without extension"),
					       message, "");
		if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
		{
			cmd = cxp_entry_dialog_get_entry_text (CXP_ENTRY_DIALOG
							       (dialog));
		}
		gtk_widget_destroy (dialog);
		g_free (message);
	}

	if (cmd != NULL)
	{
		fullcmd = g_strdup_printf ("%s '%s'", cmd, fullpath);
		g_spawn_command_line_async (fullcmd, NULL);
		g_free (cmd);
		g_free (fullcmd);
	}

	g_free (filename);
}
