/* $Id: cxp-utils.c,v 1.13 2005/05/18 17:33:07 yasumichi Exp $ */
/**
 * \if japanese
 * @file
 * @brief 桼ƥƥؿ
 *
 * @author 
 * @date Fri Jan 21 2005
 * $Revision: 1.13 $
 * \endif
 * \if english
 * @file
 * @brief Utility function group
 *
 * @author Yasumichi Akahoshi
 * @date Fri Jan 21 2005
 * $Revision: 1.13 $
 * \endif
 */


/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <cxp-utils.h>
#include <errno.h>

/*
 * This file contains support functions.It is generated by Glade.
 * add_pixmap_directory, find_pixmap_file, create_pixmap
 */

static GList *pixmaps_directories = NULL;	/**< It have directory list. */
static gchar *fname_filter = NULL;

/**
 * \if japanese
 * ե򸡺ǥ쥯ȥɲäޤGlade륵ݡȴؿǤ
 * @param [in] directory ɲäǥ쥯ȥ̾
 * \endif
 * \if english
 * The directory that retrieves the picture file is added. It is a support
 * function that Glade generates. 
 * @param [in] directory Directory Added directory name
 * \endif
 */
void add_pixmap_directory (const gchar * directory)
{
	pixmaps_directories = g_list_prepend (pixmaps_directories,
					      g_strdup (directory));
}

/**
 * \if japanese
 * ե̾add_pixmap_directoryǲä줿ǥ쥯ȥ򸡺
 * ĤäեΥѥ֤ޤGlade륵ݡȴؿǤ
 * @param [in] filename Ѥե̾
 * @return ĤäեΥѥĤʤϡNULL
 * \endif
 * \if english
 * The directory of the file name added with add_pixmap_directory is retrieved,
 * and passing the found file is returned. It is a support function that Glade
 * generates.
 * @param [in] filename Name of picture file that wants to be used
 * @return Passing of found file. NULL when not found.
 * \endif
 */
static gchar *find_pixmap_file (const gchar * filename)
{
	GList *elem;

	/* We step through each of the pixmaps directory to find it. */
	elem = pixmaps_directories;
	while (elem)
	{
		gchar *pathname =
			g_strdup_printf ("%s%s%s", (gchar *) elem->data,
					 G_DIR_SEPARATOR_S, filename);

		if (g_file_test (pathname, G_FILE_TEST_EXISTS))
			return pathname;
		g_free (pathname);
		elem = elem->next;
	}
	return NULL;
}

/**
 * \if japanese
 * ե뤫顢pixmap롣
 * @param [in] widget åȤؤΥݥ
 * @param [in] filename ե̾
 * @return ˺å
 * \endif
 * \if english
 * This is an internally used function to create pixmaps.
 * @param widget [in] Pointer to widget.
 * @param filename [in] File name of pixmap.
 * @return Pointer to new widget of pixmap.
 * \endif
 */
GtkWidget *create_pixmap (GtkWidget * widget, const gchar * filename)
{
	gchar *pathname = NULL;
	GtkWidget *pixmap;

	if (!filename || !filename[0])
		return gtk_image_new ();

	pathname = find_pixmap_file (filename);

	if (!pathname)
	{
		g_warning ("Couldn't find pixmap file: %s", filename);
		return gtk_image_new ();
	}

	pixmap = gtk_image_new_from_file (pathname);
	g_free (pathname);
	return pixmap;
}

/**
 * \if japanese
 * ˲ɤ߹
 * @param [in] filename ɤ߹ߤե
 * @return β᡼
 * \endif
 * \if english
 * @param [in] filename Picture file that wants to read
 * @return Image of image in memory
 * \endif
 */
GdkPixbuf *create_pixbuf (const gchar * filename)
{
	gchar *pathname = NULL;
	GdkPixbuf *pixbuf;
	GError *error = NULL;

	if (!filename || !filename[0])
		return NULL;

	pathname = find_pixmap_file (filename);

	if (!pathname)
	{
		g_warning ("Couldn't find pixmap file: %s", filename);
		return NULL;
	}

	pixbuf = gdk_pixbuf_new_from_file (pathname, &error);
	if (!pixbuf)
	{
		g_warning ("Failed to load pixbuf file: %s: %s\n", pathname,
			   error->message);
		g_error_free (error);
	}
	g_free (pathname);
	return pixbuf;
}

/**
 * \if japanese
 * ե̾ǥ󥳡ɤ뤿γե륿ꤷޤ
 * @param [in] value ե륿Υޥɥ饤
 * \endif
 * \if english
 * An external filter to encode the file name with locale is set. 
 * @param [in] value Command line of external filter
 * \endif
 */
void cxp_utils_set_filename_filter (const gchar *value)
{
	fname_filter = g_strdup (value);
}

/**
 * \if japanese
 * ե̾ǥ󥳡ɤ뤿˳ե륿Ȥޤ
 * \endif
 * \if english
 * The setting that uses an external filter to encode the file name with locale
 * is released. 
 * \endif
 */
void cxp_utils_free_filename_filter (void)
{
	g_free (fname_filter);
}

/**
 * \if japanese
 * ե̾κǸιǤUTF-8ǥ󥳡ɤʸޤ⤷
 * ե̾κǸ夬ǥ쥯ȥζڤʸǽäƤ硢Τ
 * ǸΥåǤǤ⤷ե̾ǥ쥯ȥζڤʸ
 * ΤߤǤä顢ñζڤʸ֤ޤ⤷ե̾ǤС
 * "."ޤ
 * @param [in] file_name ե̾
 * @return ˳ݤ줿ե̾κǸιǤǥ뤫UTF-8Ѵ
 * 줿ʸǤ
 * \endif
 * \if english
 * Gets the last component of the filename converted to UTF-8. If file_name ends
 * with a directory separator it gets the component before the last slash. If 
 * file_name consists only of directory separators (and on Windows, possibly a
 * drive letter), a single separator is returned. If file_name is empty, it gets
 * ".".
 * @param file_name  the name of the file.
 * @return a newly allocated string containing the last component of the
 * filename converted from locale to UTF-8.
 * \endif
 */
gchar *cxp_path_get_basename_of_utf8 (const gchar * file_name)
{
	gchar *base_name;
	gchar *basename_locale = NULL;
	gchar *basename_utf8;
	gchar *cmd;
	gint exit_status;

	if (fname_filter == NULL)
	{
		basename_locale = g_path_get_basename (file_name);
	}
	else
	{
		base_name = g_path_get_basename (file_name);
		cmd = g_strdup_printf ("%s '%s'", fname_filter, base_name);
		g_spawn_command_line_sync (cmd, &basename_locale, NULL, NULL, NULL);
		g_free (cmd);
		g_strchomp (basename_locale);
	}
	basename_utf8 =
		g_locale_to_utf8 (basename_locale, -1, NULL, NULL, NULL);
	g_free (basename_locale);

	return basename_utf8;
}

/**
 * \if japanese
 * ե̾Υǥ쥯ȥʬUTF-8ѴΤޤ⤷ե̾
 * ǥ쥯ȥʬޤǤʤС"."֤ޤ֤줿ʸɬפʤ
 * ʤ꼡衢Ƥ
 * @param [in] file_name ե̾
 * @return 뤫UTF-8Ѵ줿ե̾Υǥ쥯ȥʬ
 * \endif
 * \if english
 * Gets the directory components of a file name converted to UTF-8. If the file 
 * name has no directory components "." is returned. The returned string should
 * be freed when no longer needed.
 * @param file_name	the name of the file.
 * @return the directory components of the file converted from locale to UTF-8.
 * \endif
 */
gchar *cxp_path_get_dirname_of_utf8 (const gchar * file_name)
{
	gchar *gcDirnameLocale;
	gchar *gcDirnameUtf8;

	gcDirnameLocale = g_path_get_dirname (file_name);
	gcDirnameUtf8 =
		g_locale_to_utf8 (gcDirnameLocale, -1, NULL, NULL, NULL);
	g_free (gcDirnameLocale);

	return gcDirnameUtf8;
}

/**
 * \if japanese
 * դȻ"2005/01/28 10:05:00"ͤʸѴޤ
 * @param [in] timep ջɽtime_tؤΥݥ
 * @return ˳ݤ줿Ѵʸ
 * \endif
 * \if english
 * This function format date and time.
 * @param timep pointer to time_t.
 * @return newly alocate formatted string.
 * \endif
 */
gchar *cxp_strftime (const time_t * timep)
{
	gchar *str;
	struct tm *tm;

	tm = localtime (timep);
	str = g_new (gchar, 20);
	strftime (str, 20, "%Y/%m/%d %H:%M:%S", tm);

	return str;
}

void *cxp_error_dialog_run_about_file (gchar *filename)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new (NULL,
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_ERROR,
			GTK_BUTTONS_CLOSE,
			"%s:%s",
			filename,
			g_strerror (errno));
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

/**
 * \if japanese
 * ե
 * \endif
 */
gboolean cxp_remove_file (const gchar *fullpath)
{
	GtkWidget *dialog;
	gchar *filename;

	/* for "rm -rf" */
	gchar *error_msg;
	gchar *cmd;
	gchar *standard_error;
	gint exit_status;

	filename = cxp_path_get_basename_of_utf8 (fullpath);

	if (g_file_test (fullpath, G_FILE_TEST_IS_DIR)
	    && !g_file_test (fullpath, G_FILE_TEST_IS_SYMLINK))
	{
		errno = 0;
		if ((exit_status = rmdir (fullpath)) != 0)
		{
			switch (errno)
			{
			case ENOTEMPTY:
				dialog = gtk_message_dialog_new (NULL,
								 GTK_DIALOG_DESTROY_WITH_PARENT,
								 GTK_MESSAGE_ERROR,
								 GTK_BUTTONS_YES_NO,
								 "%s:%s\nDo you really delete it?",
								 filename,
								 g_strerror
								 (errno));
				if (gtk_dialog_run (GTK_DIALOG (dialog)) ==
				    GTK_RESPONSE_YES)
				{
					cmd = g_strdup_printf ("rm -rf '%s'",
							       fullpath);
					g_spawn_command_line_sync (cmd, NULL,
								   &standard_error,
								   &exit_status,
								   NULL);
					if (exit_status != 0)
					{
						error_msg =
							g_locale_to_utf8
							(standard_error, -1,
							 NULL, NULL, NULL);
						dialog = gtk_message_dialog_new
							(NULL,
							 GTK_DIALOG_DESTROY_WITH_PARENT,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_CLOSE,
							 "%s", error_msg);
						gtk_dialog_run (GTK_DIALOG
								(dialog));
						gtk_widget_destroy (dialog);
						g_free (error_msg);
					}
				}
				gtk_widget_destroy (dialog);
				break;
			default:
				cxp_error_dialog_run_about_file (filename);
			}
		}
	}
	else
	{
		errno = 0;
		if ((exit_status = unlink (fullpath)) != 0)
		{
			cxp_error_dialog_run_about_file (filename);
		}
	}

	g_free (filename);

	return	(exit_status == 0)?TRUE:FALSE;
}

gchar* cxp_utils_make_unique_key (void)
{
	GTimeVal time_val;

	g_get_current_time (&time_val);
	return g_strdup_printf("%ld", time_val.tv_sec);
}
