/**
 * @file
 * @brief A widget of Property dialog
 *
 * A widget of Property dialog
 * @author Yasumichi Akahoshi <yasumichi@users.sourceforge.jp>
 * @date Sun Apr 3 02:53:00 2005
 * $Revision: 1.13 $
 ****************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gconf/gconf-client.h>
#include <cxp.h>
#include <glib/gi18n.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <grp.h>
#include <pwd.h>
#include "cxp-pref-dialog.h"

enum
{
	ENTRY_PREVIEW_TEXT = 0,
	ENTRY_PREVIEW_BINARY,
	ENTRY_PREVIEW_DIRECTORY,
	ENTRY_ATTACH_MAIL,
	ENTRY_FILENAME_FILTER,
	ENTRY_COUNT,
};

gchar *config_key[] = {
	"/apps/cxp/cxplorer/PreviewText",
	"/apps/cxp/cxplorer/PreviewBinary",
	"/apps/cxp/cxplorer/PreviewDir",
	"/apps/cxp/cxplorer/AttachCommand",
	"/apps/cxp/cxplorer/FilenameFilter"
};

/**
 * the private structure 
 */
typedef struct
{
	GConfClient *client;
	GtkWidget *ml_entry;
	GtkWidget *entry[ENTRY_COUNT];
	gboolean dispose_has_run;
} CxpPrefDialogPrivate;

#define CXP_PREF_DIALOG_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CXP_TYPE_PREF_DIALOG, CxpPrefDialogPrivate))

static GObjectClass *parent_class = NULL;

enum
{
	CXP_PREF_DIALOG_GCONF_CLIENT = 1,
};

static void cxp_pref_dialog_class_init (CxpPrefDialogClass *klass);
static void cxp_pref_dialog_instance_init (GTypeInstance * instance,
					       gpointer g_class);
static GObject * cxp_pref_dialog_constructor (GType type, guint n_construct_properties, GObjectConstructParam *construct_properties);
static void cxp_pref_dialog_construct_child (CxpPrefDialog *self);
static void cxp_pref_dialog_dispose (GObject * obj);
static void cxp_pref_dialog_finalize (GObject * obj);
static void cxp_pref_dialog_ok_button_clicked (GtkWidget *button, CxpPrefDialog *dialog);
static void cxp_pref_dialog_custom_button_clicked (GtkWidget *button, CxpPrefDialog *dialog);

static void cxp_pref_dialog_class_init (CxpPrefDialogClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->constructor = cxp_pref_dialog_constructor;
	gobject_class->dispose = cxp_pref_dialog_dispose;
	gobject_class->finalize = cxp_pref_dialog_finalize;

	parent_class = g_type_class_peek_parent (klass);

	g_type_class_add_private (klass, sizeof (CxpPrefDialogPrivate));
}

static void cxp_pref_dialog_instance_init (GTypeInstance * instance,
					       gpointer g_class)
{
	CxpPrefDialogPrivate *private = CXP_PREF_DIALOG_GET_PRIVATE(instance);

	private->client = gconf_client_get_default ();
	private->dispose_has_run = FALSE;
}

static GObject * cxp_pref_dialog_constructor (GType type, guint n_construct_properties, GObjectConstructParam *construct_properties)
{
	GObject *obj;
	CxpPrefDialogClass *klass;

	klass = CXP_PREF_DIALOG_CLASS (g_type_class_peek (CXP_TYPE_PREF_DIALOG));
	obj = parent_class->constructor (type,
			n_construct_properties,
			construct_properties);

	cxp_pref_dialog_construct_child (CXP_PREF_DIALOG(obj));

	return obj;
}

static void cxp_pref_dialog_construct_child (CxpPrefDialog *self)
{
	CxpPrefDialogPrivate *private = CXP_PREF_DIALOG_GET_PRIVATE(self);
	GtkWidget *cancel_button;
	GtkWidget *ok_button;
	GtkWidget *custom_button;
	GtkWidget *notebook;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *hbox;
	gchar *entry_title[] = {
		N_("Text File"),
		N_("Binary File"),
		N_("Directory")
	};
	gchar *filter;
	gint index;

	gtk_window_set_destroy_with_parent(GTK_WINDOW(self), TRUE);
	gtk_window_set_title(GTK_WINDOW(self), _("Preferences"));

	notebook = gtk_notebook_new ();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(self)->vbox), notebook, FALSE, FALSE, 0);
	gtk_widget_show (notebook);

	table = gtk_table_new (4, 3, FALSE);
	label = gtk_label_new (_("max lines"));
	gtk_table_attach_defaults (GTK_TABLE(table), label, 0, 1, 0, 1);
	private->ml_entry = gtk_spin_button_new_with_range (10.0, 1000.0, 1.0);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON(private->ml_entry), gconf_client_get_int (private->client, "/apps/cxp/cxplorer/PreviewLines", NULL));
	gtk_table_attach_defaults (GTK_TABLE(table), private->ml_entry, 1, 2, 0, 1);
	for(index=0; index < 3; index++)
	{
		label = gtk_label_new (_(entry_title[index]));
		gtk_table_attach_defaults (GTK_TABLE(table), label, 0, 1, index+1, index+2);
		private->entry[index] = gtk_entry_new ();
		gtk_entry_set_text (GTK_ENTRY(private->entry[index]), gconf_client_get_string (private->client, config_key[index], NULL));
		gtk_entry_set_width_chars (GTK_ENTRY(private->entry[index]), 40);
		gtk_table_attach_defaults (GTK_TABLE(table), private->entry[index], 1, 2, index+1, index+2);
	}
	gtk_widget_show_all (table);
	
	label = gtk_label_new (_("Preview Command"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), table, label);

	vbox = gtk_vbox_new (FALSE, 2);
	hbox = gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, TRUE, 2);
	label = gtk_label_new (_("Mail"));
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 2);
	private->entry[ENTRY_ATTACH_MAIL] = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY(private->entry[ENTRY_ATTACH_MAIL]), gconf_client_get_string (private->client, config_key[ENTRY_ATTACH_MAIL], NULL));
	gtk_entry_set_width_chars (GTK_ENTRY(private->entry[ENTRY_ATTACH_MAIL]), 40);
	gtk_box_pack_start (GTK_BOX(hbox), private->entry[ENTRY_ATTACH_MAIL], FALSE, TRUE, 2);
	custom_button = gtk_button_new_with_label (_("Edit of additional menu..."));
	gtk_box_pack_start (GTK_BOX(vbox), custom_button, FALSE, FALSE, 2);
	g_signal_connect (custom_button, "clicked", G_CALLBACK(cxp_pref_dialog_custom_button_clicked), self);
	gtk_widget_show_all (vbox);

	label = gtk_label_new (_("Send To"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);


	vbox = gtk_vbox_new (FALSE, 2);
	label = gtk_label_new (_("An external filter is used to display the file name"));
	gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.0);
	gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, TRUE, 2);
	hbox = gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, TRUE, 2);
	label = gtk_label_new (_("Command"));
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 2);
	private->entry[ENTRY_FILENAME_FILTER] = gtk_entry_new ();
	if ((filter = gconf_client_get_string (private->client, config_key[ENTRY_FILENAME_FILTER], NULL)) != NULL)
	{
		gtk_entry_set_text (GTK_ENTRY(private->entry[ENTRY_FILENAME_FILTER]), filter);
	}
	gtk_entry_set_width_chars (GTK_ENTRY(private->entry[ENTRY_FILENAME_FILTER]), 40);
	gtk_box_pack_start (GTK_BOX(hbox), private->entry[ENTRY_FILENAME_FILTER], FALSE, TRUE, 2);
	gtk_widget_show_all (vbox);
	
	label = gtk_label_new (_("External filter"));
	gtk_notebook_append_page (GTK_NOTEBOOK(notebook), vbox, label);

	cancel_button = gtk_button_new_from_stock("gtk-cancel");
	gtk_dialog_add_action_widget (GTK_DIALOG (self), cancel_button, GTK_RESPONSE_CANCEL);
	gtk_widget_show (cancel_button);

	ok_button = gtk_button_new_from_stock("gtk-ok");
	gtk_dialog_add_action_widget (GTK_DIALOG (self), ok_button, GTK_RESPONSE_OK);
	gtk_widget_show (ok_button);
	g_signal_connect (ok_button, "clicked", G_CALLBACK(cxp_pref_dialog_ok_button_clicked), self);
}

static void cxp_pref_dialog_dispose (GObject * obj)
{
	CxpPrefDialogPrivate *private = CXP_PREF_DIALOG_GET_PRIVATE(obj);

	if (private->dispose_has_run)
	{
		/* If dispose did already run, return. */
		return;
	}
	/* Make sure dispose does not run twice. */
	private->dispose_has_run = TRUE;

	/* 
	 * In dispose, you are supposed to free all types referenced from this
	 * object which might themselves hold a reference to self. Generally,
	 * the most simple solution is to unref all members on which you own a 
	 * reference.
	 */
	g_object_unref (private->client);

	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void cxp_pref_dialog_finalize (GObject * obj)
{
	/* Chain up to the parent class */
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void cxp_pref_dialog_ok_button_clicked (GtkWidget *button, CxpPrefDialog *dialog)
{
	CxpPrefDialogPrivate *private = CXP_PREF_DIALOG_GET_PRIVATE(dialog);
	gint index;

	gconf_client_set_int (private->client, "/apps/cxp/cxplorer/PreviewLines", gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(private->ml_entry)), NULL);
	for (index=0; index<ENTRY_COUNT; index++)
	{
		gconf_client_set_string (private->client, config_key[index], g_strdup(gtk_entry_get_text(GTK_ENTRY(private->entry[index]))), NULL);
	}

	cxp_utils_set_filename_filter (gtk_entry_get_text(GTK_ENTRY(private->entry[ENTRY_FILENAME_FILTER])));
}

static void cxp_pref_dialog_custom_button_clicked (GtkWidget *button, CxpPrefDialog *dialog)
{
	GtkWidget *window;

	window = cxp_gconf_pair_editor_new (GTK_WINDOW(dialog), _("Edit of additional menu"), "/apps/cxp/SendTo", _("Caption"), _("Command"));
	gtk_widget_show (window);
	gtk_window_set_modal (GTK_WINDOW(window), TRUE);
}

GType cxp_pref_dialog_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (CxpPrefDialogClass),
			NULL,	/* base_init */
			NULL,	/* base_finalize */
			(GClassInitFunc) cxp_pref_dialog_class_init,	/* class_init */
			NULL,	/* class_finalize */
			NULL,	/* class_data */
			sizeof (CxpPrefDialog),
			0,	/* n_preallocs */
			cxp_pref_dialog_instance_init	/* instance_init */
		};
		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "CxpPrefDialogType",
					       &info, 0);
	}
	return type;
}

/**
 * Creates a new dialog box which show detail of file.
 */
GtkWidget *cxp_pref_dialog_new (GtkWindow *parent)
{
	CxpPrefDialog *dialog;

	dialog = g_object_new(CXP_TYPE_PREF_DIALOG, NULL);
	gtk_window_set_transient_for (GTK_WINDOW(dialog), parent);

	return	GTK_WIDGET(dialog);
}

