# キャラクタクラス
require 'uuidtools'
require 'darkhall/const'
require 'darkhall/character'
require 'darkhall/spell'

module DarkHall
	class Member < Character
		bool_attr_accessor :existing
		attr_accessor :uuid, :name, :number, :hp_max_base, :hp_max_penalty, :mp_max
		attr_accessor :mp_random_bonus
		attr_accessor :items, :weapon_index, :shield_index, :armor_index, :accessory_indexes
		attr_accessor :level, :exp, :last_got_exp, :gold, :last_got_gold, :spells, :spell_learning, :messages
		attr_accessor :mastery
		attr_accessor :elapsed_week, :birth_week, :age
		attr_accessor :last_select_ids
		attr_accessor :action
		attr_accessor :profiles
		attr_accessor :arrow_index
		attr_accessor :senility, :regenerated, :inherited_member_ids, :remaining_week_before_revived
		attr_accessor :life_end_week
		
		attr_accessor :bonus_points
		attr_accessor :skill_levels, :skill_points
		
		attr_accessor :backgrounds
		
		bool_attr_accessor :ai_act, :soul
		
		alias spell_ids spells
		alias spell_ids= spells=
		
		STORED_SIMPLE_VARIABLE_NAMES =
			%w(uuid name hp hp_max_base hp_max_penalty
			   exp gold level age elapsed_week birth_week
			   items weapon_index armor_index accessory_indexes spells ai_act
				 senility remaining_week_before_revived regenerated inherited_member_ids soul existing
				 life_end_week)
		
		STORED_HASH_VARIABLE_NAMES =
			%w(bonus_points skill_levels skill_points
			   mp mp_max mp_random_bonus spell_learning
				 profiles mastery backgrounds)



		def initialize(level = 1, age_type = AGE::YOUNG, randomize = true)
			super()
			# 存在フラグ
			@existing = true
			
			# レベルと経験値
			@level = level
			@exp = EXP_TABLE[level]
			@last_got_exp = 0
			
			# 誕生・蘇生・老衰
			@elapsed_week = 0
			@regenerated = 0
			@inherited_member_ids = []
			@remaining_week_before_revived = nil
			@senility = 0
			@life_end_week = nil

			
			# HP/MP設定
			@hp_max_base = 10 + hp_max_growth * 8
			if randomize then
				@hp_max_base += (rand(4) + rand(4))
			else
				@hp_max_base += 3
			end
			(level - 1).times do
				grow_hp(randomize)
			end
			@hp_max_penalty = 0
			@hp = @hp_max_base
			@soul = false
			@mp_max = {}
			@mp = {}
			@mp_random_bonus = {}
			
			# 所持品
			@gold = 30
			@items = []
			@weapon_index = nil
			@armor_index = nil
			@shield_index = nil
			@accessory_indexes = []

			

			
			# 誕生週の設定
			case age_type
			when AGE::CHILD
				default_age = 10
			when AGE::YOUNG
				default_age = 17
			when AGE::ELDER
				default_age = 25
			when AGE::OLD
				default_age = 60
			else
				raise "Unknown Age Symbol (#{age_type.inspect})"
			end
			week = GS.week || 0
			@birth_week = week - (WEEK_PER_YEAR * default_age + rand(180))
			@age = self.get_collect_age(week)

			
			@action = nil
			@arrow_index = 0

			@mastery = {MST::WEAPON => {}, MST::SHIELD => {}, MST::ARMOR => {}, MST::SPELL => {}}

			
			@spells = []
			@spell_learning = {}
			@bonus_points = {}
			ABILITY_TYPES.each do |type|
				@bonus_points[type] = (randomize ? rand(2) : 0)
			end
			
			@skill_levels = Hash.new(0)
			@skill_points = Hash.new(0)
			
			set_all_mp
			
			self.set_skills

			



			@profiles = {}

			MemberProfileType.constants.each do |x|
				@profiles[x.to_s] = 0
			end
			@profiles[MPT::CreatedWeek] = GS.week
			@profiles[MPT::LostWeek] = nil
			
			@backgrounds = {}
			
			@ai_act = false
			
			setup_on_load
			
		end
		
		def setup_on_load
			@action = nil
			@last_select_ids ||= {}
			DB.spells.each do |spell|
				@spell_learning[spell.id] ||= 0
				set_mp(spell.id) unless @mp.include?(spell.id)
			end

			
			# 後方互換
			@regenerated ||= 0
			@inherited_member_ids ||= []
			if @existing.nil? then
				@existing = true
			end
			@senility ||= 0
			@uuid ||= UUID.random_create.to_s

			
		end
		
		# 罠による死亡用
		def data_id
			"Other"
		end

		def exist?
			existing?
		end
		
		def dead?
			@dead or @soul
		end
		
		def alive?
			not @dead and not @soul
		end
		
		def life_end?
			(@life_end_week ? true : false)
		end
		
		def kill_to_soul
			kill
			@soul = true
		end
		
		
		def set_starting_equipments(weapon_id, armor_id = nil, *accessory_ids)
			@items.clear
			@weapon_index = nil
			@armor_index = nil
			@accessory_indexes.clear
		
			if weapon_id then
				get_item(Weapon.new(weapon_id)) and equip(@items.size - 1)
			end
			
			if armor_id then
				get_item(Armor.new(armor_id)) and equip(@items.size - 1)
			end
			
			accessory_ids.each do |id|
				get_item(Accessory.new(id)) and equip(@items.size - 1)
			end
			
			return self
			
		end

		def position_caption
			if @position and ai_act? then
				"#{super}A"
			else
				super
			end
		end


			
		def hp_max
			@hp_max_base - @hp_max_penalty
		end
		
		# キャラクターにHPダメージを与える
		def life_damage(value, killing = true)
			@hp_max_penalty += value
			if hp_max <= 0 then
				@hp_max_penalty = @hp_max_base - 1# 0以下にはならない
			end
			
			super
			
			return
		end

		
		def use_mp(value, spell_id)
			super
			use_rate = value.to_f / @mp_max[spell_id]
			
			# ブースト分は含まない
			if magic_boost? then
				use_rate *= 0.5
			end
			
			inc = use_rate * 10.0 + 1.0
			LOGGER.puts sprintf("\t(呪文熟練度 +%.1g%%)", inc)
			
			increase_spell_mastery(spell_id, inc)
			
			@profiles[MPT::UsedMP] += value
			
		end

		
		def revive_cost
			((20 + hp_max_base * 2.0) * GS.rule.revive_cost_rate).to_i
		end
		
		def regenerate_cost
			revive_cost * 2
		end
		
		def inherit_cost
			0
		end
		
		def inherited
			@inherited_member_ids.size
		end
		
		def revive_border
			75 - @regenerated * 2
		end
		
		def on_regenerated
			# 再生回数をカウント（老衰などに影響）
			@regenerated += 1
			
			# 経験値の20%分を失う（レベルダウンはしない）
			@exp = lose_exp_rate_on_total(0.2)
			
			# レベルアップ1回分+10の最大HPボーナス
			grow_hp
			@hp_max_base += 10
			@hp += 10
		end


		def weapon_hitting
			re = super
			equipments.compact.each{|x| re += x.hitting_bonus}
			re += (get_weapon_mastery * 0.03).to_i
			re += 2 if get_weapon_mastery >= 100.0
			
			re
		end
		
		def perfect_guard_border
			20 + (get_shield_mastery * 0.1).to_i
		end
		
		
		def avoid
			re = super
			equipments.compact.each{|x| re += x.avoid_bonus}
			
			re
		end
		
		def speed
			super + (get_armor_mastery * 0.05).to_i
		end
		
		def surviving_bonus
			status.map{|x| x.surviving_bonus}.total || 0
		end
		
		def fast_escaping_bonus
			status.map{|x| x.fast_escaping_bonus}.total || 0
		end
		
		def enemy_ousting_bonus
			status.map{|x| x.enemy_ousting_bonus}.total || 0
		end
		
		

		def damage_resists
			re = {}
			equipments.each do |item|
				if item and item.damage_resists then
					re.merge!(item.damage_resists)
				end
			end
			
			re
		end
		
		def state_resists
			re = {}
			equipments.each do |item|
				if item and item.state_resists then
					re.merge!(item.state_resists)
				end
			end
			
			re
		end
		
		def critical_guard
			false
		end
		alias critical_guard? critical_guard



		def ghost_body
			return false
		end
		alias ghost_body? ghost_body


		def weapon_mastery
			@mastery[MST::WEAPON]
		end
		
		def armor_mastery
			@mastery[MST::ARMOR]
		end

		def shield_mastery
			@mastery[MST::SHIELD]
		end

		def spell_mastery
			@mastery[MST::SPELL]
		end

		def get_mastery(type, data_id)
			@mastery[type][data_id] || 0.0
		end
		
		
		def get_weapon_mastery
			return get_mastery(MST::WEAPON, self.weapon.data_id)
		end

		def get_armor_mastery
			return (self.armor ? get_mastery(MST::ARMOR, self.armor.data_id) : 100.0)
		end

		def get_shield_mastery
			return get_mastery(MST::SHIELD, self.shield.data_id)
		end
		
		def get_spell_mastery(spell)
			return get_mastery(MST::SPELL, spell)
		end




		def increase_mastery(type, data_id, inc, max = 100.0)
			@mastery[type][data_id] ||= 0.0
			if @mastery[type][data_id] < max then
				@mastery[type][data_id] += inc
				@mastery[type][data_id] = max if @mastery[type][data_id] > max
			end
		end
		
		def increase_weapon_mastery(inc)
			increase_mastery(MST::WEAPON, self.weapon.data_id, inc, (self.weapon.prefix == IP::Master ? 200.0 : 100.0))
		end
		
		def increase_armor_mastery(inc)
			if self.armor then
				increase_mastery(MST::ARMOR, self.armor.data_id, inc, (self.armor.prefix == IP::Master ? 200.0 : 100.0))
			end
		end


		def increase_shield_mastery(inc)
			increase_mastery(MST::SHIELD, self.shield.data_id, inc, (self.shield.prefix == IP::Master ? 200.0 : 100.0))
		end
		
		def increase_spell_mastery(spell, inc)
			increase_mastery(MST::SPELL, spell, inc)
		end

		
		
		def on_use_weapon(hitting_number)
			# 矢の消費
			if weapon.data.kind_of?(BowModel) then
				arrows[@arrow_index].number -= hitting_number
				if arrows[@arrow_index].number <= 0 then
					lose_item(arrows[@arrow_index])
				end
			end
			
			# 武器の熟練
			increase_weapon_mastery(0.4 + 0.1 * rand(3))
			
			# 「隠れる」効果の解除
			remove_state(HidingState)

		end
		
		def on_use_shield
			super
			increase_shield_mastery(0.6 + 0.1 * rand(3))
		end
				
		def on_use_skill(symbol, difficult_level, rate)
			get_sp = Roll.randomize(SP_NEXT_TABLE[difficult_level + 1] * rate).to_i
			@skill_points[symbol] += get_sp
			return get_sp
		end
		
		def on_round_end
			super
			increase_armor_mastery(0.2 + 0.1 * rand(3))
		end
		
		def skill_levelup?(symbol)
			return @skill_points[symbol] > SP_TABLE[@skill_levels[symbol] + 1]
		end
		
		def skill_levelup(symbol)
			while self.skill_levelup?(symbol) do
				@skill_levels[symbol] += 1
			end
			return nil
		end
		
		def commandable?
			not ai_act? and active?
		end
		
		def set_ai_action
			set_action(pick_ai_action)
		end
		
		def pick_ai_action
			if self.forward? then
				targets = Util.random_pick($troop.front_groups)
				AttackAction.new(self, targets)
			else
				GuardAction.new(self)
			end
		end
		

		
		
		
		
		
		def set_new_parameters(age_symbol)
			raise "obselete method."
		end
		
		# 注：小文字しか受け付けない
		def get_ability_base(type)
			growth = get_ability_growth(type)
			
			Calc.ability_base(growth, @level) + @bonus_points[type] - @senility
		end
		
		

		
		def get_ability_bonus(type)
			method_name = BONUS_METHOD_NAME_TABLE[type]
			bonus = 0
			bonus += self.weapon.send(method_name)
			bonus += self.armor.send(method_name) if self.armor

			self.accessories.compact.each do |accessory|
				bonus += accessory.send(method_name)
			end
			
			self.status.each do |state|
				bonus += state.send(method_name)
			end
			
			
			return bonus.to_i
		end
		
		
		

		
		
		


		

		
		def set_skills
			@skill_levels[SKILL::UNLOCK] = 0
			@skill_points[SKILL::UNLOCK] = 0
			@skill_levels[SKILL::REMOVE_TRAP] = 0
			@skill_points[SKILL::REMOVE_TRAP] = 0
		end
		
		def set_mp(spell_id)
			mastery = get_spell_mastery(spell_id)
			@mp_random_bonus[spell_id] ||= rand(3)
			@mp_max[spell_id] = (psy_base / 2 * (1.0 + mastery / 200) + @mp_random_bonus[spell_id] - 3).to_i
			@mp[spell_id] = @mp_max[spell_id]
		end
		
		def set_all_mp
			DB.spells.each do |spell|
				set_mp(spell.id)
			end
		end
		
		def order
			GS.party.members.index(self)
		end
		
		

		def defense
			re = (@armor_index ? self.armor.defense : 1)
			self.accessories.compact.each do |acc|
				re += acc.defense
			end
			
			re
		end
	
		def weapon
			(@weapon_index ? @items[@weapon_index] : Weapon::HAND)
		end
		
		def armor
			(@armor_index ? @items[@armor_index] : nil)
		end

		def shield
			self.accessories.find{|x| x && x.data.kind_of?(ShieldModel)}
		end
		


		def accessories
			result = [nil, nil]
			result[0] = @items[@accessory_indexes[0]] if @accessory_indexes[0]
			result[1] = @items[@accessory_indexes[1]] if @accessory_indexes[1]

			return result
		end
		
		
		def equipments
			return [weapon, armor, accessories[0], accessories[1]]
		end
		
		def equipped?(item)
			return equipments.include?(item)
		end
		
		def equipped_accessory?(symbol)
			return accessories.find{|x| x && x.data_id == symbol}
		end
		
		# item_indexで指定したアイテムを装備する。成功すればtrue
		# 装備不可だったり指定したアイテムが存在しないならfalse
		def equip(item_index)
			new = @items[item_index]
			return false unless new or new.usable_by?(self)
			
			case new
			when Weapon
				@weapon_index = item_index
			when Armor
				@armor_index = item_index
			when Accessory
				case self.accessories.compact.size
				when 0
					@accessory_indexes[0] = item_index
				when 1
					@accessory_indexes.compact!
					if self.shield && new.data.kind_of?(ShieldModel) then
						@accessory_indexes[0] = item_index
					else
						@accessory_indexes[1] = item_index
					end
				when 2
					if self.shield && self.shield == self.accessories[0] && new.data.kind_of?(ShieldModel) then
						@accessory_indexes[0] = item_index
					else
						@accessory_indexes[1] = item_index
					end
				end
			end
			return true
		end
		
		def unequip_all_items
			@weapon_index = nil
			@armor_index = nil
			@accessory_indexes = [nil, nil]
		end
		
		
		def arrows
			return @items.find_all{|x| x.data.kind_of?(ArrowModel)}
		end
		
		
		def bolts
			return @items.find_all{|x| x.data.kind_of?(BoltModel)}
		end

		
	
		
										 
		
		def masterable?(model)
		end
		
		def damage(value, killing = true)
			super
			PARTY_WINDOW.update if PARTY_WINDOW.maked_surface?
		end
		
		def dead_check(damage)
			if @hp <= 0 then
				hp_rate = (@hp * -1).to_f / self.hp_max.to_f
				damage_rate = damage.to_f / self.hp_max.to_f
				
				hp_penalty = (hp_rate * 100 / 2).to_i
				damage_penalty = (damage_rate * 100).to_i
				suv_bonus = self.surviving_bonus
				
				dead_border = 20 + hp_penalty + damage_penalty - suv_bonus
				dice = CheckDice.new(dead_border.to_i)
				
	
				LOGGER.puts "\t死亡判定:#{dice.success_percentage}% (HP:#{@hp}/#{hp_max} Damage:#{damage} border:20 + #{hp_penalty} + #{damage_penalty} - #{suv_bonus})"
				if dice.roll then
					LOGGER.puts "\t#{name}は倒れた"
					self.dead_sequence
				end
			end
		end
		
		
		def status
			re = @added_status
			equipments.compact.each do |eq|
				re += eq.constant_status
			end
			
			re
		end

		# Return: success?
		def revive
			if super then
				@soul = false
				@profiles[MPT::Revived] += 1
				return true
			else
				return false
			end
		end
		
		def protection_check(damage_types)
			equipments.compact.each do |eq|
				if eq.protection_damage_types.find{|x| damage_types.include?(x)} and
				rand(100) < eq.protection_rate then
					return eq.name
				end
			end
			
			return nil
		end
		
		def dead_sequence
			super
			GS.party.members.delete(self)
			GS.party.members.push(self)
		end
		
		def rest_exp_to_next
			if level_max? then
				nil
			else
				EXP_TABLE[@level+1] - @exp
			end
		end
		
		def get_exp(exp)
			unless level_max? then
				@exp += exp
				@last_got_exp = exp
			end
		end
		
		def lose_exp_rate(rate)
			@exp -= (rest_exp_to_next * rate).to_i
			@exp = 0 if @exp < 0
		end
		
		def lose_exp_rate_on_total(rate)
			@exp -= (@exp * rate).to_i
			@exp = 0 if @exp < 0
		end

		
		def level_max?
			@level >= LEVEL_MAX
		end
		
		def levelup?
			not level_max? and @exp >= EXP_TABLE[@level+1]
		end
		
		def levelup
			old = self.dup
			old.mp_max = self.mp_max.dup
			
			@level += 1
			
			grow_hp
			set_all_mp
			
			# 新しい術の習得
			founds = @spells.find_all{|x| (@spell_learning[x] || 0) + 1 < SPELL_DATA[x].trick_ids.size}
			
			unless founds.empty? then
				spell_id = Util.random_pick(founds)
				@spell_learning[spell_id] ||= 0
				@spell_learning[spell_id] += 1
				#got_trick_id = SPELL_DATA[spell].trick_ids[@spell_learning[spell] - 1]
			end
			
			return {:spell_id => spell_id}
		end
		
		
		def grow_hp(randomize = true)
			# 成長基本値
			hp_inc = (hp_max_growth * 2)
			
			# レベル補正
			#hp_inc += (@level * 0.1).to_i
			
			# ランダム補正
			if randomize then
				hp_inc += (rand(3) + rand(3))
			else
				hp_inc += 2
			end
			
			# 継承ボーナス（最大4回）
			inh = [inherited, 4].min
			if inh >= 1 then
				hp_inc += (2 + inh * 2)
			end
			
			@hp_max_base += hp_inc
			@hp += hp_inc
		end
		
		
		def get_collect_age(current_week = GS.week)
			return ((current_week - @birth_week) / WEEK_PER_YEAR).to_i
		end
		
		def aging?
			(@age != self.get_collect_age) and not life_end?
		end
		
		def get_gold(got, visible_and_recorded = true)
			@gold += got
			@gold = GOLD_MAX if @gold > GOLD_MAX
			if visible_and_recorded then
				@last_got_gold = got
				GS.player_profiles[PPT::GotGold] += got
				Game.increment_player_record(PRT::GotGold, PPT::GotGold, got)
			end
		end
		
		def lose_gold(lost)
			@gold -= lost
			@gold = 0 if @gold < 0
		end
		
		def get_message(scene)
			if @messages.include?(scene) then
				return @messages[scene][rand(@messages[scene].size)]
			else
				return nil
			end
		end
		
		def getable_item?(item)
			return false if @items.size >= ITEM_MAX
			return true
		end
		
		def find_item(item_symbol)
			items.find{|x| x.data_id == item_symbol}
		end
		
		
		
		def get_item(item)
			return false unless getable_item?(item)
			Game.on_found('item', item.data_id)
			@items << item
			self.bundle
			return true
		end
		
		def bundle
			through_flag = false
		
			until through_flag do
				through_flag = catch(:bundle_check) do
					@items.each_index do |i|
						if @items[i].kind_of?(BundleItem) then
							@items.each_index do |j|
								next if i == j
								if @items[i].data_id == @items[j].data_id then
									@items[i].number += @items[j].number
									self.lose_item(j)
									# リトライ
									throw(:bundle_check, false)
								end
							end
						end
					end
					
					# チェック通過
					throw(:bundle_check, true)
				end
			end
			
			# アイテム最大数チェック
			@items.each do |item|
				if item.kind_of?(BundleItem) && (item.number > item.number_max) then
					@items << BundleItem.new(item.data_id, item.number - item.number_max)
					item.number = item.number_max
				end
			end
		end
		
		# 所持アイテムのバンドルと並び替えを行う
		def sort_items
			bundle
			
			item_transaction do
				@items = @items.sort_by{|x| DB.items.index(x.data)}
			end
		end
		
		# アイテム一つを消失させる
		# 消失対象はItemインスタンス、ID、所持アイテム番号のどれかで指定
		def lose_item(target)
			lost = nil
			
			item_transaction{
				case target
				when String, Symbol
					found = find_item(target)
					lost = @items.delete(found) if found
				when Fixnum
					lost = @items.slice!(target)
				when Item
					lost = @items.delete(target)
				end
			}
			
			return lost
		end
		
		# ブロックを実行した後、装備状態を復元する
		# （アイテムの並び替えなどを行うときに使う）
		def item_transaction
			weapon_object_id = (self.weapon && self.weapon.object_id)
			armor_object_id = (self.armor && self.armor.object_id)
			accessory0_object_id = (self.accessories[0] && self.accessories[0].object_id)
			accessory1_object_id = (self.accessories[1] && self.accessories[1].object_id)

			yield

			object_ids = @items.map{|x| x.object_id}
			@weapon_index = object_ids.index(weapon_object_id)
			@armor_index = object_ids.index(armor_object_id)
			@accessory_indexes[0] = object_ids.index(accessory0_object_id)
			@accessory_indexes[1] = object_ids.index(accessory1_object_id)

		end
		
		def analysis_vision_usable?
			alive? and spell_caster? and @spells.include?('Vision') and
			@spell_learning['Vision'] and @spell_learning['Vision'] >= 1 and
			usable_mp?(TRICK_DATA['AnalysisVision'].mp_cost, 'Vision')
			
		end
		
		
		
		

		def dump
			hash = super
			
			(STORED_SIMPLE_VARIABLE_NAMES + STORED_HASH_VARIABLE_NAMES).each do |name|
				hash[name] = instance_variable_get("@#{name}")
			end
			
			
			return hash
		end
		
		
		
		def self.create_from_mpac_model(hash)
			obj = super(hash)
			STORED_SIMPLE_VARIABLE_NAMES.each do |name|
				eval("obj.#{name} = hash[name]")
			end
			STORED_HASH_VARIABLE_NAMES.each do |name|
				new_data = obj.send(name)
				(hash[name] || {}).each_pair do |key, value|
					new_data[key] = value
				end
				eval("obj.#{name.to_s} = new_data")
			end
			
			
			hash['mastery'].each_pair do |type, data|
				hash['mastery'][type].each_pair do |kind, value|
					obj.mastery[type][kind] = value
				end
			end

			return obj
		end

	end
	
	
	
	class Fighter < Member
		
		
		def class_caption
			return s_('ClassName|軽戦士')
		end
		
		def class_initial
			return s_('ClassInitial|軽')
		end

		
		def spell_caster?
			return false
		end
		
		
	end
	
	class Grappler < Member
		
		
		def class_caption
			return s_('ClassName|格闘家')
		end
		
		def class_initial
			return s_('ClassInitial|格')
		end
		
		def spell_caster?
			return false
		end
		
		def weapon_power
			atk = super
			atk *= (1.00 + 0.10 + @level * 0.05)
			atk = atk.to_i
			
			return atk
		end
		
		def defense
			d = super
			d += (1 + @level * 0.25).to_i
			
			return d
		end

	end

	
	class HeavyFighter < Member
	
	
	
		def class_caption
			return s_('ClassName|重戦士')
		end
		
		def class_initial
			return s_('ClassInitial|重')
		end
		
		def spell_caster?
			return false
		end
	
	end
	
	class Archer < Member
		attr_accessor :targeting
	
	
		def class_caption
			return s_('ClassName|弓闘士')
		end
		
		def class_initial
			return s_('ClassInitial|弓')
		end
		
		def spell_caster?
			return false
		end
		
		
		def arrow
			if weapon.data.kind_of?(BowModel) then
				arrows[arrow_index]
			else
				nil
			end
		end

		
	
		
	
	end
	
	
	# 勉強によって人ならぬ力、魔術を扱う能力を手に入れた者
	class Magician < Member
	
	
		def class_caption
			return s_('ClassName|術師')
		end
	
	
		def class_initial
			return s_('ClassInitial|術')
		end
		
		
	
		
		def spell_caster?
			return true
		end
		
		def spell_max
			if @level < 5 then 2
			elsif @level < 10 then 3
			elsif @level < 15 then 4
			elsif @level < 20 then 5
			else 6
			end
		end
		
	
	end
	
	class Explorer < Member
		
		def set_skills
			super
			@skill_levels[SKILL::UNLOCK] = 5
			@skill_points[SKILL::UNLOCK] = SP_TABLE[5]
			@skill_levels[SKILL::REMOVE_TRAP] = 5
			@skill_points[SKILL::REMOVE_TRAP] = SP_TABLE[5]
		end

		def on_use_skill(symbol, difficult_level, rate)
			super(symbol, difficult_level, rate * 2.0)
		end

		
		def class_caption
			return s_('ClassName|探検家')
		end
	
	
		def class_initial
			return s_('ClassInitial|探')
		end
	
		
		def spell_caster?
			return false
		end
		
	
	end
	
	class Prayer < Member
		
		def class_caption
			return s_('ClassName|僧')
		end
		
		def class_initial
			return s_('ClassInitial|僧')
		end

		def spell_caster?
			return false
		end
		

	end
	
	
	class Academian < Member
		
		def class_caption
			s_('ClassName|学者')
		end
		
		def class_initial
			s_('ClassInitial|学')
		end

		def spell_caster?
			true
		end
		
		def spell_max
			if @level < 5 then 2
			elsif @level < 10 then 3
			elsif @level < 15 then 4
			elsif @level < 20 then 5
			else 6
			end
		end
		

	end

	
	
	
	
	class Assasin < Member
		
		def class_caption
			return "暗殺者"
		end
	
	
		def class_initial
			return "暗"
		end
	
		
		def spell_caster?
			return true
		end
	end
	
	
	class Samurai < Member
		
		def class_caption
			return "侍"
		end
	
	
		def class_initial
			return "侍"
		end
	
		
		def spell_caster?
			return true
		end
	end
	
	
	class Wizard < Member
		def class_caption
			return "魔道師"
		end
	
	
		def class_initial
			return "魔"
		end
	
		
		def spell_caster?
			return true
		end
	
	end
	
	
	class Dancer < Member
		def class_caption
			return "踊り子"
		end
	
	
		def class_initial
			return "踊"
		end
	
		
		def spell_caster?
			return false
		end
	
	end
	

	
	
	# 成長率
	class Fighter
		GLOWTH = {
				STR => GLOW_A, TEC => GLOW_A, AGL => GLOW_A,
				WIL => GLOW_C, PSY => GLOW_D
			}
	
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_B; end
	end
	
	class Grappler
		GLOWTH = {
				STR => GLOW_S, TEC => GLOW_B, AGL => GLOW_S,
				WIL => GLOW_C, PSY => GLOW_C
			}
	
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_S; end
	end

	
	class HeavyFighter
		GLOWTH = {
				STR => GLOW_S, TEC => GLOW_C, AGL => GLOW_C,
				WIL => GLOW_D, PSY => GLOW_D
		}
		
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_A; end

	end
	
	class Archer
		GLOWTH = {
				STR => GLOW_A, TEC => GLOW_A, AGL => GLOW_D,
				WIL => GLOW_S, PSY => GLOW_C
			}
		
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_C; end
	end
	
	class Explorer
		GLOWTH = {
				STR => GLOW_B, TEC => GLOW_S, AGL => GLOW_B,
				WIL => GLOW_C, PSY => GLOW_B
			}
		
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_B; end
	end
	
	class Magician
		GLOWTH = {
				STR => GLOW_C, TEC => GLOW_C, AGL => GLOW_C,
				WIL => GLOW_B, PSY => GLOW_S
			}
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_D; end
	end
	
	
	
	class Academian
		GLOWTH = {
				STR => GLOW_E, TEC => GLOW_E, AGL => GLOW_D,
				WIL => GLOW_S, PSY => GLOW_C
			}
		
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_E; end

	end
	
	class Prayer
		GLOWTH = {
				STR => GLOW_B, TEC => GLOW_B, AGL => GLOW_C,
				WIL => GLOW_B, PSY => GLOW_A
			}
		
		def get_ability_growth(type); GLOWTH[type]; end
		def hp_max_growth; GLOW_B; end

	end


	
	
	
	# 初期装備
	class Fighter
		def initialize(*args)
			super
			set_starting_equipments('NoviceSword', 'NoviceArmor', 'NoviceGuard')
		end

	end
	
	class HeavyFighter
		def initialize(*args)
			super
			set_starting_equipments('NoviceSword', 'NoviceArmor', 'NoviceGuard')
		end

	end

	class Archer
		def initialize(*args)
			super
			set_starting_equipments('NoviceBow', 'NoviceArmor')
			get_item(BundleItem.new('WoodArrow', 50))
		end

	end
	
	class Magician
		def initialize(*args)
			super
			set_starting_equipments('NoviceClub', 'NoviceCloak')
		end

	end
	
	class Academian
		def initialize(*args)
			super
			set_starting_equipments('NoviceClub', 'NoviceCloak')
		end

	end


	class Explorer
		def initialize(*args)
			super
			set_starting_equipments('NoviceClub', 'NoviceCloak')
		end

	end
	class Prayer
		def initialize(*args)
			super
			set_starting_equipments('NoviceClub', 'NoviceCloak')
		end

	end
	
end
	
	
	
	
