#include "xml_data_tree.h"

#include <iostream>
#include <string>
#include <vector>
#include <sstream>

#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/framework/MemBufInputSource.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>
#include <xercesc/sax/HandlerBase.hpp>

#include <xercesc/dom/DOM.hpp>
#include <xercesc/dom/DOMNode.hpp>

#include "local_encoded_string.h"
#include "universal_name.h"
#include "path_converter.h"
#include "compat_stringstream.h"

#define ADD_MULTI_ELEMENT_INDEX


using namespace xercesc;

namespace fuse_xml {

class MyErrorHandler : public ErrorHandler
{
public:
    bool handleError( const DOMError & error )
    {
        std::cerr << Local_Encoded_String( error.getLocation() -> getURI() )
                  << ": " << error.getLocation() -> getLineNumber()
                  << ": " << error.getLocation() -> getColumnNumber()
                  << ": " << Local_Encoded_String( error.getMessage() )
                  << std::endl;

        return error.getSeverity() != DOMError::DOM_SEVERITY_FATAL_ERROR;
    };

protected:
    void print( const std::string & type, const SAXParseException & e ) const
    {
        std::cerr << type << ": "
                  << "line " << e.getLineNumber()
                  << ", column " << e.getColumnNumber()
                  << ": " << Local_Encoded_String( e.getMessage() )
                  << std::endl;
    }

    virtual void warning( const SAXParseException & e )
    {
        this -> print( "warning", e );
    }

    virtual void error( const SAXParseException & e )
    {
        this -> print( "error", e );
    }

    virtual void fatalError( const SAXParseException & e )
    {
        this -> print( "fatal error", e );
    }

    virtual void resetErrors()
    {
    }
};


//
// XMLDataNode
//
XMLDataNode::DataType XMLDataNode::getType() const
{
    return this -> type;
}

bool XMLDataNode::isElement() const
{
    return this -> type == ELEMENT;
}
bool XMLDataNode::isText() const
{
    return this -> type == TEXT;
}


//
// XMLDataTree
//

std::string PathElement::toPathString() const
{
#ifdef ADD_MULTI_ELEMENT_INDEX
    if ( this -> nth == -1 )
    {
        return this -> name.getShortFQN();
    }
    else
    {
        compat_stringstream buf;
        buf << "[" << this -> nth << "]";

        return this -> name.getShortFQN() + buf.str();
    }
#else
    return this -> name.getShortFQN();
#endif
}

// XXX: should make a class
static
std::string convert_path_to_string( const std::vector<PathElement> & path )
{
    std::string path_string;

    for ( size_t i = 0 ; i < path.size() ; ++ i )
    {
        path_string += "/";
        path_string += Path_Converter::escape_slash( path[i].toPathString() );
    }

    return path_string;
}


static void traverse_node( const DOMNode * node,
                           const std::vector<PathElement> & path,
                           std::map<std::string, ref_count_ptr<XMLDataNode> > * files,
                           int nth );


XMLDataTree * XMLDataTree::parse( const char * path )
{
    //
    // initialize
    //
    try
    {
        XMLPlatformUtils::Initialize();
    }
    catch( const XMLException & e )
    {
        std::cerr << Local_Encoded_String( e.getMessage() ) << std::endl;

        return static_cast<XMLDataTree *>( 0 );
    }


    //
    // Make Parser
    //
    XercesDOMParser * parser = new XercesDOMParser;
    ErrorHandler * error_handler = new MyErrorHandler;
    parser -> setErrorHandler( error_handler );

//  parser -> setValidationScheme( XercesDOMParser::Val_Always );
    parser -> setDoNamespaces( true );
    parser -> setIncludeIgnorableWhitespace( false );
    parser -> setDoSchema( false );
//  parser -> setCreateEntityReferenceNodes( false );

    XMLDataTree * tree = new XMLDataTree();
    
    try
    {
        LocalFileInputSource buf( XML_Parser_Encoded_String( path ) );

        parser -> parse( path );

        int n_errors = parser -> getErrorCount();

        if ( n_errors != 0 )
        {
            std::cerr << "parse failed" << std::endl;

            return static_cast<XMLDataTree *>( 0 );
        }

        DOMDocument * doc = parser -> getDocument();
        DOMElement * top_level_element = doc -> getDocumentElement();

        std::vector<PathElement> child_elements;
        child_elements.push_back( PathElement
                                  ( Universal_Name
                                    ( Local_Encoded_String
                                      ( top_level_element -> getNamespaceURI() ),
                                      Local_Encoded_String
                                      ( top_level_element -> getLocalName() ) ),
                                    -1 ) );

        tree -> files["/"] = new XMLDataNode( child_elements );

        traverse_node( top_level_element, std::vector<PathElement>(),
                       &(tree -> files), -1 );
    }
    catch( const XMLException &  e )
    {
        std::cerr << Local_Encoded_String( e.getMessage() ) << std::endl;

        return static_cast<XMLDataTree *>( 0 );
    }
    catch( const DOMException &  e )
    {
        std::cerr << Local_Encoded_String( e.getMessage() ) << std::endl;

        return static_cast<XMLDataTree *>( 0 );
    }
    catch( std::exception & e )
    {
        std::cerr << "exception: " << e.what() << std::endl;
    }
    catch(...)
    {
        std::cerr << "exception" << std::endl;
    }


    //
    // Delete Parser
    //
    delete parser;
    delete error_handler;

    //
    // finalize
    //
    XMLPlatformUtils::Terminate();

    return tree;
}

void traverse_node( const DOMNode * node,
                    const std::vector<PathElement> & path,
                    std::map<std::string, ref_count_ptr<XMLDataNode> > * files,
                    int nth )
{
    if ( node == static_cast<const DOMNode *>(0) )
    {
        return;
    }

    if ( node -> getNodeType() == node -> ELEMENT_NODE )
    {
        Local_Encoded_String local_name( node -> getLocalName() );
        Local_Encoded_String namespace_uri( node -> getNamespaceURI() );

        std::vector<PathElement> this_path = path;

        this_path.push_back( PathElement
                             ( Universal_Name( std::string( namespace_uri ),
                                               std::string( local_name ) ),
                               nth ) );

        const std::string path_string = convert_path_to_string( this_path );

#if 0
        std::cerr << "path_string = [" << path_string << "]" << std::endl;
#endif

        const DOMNodeList * child_nodes = node -> getChildNodes();

        //
        // check child tags
        //
        std::map<Universal_Name, int> n_tags;
        std::map<Universal_Name, int> tag_count;

        for ( size_t i = 0 ; i < child_nodes->getLength() ; ++ i )
        {
            const DOMNode * c = child_nodes -> item( i );

#if 1
            if ( c -> getNodeType() != node -> ELEMENT_NODE )
            {
                continue;
            }
#endif

            const Universal_Name child_name( Local_Encoded_String
                                             ( c -> getNamespaceURI() ),
                                             Local_Encoded_String
                                             ( c -> getLocalName() ) );

            std::map<Universal_Name, int>::iterator
                it = n_tags.find( child_name );

            if ( it == n_tags.end() )
            {
                n_tags[ child_name ] = 1;
                tag_count[ child_name ] = 0;
            }
            else
            {
                (*it).second ++;
            }
        }

#if 0
        // debug print
        for ( std::map<Universal_Name, int>::const_iterator
                  it = n_tags.begin() ; it != n_tags.end() ; ++ it )
        {
            std::cerr << (*it).first.getFQN() << ": " << (*it).second
                      << std::endl;
        }
#endif

        std::vector<PathElement> child_elements;
        for ( size_t i = 0 ; i < child_nodes->getLength() ; ++ i )
        {
            const DOMNode * c = child_nodes -> item( i );

            int child_nth = -1;

            if ( c -> getNodeType() == c -> ELEMENT_NODE )
            {
                const Universal_Name child_name( Local_Encoded_String
                                                 ( c -> getNamespaceURI() ),
                                                 Local_Encoded_String
                                                 ( c -> getLocalName() ) );

                std::map<Universal_Name, int>::iterator
                    it = n_tags.find( child_name );

                if ( it != n_tags.end()
                     && (*it).second >= 2 )
                {
                    child_nth = tag_count[ child_name ];
                    tag_count[ child_name ] ++;
                }

                child_elements.push_back( PathElement( child_name, child_nth ) );
            }

            traverse_node( c, this_path, files, child_nth );
        }

        (*files)[ path_string ] = new XMLDataNode( child_elements );

#if 0
        std::cerr << "!!!" << "path = [" << path_string << "]:"
                  << " dir" << std::endl;
#endif


        // XXX: should be more smart
        child_elements.clear();
        for ( size_t i = 0 ; i < child_nodes-> getLength() ; ++ i )
        {
            const DOMNode * c = child_nodes -> item( i );

            int child_nth = -1;

            if ( c -> getNodeType() == c -> ELEMENT_NODE )
            {
                const Universal_Name child_name( Local_Encoded_String
                                                 ( c -> getNamespaceURI() ),
                                                 Local_Encoded_String
                                                 ( c -> getLocalName() ) );

                std::map<Universal_Name, int>::iterator
                    it = n_tags.find( child_name );

                if ( it != n_tags.end()
                     && (*it).second >= 2 )
                {
                    child_nth = tag_count[ child_name ];
                    tag_count[ child_name ] ++;
                }

                child_elements.push_back( PathElement( child_name, child_nth ) );
            }

            traverse_node( c, this_path, files, child_nth );
        }

#if 0
        for ( size_t i = 0 ; i < child_nodes->getLength() ; ++ i )
        {
            const DOMNode * c = child_nodes -> item( i );
            int child_nth = -1;

            if ( c -> getNodeType() == c -> ELEMENT_NODE )
            {
                const Universal_Name child_name( Local_Encoded_String
                                                 ( c -> getNamespaceURI() ),
                                                 Local_Encoded_String
                                                 ( c -> getLocalName() ) );

                std::map<Universal_Name, int>::const_iterator
                    it = n_tags.find( child_name );

                if ( it != n_tags.end() )
                {
                    // XXX: not implemented yet
                    child_nth = (*it).second;
                }
            }

            traverse_node( c, this_path, files, child_nth );
        }
#endif
    }
    else if ( node -> getNodeType() == node -> TEXT_NODE )
    {
        const std::string value = Local_Encoded_String( node -> getTextContent() );
#if 0
        Local_Encoded_String node_name( node -> getNodeName() );
        std::cerr << "text node [" << node_name << "] = "
                  << "[" << value << "]" << std::endl;
#endif

        bool whitespace_only = true;
        for ( size_t i = 0 ; i < value.length() ; ++ i )
        {
            if ( value[i] != ' '
                 && value[i] != '\t'
                 && value[i] != '\r'
                 && value[i] != '\n' )
            {
                whitespace_only = false;
                break;
            }
        }

        if ( whitespace_only )
        {
            return;
        }

        std::string path_string = convert_path_to_string( path );

#if 0
        std::cerr << "@@@" << "path = [" << path_string << "] = [" << value << "]"
                  << std::endl;
#endif

        (*files)[ path_string ] = new XMLDataNode( value );

#if 0
        std::cerr << "!!!" << "path = [" << path_string << "]:"
                  << " file content = [" << value << "]" << std::endl;
#endif
    }
}


} // end of namespace fuse_xml
