﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using DataUnit.Core;
using System.IO;
using log4net;
using System.Reflection;

namespace DataUnit.Windows
{
    public partial class frmSql : Form
    {

        private bool _isWaiting = false;
        private bool _isDataExits = false;
        private Config _config = null;
        private DateTime _nextExecTime;
        private DateTime _StartTime;

        private static readonly ILog _logger
           = LogManager.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

        private frmSql()
        {
            InitializeComponent();
        }

        public frmSql(Config config):this()
        {
            _config = config;
        }

        private void frmSql_Load(object sender, EventArgs e)
        {
            cmbUnit.SelectedIndex = 0;
            this.txtDirSql.Text = _config.SqlPath;
            this.txtDirResult.Text = _config.OutputPath;
            btnExpResult.Enabled = Directory.Exists(this.txtDirResult.Text);
        }

        private void btnWait_Click(object sender, EventArgs e)
        {
            ChangeControlStatus();
        }

        private void SetNextExecTime()
        {
            long nanoSecond = 0;
            switch (this.cmbUnit.SelectedIndex)
            {
                case 0:
                    nanoSecond = Convert.ToInt64(nudSapn.Value) * 60 * 1000 * 10000;
                    break;

                case 1:
                    nanoSecond = Convert.ToInt64(nudSapn.Value) * 60 * 60 * 1000 * 10000;
                    break;

                case 2:
                    nanoSecond = Convert.ToInt64(nudSapn.Value) * 24 * 60 * 60 * 1000 * 10000;
                    break;

                default:
                    Debug.Assert(false);
                    break;
            }

            TimeSpan ts = new TimeSpan(nanoSecond);
            _nextExecTime = GetNowDateTime().Add(ts);
            lblNext.Text = String.Format("次回実行予定時刻：{0:yyyy/MM/dd HH:mm}", _nextExecTime);
            lblNext.Visible = true;
        }

        private void ChangeControlStatus()
        {
            _isWaiting = !_isWaiting;
            if (_isWaiting)
            {
                SetNextExecTime();
                btnWait.Text = "　タイマー解除";
                btnWait.ImageIndex = 1;
                rtxResult.Text = String.Format("{0:yyyy/MM/dd HH:mm:ss} 監視を開始します。\r\n", DateTime.Now);
                this.Text = "DataUnit";
                pgbSql.ForeColor = SystemColors.Highlight;
                btnExpSql.Enabled = Directory.Exists(this.txtDirResult.Text);
                btnExpResult.Enabled = Directory.Exists(this.txtDirResult.Text);
            }
            else
            {
                lblNext.Visible = false;
                btnWait.Text = "　タイマー実行";
                btnWait.ImageIndex = 0;
                btnExpSql.Enabled = Directory.Exists(this.txtDirResult.Text);
                btnExpResult.Enabled = Directory.Exists(this.txtDirResult.Text);
            }

            pgbSql.Value = 0;
            this.btnSQL.Enabled = !_isWaiting;
            this.btnClose.Enabled = !_isWaiting;
            this.timExec.Enabled = _isWaiting;
            this.cmbUnit.Enabled = !_isWaiting;
            this.nudSapn.Enabled = !_isWaiting;
        }

        private void timExec_Tick(object sender, EventArgs e)
        {
            
            if (_nextExecTime <= GetNowDateTime())
            {
                timExec.Enabled = false;
                ExecSQL();
                SetNextExecTime();
                timExec.Enabled = true;
            }
            
        }

        private void ExecSQL()
        {
            try
            {
                _StartTime = DateTime.Now;
                _isDataExits = false;
                Host host = new Host(_config);
                host.Status += new Host.StatusEventHandler(host_Status);
                host.CheckData();
                if (_isDataExits)
                {
                    pgbSql.ForeColor = Color.Yellow;
                }
                else
                {
                    pgbSql.ForeColor = Color.GreenYellow;
                }
            }
            catch (Exception ex)
            {
                _logger.Error("エラーが発生しました。", ex);
                string msg = String.Format("\r\nエラーが発生しました。\r\n{0}\r\n{1}\r\n", ex.Message,ex.ToString());
                pgbSql.ForeColor = Color.Red;
                rtxResult.Text += msg;
                ScrollRichText();
            }
            finally
            {
                this.Text += GetFinishedTitle();
            }
        }

        private void host_Status(object sender, StatusEventArgs e)
        {
            pgbSql.Maximum = e.AllCount;
            pgbSql.Value = e.CurrentCount;

            if (e.Kind == StatusKind.ExistData)
            {
                _isDataExits = true;
            }

            this.Text = GetExecTitle(e.CurrentCount, e.AllCount);
            string msg = String.Format("{0:yyyy/MM/dd HH:mm:ss} {1}\r\n", DateTime.Now,e.Message);
            if (e.Kind == StatusKind.Start)
            {
                msg += String.Format("{0:yyyy/MM/dd HH:mm:ss} 全{1}件のSQLを実行します。\r\n", DateTime.Now, e.AllCount.ToString());
            }
            rtxResult.Text += msg;
            ScrollRichText();
            Application.DoEvents();
        }

        private DateTime GetNowDateTime()
        {
            DateTime now = DateTime.Now;
            return new DateTime(now.Year, now.Month, now.Day, 
                                now.Hour, now.Minute, 0);
        }

        private void frmTimer_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (!btnClose.Enabled)
            {
                const string MSG = "タイマー実行中です。\r\nタイマーを解除してから画面を閉じてください。\r\n";
                rtxResult.Text += MSG;
                ScrollRichText();
                e.Cancel = true;
            }
        }

        private void ScrollRichText()
        {
            rtxResult.SelectionStart = rtxResult.Text.Length;
            rtxResult.Focus();
            rtxResult.ScrollToCaret();
        }

        private void btnClose_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void btnSQL_Click(object sender, EventArgs e)
        {
            this.Cursor = Cursors.WaitCursor;
            rtxResult.Text = "";
            btnClose.Enabled = false;
            btnWait.Enabled = false;
            btnSQL.Enabled = false;
            btnExpSql.Enabled = false;
            btnExpResult.Enabled = false;

            ExecSQL();

            btnClose.Enabled = true;
            btnWait.Enabled = true;
            btnSQL.Enabled = true;
            this.Cursor = Cursors.Default;
            btnExpSql.Enabled = Directory.Exists(this.txtDirResult.Text);
            btnExpResult.Enabled = Directory.Exists(this.txtDirResult.Text);

        }

        private string GetExecTitle(int count, int allCount)
        {
            string title = "DataUnit ({0}/{1}) {2}";
            return String.Format(title,
                                count.ToString(),
                                allCount.ToString(),
                                _StartTime.ToString("HH:mm:ss"));
            
        }

        private string GetFinishedTitle()
        {
            DateTime endTime = DateTime.Now;
            TimeSpan ts = endTime - _StartTime;
            string span = "";
            if (ts.Hours > 0)
            {
                span = ts.Hours.ToString("d") + "時間";
            }
            if (ts.Minutes > 0)
            {
                span = span + ts.Minutes.ToString("d") + "分";
            }
            span = span + ts.Seconds.ToString("d") + "秒";
            return "→" + endTime.ToString("HH:mm:ss") + " [" + span + "]";

        }

        private void btnExpSql_Click(object sender, EventArgs e)
        {
            UICommon.ShowExplorer(txtDirSql.Text);
        }

        private void btnExpResult_Click(object sender, EventArgs e)
        {
            UICommon.ShowExplorer(txtDirResult.Text);
        }
    }
}
