/**
 * Copyright (c) 2004, yher2 project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.junit.db.data;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Date;

import net.yher2.commons.lang.NumberUtils;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;

/**
 * @auther  hisaboh
 */
public abstract class ColumnData {
	public static ColumnData getInstance(Object value) {
		if (value == null) return new EmptyColumn();
		else if (value instanceof Integer) return new IntegerColumn((Integer)value);
		else if (value instanceof Double) return new NumberColumn((Double)value);
		else if (value instanceof Date) return new DateColumn((Date)value);
		else if (value instanceof String) return new StringColumn((String)value);
		else if (value instanceof Boolean) return new BooleanColumn((Boolean)value);
		else if (value instanceof HSSFCell) return parse((HSSFCell)value);
		else throw new IllegalArgumentException("Illegal arguments="+value);
	}
	private static ColumnData parse(HSSFCell cell) {
		if (cell == null) {
			return new EmptyColumn();
		}
		switch (cell.getCellType()) {
		case HSSFCell.CELL_TYPE_BLANK:
			return new EmptyColumn();
		case HSSFCell.CELL_TYPE_BOOLEAN:
			return new BooleanColumn(cell.getBooleanCellValue());
		case HSSFCell.CELL_TYPE_NUMERIC:
			if (HSSFDateUtil.isCellDateFormatted(cell)) {
				return new DateColumn(cell.getDateCellValue());
			} else {
				if (NumberUtils.isInteger(cell.getNumericCellValue())) {
					return new IntegerColumn((int)cell.getNumericCellValue());
				} else {
					return new NumberColumn(cell.getNumericCellValue());
				}
			}
		case HSSFCell.CELL_TYPE_STRING:
			return new StringColumn(cell.getStringCellValue().trim());
		case HSSFCell.CELL_TYPE_ERROR:
		case HSSFCell.CELL_TYPE_FORMULA:
		default:
			throw new IllegalStateException("There is illegal column at line.");
		}
	}
	
	public abstract Object getValue();

	/**
	 * @param i
	 * @param ps
	 */
	protected abstract void set(int index, PreparedStatement ps) throws SQLException;
}
