package org.dbunitng.listeners.internal;

import static org.dbunitng.annotations.DatabaseOperationType.*;

import java.sql.SQLException;

import javax.sql.DataSource;

import org.dbunit.DataSourceDatabaseTester;
import org.dbunit.IDatabaseTester;
import org.dbunit.JdbcDatabaseTester;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.IDataSet;
import org.dbunit.operation.DatabaseOperation;
import org.dbunitng.annotations.DatabaseOperationType;
import org.dbunitng.exception.DbUnitNGRuntimeException;

/**
 * データベースへの操作を表す。
 * 
 * @author jyukutyo
 * 
 */
public class DbUnitNGDatabaseOperation {

	/** DbUnitのtester */
	private IDatabaseTester tester;

	/**
	 * コンストラクタ.
	 * 
	 * @param config
	 *            データベース接続設定
	 */
	public DbUnitNGDatabaseOperation(DbUnitNGConfig config) {
		tester =
			new JdbcDatabaseTester(config.getDriver(), config.getUrl(), config
				.getUserName(), config.getPassword());
	}

	/**
	 * コンストラクタ.
	 * 
	 * @param dataSource
	 *            データソース
	 */
	public DbUnitNGDatabaseOperation(DataSource dataSource) {
		tester = new DataSourceDatabaseTester(dataSource);
	}

	/**
	 * データベースへの操作を実行する.
	 * 
	 * @param type
	 *            データベース操作タイプ
	 * @param dataSet
	 *            DbUnitのデータセット
	 */
	public void execute(DatabaseOperationType type, IDataSet dataSet) {
		if (NONE == type) {
			return;
		}

		IDatabaseConnection connection = null;
		try {
			connection = tester.getConnection();
			DatabaseOperation operation;
			if (CLEAN_INSERT == type) {
				operation = DatabaseOperation.CLEAN_INSERT;
			} else if (INSERT == type) {
				operation = DatabaseOperation.INSERT;
			} else if (UPDATE == type) {
				operation = DatabaseOperation.UPDATE;
			} else if (REFRESH == type) {
				operation = DatabaseOperation.REFRESH;
			} else if (DELETE == type) {
				operation = DatabaseOperation.DELETE;
			} else if (DELETE_ALL == type) {
				operation = DatabaseOperation.DELETE_ALL;
			} else if (TRUNCATE_TABLE == type) {
				operation = DatabaseOperation.TRUNCATE_TABLE;
			} else {
				throw new DbUnitNGRuntimeException(
					"DatabaseOperationType is not undifined. " + type);
			}

			operation.execute(connection, dataSet);

		} catch (Exception e) {
			throw new DbUnitNGRuntimeException(
				"DatabaseOperation can not be executed.",
				e);
		} finally {
			if (connection != null) {
				try {
					connection.close();
				} catch (SQLException e) {
					// empty
				}
			}
		}
	}
}
