/*
 * Copyright (C) 2006 Sony Computer Entertainment Inc.
 * Av backend support for PS3PF 
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/kernel.h>
#include <linux/ioctl.h>
#include <asm/lv1call.h>
#include <asm/atomic.h>
#include <asm/ps3av.h>

//#define PS3AV_DEBUG
#ifdef PS3AV_DEBUG
#define DPRINTK(fmt, args...) \
	do { printk("ps3av " fmt, ## args); } while (0)
#else
#define DPRINTK(fmt, args...) do { } while (0)
#endif

#ifdef CONFIG_FB_PS3
/* drivers/video/ps3fb.c */
extern int ps3fb_set_resolution(uint32_t, uint32_t);
extern int ps3fb_get_res(uint32_t, uint32_t, uint32_t *, uint32_t *);
#endif
extern int ps3pf_boot_data_get_video_mode(void);

#define BUFSIZE          4096 /* vuart buf size */
#define PS3AV_BUF_SIZE   512  /* max packet size */


static int timeout = 5000; // in msec ( 5 sec )
module_param(timeout, int, 0644);

struct ps3av ps3av;
static atomic_t ps3av_open_count = ATOMIC_INIT(0);

/* color space */
#define YUV444 PS3AV_CMD_VIDEO_CS_YUV444_8
#define RGB8   PS3AV_CMD_VIDEO_CS_RGB_8
/* format */
#define XRGB   PS3AV_CMD_VIDEO_FMT_X8R8G8B8
/* aspect */
#define A_N    PS3AV_CMD_AV_ASPECT_4_3
#define A_W    PS3AV_CMD_AV_ASPECT_16_9
struct avset_video_mode {
	uint32_t cs;
	uint32_t fmt;
	uint32_t vid;
	uint32_t aspect;
	uint32_t x;
	uint32_t y;
	uint32_t interlace;
	uint32_t freq;
} video_mode_table[] = {
 {     0,    0,                        0,         0,    0,    0, 0, 0}, /* auto */
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_480I,       A_N,  720,  480, 1, 60}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_480P,       A_N,  720,  480, 0, 60}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_720P_60HZ,  A_N, 1280,  720, 0, 60}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_1080I_60HZ, A_W, 1920, 1080, 1, 60}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_1080P_60HZ, A_W, 1920, 1080, 0, 60}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_576I,       A_N,  720,  576, 1, 50}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_576P,       A_N,  720,  576, 0, 50}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_720P_50HZ,  A_N, 1280,  720, 0, 50}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_1080I_50HZ, A_W, 1920, 1080, 1, 50}, 
 {YUV444, XRGB, PS3AV_CMD_VIDEO_VID_1080P_50HZ, A_W, 1920, 1080, 0, 50}, 
 {  RGB8, XRGB, PS3AV_CMD_VIDEO_VID_WXGA,       A_W, 1280,  768, 0, 60}, 
 {  RGB8, XRGB, PS3AV_CMD_VIDEO_VID_SXGA,       A_N, 1280, 1024, 0, 60}, 
 {  RGB8, XRGB, PS3AV_CMD_VIDEO_VID_WUXGA,      A_W, 1920, 1200, 0, 60},
};

struct ps3av_send_hdr {
	uint16_t version;
	uint16_t size;
	uint32_t cid;
} __attribute__ ((packed));

struct ps3av_reply_hdr {
	uint16_t version;
	uint16_t size;
	uint32_t cid;
	uint32_t status;
} __attribute__ ((packed));

/* supported CIDs */
uint32_t cmd_table[] = {
	/* init */
	PS3AV_CID_AV_INIT,
	PS3AV_CID_AV_FIN,
	PS3AV_CID_VIDEO_INIT,
	PS3AV_CID_AUDIO_INIT,

	/* set */
	PS3AV_CID_AV_ENABLE_EVENT,
	PS3AV_CID_AV_DISABLE_EVENT,

	PS3AV_CID_AV_VIDEO_CS,
	PS3AV_CID_AV_VIDEO_MUTE,
	PS3AV_CID_AV_VIDEO_DISABLE_SIG,
	PS3AV_CID_AV_AUDIO_PARAM,
	PS3AV_CID_AV_AUDIO_MUTE,
	PS3AV_CID_AV_HDMI_MODE,
	PS3AV_CID_AV_TV_MUTE,

	PS3AV_CID_VIDEO_MODE,
	PS3AV_CID_VIDEO_FORMAT,
	PS3AV_CID_VIDEO_PITCH,

	PS3AV_CID_AUDIO_MODE,
	PS3AV_CID_AUDIO_MUTE,
	PS3AV_CID_AUDIO_ACTIVE,
	PS3AV_CID_AUDIO_INACTIVE,
	PS3AV_CID_AVB_PARAM,

	/* get */
	PS3AV_CID_AV_GET_HW_CONF,
	PS3AV_CID_AV_GET_MONITOR_INFO,

	/* event */
	PS3AV_CID_EVENT_UNPLUGGED,
	PS3AV_CID_EVENT_PLUGGED,
	PS3AV_CID_EVENT_HDCP_DONE,
	PS3AV_CID_EVENT_HDCP_FAIL,
	PS3AV_CID_EVENT_HDCP_AUTH,
	PS3AV_CID_EVENT_HDCP_ERROR,

	0
};

#define PS3AV_EVENT_CMD_MASK           0x10000000
#define PS3AV_EVENT_ID_MASK            0x0000ffff
#define PS3AV_CID_MASK                 0xffffffff
#define PS3AV_REPLY_BIT                0x80000000

#define ps3av_event_get_port_id(cid)   ((cid >> 16) & 0xff)


static uint32_t *search_cmd_table(uint32_t cid, uint32_t mask)
{
	uint32_t *table;
	int i;

	table = cmd_table;
	for (i = 0; ; table++, i++) {
		if ((*table & mask) == (cid & mask)) { 
			break;
		}
		if (*table == 0) {
			return NULL;
		}
	}
	return table;
}

static int parse_event_packet(uint8_t *packet_p)
{
	uint8_t *p;
	struct ps3av_reply_hdr *hdr;
	uint32_t *table;

	p = packet_p;
	hdr = (struct ps3av_reply_hdr *)p;
	if (hdr->cid & PS3AV_EVENT_CMD_MASK) {
		table = search_cmd_table(hdr->cid, PS3AV_EVENT_CMD_MASK);
		if (table) {
			DPRINTK("recv event packet cid:%08x port:0x%x size:%d\n", 
				hdr->cid, ps3av_event_get_port_id(hdr->cid),
				hdr->size);
		} else {
			printk("%s: failed event packet, cid:%08x size:%d\n",
				__FUNCTION__,
				hdr->cid, hdr->size); 
		}
		return 1; /* receive event packet */
	}
	return 0;
}

static int video_mode2res(uint32_t id, uint32_t *xres, uint32_t *yres)
{
	int size;

	id = id & PS3AV_MODE_MASK;
	size = sizeof(video_mode_table) / sizeof(*video_mode_table);
	if (id > size - 1 || id < 0) {
		printk("mode2res %d failed\n", id);
		return -1;
	}
	*xres = video_mode_table[id].x;
	*yres = video_mode_table[id].y;
	return 0;
}

static int send_cmd_pkt(uint32_t *buf, uint32_t *recv_buf, 
			int write_len, int read_len)
{
	int res;
	uint8_t *p;
	uint32_t cmd;
	struct ps3av_send_hdr *send_hdr;
	struct ps3av_reply_hdr *reply_hdr;
	int event;

	if (!ps3av.available) {
		res = -EINVAL;
	}

	p = (uint8_t *)buf;
	send_hdr = (struct ps3av_send_hdr *)p;
	cmd = send_hdr->cid;
	/* send pkt */
	res = ps3av_vuart_write(&ps3av.vuart, p, write_len);
	if (res < 0) {
		DPRINTK("%s: vuart_write() failed (result=%d)\n",
				__FUNCTION__, res);
		goto end;
	}

	/* recv pkt */
	do {
		p = (uint8_t *)recv_buf;
		/* 1st read */
		res = ps3av_vuart_read(&ps3av.vuart, p, PS3AV_HDR_SIZE, timeout);
		if (res != PS3AV_HDR_SIZE) {
			DPRINTK("%s: ps3av_vuart_read() failed (result=%d)\n",
				__FUNCTION__, res);
			goto end;
		}
		reply_hdr = (struct ps3av_reply_hdr *)p;
		/* 2nd read */
		res = ps3av_vuart_read(&ps3av.vuart, 
				p + PS3AV_HDR_SIZE, reply_hdr->size, timeout);
		if (res < 0) {
			DPRINTK("%s: ps3av_vuart_read() failed (result=%d)\n",
				__FUNCTION__, res);
			goto end;
		}
		res += PS3AV_HDR_SIZE; /* total len */
		event = parse_event_packet(p); /* ret > 0 event packet */
	} while(event);

	reply_hdr = (struct ps3av_reply_hdr *)p;
	if ((cmd | PS3AV_REPLY_BIT) != reply_hdr->cid) {
		DPRINTK("%s: reply err (result=%x)\n",
			__FUNCTION__, reply_hdr->cid);
		res = -EINVAL;
	}

end:
	return res;
}

static int process_reply_packet(uint32_t *cmd_buf, uint32_t *recv_buf, 
				int user_buf_size)
{
	struct ps3av_reply_hdr *hdr;
	int return_len;

	hdr = (struct ps3av_reply_hdr *)recv_buf;
	if (hdr->version != PS3AV_VERSION) {
		DPRINTK("reply_packet invalid version:%x\n",
			hdr->version);
		return -EFAULT;
	}
	return_len = hdr->size + PS3AV_HDR_SIZE;
	if (return_len > user_buf_size) return_len = user_buf_size;
	memcpy(cmd_buf, recv_buf, return_len);
	return 0; /* success */
}

void ps3av_set_hdr(uint32_t cid, uint16_t size, void *buf)
{
	struct ps3av_send_hdr *hdr;

	hdr = (struct ps3av_send_hdr *)buf;
	hdr->version = PS3AV_VERSION;
	hdr->size = size - PS3AV_HDR_SIZE;
	hdr->cid = cid;
}

int ps3av_do_pkt(uint32_t cid, uint16_t send_len, 
				size_t usr_buf_size, uint32_t *buf)
{
	int res = 0;
	uint8_t recv_buf[PS3AV_BUF_SIZE];
	uint32_t *table;

	if (down_interruptible (&ps3av.sem)) {
		printk("%s:sem failed cid:%x \n", __FUNCTION__, cid);
		return -ERESTARTSYS;
	}
	table = search_cmd_table(cid, PS3AV_CID_MASK);
	if (table == NULL) {
		printk("%s: invalid_cid:%x\n", __FUNCTION__, cid);
		res = -EINVAL;
		goto err;
	}
	if (send_len < PS3AV_HDR_SIZE) {
		printk("%s: invalid send_len:%d\n", __FUNCTION__, send_len);
		goto err;
	}

	/* create header */
	ps3av_set_hdr(cid, send_len, buf);

	if (usr_buf_size < send_len || usr_buf_size > PS3AV_BUF_SIZE) {
		printk("%s: invalid packet size:%ld\n",
			__FUNCTION__, usr_buf_size);
		goto err;
	}

	/* send packet via vuart */
	res = send_cmd_pkt(buf, (uint32_t *)recv_buf, 
				send_len, usr_buf_size);
	if (res < 0) {
		printk("%s: send_cmd_pkt() failed (result=%d)\n",
			__FUNCTION__, res);
		goto err;
	}

	/* process reply packet */
	res = process_reply_packet(buf, (uint32_t *)recv_buf, usr_buf_size);
	if (res < 0) {
		printk("%s: put_return_status() failed (result=%d)\n",
			__FUNCTION__, res);
		goto err;
	}

	up (&ps3av.sem);
	return 0;

err:
	up (&ps3av.sem);
	printk("%s:failed cid:%x res:%d\n", __FUNCTION__, cid, res);
	return res;
}

/* set resolution of ps3fb */
static int gpu_fbset(uint32_t id)
{
#ifdef CONFIG_FB_PS3
	uint32_t xres, yres, x, y;

	if (video_mode2res(id, &xres, &yres)) {
		return -1;
	}
	ps3fb_get_res(xres, yres, &x, &y); /* get draw area size */
	ps3fb_set_resolution(x, y);
	return 0;
#else
	return -1;
#endif
}

static int set_av_video_mute(uint32_t mute)
{
	int i, num_of_av_port;

	num_of_av_port = ps3av.av_hw_conf.num_of_hdmi
				+ ps3av.av_hw_conf.num_of_avmulti;
	/* video mute on */
	for (i = 0; i < num_of_av_port; i++) { 
		if (ps3av_cmd_av_video_mute(1, &ps3av.av_port[i], mute) < 0) {
			return -1;
		}
	}

	return 0;
}

static int set_video_disable_sig(uint32_t id)
{
	int i, num_of_hdmi_port, num_of_av_port;

	num_of_hdmi_port = ps3av.av_hw_conf.num_of_hdmi;
	num_of_av_port = ps3av.av_hw_conf.num_of_hdmi + 
				ps3av.av_hw_conf.num_of_avmulti;

	/* tv mute */
	for (i = 0; i < num_of_hdmi_port; i++) {
		if (ps3av_cmd_av_tv_mute(ps3av.av_port[i],
			PS3AV_CMD_MUTE_ON) < 0) {
			return -1;
		}
	}
	msleep(100);

		/* video mute on */
	for (i = 0; i < num_of_av_port; i++) {
		if (ps3av_cmd_av_video_disable_sig(ps3av.av_port[i]) < 0) {
			return -1;
		}
		if (i < num_of_hdmi_port) {
			if (ps3av_cmd_av_tv_mute(ps3av.av_port[i],
				PS3AV_CMD_MUTE_OFF) < 0) {
				return -1;
			}
		}
	}
	msleep(300);

	return 0;
}

static int set_audio_mute(uint32_t mute)
{
	int i, num_of_av_port, num_of_opt_port;

	num_of_av_port = ps3av.av_hw_conf.num_of_hdmi + 
				ps3av.av_hw_conf.num_of_avmulti;
	num_of_opt_port = ps3av.av_hw_conf.num_of_spdif;

	for (i = 0; i < num_of_av_port; i++) { 
		if (ps3av_cmd_av_audio_mute(1, &ps3av.av_port[i], mute) < 0) {
			return -1;
		}
	}
	for (i = 0; i < num_of_opt_port; i++) {
		if (ps3av_cmd_audio_mute(1, &ps3av.opt_port[i], mute) < 0) {
			return -1;
		}
	}

	return 0;
}

static int set_audiomode(uint32_t ch, uint32_t fs, uint32_t word_bits,
			uint32_t format, uint32_t source)
{
	struct ps3av_pkt_avb_param avb_param;
	int i, num_of_audio;
	struct ps3av_pkt_audio_mode audio_mode;
	uint8_t *param_p;
	uint32_t len;
	int vid;

	num_of_audio = ps3av.av_hw_conf.num_of_hdmi 
				+ ps3av.av_hw_conf.num_of_avmulti 
				+ ps3av.av_hw_conf.num_of_spdif;

	avb_param.num_of_video_pkt = 0;
	avb_param.num_of_audio_pkt = PS3AV_AVB_NUM_AUDIO; /* allways 0 */
	avb_param.num_of_av_video_pkt = 0;
	avb_param.num_of_av_audio_pkt = ps3av.av_hw_conf.num_of_hdmi;

	vid = video_mode_table[atomic_read(&ps3av.ps3av_mode)].vid;

	/* audio mute */
	set_audio_mute(PS3AV_CMD_MUTE_ON);

	/* audio inactive */
	if (ps3av_cmd_audio_active(0, ps3av.audio_port) < 0) {
		DPRINTK("ps3av_cmd_audio_active OFF failed\n");
	}

	param_p = (uint8_t *)&avb_param.video;
	/* audio_pkt */
	for (i = 0; i < num_of_audio; i++) {
		ps3av_cmd_set_audio_mode((uint8_t *)&audio_mode,
					ps3av.av_port[i],
					ch, fs, word_bits, format, source);
		if (i < ps3av.av_hw_conf.num_of_hdmi) {
			/* hdmi only */
			param_p = ps3av_cmd_set_av_audio_param(param_p, 
						ps3av.av_port[i],
						&audio_mode, vid);
		}
		/* audio_mode pkt should be sent separately */
		if (ps3av_cmd_audio_mode(&audio_mode) < 0) { 
			DPRINTK("ps3av_cmd_audio_mode failed, port:%x\n", i);
		}
	}

	/* send command using avb pkt */
	len = param_p - (uint8_t *)&avb_param;
	if (ps3av_cmd_avb_param(&avb_param, len) < 0) {
		DPRINTK("ps3av_cmd_avb_param failed\n");
	}

	/* audio mute */
	set_audio_mute(PS3AV_CMD_MUTE_OFF);

	/* audio active */
	if (ps3av_cmd_audio_active(1, ps3av.audio_port) < 0) {
		DPRINTK("ps3av_cmd_audio_active ON failed\n");
	}

	return 0;
}


static int set_videomode(uint32_t id)
{
	struct ps3av_pkt_avb_param avb_param;
	int i;
	uint8_t *param_p;
	uint32_t len, av_video_cs = 0;
	struct avset_video_mode *video_mode;
	int old_mode, res, event = 0;

	old_mode = ps3av_get_mode();

	video_mode = &video_mode_table[id & PS3AV_MODE_MASK];

	avb_param.num_of_video_pkt = PS3AV_AVB_NUM_VIDEO; /* num of head */
	avb_param.num_of_audio_pkt = 0; 
	avb_param.num_of_av_video_pkt = ps3av.av_hw_conf.num_of_hdmi + 
					ps3av.av_hw_conf.num_of_avmulti;
	avb_param.num_of_av_audio_pkt = 0;

	/* send command packet */
	if (event) {
		/* event enable */
		if (ps3av_cmd_enable_event() < 0) {
			DPRINTK("ps3av_cmd_enable_event failed \n");
		}
	}

	/* av video mute */
	set_av_video_mute(PS3AV_CMD_MUTE_ON);
	/* video signal off */
	set_video_disable_sig(id);

	/* set gpu video */
	gpu_fbset(id);

	/* Retail PS3 product doesn't support this */
	if (id & PS3AV_MODE_HDCP_OFF) {
		res = ps3av_cmd_av_hdmi_mode(PS3AV_CMD_AV_HDMI_HDCP_OFF);
		if (res == PS3AV_STATUS_UNSUPPORTED_HDMI_MODE) {
			DPRINTK("Not supported\n");
		} else if (res) {
			DPRINTK("ps3av_cmd_av_hdmi_mode failed\n");
		}
	} else if (old_mode & PS3AV_MODE_HDCP_OFF){
		res = ps3av_cmd_av_hdmi_mode(PS3AV_CMD_AV_HDMI_MODE_NORMAL);
		if (res < 0 && res != PS3AV_STATUS_UNSUPPORTED_HDMI_MODE) {
			DPRINTK("ps3av_cmd_av_hdmi_mode failed\n");
		}
	}

	param_p = (uint8_t *)&avb_param.video;
	/* video_pkt */
	for (i = 0; i < avb_param.num_of_video_pkt; i++) {
		param_p = ps3av_cmd_set_video_mode(param_p,
				ps3av.head[i], video_mode->vid,
				video_mode->fmt, id);
	}
	/* av_video_pkt */
	for (i = 0; i < avb_param.num_of_av_video_pkt; i++) {
		if (id & PS3AV_MODE_DVI || id & PS3AV_MODE_RGB) {
			av_video_cs = RGB8; 
		} else {
			av_video_cs = video_mode->cs;
		}
#ifndef PS3AV_HDMI_YUV
		if (ps3av.av_port[i] == PS3AV_CMD_AVPORT_HDMI_0 
		 || ps3av.av_port[i] == PS3AV_CMD_AVPORT_HDMI_1) {
			av_video_cs = RGB8; /* use RGB for HDMI */
		 }
#endif
		param_p = ps3av_cmd_set_av_video_cs(param_p,
					ps3av.av_port[i], video_mode->vid, 
					av_video_cs, video_mode->aspect, id); 
	}
	/* send command using avb pkt */
	len = param_p - (uint8_t *)&avb_param;
	res = ps3av_cmd_avb_param(&avb_param, len);
	if (res == PS3AV_STATUS_NO_SYNC_HEAD) {
		printk("Command failed. Please try your request again. \n");
	} else if (res) {
		DPRINTK("ps3av_cmd_avb_param failed\n");
	}

	msleep(1500);
	/* av video mute */
	set_av_video_mute(PS3AV_CMD_MUTE_OFF);

	return 0;
}

static int vid2table_id(int vid)
{
	int i;

	for(i = 1; 
		i < sizeof(video_mode_table) / sizeof(*video_mode_table); i++){
		if (video_mode_table[i].vid == vid) {
			return i;
		}
	}
	return -1;
}

static int resbit2vid(uint32_t res_50, uint32_t res_60)
{
	int vid = -1;

	if (res_50 > res_60) { /* if res_50 == res_60, res_60 will be used */
		if (res_50 & PS3AV_RESBIT_1920x1080P) {
			vid = PS3AV_CMD_VIDEO_VID_1080P_50HZ;
		} else if (res_50 & PS3AV_RESBIT_1920x1080I) {
			vid = PS3AV_CMD_VIDEO_VID_1080I_50HZ;
		} else if (res_50 & PS3AV_RESBIT_1280x720P) {
			vid = PS3AV_CMD_VIDEO_VID_720P_50HZ;
		} else if (res_50 & PS3AV_RESBIT_720x576P) {
			vid = PS3AV_CMD_VIDEO_VID_576P;
		} else {
			vid = -1;
		}
	} else {
		if (res_60 & PS3AV_RESBIT_1920x1080P) {
			vid = PS3AV_CMD_VIDEO_VID_1080P_60HZ;
		} else if (res_60 & PS3AV_RESBIT_1920x1080I) {
			vid = PS3AV_CMD_VIDEO_VID_1080I_60HZ;
		} else if (res_60 & PS3AV_RESBIT_1280x720P) {
			vid = PS3AV_CMD_VIDEO_VID_720P_60HZ;
		} else if (res_60 & PS3AV_RESBIT_720x480P) {
			vid = PS3AV_CMD_VIDEO_VID_480P;
		} else {
			vid = -1;
		}
	}
	return vid;
}

static int hdmi_get_vid(struct ps3av_info_monitor *info)
{
	uint32_t res_50, res_60;
	int vid = -1;

	if (info->monitor_type != PS3AV_MONITOR_TYPE_HDMI) {
		return -1;
	}

	/* check native resolution */
	res_50 = info->res_50.native & PS3AV_RES_MASK_50;
	res_60 = info->res_60.native & PS3AV_RES_MASK_60;
	if (res_50 || res_60) {
		vid = resbit2vid(res_50, res_60);
		return vid;
	}

	/* check resolution */
	res_50 = info->res_50.res_bits & PS3AV_RES_MASK_50;
	res_60 = info->res_60.res_bits & PS3AV_RES_MASK_60;
	if (res_50 || res_60) {
		vid = resbit2vid(res_50, res_60);
		return vid;
	}

	if (ps3av.region & PS3AV_REGION_60) {
		vid = PS3AV_DEFAULT_HDMI_VID_REG_60;
	} else {
		vid = PS3AV_DEFAULT_HDMI_VID_REG_50;
	}
	return vid;
}

static int auto_videomode(struct ps3av_pkt_av_get_hw_conf *av_hw_conf, 
			int boot)
{
	int i, vid = -1;
	int dvi = 0;
	int rgb = 0;
	struct ps3av_pkt_av_get_monitor_info monitor_info;
	struct ps3av_info_monitor *info;

	/* get vid for hdmi */
	for(i = 0; i < av_hw_conf->num_of_hdmi; i++) {
		if (ps3av_cmd_video_get_monitor_info(&monitor_info, 
					PS3AV_CMD_AVPORT_HDMI_0 + i) < 0) {
			return -1;
		} else {
#ifdef PS3AV_DEBUG
			ps3av_cmd_av_monitor_info_dump(&monitor_info);
#endif
			info = (struct ps3av_info_monitor *)&monitor_info.info;
			/* check DVI */
			if (info->monitor_type == PS3AV_MONITOR_TYPE_DVI) {
				dvi = PS3AV_MODE_DVI;
				break;
			}
			/* check HDMI */
			vid = hdmi_get_vid(info);
			if (vid != -1) {
				/* got valid vid */
				break;
			}
		}
	}

	if (dvi) {
		/* DVI mode */
		vid = PS3AV_DEFAULT_DVI_VID;
	} else if (vid == -1) {
		/* no HDMI interface or HDMI is off */
		if (ps3av.region & PS3AV_REGION_60) {
			vid = PS3AV_DEFAULT_AVMULTI_VID_REG_60;
		} else {
			vid = PS3AV_DEFAULT_AVMULTI_VID_REG_50;
		}
		if (ps3av.region & PS3AV_REGION_RGB) {
			rgb = PS3AV_MODE_RGB;
		}
	} else if (boot) {
		/* HDMI: using DEFAULT HDMI_VID while booting up */
		info = (struct ps3av_info_monitor *)monitor_info.info;
		if (ps3av.region & PS3AV_REGION_60) {
			if (info->res_60.res_bits & PS3AV_RESBIT_720x480P) {
				vid = PS3AV_DEFAULT_HDMI_VID_REG_60;
			} else if (info->res_50.res_bits & PS3AV_RESBIT_720x576P) {
				vid = PS3AV_DEFAULT_HDMI_VID_REG_50;
			} else {
				vid = PS3AV_DEFAULT_HDMI_VID_REG_60; /* default */
			}
		} else {
			if (info->res_50.res_bits & PS3AV_RESBIT_720x576P) {
				vid = PS3AV_DEFAULT_HDMI_VID_REG_50;
			} else if (info->res_60.res_bits & PS3AV_RESBIT_720x480P) {
				vid = PS3AV_DEFAULT_HDMI_VID_REG_60;
			} else {
				vid = PS3AV_DEFAULT_HDMI_VID_REG_50; /* default */
			}
		}
	}

	return (vid2table_id(vid) | dvi | rgb);
}

static int get_hw_conf(struct ps3av *ps3av)
{
	int i, j, k;

	/* get av_hw_conf */
	if (ps3av_cmd_av_get_hw_conf(&ps3av->av_hw_conf) < 0) {
		return -1;
	} else {
#ifdef PS3AV_DEBUG
		ps3av_cmd_av_hw_conf_dump(&ps3av->av_hw_conf);
#endif
	}


	for (i = 0; i < PS3AV_HEAD_MAX; i++) {
		ps3av->head[i] = PS3AV_CMD_VIDEO_HEAD_A + i;
	}
	for (i = 0; i < PS3AV_OPT_PORT_MAX; i++) {
		ps3av->opt_port[i] = PS3AV_CMD_AVPORT_SPDIF_0 + i;
	}
	for (i = 0; i < ps3av->av_hw_conf.num_of_hdmi; i++) {
		ps3av->av_port[i] = PS3AV_CMD_AVPORT_HDMI_0 + i;
	}
	for (j = 0; j < ps3av->av_hw_conf.num_of_avmulti; j++) {
		ps3av->av_port[i + j] = PS3AV_CMD_AVPORT_AVMULTI_0 + j;
	}
	for (k = 0; k < ps3av->av_hw_conf.num_of_spdif; k++) {
		ps3av->av_port[i + j + k] = PS3AV_CMD_AVPORT_SPDIF_0 + k;
	}

	/* set all audio port */
	ps3av->audio_port = PS3AV_CMD_AUDIO_PORT_HDMI_0 \
				| PS3AV_CMD_AUDIO_PORT_HDMI_1 \
				| PS3AV_CMD_AUDIO_PORT_AVMULTI_0 \
				| PS3AV_CMD_AUDIO_PORT_SPDIF_0 \
				| PS3AV_CMD_AUDIO_PORT_SPDIF_1;

	return 0;
}

/* set mode using id */
int ps3av_set_video_mode(uint32_t id, int boot)
{
	int size;
	uint32_t option;
	uint32_t old_id;

	size = sizeof(video_mode_table) / sizeof(*video_mode_table);
	if ((id & PS3AV_MODE_MASK) > size - 1 || id < 0) {
		DPRINTK("%s: error id :%d\n", __FUNCTION__, id);
		goto end;
	}

	/* auto mode */
	option = id & ~PS3AV_MODE_MASK;
	if ((id & PS3AV_MODE_MASK) == 0) {
		id = auto_videomode(&ps3av.av_hw_conf, boot);
		if (id < 1) {
			printk("%s: invalid id :%d\n", __FUNCTION__, id);
			goto end;
		}
		id |= option;
	}

	/* set videomode */
	old_id = atomic_read(&ps3av.ps3av_mode);
	atomic_set(&ps3av.ps3av_mode, id);
	if(set_videomode(id)) {
		atomic_set(&ps3av.ps3av_mode, old_id);
	}

	return 0;
end:
	return -1;
}
EXPORT_SYMBOL(ps3av_set_video_mode);

int ps3av_set_audio_mode(uint32_t ch, uint32_t fs, uint32_t word_bits,
			uint32_t format, uint32_t source)
{
	int res;

	res = set_audiomode(ch, fs, word_bits, format, source);
	return res;
}
EXPORT_SYMBOL(ps3av_set_audio_mode);

int ps3av_set_mode(uint32_t id, int boot)
{
	int res;

	res = ps3av_set_video_mode(id, boot);
	if (res)  {
		goto err;
	}
	res = ps3av_set_audio_mode(PS3AV_CMD_AUDIO_NUM_OF_CH_2,
					PS3AV_CMD_AUDIO_FS_48K,
					PS3AV_CMD_AUDIO_WORD_BITS_16,
					PS3AV_CMD_AUDIO_FORMAT_PCM,
					PS3AV_CMD_AUDIO_SOURCE_SERIAL);
	if (res) {
		goto err;
	}
	return 0;
err:
	return -1;
}
EXPORT_SYMBOL(ps3av_set_mode);

int ps3av_get_mode(void)
{
	return atomic_read(&ps3av.ps3av_mode);
}
EXPORT_SYMBOL(ps3av_get_mode);

int ps3av_get_scanmode(int id)
{
	int size;

	id = id & PS3AV_MODE_MASK;
	size = sizeof(video_mode_table) / sizeof(*video_mode_table);
	if (id > size - 1 || id < 0) {
		printk("ps3av_get_scanmode %d failed\n", id);
		return -1;
	}
	return video_mode_table[id].interlace;
}
EXPORT_SYMBOL(ps3av_get_scanmode);

int ps3av_get_refresh_rate(int id)
{
	int size;

	id = id & PS3AV_MODE_MASK;
	size = sizeof(video_mode_table) / sizeof(*video_mode_table);
	if (id > size - 1 || id < 0) {
		printk("ps3av_get_refresh_rate %d failed\n", id);
		return -1;
	}
	return video_mode_table[id].freq;
}
EXPORT_SYMBOL(ps3av_get_refresh_rate);

/* get resolution by video_mode */
int ps3av_video_mode2res(uint32_t id, uint32_t *xres, uint32_t *yres)
{
	return video_mode2res(id, xres, yres);
}
EXPORT_SYMBOL(ps3av_video_mode2res);

/* mute */
int ps3av_video_mute(int mute)
{
	uint32_t mode;
	int res = -1;

	mode = mute ? PS3AV_CMD_MUTE_ON : PS3AV_CMD_MUTE_OFF;
	res = set_av_video_mute(mode);
	return res;
}
EXPORT_SYMBOL(ps3av_video_mute);

int ps3av_audio_mute(int mute)
{
	uint32_t mode;

	mode = mute ? PS3AV_CMD_MUTE_ON : PS3AV_CMD_MUTE_OFF;
	return set_audio_mute(mode);
}
EXPORT_SYMBOL(ps3av_audio_mute);

int ps3av_dev_open(void)
{
	uint64_t status = 0;

	if (atomic_read(&ps3av_open_count) == 0) {
		status = lv1_gpu_open(0);
		if (status) {
			atomic_dec(&ps3av_open_count);
		}
	}
	atomic_inc(&ps3av_open_count);

	return status;
}
EXPORT_SYMBOL(ps3av_dev_open);

int ps3av_dev_close(void)
{
	uint64_t status = 0;

	if (atomic_read(&ps3av_open_count) <= 0) {
		return -1;
	}

	if (atomic_read(&ps3av_open_count) == 1) {
		status = lv1_gpu_close();
		/* ignore error */
	}
	atomic_dec(&ps3av_open_count);

	return status;
}
EXPORT_SYMBOL(ps3av_dev_close);

static int ps3av_at_exit(struct notifier_block *self, 
				unsigned long state, void *data)
{
	/* fin avsetting modules */
	ps3av_cmd_fin();

	if (!ps3av.available) {
		return NOTIFY_OK;
	}
	ps3av_vuart_cleanup(&ps3av.vuart);
	ps3av.available = 0;

	return NOTIFY_OK;
}

static struct notifier_block ps3av_reboot_nb = {
	.notifier_call = ps3av_at_exit
};

static int ps3av_module_init(void)
{
	int res;
	uint32_t id;
	int region;

	DPRINTK("init ...\n");
	DPRINTK("  timeout=%d\n", timeout);

	memset(&ps3av, 0, sizeof(ps3av));

	init_MUTEX(&ps3av.sem);
	atomic_set(&ps3av.ps3av_mode, 0);
	res = ps3av_vuart_init(&ps3av.vuart, 0, BUFSIZE);
	if (res != 0) {
		DPRINTK("%s: ps3av_vuart_init noport (result=%d)\n",
		       __FUNCTION__,  res);
	}

	ps3av.available = 1;
	region = ps3pf_boot_data_get_video_mode();
	switch(region) {
	case PS3AV_BOOT_NTSC:
		ps3av.region = PS3AV_REGION_60;
		break;
	case PS3AV_BOOT_PAL_YUV:
		ps3av.region = PS3AV_REGION_50;
		break;
	case PS3AV_BOOT_PAL_RGB:
		ps3av.region = PS3AV_REGION_50 | PS3AV_REGION_RGB;
		break;
	default:
		ps3av.region = PS3AV_REGION_60;
		break;
	}
	register_reboot_notifier(&ps3av_reboot_nb);

	/* init avsetting modules */
	if (ps3av_cmd_init() < 0) {
		printk("%s: avset_init failed\n", __FUNCTION__);
	}
	get_hw_conf(&ps3av);
	id = auto_videomode(&ps3av.av_hw_conf, 1);
	atomic_set(&ps3av.ps3av_mode, id);

	DPRINTK("init...done\n");

	return 0;
}

module_init(ps3av_module_init);
