/*
 * Copyright (C) 2006 Sony Computer Entertainment Inc.
 * Av backend support for PS3PF 
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <asm/lv1call.h>
#include <asm/ps3av.h>

//#define PS3AV_DEBUG
#ifdef PS3AV_DEBUG
#define DPRINTK(fmt, args...) \
	do { printk("ps3av " fmt, ## args); } while (0)
#else
#define DPRINTK(fmt, args...) do { } while (0)
#endif

#ifdef CONFIG_FB_PS3
extern int ps3fb_flip_ctl(int);
#else
#define ps3fb_flip_ctl(x)
#endif

struct video_fmt {
	uint32_t format;
	uint32_t order;
} video_fmt_table[] = {
	{ PS3AV_CMD_VIDEO_FORMAT_ARGB_8BIT, PS3AV_CMD_VIDEO_ORDER_RGB },
	{ PS3AV_CMD_VIDEO_FORMAT_ARGB_8BIT, PS3AV_CMD_VIDEO_ORDER_BGR },
};

static uint32_t cs_video2av(int cs) 
{
	uint32_t res = 0;

	switch(cs) {
	case PS3AV_CMD_VIDEO_CS_RGB_8:
	case PS3AV_CMD_VIDEO_CS_RGB_10:
	case PS3AV_CMD_VIDEO_CS_RGB_12:
		res = PS3AV_CMD_AV_CS_RGB_8;
		break;
	case PS3AV_CMD_VIDEO_CS_YUV444_8:
	case PS3AV_CMD_VIDEO_CS_YUV444_10:
	case PS3AV_CMD_VIDEO_CS_YUV444_12:
		res = PS3AV_CMD_AV_CS_YUV444_8;
		break;
	case PS3AV_CMD_VIDEO_CS_YUV422_8:
	case PS3AV_CMD_VIDEO_CS_YUV422_10:
	case PS3AV_CMD_VIDEO_CS_YUV422_12:
		res = PS3AV_CMD_AV_CS_YUV422_8;
		break;
	case PS3AV_CMD_VIDEO_CS_XVYCC_8:
	case PS3AV_CMD_VIDEO_CS_XVYCC_10:
	case PS3AV_CMD_VIDEO_CS_XVYCC_12:
		res = PS3AV_CMD_AV_CS_XVYCC_8;
		break;
	default:
		res = PS3AV_CMD_AV_CS_RGB_8;
		break;
	}
	return res;
}
static uint32_t cs_video2av_bitlen(int cs)
{
	uint32_t res = 0;

	switch(cs) {
	case PS3AV_CMD_VIDEO_CS_RGB_8:
	case PS3AV_CMD_VIDEO_CS_YUV444_8:
	case PS3AV_CMD_VIDEO_CS_YUV422_8:
	case PS3AV_CMD_VIDEO_CS_XVYCC_8:
		res = PS3AV_CMD_AV_CS_8;
		break;
	case PS3AV_CMD_VIDEO_CS_RGB_10:
	case PS3AV_CMD_VIDEO_CS_YUV444_10:
	case PS3AV_CMD_VIDEO_CS_YUV422_10:
	case PS3AV_CMD_VIDEO_CS_XVYCC_10:
		res = PS3AV_CMD_AV_CS_10;
		break;
	case PS3AV_CMD_VIDEO_CS_RGB_12:
	case PS3AV_CMD_VIDEO_CS_YUV444_12:
	case PS3AV_CMD_VIDEO_CS_YUV422_12:
	case PS3AV_CMD_VIDEO_CS_XVYCC_12:
		res = PS3AV_CMD_AV_CS_12;
		break;
	default:
		res = PS3AV_CMD_AV_CS_8;
		break;
	}
	return res;
}

static uint32_t vid_video2av(int vid)
{
	uint32_t res = 0;

	switch (vid) {
	case PS3AV_CMD_VIDEO_VID_480I:
		res = PS3AV_CMD_AV_VID_480I;
		break;
	case PS3AV_CMD_VIDEO_VID_480P:
		res = PS3AV_CMD_AV_VID_480P;
		break;
	case PS3AV_CMD_VIDEO_VID_576I:
		res = PS3AV_CMD_AV_VID_576I;
		break;
	case PS3AV_CMD_VIDEO_VID_576P:
		res = PS3AV_CMD_AV_VID_576P;
		break;
	case PS3AV_CMD_VIDEO_VID_1080I_60HZ:
		res = PS3AV_CMD_AV_VID_1080I_60HZ;
		break;
	case PS3AV_CMD_VIDEO_VID_720P_60HZ:
		res = PS3AV_CMD_AV_VID_720P_60HZ;
		break;
	case PS3AV_CMD_VIDEO_VID_1080P_60HZ:
		res = PS3AV_CMD_AV_VID_1080P_60HZ;
		break;
	case PS3AV_CMD_VIDEO_VID_1080I_50HZ:
		res = PS3AV_CMD_AV_VID_1080I_50HZ;
		break;
	case PS3AV_CMD_VIDEO_VID_720P_50HZ:
		res = PS3AV_CMD_AV_VID_720P_50HZ;
		break;
	case PS3AV_CMD_VIDEO_VID_1080P_50HZ:
		res = PS3AV_CMD_AV_VID_1080P_50HZ;
		break;
	case PS3AV_CMD_VIDEO_VID_WXGA:
		res = PS3AV_CMD_AV_VID_WXGA;
		break;
	case PS3AV_CMD_VIDEO_VID_SXGA:
		res = PS3AV_CMD_AV_VID_SXGA;
		break;
	case PS3AV_CMD_VIDEO_VID_WUXGA:
		res = PS3AV_CMD_AV_VID_WUXGA;
		break;
	default:
		res = PS3AV_CMD_AV_VID_480P;
		break;
	}
	return res;
}

int ps3av_cmd_init(void)
{
	int res = -1;
	struct ps3av_pkt_av_init av_init;
	struct ps3av_pkt_video_init video_init;
	struct ps3av_pkt_audio_init audio_init;

	/* video init */
	memset((uint8_t *)&video_init, 0, sizeof(video_init));

	if ((res = ps3av_do_pkt(PS3AV_CID_VIDEO_INIT,
				PS3AV_HDR_SIZE + sizeof(video_init.cid),
				sizeof(video_init),
				(uint32_t *)&video_init)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&video_init)) != 0) {
			printk("PS3AV_CID_VIDEO_INIT:failed status %x\n", 
				res);
			goto err;
		}
	}

	/* audio init */
	memset((uint8_t *)&audio_init, 0, sizeof(audio_init));

	if ((res = ps3av_do_pkt(PS3AV_CID_AUDIO_INIT, 
				PS3AV_HDR_SIZE + sizeof(audio_init.cid),
				sizeof(audio_init),
				(uint32_t *)&audio_init)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&audio_init)) != 0) {
			printk("PS3AV_CID_AUDIO_INIT:failed status %x\n", 
				res);
			goto err;
		}
	}

	/* av init */
	memset((uint8_t *)&av_init, 0, sizeof(av_init));
	av_init.event_bit = 0;

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_INIT, 
				sizeof(av_init),
				sizeof(av_init),
				(uint32_t *)&av_init)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&av_init)) != 0) {
			printk("PS3AV_CID_AV_INIT:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_fin(void)
{
	int res = -1;
	struct ps3av_pkt_av_fin av_fin;

	memset((uint8_t *)&av_fin, 0, sizeof(av_fin));

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_FIN, 
				PS3AV_HDR_SIZE + sizeof(av_fin.cid),
				sizeof(av_fin),
				(uint32_t *)&av_fin)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&av_fin)) != 0) {
			printk("PS3AV_CID_AV_FIN:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_av_video_mute(int num_of_port, uint32_t *port, uint32_t mute)
{
	int i, send_len, res = -1;
	struct ps3av_pkt_av_video_mute av_video_mute;

	if (num_of_port > PS3AV_MUTE_PORT_MAX) {
		goto err;
	}

	memset((uint8_t *)&av_video_mute, 0, sizeof(av_video_mute));
	for (i = 0; i < num_of_port; i++) {
		av_video_mute.mute[i].avport = port[i];
		av_video_mute.mute[i].mute = mute;
	}

	send_len = PS3AV_HDR_SIZE + sizeof(av_video_mute.cid) + 
			sizeof(struct ps3av_av_mute) * num_of_port; 
	if ((res = ps3av_do_pkt(PS3AV_CID_AV_VIDEO_MUTE,
				send_len,
				sizeof(av_video_mute), 
				(uint32_t *)&av_video_mute)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&av_video_mute)) != 0) {
			printk("PS3AV_CID_AV_VIDEO_MUTE:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_av_video_disable_sig(uint32_t port)
{
	int res = -1;
	struct ps3av_pkt_av_video_disable_sig av_video_sig;


	memset((uint8_t *)&av_video_sig, 0, sizeof(av_video_sig));
	av_video_sig.avport = port;

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_VIDEO_DISABLE_SIG,
				sizeof(av_video_sig),
				sizeof(av_video_sig), 
				(uint32_t *)&av_video_sig)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&av_video_sig)) != 0) {
			printk("PS3AV_CID_AV_VIDEO_DISABLE_SIG:failed status %x port:%x\n", 
				res, port);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_av_tv_mute(uint32_t avport, uint32_t mute)
{
	int res = -1;
	struct ps3av_pkt_av_tv_mute tv_mute;

	memset((uint8_t *)&tv_mute, 0, sizeof(tv_mute));
	tv_mute.avport = avport;
	tv_mute.mute = mute;

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_TV_MUTE,
				sizeof(tv_mute),
				sizeof(tv_mute), 
				(uint32_t *)&tv_mute)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&tv_mute)) != 0) {
			printk("PS3AV_CID_AV_TV_MUTE:failed status %x port:%x\n", 
				res, avport);
			goto err;
		}
	}
	return 0;
err:
	return res;
}

int ps3av_cmd_enable_event(void)
{
	int res = -1;
	struct ps3av_pkt_av_event av_event;

	memset((uint8_t *)&av_event, 0, sizeof(av_event));
	av_event.event_bit = PS3AV_CMD_EVENT_BIT_UNPLUGGED |
				PS3AV_CMD_EVENT_BIT_PLUGGED |
				PS3AV_CMD_EVENT_BIT_HDCP_DONE ;

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_ENABLE_EVENT, 
				sizeof(av_event),
				sizeof(av_event), 
				(uint32_t *)&av_event)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&av_event)) != 0) {
			printk("PS3AV_CID_AV_ENABLE_EVENT:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_av_hdmi_mode(uint8_t mode)
{
	int res = -1;
	struct ps3av_pkt_av_hdmi_mode hdmi_mode;

	memset((uint8_t *)&hdmi_mode, 0, sizeof(hdmi_mode));
	hdmi_mode.mode = mode;

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_HDMI_MODE,
				sizeof(hdmi_mode),
				sizeof(hdmi_mode),
				(uint32_t *)&hdmi_mode)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&hdmi_mode)) != 0) {
			if (res != PS3AV_STATUS_UNSUPPORTED_HDMI_MODE) {
				printk("PS3AV_CID_AV_HDMI_MODE:failed status %x\n", 
				res);
			}
			goto err;
		}
	}

	return 0;
err:
	return res;
}

uint8_t *ps3av_cmd_set_av_video_cs(uint8_t *p, uint32_t avport, int video_vid, 
				int cs_out, int aspect, uint32_t id)
{
	struct ps3av_pkt_av_video_cs *av_video_cs;

	av_video_cs = (struct ps3av_pkt_av_video_cs *)p;
	if (video_vid == -1) video_vid = PS3AV_CMD_VIDEO_VID_720P_60HZ;
	if (cs_out == -1) cs_out = PS3AV_CMD_VIDEO_CS_YUV444_8;
	if (aspect == -1) aspect = 0;

	memset((uint8_t *)av_video_cs, 0, sizeof(*av_video_cs));
	ps3av_set_hdr(PS3AV_CID_AV_VIDEO_CS, sizeof(*av_video_cs),
			av_video_cs); 
	av_video_cs->avport = avport;
	/* should be same as video_mode.resolution */
	av_video_cs->av_vid = vid_video2av(video_vid);
	av_video_cs->av_cs_out = cs_video2av(cs_out);
	/* should be same as video_mode.video_cs_out */
	av_video_cs->av_cs_in = cs_video2av(PS3AV_CMD_VIDEO_CS_RGB_8);
	av_video_cs->bitlen_out = cs_video2av_bitlen(cs_out);
	av_video_cs->aspect = aspect;
	if (id & PS3AV_MODE_DITHER) {
		av_video_cs->dither = PS3AV_CMD_AV_DITHER_ON 
					| PS3AV_CMD_AV_DITHER_8BIT; 
	} else {
		/* default off */
		av_video_cs->dither = PS3AV_CMD_AV_DITHER_OFF; 
	}

	return p + sizeof(*av_video_cs); 
}

uint8_t *ps3av_cmd_set_video_mode(uint8_t *p, uint32_t head, int video_vid, 
				int video_fmt, uint32_t id)
{
	struct ps3av_pkt_video_mode *video_mode;
	uint32_t x, y;

	video_mode = (struct ps3av_pkt_video_mode *)p;
	if (video_vid == -1) video_vid = PS3AV_CMD_VIDEO_VID_720P_60HZ;
	if (video_fmt == -1) video_fmt = PS3AV_CMD_VIDEO_FMT_X8R8G8B8;

	if (ps3av_video_mode2res(id, &x, &y)) {
		return p;
	}
	/* video mode */
	memset((uint8_t *)video_mode, 0, sizeof(*video_mode));
	ps3av_set_hdr(PS3AV_CID_VIDEO_MODE, sizeof(*video_mode), video_mode); 
	video_mode->video_head      = head;
	if (video_vid == PS3AV_CMD_VIDEO_VID_480I
		&& head == PS3AV_CMD_VIDEO_HEAD_B) {
		video_mode->video_vid = PS3AV_CMD_VIDEO_VID_480I_A;
	} else {
		video_mode->video_vid = video_vid;
	}
	video_mode->width           = (uint16_t)x;
	video_mode->height          = (uint16_t)y;
	video_mode->pitch           = video_mode->width * 4; /* line_length */
	video_mode->video_out_format= PS3AV_CMD_VIDEO_OUT_FORMAT_RGB_12BIT;
	video_mode->video_format    = video_fmt_table[video_fmt].format;
	video_mode->video_order     = video_fmt_table[video_fmt].order;

	DPRINTK("video_mode:vid:%x width:%d height:%d pitch:%d out_formt:%d format:%x order:%x\n", 
		video_vid, video_mode->width, video_mode->height, 
		video_mode->pitch, video_mode->video_out_format,
		video_mode->video_format, video_mode->video_order);
	return p + sizeof(*video_mode);
}

int ps3av_cmd_video_format_black(uint32_t head, uint32_t video_fmt, uint32_t mute)
{
	int res = -1;
	struct ps3av_pkt_video_format video_format;

	memset((uint8_t *)&video_format, 0, sizeof(video_format));
	video_format.video_head = head;
	if (mute != PS3AV_CMD_MUTE_OFF) {
		video_format.video_format = PS3AV_CMD_VIDEO_FORMAT_BLACK;
	} else {
		video_format.video_format = video_fmt_table[video_fmt].format;
	}
	video_format.video_order = video_fmt_table[video_fmt].order;
	if ((res = ps3av_do_pkt(PS3AV_CID_VIDEO_FORMAT,
				sizeof(video_format),
				sizeof(video_format),
				(uint32_t *)&video_format)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&video_format)) != 0) {
			printk("PS3AV_CID_VIDEO_FORMAT:failed status %x\n", res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}


int ps3av_cmd_av_audio_mute(int num_of_port, uint32_t *port, uint32_t mute)
{
	int i, send_len, res = -1;
	struct ps3av_pkt_av_audio_mute av_audio_mute;

	if (num_of_port > PS3AV_MUTE_PORT_MAX) {
		goto err;
	}

	/* audio mute */
	memset((uint8_t *)&av_audio_mute, 0, sizeof(av_audio_mute));
	for (i = 0; i < num_of_port; i++) {
		av_audio_mute.mute[i].avport = port[i];
		av_audio_mute.mute[i].mute = mute;
	}

	send_len = PS3AV_HDR_SIZE + sizeof(av_audio_mute.cid) +
			sizeof(struct ps3av_av_mute) * num_of_port;
	if ((res = ps3av_do_pkt(PS3AV_CID_AV_AUDIO_MUTE,
				send_len,
				sizeof(av_audio_mute),
				(uint32_t *)&av_audio_mute)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&av_audio_mute)) != 0) {
			printk("PS3AV_CID_AV_AUDIO_MUTE:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

static uint8_t cnv_mclk(uint32_t fs)
{
	uint8_t mclk;

	switch(fs) {
	case PS3AV_CMD_AUDIO_FS_44K:
		mclk = PS3AV_CMD_AV_MCLK_512;
		break;
	case PS3AV_CMD_AUDIO_FS_48K:
		mclk = PS3AV_CMD_AV_MCLK_512;
		break;
	case PS3AV_CMD_AUDIO_FS_88K:
		mclk = PS3AV_CMD_AV_MCLK_256;
		break;
	case PS3AV_CMD_AUDIO_FS_96K:
		mclk = PS3AV_CMD_AV_MCLK_256;
		break;
	case PS3AV_CMD_AUDIO_FS_176K:
		mclk = PS3AV_CMD_AV_MCLK_128;
		break;
	case PS3AV_CMD_AUDIO_FS_192K:
		mclk = PS3AV_CMD_AV_MCLK_128;
		break;
	default:
		printk("cnv_mclk failed, fs:%x\n", fs);
		mclk = 0;
		break;
	}
	return mclk;
}

const uint32_t ns_table[][5] ={
	/*   D1,    D2,    D3,    D4,    D5 */
	{  6272,  6272, 17836, 17836,  8918 }, /*  44K */
	{  6144,  6144, 11648, 11648,  5824 }, /*  48K */
	{ 12544, 12544, 35672, 35672, 17836 }, /*  88K */
	{ 12288, 12288, 23296, 23296, 11648 }, /*  96K */
	{ 25088, 25088, 71344, 71344, 35672 }, /* 176K */
	{ 24576, 24576, 46592, 46592, 23296 }, /* 192K */
};

static void cnv_ns(uint8_t *ns, uint32_t fs, uint32_t video_vid)
{
	uint32_t av_vid, ns_val;
	uint8_t *p = ns;
	int d;

	d = ns_val = 0;
	av_vid = vid_video2av(video_vid);
	switch (av_vid) {
	case PS3AV_CMD_AV_VID_480I:
	case PS3AV_CMD_AV_VID_576I:
		d = 0;
		break;
	case PS3AV_CMD_AV_VID_480P:
	case PS3AV_CMD_AV_VID_576P:
		d = 1;
		break;
	case PS3AV_CMD_AV_VID_1080I_60HZ:
	case PS3AV_CMD_AV_VID_1080I_50HZ:
		d = 2;
		break;
	case PS3AV_CMD_AV_VID_720P_60HZ:
	case PS3AV_CMD_AV_VID_720P_50HZ:
		d = 3;
		break;
	case PS3AV_CMD_AV_VID_1080P_60HZ:
	case PS3AV_CMD_AV_VID_1080P_50HZ:
	case PS3AV_CMD_AV_VID_WXGA:
	case PS3AV_CMD_AV_VID_SXGA:
	case PS3AV_CMD_AV_VID_WUXGA:
		d = 4;
		break;
	default:
		printk("cnv_ns faild, vid:%x\n", video_vid);
		break;
	}

	switch(fs) {
	case PS3AV_CMD_AUDIO_FS_44K:
		ns_val = ns_table[0][d];
		break;
	case PS3AV_CMD_AUDIO_FS_48K:
		ns_val = ns_table[1][d];
		break;
	case PS3AV_CMD_AUDIO_FS_88K:
		ns_val = ns_table[2][d];
		break;
	case PS3AV_CMD_AUDIO_FS_96K:
		ns_val = ns_table[3][d];
		break;
	case PS3AV_CMD_AUDIO_FS_176K:
		ns_val = ns_table[4][d];
		break;
	case PS3AV_CMD_AUDIO_FS_192K:
		ns_val = ns_table[5][d];
		break;
	default:
		printk("cnv_ns faild, fs:%x\n", fs);
		break;
	}
	*p++ = ns_val & 0x000000FF;
	*p++ = (ns_val & 0x0000FF00) >> 8;
	*p =   (ns_val & 0x00FF0000) >> 16;
}

static uint8_t cnv_enable(uint32_t source, uint8_t *enable)
{
	uint8_t *p, ret = 0;

	if (source == PS3AV_CMD_AUDIO_SOURCE_SPDIF) {
		ret = 0x03;
	} else if(source == PS3AV_CMD_AUDIO_SOURCE_SERIAL) {
		p = enable;
		ret = ((p[0] << 4) + (p[1] << 5) 
			+ (p[2] << 6) + (p[3] << 7)) | 0x01;
	} else {
		printk("cnv_enable faild, source:%x\n", source);
	}
	return ret;
}

static uint8_t cnv_fifomap(uint8_t *map)
{
	uint8_t *p, ret = 0;

	p = map;
	ret = p[0] + (p[1] << 2) + (p[2] << 4) + (p[3] << 6);
	return ret;
}

static uint8_t cnv_inputlen(uint32_t word_bits)
{
	uint8_t ret = 0;

	switch (word_bits) {
	case PS3AV_CMD_AUDIO_WORD_BITS_16:
		ret = PS3AV_CMD_AV_INPUTLEN_16;
		break;
	case PS3AV_CMD_AUDIO_WORD_BITS_20:
		ret = PS3AV_CMD_AV_INPUTLEN_20;
		break;
	case PS3AV_CMD_AUDIO_WORD_BITS_24:
		ret = PS3AV_CMD_AV_INPUTLEN_24;
		break;
	default:
		printk("cnv_inputlen failed word_bits:%x\n", word_bits);
		break;
	}
	return ret;
}

static uint8_t cnv_layout(uint32_t num_of_ch)
{
	uint8_t ret = 0;

	if (num_of_ch > PS3AV_CMD_AUDIO_NUM_OF_CH_8) {
		printk("cnv_layout failed, num_of_ch:%x\n", num_of_ch);
		ret = 0;
		return ret;
	}
	if (num_of_ch == PS3AV_CMD_AUDIO_NUM_OF_CH_2) {
		ret = 0x0;
	} else {
		ret = 0x1; 
	}
	return ret;
}

static void cnv_info(uint8_t *info, struct ps3av_pkt_audio_mode *mode)
{
	struct ps3av_audio_info_frame *p;

	p = (struct ps3av_audio_info_frame *)info;
	p->pb1.cc = mode->audio_num_of_ch + 1;	/* CH2:0x01 --- CH8:0x07*/
	p->pb1.ct = 0;
	p->pb2.sf = 0;
	p->pb2.ss = 0;

	p->pb3 = 0; /* check mode->audio_format ?? */
	p->pb4 = mode->audio_layout;
	p->pb5.dm = mode->audio_downmix;
	p->pb5.lsv = mode->audio_downmix_level;
}

static void cnv_chstat(uint8_t *chstat, uint8_t *cs_info)
{
	memcpy(chstat, cs_info, 5);
}

uint8_t *ps3av_cmd_set_av_audio_param(uint8_t *p, uint32_t port,
		struct ps3av_pkt_audio_mode *audio_mode, uint32_t video_vid)
{
	struct ps3av_pkt_av_audio_param *param;

	param = (struct ps3av_pkt_av_audio_param *)p;

	memset((uint8_t *)param, 0, sizeof(*param));
	ps3av_set_hdr(PS3AV_CID_AV_AUDIO_PARAM, sizeof(*param), param); 

	param->avport = port;
	param->mclk = cnv_mclk(audio_mode->audio_fs) | 0x80;
	cnv_ns(param->ns, audio_mode->audio_fs, video_vid);
	param->enable =cnv_enable(audio_mode->audio_source,
				audio_mode->audio_enable);
	param->swaplr = 0x09;
	param->fifomap = cnv_fifomap(audio_mode->audio_map);
	param->inputctrl = 0x49;
	param->inputlen = cnv_inputlen(audio_mode->audio_word_bits);
	param->layout = cnv_layout(audio_mode->audio_num_of_ch);
	cnv_info(param->info, audio_mode);
	cnv_chstat(param->chstat, audio_mode->audio_cs_info);

	return p + sizeof(*param);
}

/* default cs val */
static uint8_t mode_cs_info[] = {
	0x00, 0x09, 0x00, 0x02, 0x01, 0x00, 0x00, 0x00
};
#define CS_44	0x00
#define CS_48	0x02
#define CS_88	0x08
#define CS_96	0x0a
#define CS_176	0x0c
#define CS_192	0x0e
#define CS_MASK	0x0f
#define CS_BIT	0x40

uint8_t *ps3av_cmd_set_audio_mode(uint8_t *p, uint32_t avport, 
			uint32_t ch, uint32_t fs, uint32_t word_bits, 
			uint32_t format, uint32_t source)
{
	struct ps3av_pkt_audio_mode *audio;
	int spdif_through, spdif_bitstream;
	int i;

	if (!(ch | fs | format | word_bits | source)) {
		ch = PS3AV_CMD_AUDIO_NUM_OF_CH_2;
		fs = PS3AV_CMD_AUDIO_FS_48K;
		word_bits = PS3AV_CMD_AUDIO_WORD_BITS_16;
		format = PS3AV_CMD_AUDIO_FORMAT_PCM;
		source = PS3AV_CMD_AUDIO_SOURCE_SERIAL;
	}
	spdif_through = spdif_bitstream = 0; /* XXX not supported */

	audio = (struct ps3av_pkt_audio_mode *)p;
	/* audio mode */
	memset((uint8_t *)audio, 0, sizeof(*audio));
	ps3av_set_hdr(PS3AV_CID_AUDIO_MODE, sizeof(*audio), audio);

	audio->avport             = (uint8_t)avport;
	audio->mask               = 0x0FFF; /* XXX set all */
	audio->audio_num_of_ch    = ch;
	audio->audio_fs           = fs;
	audio->audio_word_bits    = word_bits;
	audio->audio_format       = format;
	audio->audio_source       = source;

	switch (ch) {
	case PS3AV_CMD_AUDIO_NUM_OF_CH_8:
		audio->audio_enable[3] = 1;
		/* fall through */
	case PS3AV_CMD_AUDIO_NUM_OF_CH_6:
		audio->audio_enable[2] = 1;
		audio->audio_enable[1] = 1;
		/* fall through */
	case PS3AV_CMD_AUDIO_NUM_OF_CH_2:
	default:
		audio->audio_enable[0] = 1;
	}

	/* audio swap L/R */
	for (i = 0; i < 4; i++) {
		audio->audio_swap[i] = PS3AV_CMD_AUDIO_SWAP_0; /* no swap */
	}

	/* audio serial input mapping */
	audio->audio_map[0] = PS3AV_CMD_AUDIO_MAP_OUTPUT_0;
	audio->audio_map[1] = PS3AV_CMD_AUDIO_MAP_OUTPUT_1;
	audio->audio_map[2] = PS3AV_CMD_AUDIO_MAP_OUTPUT_2;
	audio->audio_map[3] = PS3AV_CMD_AUDIO_MAP_OUTPUT_3;

	/* audio speaker layout */
	if (avport == PS3AV_CMD_AVPORT_HDMI_0 ||
	    avport == PS3AV_CMD_AVPORT_HDMI_1) {
		switch (ch) {
		case PS3AV_CMD_AUDIO_NUM_OF_CH_8:
			audio->audio_layout = PS3AV_CMD_AUDIO_LAYOUT_8CH;
			break;
		case PS3AV_CMD_AUDIO_NUM_OF_CH_6:
			audio->audio_layout = PS3AV_CMD_AUDIO_LAYOUT_6CH;
			break;
		case PS3AV_CMD_AUDIO_NUM_OF_CH_2:
		default:
			audio->audio_layout = PS3AV_CMD_AUDIO_LAYOUT_2CH;
			break;
		}
	} else {
		audio->audio_layout = PS3AV_CMD_AUDIO_LAYOUT_2CH; 
	}

	/* audio downmix permission */
	audio->audio_downmix = PS3AV_CMD_AUDIO_DOWNMIX_PERMITTED;
	/* audio downmix level shift (0:0dB to 15:15dB) */
	audio->audio_downmix_level = 0; /* 0dB */

	/* set ch status */
	for (i = 0; i < 8; i++) {
		audio->audio_cs_info[i] = mode_cs_info[i];
	}
	switch(fs) {
	case PS3AV_CMD_AUDIO_FS_44K:
		audio->audio_cs_info[3] &= ~CS_MASK; 
		audio->audio_cs_info[3] |= CS_44;
		break;
	case PS3AV_CMD_AUDIO_FS_88K:
		audio->audio_cs_info[3] &= ~CS_MASK; 
		audio->audio_cs_info[3] |= CS_88;
		break;
	case PS3AV_CMD_AUDIO_FS_96K:
		audio->audio_cs_info[3] &= ~CS_MASK; 
		audio->audio_cs_info[3] |= CS_96;
		break;
	case PS3AV_CMD_AUDIO_FS_176K:
		audio->audio_cs_info[3] &= ~CS_MASK; 
		audio->audio_cs_info[3] |= CS_176;
		break;
	case PS3AV_CMD_AUDIO_FS_192K:
		audio->audio_cs_info[3] &= ~CS_MASK; 
		audio->audio_cs_info[3] |= CS_192;
		break;
	default:
		break;
	}

	/* pass through setting */
	if (spdif_through && 
		(avport == PS3AV_CMD_AVPORT_SPDIF_0 
			|| avport == PS3AV_CMD_AVPORT_SPDIF_1)) {
		audio->audio_word_bits = PS3AV_CMD_AUDIO_WORD_BITS_16;
		audio->audio_source = PS3AV_CMD_AUDIO_SOURCE_SPDIF;
		if (spdif_bitstream) {
			audio->audio_format =
					PS3AV_CMD_AUDIO_FORMAT_BITSTREAM;
			audio->audio_cs_info[0] |= CS_BIT; 
		}
	}

	return p + sizeof(*audio);
}

int ps3av_cmd_audio_mode(struct ps3av_pkt_audio_mode *audio_mode)
{
	int res = -1;

	if ((res = ps3av_do_pkt(PS3AV_CID_AUDIO_MODE,
				sizeof(*audio_mode),
				sizeof(*audio_mode),
				(uint32_t *)audio_mode)) < 0) {
		goto err;
	} else {
		if ((res = get_status(audio_mode)) != 0) {
			printk("PS3AV_CID_AUDIO_MODE:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_audio_mute(int num_of_port, uint32_t *port, uint32_t mute)
{
	int i, send_len, res = -1;
	struct ps3av_pkt_audio_mute audio_mute;

	if (num_of_port > PS3AV_OPT_PORT_MAX) goto err;

	/* audio mute */
	memset((uint8_t *)&audio_mute, 0, sizeof(audio_mute));
	for (i = 0; i < num_of_port; i++) {
		audio_mute.mute[i].avport = port[i];
		audio_mute.mute[i].mute = mute;
	}

	send_len = PS3AV_HDR_SIZE + sizeof(audio_mute.cid) +
			sizeof(struct ps3av_audio_mute) * num_of_port;
	if ((res = ps3av_do_pkt(PS3AV_CID_AUDIO_MUTE,
				send_len,
				sizeof(audio_mute),
				(uint32_t *)&audio_mute)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&audio_mute)) != 0) {
			printk("PS3AV_CID_AUDIO_MUTE:failed status %x\n", 
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_audio_active(int active, uint32_t port)
{
	int res = -1;
	struct ps3av_pkt_audio_active audio_active;
	uint32_t cid;

	/* audio active */
	memset((uint8_t *)&audio_active, 0, sizeof(audio_active));
	audio_active.audio_port = port;
	cid = active ? PS3AV_CID_AUDIO_ACTIVE : PS3AV_CID_AUDIO_INACTIVE;

	if ((res = ps3av_do_pkt(cid,
				sizeof(audio_active),
				sizeof(audio_active),
				(uint32_t *)&audio_active)) < 0) {
		goto err;
	} else {
		if ((res = get_status(&audio_active)) != 0) {
			printk("PS3AV_CID_AUDIO_ACTIVE:%x failed status %x\n", cid,
				res);
			goto err;
		}
	}

	return 0;
err:
	return res;
}

int ps3av_cmd_avb_param(struct ps3av_pkt_avb_param *avb, uint32_t send_len)
{
	int res = -1;

	/* avb packet */
	ps3fb_flip_ctl(0); /* flip off */
	if ((res = ps3av_do_pkt(PS3AV_CID_AVB_PARAM,
				send_len,
				sizeof(*avb),
				(uint32_t *)avb)) < 0) {
		goto err;
	} else {
		if ((res = get_status(avb)) != 0) {
			DPRINTK("PS3AV_CID_AVB_PARAM:failed status %x\n", 
				res);
			goto err;
		}
	}

	ps3fb_flip_ctl(1); /* flip on */ 
	return 0;
err:
	ps3fb_flip_ctl(1); /* flip on */
	return res;
}

int ps3av_cmd_av_get_hw_conf(struct ps3av_pkt_av_get_hw_conf *hw_conf)
{
	int res = -1;
	struct ps3av_pkt_av_get_hw_conf *p = hw_conf;

	memset((uint8_t *)p, 0, sizeof(*p));

	if ((res = ps3av_do_pkt(PS3AV_CID_AV_GET_HW_CONF,
				PS3AV_HDR_SIZE + sizeof(p->cid),
				sizeof(*p),
				(uint32_t *)p)) < 0) {
		goto err;
	} else {
		if ((res = get_status(p)) != 0) {
			printk("PS3AV_CID_AV_GET_HW_CONF:failed status %x\n", 
				res);
			goto err;
		}
	}
	return 0;
err:
	return res;
}

int ps3av_cmd_av_hw_conf_dump(struct ps3av_pkt_av_get_hw_conf *hw_conf)
{
	struct ps3av_pkt_av_get_hw_conf *p = hw_conf;

	printk("av_h_conf:num of hdmi:%d\n", p->num_of_hdmi);
	printk("av_h_conf:num of avmulti:%d\n", p->num_of_avmulti);
	printk("av_h_conf:num of spdif:%d\n", p->num_of_spdif);

	return 0;
}

int ps3av_cmd_video_get_monitor_info(struct ps3av_pkt_av_get_monitor_info 
					*monitor_info, uint32_t avport)
{
	int res = -1;
	struct ps3av_pkt_av_get_monitor_info *p = monitor_info;

	memset((uint8_t *)p, 0, sizeof(*p));
	p->avport = avport;
	if ((res = ps3av_do_pkt(PS3AV_CID_AV_GET_MONITOR_INFO,
				PS3AV_HDR_SIZE + sizeof(p->cid) + 
				sizeof(p->avport) + sizeof(p->reserved),
				sizeof(*p),
				(uint32_t *)p)) < 0) {
		goto err;
	} else {
		if ((res = get_status(p)) != 0) {
			printk("PS3AV_CID_AV_GET_MONITOR_INFO:failed status %x\n", 
				res);
			goto err;
		}
	}
	return 0;
err:
	return res;
}

int ps3av_cmd_av_monitor_info_dump(struct ps3av_pkt_av_get_monitor_info *monitor_info)
{
	struct ps3av_pkt_av_get_monitor_info *p = monitor_info;
	struct ps3av_info_monitor *info;
	struct ps3av_info_audio *audio;
	int i;

	info = (struct ps3av_info_monitor *)(p->info);
	audio = (struct ps3av_info_audio *)(info + 1);

	printk("Monitor Info: size%d\n", p->size);

	printk("avport:%02x\n", info->avport);
	printk("monitor_id:");
	for (i = 0; i < 10; i++) printk("%02x ", info->monitor_id[i]);
	printk("\nmonitor_type:%02x\n", info->monitor_type);
	printk("monitor_name:");
	for (i = 0; i < 16; i++) printk("%c", info->monitor_name[i]);

	/* resolution */
	printk("\nresolution_60: bits:%08x native:%08x\n", 
		info->res_60.res_bits, info->res_60.native);
	printk("resolution_50: bits:%08x native:%08x\n", 
		info->res_50.res_bits, info->res_50.native);
	printk("resolution_other: bits:%08x native:%08x\n", 
		info->res_other.res_bits, info->res_other.native);
	printk("resolution_vesa: bits:%08x native:%08x\n", 
		info->res_vesa.res_bits, info->res_vesa.native);

	/* color space */
	printk("color space    rgb:%02x\n", info->cs.rgb);
	printk("color space yuv444:%02x\n", info->cs.yuv444);
	printk("color space yuv422:%02x\n", info->cs.yuv422);

	/* color info */
	printk("color info   red:X %04x Y %04x\n", 
			info->color.red_x, info->color.red_y);
	printk("color info green:X %04x Y %04x\n", 
			info->color.green_x, info->color.green_y);
	printk("color info  blue:X %04x Y %04x\n", 
			info->color.blue_x, info->color.blue_y);
	printk("color info white:X %04x Y %04x\n", 
			info->color.white_x, info->color.white_y);
	printk("color info gamma: %08x\n", info->color.gamma);

	/* other info */
	printk("supported_AI:%02x\n", info->supported_ai);
	printk("speaker_info:%02x\n", info->speaker_info);
	printk("num of audio:%02x\n", info->num_of_audio_block);

	/* audio block */
	for (i = 0;i < info->num_of_audio_block; i++)
	{
		printk("audio[%d] type:%02x max_ch:%02x fs:%02x sbit:%02x\n",
			i, audio->type, audio->max_num_of_ch,
			audio->fs, audio->sbit);
		audio++;
	}

	return 0;
}

#define PS3AV_AV_LAYOUT_0 (PS3AV_CMD_AV_LAYOUT_32 \
		| PS3AV_CMD_AV_LAYOUT_44 \
		| PS3AV_CMD_AV_LAYOUT_48)

#define PS3AV_AV_LAYOUT_1 (PS3AV_AV_LAYOUT_0 \
		| PS3AV_CMD_AV_LAYOUT_88 \
		| PS3AV_CMD_AV_LAYOUT_96 \
		| PS3AV_CMD_AV_LAYOUT_176 \
		| PS3AV_CMD_AV_LAYOUT_192)

static int ps3av_audio_output[] =
	{ 0, 
	  PS3AV_AV_LAYOUT_0,	/* PS3AV_CMD_VIDEO_VID_480I */
	  PS3AV_AV_LAYOUT_0,
	  PS3AV_AV_LAYOUT_0,
	  PS3AV_AV_LAYOUT_0,
	  PS3AV_AV_LAYOUT_0,
	  PS3AV_AV_LAYOUT_0,
	  PS3AV_AV_LAYOUT_1,	/* PS3AV_CMD_VIDEO_VID_1080P_60HZ */
	  PS3AV_AV_LAYOUT_1,
	  PS3AV_AV_LAYOUT_1,
	  PS3AV_AV_LAYOUT_1,
	  PS3AV_AV_LAYOUT_1,
	  PS3AV_AV_LAYOUT_1,
	  0,			/* PS3AV_CMD_VIDEO_VID_WXGA */
	  0,
	  0
};

int ps3av_cmd_av_get_audio_output(int in)
{
	/* input video_vid */
	int out = 0;

	if (sizeof(ps3av_audio_output)/sizeof(*ps3av_audio_output) < in) {
		out = 0;
	} else {
		out = ps3av_audio_output[in];
	}
	return out;
}


/************************* vuart ***************************/
extern unsigned long p_to_lp(long pa);

#define POLLING_INTERVAL  25 // in msec

int ps3av_vuart_init(struct vuart *vuart, unsigned long port, int bufsize)
{
	unsigned long status;
	unsigned long value;
	void *addr;

	DPRINTK("vuart_init...\n");

	memset(vuart, 0, sizeof(struct vuart));
	vuart->port = port;
	vuart->bufsize = bufsize;
	vuart->page_shift = 12;

	status = lv1_get_virtual_uart_param(vuart->port, 0, &value);
	if( status != 0 ) {
		DPRINTK("%s: lv1_get_virtual_uart_param(port=%ld): "
		       "failed (status=%ld)\n",
		       __FUNCTION__, vuart->port, status);
		return -EIO;
	}
	addr = (void *)kmalloc(vuart->bufsize, GFP_KERNEL);
	if (addr == NULL) {
		DPRINTK("kmalloc failed \n");
		return -ENOMEM;
	}
	vuart->lpar_addr = p_to_lp(__pa(addr));
	vuart->buf = addr;

	return 0;
}

int ps3av_vuart_cleanup(struct vuart *vuart)
{
	DPRINTK("vuart_cleanup...\n");
	if (vuart->buf) {
		kfree(vuart->buf);
	}

	return 0;
}

int ps3av_vuart_write(struct vuart *vuart, void *buf, unsigned long size)
{
	unsigned long status;
	unsigned long length;

	if (vuart->bufsize < size) {
		DPRINTK("%s: truncate size, %ld -> %ld\n", __FUNCTION__,
			size, vuart->bufsize);
		size = vuart->bufsize;
	}
	
	memset(vuart->buf, 0, vuart->bufsize); // failsafe
	memcpy(vuart->buf, buf, size);

	status = lv1_write_virtual_uart(vuart->port,
					vuart->lpar_addr,
					size,
					&length);
	if (status) {
		printk("%s: lv1_write_virtual_uart(port=%ld, size=%ld): "
		       "failed (status=%ld)\n",
		       __FUNCTION__, vuart->port, size, status);
		return -EIO;
	}

	return (int)length;
}

int ps3av_vuart_read(struct vuart *vuart, void *buf, 
					unsigned long size, int timeout)
{
	unsigned long status;
	unsigned long length;
	int loopcnt;

	if (vuart->bufsize < size) {
		DPRINTK("%s: truncate size, %ld -> %ld\n", __FUNCTION__,
			size, vuart->bufsize);
		size = vuart->bufsize;
	}

	memset(vuart->buf, 0, vuart->bufsize); // failsafe

	length = 0;
	loopcnt = 0;

	timeout = (timeout + POLLING_INTERVAL - 1) / POLLING_INTERVAL;
	while (loopcnt++ <= timeout) {
		status = lv1_read_virtual_uart(vuart->port, vuart->lpar_addr,
					       size, &length);
		if (status) {
			printk("%s: lv1_read_virtual_uart(port=%ld): "
			       "failed (status=%ld)\n",
			       __FUNCTION__, vuart->port, status);
			return -EIO;
		}
		if (0 < length) {
			break;
		}
		msleep(POLLING_INTERVAL);
	}

	if (length == 0) {
		DPRINTK("vuart_read(): TIMEOUT\n");
		return -EWOULDBLOCK;
	}

	memcpy(buf, vuart->buf, length);

	return (int)length;
}

