/*
 * sysmgr.c
 *
 *  communicate with sysmgr at PME
 *
 * Copyright (C) 2006 Sony Computer Entertainment Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/stddef.h>
#include <linux/string.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>

#include <asm/atomic.h>
#include <asm/io.h>
#include <asm/lv1call.h>

#include "vuart.h"

extern unsigned long p_to_lp(long pa);


struct sysmgr_header {
	u8	version;
	u8	size;
	u8	reserved1;
	u8	reserved2;
	u32	payload_size;
	u16	service_id;
	u16	reserved3;
	u32	request_tag;
};
#define SYSMGR_SERVICE_ID_REQUEST			1
#define SYSMGR_SERVICE_ID_RESPONSE			2
#define SYSMGR_SERVICE_ID_COMMAND			3
#define SYSMGR_SERVICE_ID_EXTERNAL_EVENT		4
#define SYSMGR_SERVICE_ID_SET_NEXT_OPERATION		5
#define SYSMGR_SERVICE_ID_SET_ATTRIBUTE			8
#define SYSMGR_SERVICE_ID_REQUEST_PARAMETER		9
#define SYSMGR_SERVICE_ID_DELIVER_INTER_LPER_PARAMETER	10
#define SYSMGR_SERVICE_ID_DELIVER_BOOT_PARAMETER	11
#define SYSMGR_SERVICE_ID_REQUEST_TEMPERATURE		13
#define SYSMGR_SERVICE_ID_DELIVER_TEMPERATURE		14
#define SYSMGR_SERVICE_ID_REQUEST_TZONE_PRESENCE	15
#define SYSMGR_SERVICE_ID_DELIVER_TZONE_PRESENCE	16
#define SYSMGR_SERVICE_ID_REQUEST_LED_STATUS		17
#define SYSMGR_SERVICE_ID_DELIVER_LED_STATUS		18
#define SYSMGR_SERVICE_ID_END				19

#define SYSMGR_GOS_ID_NONE		0x00
#define SYSMGR_GOS_ID_LINUX		0x03


struct sysmgr_request {
	u8	version;
	u8	type;
	u8	id;
	u8	reserved03;
	u32	reserved04;
	u64	reserved08;
};
#define SYSMGR_REQUEST_SHUTDOWN		0x01
#define SYSMGR_REQUEST_ID_MYSELF	0x00

struct sysmgr_response {
	u8	version;
	u8	reserved01;
	u16	reserved02;
	u8	status;
	u8	reserved05;
	u16	reserved06;
	u64	reserved08;
};
#define SYSMGR_RESPONSE_ACK		0x00
#define SYSMGR_RESPONSE_NAK		0xff


struct sysmgr_command {
	u8	version;
	u8	type;
	u16	reserved02;
	u32	reserved04;
	u64	reserved08;
};
#define SYSMGR_COMMAND_SHUTDOWN		0x01


struct sysmgr_external_event {
	u8	version;
	u8	type;
	u16	reserved02;
	u32	parameter1;
	u64	reserved08;
};
#define SYSMGR_EVENT_S				0x01
#define SYSMGR_EVENT_L				0x02
#define SYSMGR_EVENT_POWER_PRESSED		0x03
#define SYSMGR_EVENT_POWER_RELEASED		0x04
#define SYSMGR_EVENT_RESET_PRESSED		0x05
#define SYSMGR_EVENT_RESET_RELEASED		0x06
#define SYSMGR_EVENT_THERMAL_ALERT		0x07
#define SYSMGR_EVENT_THERMAL_ALERT_CLEARED	0x08
#define SYSMGR_EVENT_SIRCS_PRESSED		0x09
#define SYSMGR_EVENT_SIRCS_RELEASED		0x0a


struct sysmgr_set_next_operation {
	u8	version;
	u8	type;
	u8	gos_id;
	u8	reserved03;
	u32	wake_source;
	u64	reserved08;
};
#define SYSMGR_NEXT_OPERATION_SHUTDOWN	0x01
#define SYSMGR_NEXT_OPERATION_REBOOT	0x02
#define SYSMGR_NEXT_OPERATION_LPAR_BOOT	0x82
#define SYSMGR_WAKE_SOURCE_SYSTEM_DEFAULT	0x00000000
#define SYSMGR_WAKE_SOURCE_POWER_BUTTON		0x00000004
#define SYSMGR_WAKE_SOURCE_RESET_BUTTON		0x00000008
#define SYSMGR_WAKE_SOURCE_EJECT_BUTTON		0x00000010
#define SYSMGR_WAKE_SOURCE_DISC_LOADED		0x00000020
#define SYSMGR_WAKE_SOURCE_RTC_ALARM		0x00000040
#define SYSMGR_WAKE_SOURCE_RTC_ALARM_ERROR	0x00000080
#define SYSMGR_WAKE_SOURCE_SIRCS		0x00000100
#define SYSMGR_WAKE_SOURCE_CONTROLER		0x00000200
#define SYSMGR_WAKE_SOURCE_LAN			0x00000400
#define SYSMGR_WAKE_SOURCE_USB			0x00000800
#define SYSMGR_WAKE_SOURCE_PCI			0x00001000
#define SYSMGR_WAKE_SOURCE_HDMI_CEC		0x00002000
#define SYSMGR_WAKE_SOURCE_POR			0x10000000


struct sysmgr_set_attribute {
	u8	version;
	u8	reserved01;
	u16	reserved02;
	u32	attribute;
};
#define SYSMGR_ATTRIBUTE_EVENT_POWER		0x00000001
#define SYSMGR_ATTRIBUTE_EVENT_RESET		0x00000002
#define SYSMGR_ATTRIBUTE_EVENT_THERMAL		0x00000004
#define SYSMGR_ATTRIBUTE_EVENT_SIRCS		0x00000008


struct sysmgr_request_parameter {
	u8	version;
	u8	type;
	u16	reserved02;
	u32	reserved04;
};
#define SYSMGR_REQUEST_PARAMETER_INTER_LPAR	1
#define SYSMGR_REQUEST_PARAMETER_BOOT		2

struct sysmgr_inter_lpar_parameter {
	u8	version;
	u8	reserved01;
	u16	param_length;
	u32	reserved04;
	char	parameter[1];
};

struct sysmgr_boot_parameter {
	u8	version;
	u8	gos_id;
	u8	status;
	u8	reserved1;
	u32	wake_source;
	u64	parameter;
};
#define SYSMGR_BOOT_STATUS_OK		0x00


struct sysmgr_request_temperature {
	u8	version;
	u8	tzone_id;
	u16	reserved02;
	u32	reserved04;
};

struct sysmgr_deliver_temperature {
	u8	version;
	u8	tzone_id;
	u8	status;
	u8	reserved03;
	u8	temperature_i;
	u8	temperature_d;
	u16	reserved06;
};
#define SYSMGR_TEMP_STATUS_OK			0x00
#define SYSMGR_TEMP_STATUS_BUSY			0x01
#define SYSMGR_TEMP_STATUS_INVALID_COMMAND	0x02
#define SYSMGR_TEMP_STATUS_INVALID_ZONE_ID	0x03
#define SYSMGR_TEMP_STATUS_DEVICE_ERROR		0x04

struct sysmgr_request_tzone_presence {
	u8	version;
	u8	reserved01;
	u16	reserved02;
	u32	reserved04;
};

struct sysmgr_deliver_tzone_presence {
	u8	version;
	u8	reserved01;
	u16	reserved02;
	u32	reserved04;
	u32	bitmap[8];
};


struct sysmgr_request_led_status {
	u8	version;
	u8	reserved01;
	u16	reserved02;
	u32	reserved04;
};

struct sysmgr_deliver_led_status {
	u8	version;
	u8	power;
	u8	status;
	u8	disc;
	u32	reserved04;
};



typedef void (*sysmgr_recv_func_t)(struct sysmgr_header *header, void *data);


static int set_next_op(int op);
static void sysmgr_shutdown_start(void);
static void sysmgr_reboot_start(void);
void sysmgr_shutdown_ready(void);
void sysmgr_reboot_ready(void);
void sysmgr_spin_forever(void);


static unsigned int sysmgr_port = 2;

static atomic_t sysmgr_shutdown_command_recved = ATOMIC_INIT(0);
static atomic_t sysmgr_shutdown_readied = ATOMIC_INIT(0);

static spinlock_t state_lock;
static int next_state = 0;

static int sysmgr_send(void *buf, unsigned long size)
{
	unsigned long st;
	unsigned long length;

	st = lv1_write_virtual_uart(sysmgr_port,
				    p_to_lp(virt_to_phys(buf)),
				    size,
				    &length);
	if (st != 0) {
		printk(KERN_ERR "SYSMGR: SEND ERROR: st=%ld\n", st);
		return 0;
	} else if (length != size) {
		printk(KERN_ERR
		       "SYSMGR: SEND ERROR: expect size = %ld, length = %ld\n",
		       size, length);
		return 0;
	}

	return 1;
}

static int sysmgr_recv(void *buf, unsigned long size)
{
	unsigned long st;
	unsigned long length;

	do {
		st = lv1_read_virtual_uart(sysmgr_port,
					   p_to_lp(virt_to_phys(buf)),
					   size,
					   &length);
	} while (st == 0 && length == 0);

	if (st != 0) {
		printk(KERN_ERR "SYSMGR: RECV ERROR: st=%ld\n", st);
		return 0;
	} else if (length != size) {
		printk(KERN_ERR
		       "SYSMGR: RECV ERROR: expect size = %ld, length = %ld\n",
		       size, length);
		return 0;
	}

	return 1;
}

int sysmgr_send_request(unsigned int type)
{
	static struct sysmgr_packet {
		struct sysmgr_header header;
		struct sysmgr_request request;
	} sysmgr_packet;

	sysmgr_packet.header.version = 1;
	sysmgr_packet.header.size = 16;
	sysmgr_packet.header.payload_size =
		sizeof(struct sysmgr_request);
	sysmgr_packet.header.service_id = SYSMGR_SERVICE_ID_REQUEST;
	sysmgr_packet.request.version = 1;
	sysmgr_packet.request.type = type;
	sysmgr_packet.request.id = SYSMGR_REQUEST_ID_MYSELF;

	if (sysmgr_send(&sysmgr_packet,
			sysmgr_packet.header.size +
			sysmgr_packet.header.payload_size) == 0) {
		return -EPERM;
	}
	return 0;
}

int sysmgr_send_response(unsigned int stat)
{
	static struct sysmgr_packet {
		struct sysmgr_header header;
		struct sysmgr_response response;
	} sysmgr_packet;

	sysmgr_packet.header.version = 1;
	sysmgr_packet.header.size = 16;
	sysmgr_packet.header.payload_size =
		sizeof(struct sysmgr_response);
	sysmgr_packet.header.service_id = SYSMGR_SERVICE_ID_RESPONSE;
	sysmgr_packet.response.version = 1;
	sysmgr_packet.response.status = stat;

	if (sysmgr_send(&sysmgr_packet,
			sysmgr_packet.header.size +
			sysmgr_packet.header.payload_size) == 0) {
		return -EPERM;
	}
	return 0;
}

int sysmgr_send_set_next_operation(int type, int gos_id)
{
	static struct sysmgr_packet {
		struct sysmgr_header header;
		struct sysmgr_set_next_operation set_next_operation;
	} sysmgr_packet;

	sysmgr_packet.header.version = 1;
	sysmgr_packet.header.size = 16;
	sysmgr_packet.header.payload_size =
		sizeof(struct sysmgr_set_next_operation);
	sysmgr_packet.header.service_id = SYSMGR_SERVICE_ID_SET_NEXT_OPERATION;
	sysmgr_packet.set_next_operation.version = 2;
	sysmgr_packet.set_next_operation.type = type;
	sysmgr_packet.set_next_operation.gos_id = gos_id;
	sysmgr_packet.set_next_operation.wake_source =
		SYSMGR_WAKE_SOURCE_SYSTEM_DEFAULT;

	if (sysmgr_send(&sysmgr_packet,
			sysmgr_packet.header.size +
			sysmgr_packet.header.payload_size) == 0) {
		return -EPERM;
	}
	return 0;
}

int sysmgr_send_set_attribute(unsigned int attribute)
{
	static struct sysmgr_packet {
		struct sysmgr_header header;
		struct sysmgr_set_attribute set_attribute;
	} sysmgr_packet;

	sysmgr_packet.header.version = 1;
	sysmgr_packet.header.size = 16;
	sysmgr_packet.header.payload_size =
		sizeof(struct sysmgr_set_attribute);
	sysmgr_packet.header.service_id = SYSMGR_SERVICE_ID_SET_ATTRIBUTE;
	sysmgr_packet.set_attribute.version = 1;
	sysmgr_packet.set_attribute.attribute = attribute;

	if (sysmgr_send(&sysmgr_packet,
			sysmgr_packet.header.size +
			sysmgr_packet.header.payload_size) == 0) {
		return -EPERM;
	}
	return 0;
}

int sysmgr_send_request_parameter(unsigned int type)
{
	static struct sysmgr_packet {
		struct sysmgr_header header;
		struct sysmgr_request_parameter request_parameter;
	} sysmgr_packet;

	sysmgr_packet.header.version = 1;
	sysmgr_packet.header.size = 16;
	sysmgr_packet.header.payload_size =
		sizeof(struct sysmgr_request_parameter);
	sysmgr_packet.header.service_id = SYSMGR_SERVICE_ID_REQUEST_PARAMETER;
	sysmgr_packet.request_parameter.version = 1;
	sysmgr_packet.request_parameter.type = type;

	if (sysmgr_send(&sysmgr_packet,
			sysmgr_packet.header.size +
			sysmgr_packet.header.payload_size) == 0) {
		return -EPERM;
	}
	return 0;
}

static void sysmgr_recv_command(struct sysmgr_header *header,
				void *data)
{
	struct sysmgr_command *cmd = data;
	extern void ctrl_alt_del(void);

	printk(KERN_INFO "SYSMGR: recv: command(%d:%s)\n",
	       cmd->type,
	       (cmd->type == SYSMGR_COMMAND_SHUTDOWN ?
		"shutdown": "unknown"));

	if (atomic_read(&sysmgr_shutdown_readied) == 0) {
		set_next_op(SYSMGR_NEXT_OPERATION_LPAR_BOOT);

		/* notify to userland as ctrl-alt-del event */
		ctrl_alt_del();
	}

	atomic_inc(&sysmgr_shutdown_command_recved);
}

static void sysmgr_recv_external_event(struct sysmgr_header *header,
				       void *data)
{
	struct sysmgr_external_event *event = data;

	printk(KERN_INFO "SYSMGR: recv: event(type:%d parameter:%d)\n",
	       event->type, event->parameter1);

	if (event->type == SYSMGR_EVENT_S || event->type == SYSMGR_EVENT_L ||
	    (event->type == SYSMGR_EVENT_POWER_PRESSED)) {
		printk(KERN_NOTICE "SYSMGR: starting shutdown sequence.\n");
		sysmgr_shutdown_start();
	} else if (event->type == SYSMGR_EVENT_RESET_PRESSED) {
		printk(KERN_NOTICE "SYSMGR: starting reboot sequence.\n");
		sysmgr_reboot_start();
	}
}

static void sysmgr_recv_inter_lpar_parameter(struct sysmgr_header *header,
					     void *data)
{
	struct sysmgr_inter_lpar_parameter *param = data;

	printk(KERN_INFO "SYSMGR: recv: inter_lpar_parameter(size=%d)\n",
	       param->param_length);

	if (header->payload_size < 8 ||
	    header->payload_size != 8 + param->param_length) {
		printk(KERN_ERR
		       "SYSMGR: ERROR: inter_lpar_parameter size mismatch\n");
		return;
	}

	/* ignore */
}

static void sysmgr_recv_boot_parameter(struct sysmgr_header *header,
				       void *data)
{
	struct sysmgr_boot_parameter *param = data;

	printk(KERN_INFO "SYSMGR: recv: boot_parameter\n");

	if (header->payload_size != sizeof(*param)) {
		printk(KERN_ERR
		       "SYSMGR: ERROR: boot_parameter size mismatch\n");
		return;
	}

	/* ignore */
}

static sysmgr_recv_func_t sysmgr_recv_funcs[SYSMGR_SERVICE_ID_END] =
{
	[SYSMGR_SERVICE_ID_COMMAND]
		= sysmgr_recv_command,
	[SYSMGR_SERVICE_ID_EXTERNAL_EVENT]
		= sysmgr_recv_external_event,
	[SYSMGR_SERVICE_ID_DELIVER_INTER_LPER_PARAMETER]
		= sysmgr_recv_inter_lpar_parameter,
	[SYSMGR_SERVICE_ID_DELIVER_BOOT_PARAMETER]
		= sysmgr_recv_boot_parameter,
};

static void sysmgr_recv_packet(void)
{
	struct sysmgr_header header;
	unsigned long st, value;
	char buf[1024];

	st = lv1_get_virtual_uart_param(
		sysmgr_port, VUART_PARAM_RECV_BUF_SIZE, &value);
	if (st != 0) {
		printk(KERN_ERR "SYSMGR: RECV: Failed to get param.\n");
		return;
	}
	if (value < sizeof(header)) {
		/* phantom interrupt: ingnored */
		return;
	}

	if (sysmgr_recv(&header, sizeof(header)) == 0) {
		printk(KERN_ERR "SYSMGR: RECV: Failed to recv header.\n");
		return;
	}

	if (header.payload_size > sizeof(buf)) {
		BUG();
	}
	if (sysmgr_recv(buf, header.payload_size) == 0) {
		printk(KERN_ERR "SYSMGR: RECV: Failed to recv payload.\n");
		return;
	}

	if (header.version != 0x01) {
		printk(KERN_ERR "SYSMGR: RECV: unknown header\n");
		return;
	}
	if (header.service_id < SYSMGR_SERVICE_ID_END &&
	    sysmgr_recv_funcs[header.service_id]) {
		sysmgr_recv_funcs[header.service_id](&header, buf);
	}
}

static int set_next_op(int op)
{
	long flag;

	spin_lock_irqsave(&state_lock, flag);
	if (next_state != 0) {
		spin_unlock_irqrestore(&state_lock, flag);
		return next_state;
	}
	next_state = op;
	spin_unlock_irqrestore(&state_lock, flag);

	if (op == SYSMGR_NEXT_OPERATION_LPAR_BOOT) {
		sysmgr_send_set_next_operation(op, SYSMGR_GOS_ID_LINUX);
	} else {
		sysmgr_send_set_next_operation(op, SYSMGR_GOS_ID_NONE);
	}
	return next_state;
}

static void sysmgr_shutdown_start(void)
{
	set_next_op(SYSMGR_NEXT_OPERATION_SHUTDOWN);

	if (atomic_read(&sysmgr_shutdown_command_recved) == 0) {
		printk(KERN_NOTICE "SYSMGR: sending shutdown request.\n");
		sysmgr_send_request(SYSMGR_REQUEST_SHUTDOWN);
	}
}

static void sysmgr_reboot_start(void)
{
	set_next_op(SYSMGR_NEXT_OPERATION_REBOOT);

	if (atomic_read(&sysmgr_shutdown_command_recved) == 0) {
		printk(KERN_NOTICE "SYSMGR: sending shutdown request.\n");
		sysmgr_send_request(SYSMGR_REQUEST_SHUTDOWN);
	}
}

static void sysmgr_wait_and_ack_shutdown(void)
{
	printk(KERN_NOTICE "SYSMGR: waiting for shutdown command.\n");
	while (atomic_read(&sysmgr_shutdown_command_recved) == 0) {
		/* NOTE: vuart-interrupt will be stopped.
		         we need to poll packets.*/
		sysmgr_recv_packet();
	}

	printk(KERN_NOTICE "SYSMGR: reply ACK for shutdown command.\n");
	sysmgr_send_response(SYSMGR_RESPONSE_ACK);

	printk(KERN_NOTICE "spinning forever...\n");
	for (;;) ;
}

void sysmgr_shutdown_ready(void)
{
	printk(KERN_NOTICE "SYSMGR: kernel is ready to shutdown.\n");

	atomic_inc(&sysmgr_shutdown_readied);
	sysmgr_shutdown_start();

	sysmgr_wait_and_ack_shutdown();
}

void sysmgr_reboot_ready(void)
{
	printk(KERN_NOTICE "SYSMGR: kernel is ready to reboot.\n");

	atomic_inc(&sysmgr_shutdown_readied);
	sysmgr_reboot_start();

	sysmgr_wait_and_ack_shutdown();
}

void sysmgr_spin_forever(void)
{
	printk("SYSMGR: kernel is spinning forever.\n");

	atomic_inc(&sysmgr_shutdown_readied);
	sysmgr_send_set_next_operation(
		SYSMGR_NEXT_OPERATION_SHUTDOWN,
		SYSMGR_GOS_ID_NONE);

	sysmgr_wait_and_ack_shutdown();
}

static int sysmgr_at_exit(struct notifier_block *self, unsigned long state,
			  void *data)
{
	printk(KERN_INFO "SYSMGR: cleanup.(close event mask)\n");
	sysmgr_send_set_attribute(0);
	return NOTIFY_OK;
}

static struct notifier_block sysmgr_reboot_nb = {
	.notifier_call = sysmgr_at_exit
};

int __init sysmgr_init(void)
{
	printk(KERN_INFO "SYSMGR: sysmgr_init() called\n");

	vuart_set_callback(sysmgr_port, sysmgr_recv_packet);
	vuart_set_recv_watermark(sysmgr_port, sizeof(struct sysmgr_header));
	vuart_set_irq_mask(sysmgr_port, VUART_IRQ_MASK_RECV);
	sysmgr_send_set_attribute(SYSMGR_ATTRIBUTE_EVENT_POWER |
				  SYSMGR_ATTRIBUTE_EVENT_RESET);

	register_reboot_notifier(&sysmgr_reboot_nb);

	return 0;
}
__initcall(sysmgr_init);
