/// debugostream - Version 1.0.3
/// license: Boost Software Linsence 1.0
/// author: takenoko
/// created: 2011/11/29
/// modified: 2012/02/18
/// 
/// = FORMAT =
/// %f filename(strip path)
/// %F filename
/// %l line number
/// %P process id
/// %T thread id
/// %d date
/// %t time
/// %% %
/// 
/// = SYNOPSIS =
///
/// Example.1
/// #include <debugostream.hpp>
/// int _tmain() {
///     DLOG << _T("HELLO WORLD");
///     DLOGA << "HELLO WORLD";
///     DLOGW << L"HELLO WORLD";
///     return 0;
/// }
///
/// Example.2
/// #include <debugostream.hpp>
/// int _tmain() {
///     DLOG_PREFIX("%f(%l) [");
///     DLOG_POSTFIX("]");
///     DLOG << _T("HELLO WORLD");
///     return 0;
/// }
///
/// Example.3
/// #include <debugostream.hpp>
/// int _tmain() {
///     int *p = NULL;
///     DLOG_IF(p == NULL) << _T("p is NULL");
///     return 0;
/// }
///
/// Example.4
/// #include <debugostream.hpp>
/// int _tmain() {
///     int i = 12345;
///     DLOG_CHECK(i);
///     return 0;
/// }
#pragma once

#include <windows.h>
#include <tchar.h>
#include <process.h>
#include <iostream>
#include <cstdarg>

namespace debugostream {

	template <class writerT,class charT, class traits = std::char_traits<charT>>
	class generics_streambuf : public std::basic_streambuf<charT, traits> {
	private:
		enum { BUFFER_SIZE = 4096 };
		charT buffer[BUFFER_SIZE];
		writerT writer;

	private:
		int write(charT * beg,charT * end) {
			if (beg == end)
				return 0;
			*end = 0;
			return writer(beg,end);
		}

	public:
		generics_streambuf() {
			setp(buffer,(buffer + BUFFER_SIZE - 1) - 1);
		}

		generics_streambuf(writerT writer_) : writer(writer_) {
			setp(buffer,(buffer + BUFFER_SIZE - 1) - 1);
		}

		~generics_streambuf() {
			sync();
		}

	protected:
		virtual int_type overflow(int_type c = traits::eof()) {
			if (c == traits::eof())
				return static_cast<int_type>(sync());
			*pptr() = traits::to_char_type(c);
			pbump(1);
			return static_cast<int_type>(sync());
		}

		virtual int sync() {
			int ret = write(pbase(),pptr());
			setp(pbase(),epptr());
			return ret;
		}
	};

	template <class writerT,class charT, class traits = std::char_traits<charT> >
	class generics_ostream : public std::basic_ostream<charT, traits> {
	public:
		generics_ostream() : std::basic_ostream<charT, traits>(new generics_streambuf<writerT,charT, traits>()) {}
		generics_ostream(writerT f) : std::basic_ostream<charT, traits>( new generics_streambuf<writerT,charT, traits>(f) ) { }
		~generics_ostream() {
			delete rdbuf();}
	};

#pragma warning(push)
#pragma warning(disable:4100)
	class debuga_writer {
	public:
		int operator()(char *beg,char *end) {
			OutputDebugStringA(beg);
			return 0;
		}
	};

	class debugw_writer {
	public:
		int operator()(wchar_t *beg,wchar_t *end) {
			OutputDebugStringW(beg);
			return 0;
		}
	};
#pragma warning(pop)

	typedef generics_streambuf<debuga_writer,char> debuga_streambuf;
	typedef generics_ostream<debuga_writer,char> debuga_ostream;

	typedef generics_streambuf<debugw_writer,wchar_t> debugw_streambuf;
	typedef generics_ostream<debugw_writer,wchar_t> debugw_ostream;

	class critical_section {
	private:
		CRITICAL_SECTION cs;

	public:
		critical_section() {
			InitializeCriticalSection(&cs);
		}

		~critical_section() {
			DeleteCriticalSection(&cs);
		}

		void enter() {
			EnterCriticalSection(&cs);
		}

		void leave() {
			LeaveCriticalSection(&cs);
		}
	};

	class scoped_lock {
	private:
		critical_section *cs;
		scoped_lock();
		scoped_lock(const scoped_lock &c);
		scoped_lock &operator=(scoped_lock &rhs);

	public:
		scoped_lock(critical_section *c) : cs(c) {
			cs->enter();
		}

		~scoped_lock() {
			cs->leave();
		}
	};

	class time_string {
	private:
		char buffa[32];
		wchar_t buffw[32];
	public:
		operator char*() {
			SYSTEMTIME systime;
			GetLocalTime(&systime);
			wsprintfA(buffa,"%02d:%02d:%02d",systime.wHour,systime.wMinute,systime.wSecond);
			return buffa;
		}
		operator wchar_t*() {
			SYSTEMTIME systime;
			GetLocalTime(&systime);
			wsprintfW(buffw,L"%02d:%02d:%02d",systime.wHour,systime.wMinute,systime.wSecond);
			return buffw;
		}
	};

	class date_string {
	private:
		char buffa[32];
		wchar_t buffw[32];
	public:
		operator char*() {
			SYSTEMTIME systime;
			GetLocalTime(&systime);
			wsprintfA(buffa,"%4d/%02d/%02d",systime.wYear,systime.wMonth,systime.wDay);
			return buffa;
		}
		operator wchar_t*() {
			SYSTEMTIME systime;
			GetLocalTime(&systime);
			wsprintfW(buffw,L"%4d/%02d/%02d",systime.wYear,systime.wMonth,systime.wDay);
			return buffw;
		}
	};
	class error_message {
	private:
		char *buffa;
		wchar_t *buffw;
	public:
		error_message() :buffa(0),buffw(0) {


		}

		~error_message() {
			if (!buffa) {
				LocalFree(buffa);
			}
			if (!buffw) {
				LocalFree(buffw);
			}
		}
		operator char*() {
			FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER |
			FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_MAX_WIDTH_MASK,
				NULL,GetLastError(),
				MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
				(LPSTR) &buffa,0,NULL);
			return buffa;
		}
		operator wchar_t*() {
			FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER |
			FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_MAX_WIDTH_MASK,
				NULL,GetLastError(),
				MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
				(LPWSTR) &buffw,0,NULL);
			
			return buffw;
		}
	};

	class format {
	public:

		static void printf(char *format,...) {
			char buff[4096];
			va_list args;
			va_start(args,format);
			wvsprintfA(buff,format,args);
			OutputDebugStringA(buff);
			va_end(args);
		}

		static void printf(wchar_t *format,...) {
			wchar_t buff[4096];
			va_list args;
			va_start(args,format);
			wvsprintfW(buff,format,args);
			OutputDebugStringW(buff);
			va_end(args);
		}

		static int sprintf(char *dest,char *format,...) {
			va_list args;
			va_start(args,format);
			wvsprintfA(dest,format,args);
			va_end(args);
			return 0;
		}

		static int sprintf(wchar_t *dest,wchar_t *format,...) {
			va_list args;
			va_start(args,format);
			wvsprintfW(dest,format,args);
			va_end(args);
			return 0;
		}

		static char *findFilename(char *path) {
			char *last = path;
			char *p = path;
			for (;;) {
				if (*p == '\\') {
					last = p + 1;
				} else if (*p == 0) {
					break;
				}
				p = CharNextExA(CP_OEMCP,p,0);
			}
			return last;
		}

		static wchar_t *findFilename(wchar_t *path) {
			wchar_t *last = path;
			wchar_t *p = path;
			for (;;) {
				if (*p == L'\\') {
					last = p + 1;
				} else if (*p == 0) {
					break;
				}
				p ++;
			}
			return last;
		}

		template <typename ostreamT,typename charT>
		static int print(ostreamT &os,charT *format,charT *file,int line,charT *func) {
			int prechar = 0;
			for (;*format != 0;format ++) {
				if (prechar != (charT) '%') {
					if (*format != (charT) '%') {
						os << *format;
					}
				} else {
					if (*format == (charT) 'F') {
						os << file;
					} else if (*format == (charT) 'f') {
						os << findFilename(file);
					} else if (*format == (charT) 'l') {
						os << line;
					} else if (*format == (charT) 'm') {
						os << func;
					} else if (*format == (charT) 'd') {
						os << (charT *) date_string();
					} else if (*format == (charT) 't') {
						os << (charT *) time_string();
					} else if (*format == (charT) '%') {
						os << (charT) '%';
					} else if (*format == (charT) 'P') {
						os << GetCurrentProcessId();
					} else if (*format == (charT) 'T') {
						os << GetCurrentThreadId();
					} else {
						os << *format;
					}
				}
				prechar = *format;
			}
			return 0;
		}
	};

	class static_objects {
	public:
		enum buffsize_e { BUFFSIZE = 1024 };

		static debuga_ostream &astream() {
			static debuga_ostream a;
			return a;
		}

		static debugw_ostream &wstream() {
			static debugw_ostream w;
			return w;
		}

		static critical_section *csection() {
			static critical_section acs;
			return &acs;
		}

		static char *getPrefixa() {
			static char buff[BUFFSIZE] = "%f(%l): %m ";
			return buff;
		}

		static wchar_t *getPrefixw() {
			static wchar_t buff[BUFFSIZE] = L"%f(%l): %m ";
			return buff;
		}

		static char *getPostfixa() {
			static char buff[BUFFSIZE] = "";
			return buff;
		}

		static wchar_t *getPostfixw() {
			static wchar_t buff[BUFFSIZE] = L"";
			return buff;
		}
	};

	class config {
	public:
		static void setPrefix(char const * const prefix) {
			scoped_lock slock(static_objects::csection());
			char *prefixa = static_objects::getPrefixa();
			if (prefix == NULL)
				prefixa[0] = 0;
			else
				lstrcpynA(prefixa,prefix,static_objects::BUFFSIZE);

			wchar_t *prefixw = static_objects::getPrefixw();
			MultiByteToWideChar(CP_OEMCP,MB_COMPOSITE,prefixa,-1,prefixw,static_objects::BUFFSIZE);
		}

		static void setPostfix(char const * const postfix) {
			scoped_lock slock(static_objects::csection());
			char *postfixa = static_objects::getPostfixa();
			if (postfix == NULL)
				postfixa[0] = 0;
			else
				lstrcpynA(postfixa,postfix,static_objects::BUFFSIZE);

			wchar_t *postfixw = static_objects::getPostfixw();
			MultiByteToWideChar(CP_OEMCP,MB_COMPOSITE,postfixa,-1,postfixw,static_objects::BUFFSIZE);
		}

		static void setPrefix(wchar_t const * const prefix) {
			scoped_lock slock(static_objects::csection());
			wchar_t *prefixw = static_objects::getPrefixw();
			if (prefix == NULL)
				prefixw[0] = 0;
			else
				lstrcpynW(prefixw,prefix,static_objects::BUFFSIZE);
			char *prefixa = static_objects::getPrefixa();
			WideCharToMultiByte(CP_OEMCP,WC_COMPOSITECHECK,prefixw,-1,prefixa,static_objects::BUFFSIZE,NULL,NULL);
		}

		static void setPostfix(wchar_t const * const postfix) {
			scoped_lock slock(static_objects::csection());
			wchar_t *postfixw = static_objects::getPostfixw();
			if (postfix == NULL)
				postfixw[0] = 0;
			else
				lstrcpynW(postfixw,postfix,static_objects::BUFFSIZE);
			char *postfixa = static_objects::getPostfixa();
			WideCharToMultiByte(CP_OEMCP,WC_COMPOSITECHECK,postfixw,-1,postfixa,static_objects::BUFFSIZE,NULL,NULL);
		}

	};

	class loggera {
	private:
		scoped_lock slock;
		char *filename;
		int lineno;
		char *function;
		char *prefix;
		char *postfix;

	public:
		loggera(char *filename,int lineno,char *function) : slock(static_objects::csection()),filename(filename),lineno(lineno),function(function),prefix(static_objects::getPrefixa()),postfix(static_objects::getPostfixa()) {
			format::print<debuga_ostream,char>(static_objects::astream(),prefix,filename,lineno,function);
		}

		~loggera() {
			format::print<debuga_ostream,char>(static_objects::astream(),postfix,filename,lineno,function);
			static_objects::astream() << std::endl;
		}

		debuga_ostream &getStream() {
			return static_objects::astream();
		}
	};

	class loggerw {
	private:
		scoped_lock slock;
		wchar_t *filename;
		int lineno;
		wchar_t *function;
		wchar_t *prefix;
		wchar_t *postfix;

	public:
		loggerw(wchar_t *filename,int lineno,wchar_t *function) : slock(static_objects::csection()),filename(filename),lineno(lineno),function(function),prefix(static_objects::getPrefixw()),postfix(static_objects::getPostfixw()) {
			format::print<debugw_ostream,wchar_t>(static_objects::wstream(),prefix,filename,lineno,function);
		}

		~loggerw() {
			format::print<debugw_ostream,wchar_t>(static_objects::wstream(),postfix,filename,lineno,function);
			static_objects::wstream() << std::endl;
		}

		debugw_ostream &getStream() {
			return static_objects::wstream();
		}
	};
}

// macros
#define DLOG_PREFIX(prefix) debugostream::config::setPrefix(prefix)
#define DLOG_POSTFIX(postfix) debugostream::config::setPostfix(postfix)
#define DLOGA_ERROR_MESSAGE (char *) debugostream::error_message()
#define DLOGW_ERROR_MESSAGE (wchar_t *) debugostream::error_message()

#ifndef NDEBUG
#define DLOGA debugostream::loggera(__FILE__,__LINE__,__FUNCTION__).getStream()
#define DLOGW debugostream::loggerw(__FILEW__,__LINE__,__FUNCTIONW__).getStream()
#define DLOGA_IF(condition) !(condition) ? (void) 0 : DLOGA
#define DLOGW_IF(condition) !(condition) ? (void) 0 : DLOGW
#define DLOGA_CHECK(variable) DLOGA << #variable " = " << variable
#define DLOGW_CHECK(variable) DLOGW << L#variable L" = " << variable
#define DLOGA_CSTRING(variable) DLOGA << #variable " = [" << (LPCSTR) variable << "]"
#define DLOGW_CSTRING(variable) DLOGW << L#variable L" = [" << (LPCWSTR) variable << L"]"
#else
#define DLOGA true ? (void) 0 : debugostream::loggera(__FILE__,__LINE__,__FUNCTION__).getStream()
#define DLOGW true ? (void) 0 : debugostream::loggerw(__FILEW__,__LINE__,__FUNCTIONW__).getStream()
#define DLOGA_IF(condition) (true || (condition)) ? (void) 0 : DLOGA
#define DLOGW_IF(condition) (true || (condition)) ? (void) 0 : DLOGW
#define DLOGA_CHECK(variable) (true || (variable)) ? (void) 0 : DLOGA << variable
#define DLOGW_CHECK(variable) (true || (variable)) ? (void) 0 : DLOGW << variable
#define DLOGA_CSTRING(variable) (true || (variable)) ? (void) 0 : DLOGA << variable
#define DLOGW_CSTRING(variable) (true || (variable)) ? (void) 0 : DLOGW << variable
#endif

#ifdef _UNICODE
#define DLOG DLOGW
#define DLOG_IF DLOGW_IF
#define DLOG_CHECK DLOGW_CHECK
#define DLOG_CSTRING DLOGW_CSTRING
#define DLOG_ERROR_MESSAGE DLOGW_ERROR_MESSAGE
#else
#define DLOG DLOGA
#define DLOG_IF DLOGA_IF
#define DLOG_CHECK DLOGA_CHECK
#define DLOG_CSTRING DLOGA_CSTRING
#define DLOG_ERROR_MESSAGE DLOGA_ERROR_MESSAGE
#endif
