/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.unix.misc;

import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;

public class OptionIterator implements Iterator<Character> {

	private String pattern;
	private Iterator<String> wrapped;
	private String option;
	private int index;
	private int errchar = -1;
	private boolean usePlus;

	String nopt, now;
	private int ochar, ptr;
	private boolean oplus, plus;

	/**
	 * 
	 * @param pattern
	 * @param iterator
	 */
	public OptionIterator(String pattern, boolean usePlus,
			Iterator<String> iterator) {
		this.pattern = pattern;
		this.wrapped = iterator;
		this.usePlus = usePlus;

		if(_wnext() != null &&
				now.startsWith("-") &&
				!now.equals("-") &&
				!now.equals("--")) {
			ptr = 1;  plus = false;
			_next();
		} else if(usePlus && now != null && now.startsWith("+")) {
			ptr = 1;  plus = true;
			_next();
		} else if(now != null && now.equals("--")) {
			ochar = -1;
			_wnext();
		} else {
			index--;
			ochar = -1;
		}
	}

	/**
	 * 
	 * @param pattern
	 * @param x
	 */
	public OptionIterator(String pattern, boolean usePlus,
			Iterable<String> x) {
		this(pattern, usePlus, x.iterator());
	}

	/**
	 * 
	 * @param pattern
	 * @param args
	 */
	public OptionIterator(String pattern, boolean usePlus,
			String... args) {
		this(pattern, usePlus, Arrays.asList(args).iterator());
	}

	/**
	 * 
	 * @param pattern
	 * @param x
	 */
	public OptionIterator(String pattern, Iterable<String> x) {
		this(pattern, false, x.iterator());
	}

	/**
	 * 
	 * @param pattern
	 * @param args
	 */
	public OptionIterator(String pattern, String... args) {
		this(pattern, false, Arrays.asList(args).iterator());
	}

	private String _wnext() {
		index++;
		if(wrapped.hasNext()) {
			now = wrapped.next();
			if(now == null)  now = "";
		} else {
			now = null;
		}
		return now;
	}

	private boolean _option() {
		int i = pattern.indexOf(ochar);

		return ++i < pattern.length() && pattern.charAt(i) == ':';
	}

	void _next() {
		if(errchar >= 0)  return;
//		errchar = -1;
		if(now == null) {
			ochar = -1;
		} else if(ptr < now.length()) {
			ochar = now.charAt(ptr++);
			if(ochar == ':' || pattern.indexOf(ochar) < 0) {
				errchar = ochar;
				ochar = '?';
			} else if(!_option()) {
				nopt = null;
			} else if(ptr < now.length()) {
				nopt = now.substring(ptr);
				ptr  = now.length();
			} else if(_wnext() != null) {
				nopt = now;
				ptr  = now.length();
			} else {
				ochar = -1;
				index--;
			}
		} else if(!wrapped.hasNext()) {
			ochar = -1;
			now = null;
		} else if(_wnext().equals("--")) {
			ochar = -1;
			_wnext();
		} else if(now.equals("-")) {
			ochar = -1;
		} else if(now.startsWith("-")) {
			ptr = 1;  plus = false;
			_next();
		} else if(usePlus && now.startsWith("+")) {
			ptr = 1;  plus = true;
			_next();
		} else {
			ochar = -1;
			index--;
		}
	}

	/**
	 * 
	 * @return
	 */
	public String getArgument() {
		return option;
	}

	/**
	 * 
	 * @return
	 */
	public int getIndex() {
		return ochar < 0 ? index + 1 : index;
	}

	/**
	 * 
	 * @return
	 */
	public int getErrorOption() {
		return errchar;
	}

	/**
	 * 
	 * @return
	 */
	public Iterator<String> filenameIterator() {
		if(ochar >= 0)  ochar = -1;
		return new Iterator<String>() {

			public boolean hasNext() {
				return now != null;
			}

			public String next() {
				String x = now;

				if(now == null) {
					throw new NoSuchElementException();
				}
				_wnext();
				return x;
			}

			public void remove() {
				throw new UnsupportedOperationException();
			}

		};
	}

	public boolean hasNext() {
		return ochar >= 0;
	}

	/**
	 * 
	 * @return
	 */
	public char nextChar() {
		int x;

		x = ochar;  oplus = plus;  option = nopt;  nopt = null;
		skip();
		return (char)x;
	}

	/**
	 * 
	 */
	public void skip() {
		if(ochar < 0) {
			throw new NoSuchElementException();
		} else if(errchar > 0) {
			errchar = -1;
		}
		_next();
	}

	public Character next() {
		return Character.valueOf(nextChar());
	}

	public void remove() {
		throw new UnsupportedOperationException();
	}

	/**
	 * 
	 * @return
	 */
	public String getLookahead() {
		return now;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isPlus() {
		return oplus;
	}

}
