/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh.conf;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import net.morilib.sh.AbstractShEnvironment;
import net.morilib.sh.AbstractShFile;
import net.morilib.sh.AbstractShFileSystem;
import net.morilib.sh.DefaultShRuntime;
import net.morilib.sh.ShBuiltInCommands;
import net.morilib.sh.ShCommandNotFoundException;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShExitException;
import net.morilib.sh.ShFacade;
import net.morilib.sh.ShFile;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShLexer;
import net.morilib.sh.ShParser;
import net.morilib.sh.ShProcess;
import net.morilib.sh.ShPromptReader;
import net.morilib.sh.ShRealFile;
import net.morilib.sh.ShRootShellEnvironment;
import net.morilib.sh.ShRuntime;
import net.morilib.sh.ShRuntimeException;
import net.morilib.sh.ShSecurityPolicy;
import net.morilib.sh.ShSignal;
import net.morilib.sh.ShStat;
import net.morilib.sh.ShSyntaxException;
import net.morilib.sh.ShToken;
import net.morilib.sh.ShTree;
import net.morilib.sh.misc.IOs;
import net.morilib.sh.misc.XtraceStream;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2014/01/02
 */
public class ShFactory {

	private static class XmlEnv extends AbstractShEnvironment {

		private Map<String, SpecialVariable> map;
		private Map<String, Attributes> attrs;

		private XmlEnv() {
			map   = new HashMap<String, SpecialVariable>();
			attrs = new HashMap<String, Attributes>();
			path  = new ArrayList<String>();
		}

		private XmlEnv(XmlEnv x) {
			map   = new HashMap<String, SpecialVariable>(x.map);
			attrs = new HashMap<String, Attributes>(x.attrs);
			path  = new ArrayList<String>(x.path);
		}

		public String find(String name) {
			SpecialVariable v;

			v = map.get(name);
			return v != null ? v.getValue(name) : null;
		}

		public void bind(String name, String value) {
			put(name, value);
			setpath();
		}

		public void put(String name, String value) {
			map.put(name, new NormalVariable(value));
		}

		public boolean contains(String name) {
			return map.containsKey(name);
		}

		public ShEnvironment getEnvironment() {
			return null;
		}

		public Properties toProperties() {
			Properties p = new Properties();

			for(Map.Entry<String, SpecialVariable> t : map.entrySet()) {
				p.setProperty(t.getKey(),
						t.getValue().getValue(t.getKey()));
			}
			return p;
		}

		public void export(String name) {
			if(!contains(name))  map.put(name, new NormalVariable(""));
		}

		public void unbind(String name) {
			map.remove(name);
		}

		public boolean isReadonly(String name) {
			return (attrs.containsKey(name) &&
					attrs.get(name).isReadonly());
		}

		public void setReadonly(String name) {
			attrs.put(name, READONLY);
		}

	}

	private static class XmlCmd implements ShBuiltInCommands {

		private Map<String, ShProcess> cmds =
				new HashMap<String, ShProcess>();

		/* (non-Javadoc)
		 * @see net.morilib.sh.ShBuiltInCommands#find(java.lang.String)
		 */
		public ShProcess find(String name) {
			return cmds.get(name);
		}

		/* (non-Javadoc)
		 * @see net.morilib.sh.ShBuiltInCommands#getCommands()
		 */
		public Map<String, ShProcess> getCommands() {
			return Collections.unmodifiableMap(cmds);
		}

	}

	private static class XmlFS extends AbstractShFileSystem {

		private XmlVDir root;

		public ShFile getRoot() {
			return root;
		}

		public ShFile getFile(String s) {
			SpecialDir x;
			String t;
			ShFile f;

			t = normalizePath(s);
			if(t.length() == 0) {
				return getCurrentDirectory();
			} else if(t.equals("/")) {
				return root;
			} else if(t.charAt(0) == '/') {
				t = t.substring(1);
				x = root;
			} else if(realcur == null) {
				x = root;
			} else {
				return getFile(getCurrentDirectory(), s);
			}

			f = x;
			for(String v : t.split("/")) {
				if(x == null || (f = x.getFile(v)) == null) {
					return new XmlInv(this, t);
				} else {
					x = f instanceof SpecialDir ? (SpecialDir)f : null;
				}
			}
			return f;
		}

		public ShFile getFile(String dir, String name) {
			return getFile(dir + "/" + name);
		}

		public ShFile getFile(ShFile dir, String name) {
			return getFile(dir.toString() + "/" + name);
		}

		public ShFile getNativeFile(String s) {
			return getFile(s);
		}

		public ShSecurityPolicy getSecurityPolicy() {
			return ShSecurityPolicy.ALL_PERMITTED;
		}

		public InputStream getProfile() {
			return null;
		}

		public InputStream getRc() {
			return null;
		}

		public ShFile getHome() {
			return root;
		}

	}

	private static class XmlVDir extends SpecialDir {

		private Map<String, ShFile> files =
				new HashMap<String, ShFile>();
		private long last = System.currentTimeMillis();

		public XmlVDir(ShFileSystem fs, String virtualpath,
				String name) {
			super(fs, virtualpath, name);
		}

		private XmlVDir createSubdir(String name) {
			return new XmlVDir(filesystem, joinpath(name), name);
		}

		private XmlRFile createRFile(String name, File file) {
			return new XmlRFile(filesystem, joinpath(name), name,
					file);
		}

		private XmlRDir createRDir(String name, File file,
				boolean subdir, boolean create) {
			return new XmlRDir(filesystem, joinpath(name), name, file,
					subdir, create);
		}

		private XmlStdIO createStdIO(String name, InputStream stdin,
				PrintStream stdout) {
			return new XmlStdIO(filesystem, joinpath(name), name,
					stdin, stdout);
		}

		private XmlSFile createSFile(String name, String s) {
			return new XmlSFile(filesystem, joinpath(name), name, s);
		}

		private XmlJFile createJFile(String name, String res) {
			return new XmlJFile(filesystem, joinpath(name), name, res);
		}

		private XmlCFile createCFile(String name,
				List<SpecialFile> files) {
			return new XmlCFile(filesystem, joinpath(name), name,
					files);
		}

		private XmlNFile createNFile(String name) {
			return new XmlNFile(filesystem, joinpath(name), name);
		}

		private ShFile createFileByReflection(
				String c) throws ShConfigException {
			return (SpecialFile)getcls(c, filesystem, joinpath(name),
					name);
		}

		private ShFile createDirByReflection(
				String c) throws ShConfigException {
			return (SpecialDir)getcls(c, filesystem, joinpath(name),
					name);
		}

		@Override
		ShFile getFile(String name) {
			return files.get(name);
		}

		public boolean isReadable() {
			return true;
		}

		public boolean isWritable() {
			return true;
		}

		public boolean isHidden() {
			return false;
		}

		public long getLastModified() {
			return last;
		}

		public Collection<ShFile> getFiles() {
			return files.values();
		}

		public void setLastModified(long time) {
			last = time;
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(last);
			r.setSize(765);
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlRDir extends SpecialDir {

		private boolean subdir, create;
		private File file;

		public XmlRDir(ShFileSystem fs, String virtualpath,
				String name, File file, boolean subdir,
				boolean create) {
			super(fs, virtualpath, name);
			this.file   = file;
			this.subdir = subdir;
			this.create = create;
		}

		@Override
		ShFile getFile(String name) {
			File[] l;

			l = file.listFiles();
			for(File f : l) {
				if(!f.getName().equals(name)) {
					// do nothing
				} else if(f.isFile()) {
					return new XmlRFile(filesystem,
							joinpath(f.getName()), f.getName(), f);
				} else if(subdir && f.isDirectory()) {
					return new XmlRDir(filesystem,
							joinpath(f.getName()), f.getName(), f,
							subdir, create);
				}
			}

			if(create) {
				return new XmlRFile(filesystem, joinpath(name), name,
						new File(file, name));
			} else {
				return null;
			}
		}

		public boolean isReadable() {
			return file.canRead();
		}

		public boolean isWritable() {
			return file.canWrite();
		}

		public boolean isHidden() {
			return file.isHidden();
		}

		public long getLastModified() {
			return file.lastModified();
		}

		public Collection<ShFile> getFiles() {
			List<ShFile> r = new ArrayList<ShFile>();
			File[] l;

			l = file.listFiles();
			for(File f : l) {
				if(f.isFile()) {
					r.add(new XmlRFile(filesystem,
							joinpath(f.getName()), f.getName(), f));
				} else if(subdir && f.isDirectory()) {
					r.add(new XmlRDir(filesystem,
							joinpath(f.getName()), f.getName(), f,
							subdir, create));
				}
			}
			return r;
		}

		public void setLastModified(long time) {
			file.setLastModified(time);
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(file.lastModified());
			r.setSize(file.length());
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlRFile extends SpecialFile {

		private File file;

		public XmlRFile(ShFileSystem fs, String vp, String name,
				File file) {
			super(fs, vp, name);
			this.file = file;
		}

		public boolean isExecutable() {
			return file.canExecute();
		}

		public boolean isReadable() {
			return file.canRead();
		}

		public boolean isWritable() {
			return file.canWrite();
		}

		public boolean isZeroFile() {
			return file.length() == 0;
		}

		public long getLastModified() {
			return file.lastModified();
		}

		public InputStream getInputStream() throws IOException {
			return new FileInputStream(file);
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			return new PrintStream(new FileOutputStream(file, append));
		}

		public void setLastModified(long time) {
			file.setLastModified(time);
		}

		public ShStat getStat() {
			return ShRealFile.getStat(this, virtualpath, file);
		}

	}

	private static class XmlStdIO extends SpecialFile {

		private InputStream stdin;
		private PrintStream stdout;

		public XmlStdIO(ShFileSystem fs, String vp, String name,
				InputStream stdin, PrintStream stdout) {
			super(fs, vp, name);
			this.stdin  = stdin;
			this.stdout = stdout;
		}

		public boolean isExecutable() {
			return false;
		}

		public boolean isReadable() {
			return stdin != null;
		}

		public boolean isWritable() {
			return stdout != null;
		}

		public boolean isZeroFile() {
			return false;
		}

		public long getLastModified() {
			return 0;
		}

		public InputStream getInputStream() throws IOException {
			if(stdin == null)  throw new IOException();
			return stdin;
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			if(stdout == null)  throw new IOException();
			return stdout;
		}

		public void setLastModified(long time) {
			// do nothing
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(0);
			r.setSize(0);
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlSFile extends SpecialFile {

		private final long time = System.currentTimeMillis();
		private byte[] buffer;

		public XmlSFile(ShFileSystem fs, String vp, String name,
				String str) {
			super(fs, vp, name);
			this.buffer = str.getBytes();
		}

		public boolean isExecutable() {
			return false;
		}

		public boolean isReadable() {
			return true;
		}

		public boolean isWritable() {
			return false;
		}

		public boolean isZeroFile() {
			return buffer.length == 0;
		}

		public long getLastModified() {
			return time;
		}

		public InputStream getInputStream() throws IOException {
			return new ByteArrayInputStream(buffer);
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			throw new IOException();
		}

		public void setLastModified(long time) {
			// do nothing
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(time);
			r.setSize(buffer.length);
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlJFile extends SpecialFile {

		private final long time = System.currentTimeMillis();
		private String res;

		public XmlJFile(ShFileSystem fs, String vp, String name,
				String res) {
			super(fs, vp, name);
			this.res = res;
		}

		private InputStream getstream() {
			return ShFactory.class.getResourceAsStream(res);
		}

		private long getlength() {
			byte[] b = new byte[1024];
			InputStream in;
			long l = 0;
			int m;

			if((in = getstream()) == null)  return 0;
			try {
				while((m = in.read(b)) >= 0)  l += m;
				return l;
			} catch (IOException e) {
				return 0;
			}
		}

		public boolean isExecutable() {
			return false;
		}

		public boolean isReadable() {
			return true;
		}

		public boolean isWritable() {
			return false;
		}

		public boolean isZeroFile() {
			InputStream in;

			if((in = getstream()) == null)  return true;
			try {
				return in.read() < 0;
			} catch (IOException e) {
				return true;
			}
		}

		public long getLastModified() {
			return time;
		}

		public InputStream getInputStream() throws IOException {
			InputStream in;

			if((in = getstream()) == null) {
				throw new IOException();
			}
			return in;
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			throw new IOException();
		}

		public void setLastModified(long time) {
			// do nothing
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(time);
			r.setSize(getlength());
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlCFile extends SpecialFile {

		private List<SpecialFile> files;

		public XmlCFile(ShFileSystem fs, String vp, String name,
				List<SpecialFile> files) {
			super(fs, vp, name);
			this.files = new ArrayList<SpecialFile>(files);
		}

		public boolean isExecutable() {
			for(SpecialFile f : files) {
				if(!f.isExecutable())  return false;
			}
			return true;
		}

		public boolean isReadable() {
			for(SpecialFile f : files) {
				if(!f.isReadable())  return false;
			}
			return true;
		}

		public boolean isWritable() {
			for(SpecialFile f : files) {
				if(!f.isWritable())  return false;
			}
			return true;
		}

		public boolean isZeroFile() {
			for(SpecialFile f : files) {
				if(!f.isZeroFile())  return false;
			}
			return true;
		}

		public long getLastModified() {
			long l = Long.MIN_VALUE;

			for(SpecialFile f : files) {
				if(l < f.getLastModified())  l = f.getLastModified();
			}
			return l < 0 ? 0 : l;
		}

		public InputStream getInputStream() throws IOException {
			throw new IOException();
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			final PrintStream[] p = new PrintStream[files.size()];

			for(int k = 0; k < p.length; k++) {
				p[k] = files.get(k).getPrintStream(append);
			}
			return new PrintStream(new OutputStream() {

				@Override
				public void write(int b) throws IOException {
					for(OutputStream s : p)  s.write(b);
				}

				@Override
				public void write(byte[] b) throws IOException {
					for(OutputStream s : p)  s.write(b);
				}

				@Override
				public void write(byte[] b, int off, int len)
						throws IOException {
					for(OutputStream s : p)  s.write(b, off, len);
				}

				@Override
				public void flush() throws IOException {
					for(OutputStream s : p)  s.flush();
				}

				@Override
				public void close() throws IOException {
					for(OutputStream s : p)  s.close();
				}

			});
		}

		public void setLastModified(long time) {
			for(SpecialFile f : files)  f.setLastModified(time);
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(getLastModified());
			r.setSize(0);
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlNFile extends SpecialFile {

		public XmlNFile(ShFileSystem fs, String vp, String name) {
			super(fs, vp, name);
		}

		public boolean isExecutable() {
			return false;
		}

		public boolean isReadable() {
			return true;
		}

		public boolean isWritable() {
			return true;
		}

		public boolean isZeroFile() {
			return true;
		}

		public long getLastModified() {
			return 0;
		}

		public InputStream getInputStream() throws IOException {
			return IOs.NULL_INPUT;
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			return new PrintStream(IOs.NULL_OUTPUT);
		}

		public void setLastModified(long time) {
			// do nothing
		}

		public ShStat getStat() {
			ShStat r = new ShStat();
			int f = 0;

			r.setFilename(virtualpath);
			r.setMtime(getLastModified());
			r.setSize(0);
			f |= isDirectory()  ? ShStat.DIRECTORY : 0;
			f |= isExecutable() ? ShStat.EXECUTABLE : 0;
			f |= isWritable()   ? ShStat.WRITABLE : 0;
			f |= isReadable()   ? ShStat.READABLE : 0;
			r.setFlags(f);
			return r;
		}

	}

	private static class XmlInv extends AbstractShFile {

		public XmlInv(ShFileSystem fs, String vp) {
			super(fs, vp);
		}

		public boolean isDirectory() {
			return false;
		}

		public boolean isExecutable() {
			return false;
		}

		public boolean isExist() {
			return false;
		}

		public boolean isFile() {
			return false;
		}

		public boolean isReadable() {
			return false;
		}

		public boolean isWritable() {
			return false;
		}

		public boolean isHidden() {
			return false;
		}

		public boolean isZeroFile() {
			return false;
		}

		public long getLastModified() {
			return 0;
		}

		public String getName() {
			return virtualpath.replaceFirst(".*/", "");
		}

		public Collection<ShFile> getFiles() {
			return Collections.<ShFile>singleton(this);
		}

		public InputStream getInputStream() throws IOException {
			throw new IOException();
		}

		public PrintStream getPrintStream(
				boolean append) throws IOException {
			throw new IOException();
		}

		public ShProcess getProcess() {
			return null;
		}

		public boolean mkdir() {
			return false;
		}

		public boolean renameTo(ShFile f) {
			return false;
		}

		public void setLastModified(long time) {
			// do nothing
		}

		public boolean rmdir() {
			return false;
		}

		public boolean deleteFile() {
			return false;
		}

		public ShStat getStat() {
			return null;
		}

	};

	//
	private XmlEnv env = new XmlEnv();
	private XmlCmd cmd = new XmlCmd();
	private XmlFS  filesystem = new XmlFS();
	private InputStream stdin;
	private PrintStream stdout, stderr;

	private ShFactory() {}
	
	private static Object getcls(String n) throws ShConfigException {
		Class<?> c;
		Object r;

		try {
			c = Class.forName(n);
			r = c.newInstance();
			return r;
		} catch(ClassNotFoundException e) {
			throw new ShConfigException(e);
		} catch(InstantiationException e) {
			throw new ShConfigException(e);
		} catch(IllegalAccessException e) {
			throw new ShConfigException(e);
		}
	}
	
	private static Object getcls(String n, ShFileSystem fs,
			String vp, String nm) throws ShConfigException {
		Constructor<?> o;
		Class<?> c;
		Object r;

		try {
			c = Class.forName(n);
			o = c.getConstructor(ShFileSystem.class,
					String.class, String.class);
			r = o.newInstance(fs, vp, nm);
			return r;
		} catch(ClassNotFoundException e) {
			throw new ShConfigException(e);
		} catch(InstantiationException e) {
			throw new ShConfigException(e);
		} catch(IllegalAccessException e) {
			throw new ShConfigException(e);
		} catch(SecurityException e) {
			throw new ShConfigException(e);
		} catch(NoSuchMethodException e) {
			throw new ShConfigException(e);
		} catch(IllegalArgumentException e) {
			throw new ShConfigException(e);
		} catch(InvocationTargetException e) {
			throw new ShConfigException(e);
		}
	}

	private ShFile parseDir(XmlVDir x, Node d,
			boolean write) throws ShConfigException {
		List<SpecialFile> t;
		PrintStream out;
		InputStream in;
		boolean b, c;
		String m, v;
		NodeList a;
		Element l;
		XmlVDir y;
		String s;
		ShFile f;
		File r;

		if(!(d instanceof Element)) {
			if(d.getNodeValue().trim().length() > 0) {
				throw new ShConfigException();
			}
			return null;
		}

		l = (Element)d;
		m = l.getAttribute("name");
		if((m == null || m.equals("")) && write) {
			throw new ShConfigException();
		} else if((s = l.getTagName()).equals("virtual-dir")) {
			if(x == null) {
				throw new ShConfigException();
			}

			a = l.getChildNodes();
			f = y = x.createSubdir(m);
			for(int k = 0; k < a.getLength(); k++) {
				parseDir(y, a.item(k), true);
			}
		} else if(s.equals("standard-io")) {
			if((v = l.getAttribute("in")) == null) {
				in = stdin;
			} else if(v.equals("stdin")) {
				in = stdin;
			} else {
				in = null;
			}

			if((v = l.getAttribute("out")) == null) {
				out = stdout;
			} else if(v.equals("stdout")) {
				out = stdout;
			} else if(v.equals("stderr")) {
				out = stderr;
			} else {
				out = null;
			}
			f = x.createStdIO(m, in, out);
		} else if(s.equals("string-file")) {
			v = l.getChildNodes().item(0).getNodeValue();
			f = x.createSFile(m, v.trim() + "\n");
		} else if(s.equals("resource-file")) {
			f = x.createJFile(m, l.getAttribute("resource"));
		} else if(s.equals("special-file")) {
			f = x.createDirByReflection(l.getAttribute("class"));
		} else if(s.equals("special-dir")) {
			f = x.createFileByReflection(l.getAttribute("class"));
		} else if(s.equals("multiplex-file")) {
			a = l.getChildNodes();
			t = new ArrayList<SpecialFile>();
			for(int k = 0; k < a.getLength(); k++) {
				if((f = parseDir(x, a.item(k), false)) == null) {
					// do nothing
				} else if(f instanceof SpecialFile) {
					t.add((SpecialFile)f);
				} else {
					throw new ShConfigException();
				}
			}
			f = x.createCFile(m, t);
		} else if(s.equals("null-file")) {
			f = x.createNFile(m);
		} else if((v = l.getAttribute("realpath")) == null ||
				v.equals("")) {
			throw new ShConfigException();
		} else if(s.equals("map-dir")) {
			b = "true".equals(l.getAttribute("subdir"));
			c = "true".equals(l.getAttribute("createfile"));
			if(!(r = new File(v)).isDirectory()) {
				throw new ShConfigException();
			}
			f = x.createRDir(m, r, b, c);
		} else if(s.equals("map-file")) {
			f = x.createRFile(m, new File(v));
		} else {
			throw new ShConfigException();
		}

		if(write)  x.files.put(m, f);
		return f;
	}

	private void parseCommand(Node d) throws ShConfigException {
		Element l;

		if(!(d instanceof Element)) {
			if(d.getNodeValue().trim().length() > 0) {
				throw new ShConfigException();
			}
		} else if(((l = (Element)d).getTagName()).equals(
				"resource-command")) {
			cmd.cmds.put(
					l.getAttribute("name"),
					(ShProcess)getcls(l.getAttribute("class")));
		} else {
			throw new ShConfigException();
		}
	}

	private void parseEnvironment(Node d) throws ShConfigException {
		Element l;
		String n, s;

		if(!(d instanceof Element)) {
			if(d.getNodeValue().trim().length() > 0) {
				throw new ShConfigException();
			}
		} else if((s = (l = (Element)d).getTagName()).equals(
				"variable")) {
			if((n = l.getAttribute("name")) == null || n.equals("")) {
				throw new ShConfigException();
			}
			s = l.getAttribute("value");
			env.put(n, s != null ? s : "");
		} else if(s.equals("property-variable")) {
			if((n = l.getAttribute("name")) == null || n.equals("")) {
				throw new ShConfigException();
			} else if((s = l.getAttribute("property")) == null ||
					s.equals("")) {
				throw new ShConfigException();
			}
			env.map.put(n, new PropertyVariable(s));
		} else if(s.equals("inherit-variable")) {
			if((n = l.getAttribute("name")) == null || n.equals("")) {
				throw new ShConfigException();
			} else if((s = l.getAttribute("from")) == null ||
					s.equals("")) {
				throw new ShConfigException();
			} else if((s = System.getenv(s)) != null) {
				env.map.put(n, new NormalVariable(s));
			}
		} else if(s.equals("special-variable")) {
			env.map.put(l.getAttribute("name"),
					(SpecialVariable)getcls(l.getAttribute("class")));
		} else {
			throw new ShConfigException();
		}
	}

	/**
	 * 
	 * @param in
	 * @param stdin
	 * @param stdout
	 * @param stderr
	 * @return
	 * @throws ShConfigException
	 */
	public static ShFactory getInstance(InputStream in,
			InputStream stdin, PrintStream stdout,
			PrintStream stderr) throws ShConfigException {
		DocumentBuilderFactory fac;
		DocumentBuilder bld;
		Document doc;
		ShFactory r;
		NodeList nl;
		Element rt;
		XmlVDir di;
		String s;
		Node nd;

		r = new ShFactory();
		r.stdin  = stdin;
		r.stdout = stdout;
		r.stderr = stderr;
		try {
			fac = DocumentBuilderFactory.newInstance();
			bld = fac.newDocumentBuilder();
			doc = bld.parse(in);
			rt  = doc.getDocumentElement();
			di  = r.filesystem.root =
					new XmlVDir(r.filesystem, "/", null);
			nd  = rt.getElementsByTagName("filesystem").item(0);
			nl  = nd.getChildNodes();
			for(int k = 0; k < nl.getLength(); k++) {
				r.parseDir(di, nl.item(k), true);
			}

			nd  = rt.getElementsByTagName("commands").item(0);
			if(nd instanceof Element &&
					(s = ((Element)nd).getAttribute("usedefault")) != null &&
					s.equals("true")) {
//				r.cmd.cmds.putAll(
//						ShDefaultBuiltInCommands
//						.getInstance().getCommands());
			}
			nl  = nd.getChildNodes();
			for(int k = 0; k < nl.getLength(); k++) {
				r.parseCommand(nl.item(k));
			}

			r.env.map.put("IFS", new NormalVariable(" \t"));
			r.env.map.put("PATH", new NormalVariable(""));
			nd  = rt.getElementsByTagName("environment").item(0);
			nl  = nd.getChildNodes();
			for(int k = 0; k < nl.getLength(); k++) {
				r.parseEnvironment(nl.item(k));
			}
			return r;
		} catch(ParserConfigurationException e) {
			throw new ShConfigException(e);
		} catch(SAXException e) {
			throw new ShConfigException(e);
		} catch(IOException e) {
			throw new ShConfigException(e);
		}
	}

	/**
	 * 
	 * @return
	 */
	public static ShFactory getInstance() {
		try {
			return ShFactory.getInstance(
					ShFactory.class.getResourceAsStream(
							"/net/morilib/sh/conf/nullenv.xml"),
					System.in, System.out, System.err);
		} catch (ShConfigException e) {
			throw new RuntimeException(e);
		}
	}

	//
	private static void _setarg(ShEnvironment n, String[] a) {
		for(int i = 1; i <= a.length; i++) {
			n.put(i + "", a[i - 1]);
		}
	}

	/**
	 * 
	 * @param script
	 * @param stdin
	 * @param stdout
	 * @param stderr
	 * @param vars
	 * @param args
	 * @return
	 * @throws IOException
	 * @throws ShSyntaxException
	 */
	@SuppressWarnings("resource")
	public int execute(InputStream script, InputStream stdin,
			PrintStream stdout, PrintStream stderr,
			Map<String, String> vars,
			String... args) throws IOException, ShSyntaxException {
		ShEnvironment v;
		XtraceStream q;
		ShRuntime run;
		XmlEnv env;

		run = new DefaultShRuntime(cmd);
		q   = new XtraceStream(stdout);
		env = new XmlEnv(this.env);
		for(Map.Entry<String, String> t : vars.entrySet()) {
			env.put(t.getKey(), t.getValue());
		}
		v   = new ShRootShellEnvironment(env);
		_setarg(v, args);

		try {
			return ShFacade.execute(v, filesystem, cmd, run, script,
					stdin, stdout, stderr, q);
		} catch(ShExitException e) {
			return e.getCode();
		}
	}

	/**
	 * 
	 * @param script
	 * @param stdin
	 * @param stdout
	 * @param stderr
	 * @param args
	 * @return
	 * @throws IOException
	 * @throws ShSyntaxException
	 */
	public int execute(InputStream script, InputStream stdin,
			PrintStream stdout, PrintStream stderr,
			String... args) throws IOException, ShSyntaxException {
		return execute(script, stdin, stdout, stderr,
				Collections.<String, String>emptyMap(), args);
	}

	/**
	 * 
	 * @param script
	 * @param stdout
	 * @param stderr
	 * @param vars
	 * @param args
	 * @return
	 * @throws IOException
	 * @throws ShSyntaxException
	 */
	public int execute(InputStream script,
			PrintStream stdout, PrintStream stderr,
			Map<String, String> vars,
			String... args) throws IOException, ShSyntaxException {
		return execute(script, IOs.NULL_INPUT, stdout, stderr, vars,
				args);
	}

	/**
	 * 
	 * @param script
	 * @param stdout
	 * @param stderr
	 * @param args
	 * @return
	 * @throws IOException
	 * @throws ShSyntaxException
	 */
	public int execute(InputStream script, PrintStream stdout,
			PrintStream stderr,
			String... args) throws IOException, ShSyntaxException {
		return execute(script, IOs.NULL_INPUT, stdout, stderr, args);
	}

	/**
	 * 
	 * @param script
	 * @param stdout
	 * @param args
	 * @return
	 * @throws IOException
	 * @throws ShSyntaxException
	 */
	public int execute(InputStream script, PrintStream stdout,
			String... args) throws IOException, ShSyntaxException {
		return execute(script, IOs.NULL_INPUT, stdout,
				new PrintStream(IOs.NULL_OUTPUT), args);
	}

	/**
	 * 
	 * @param stdin
	 * @param stdout
	 * @param stderr
	 * @return
	 * @throws IOException
	 */
	public int interactive(InputStream stdin, PrintStream stdout,
			PrintStream stderr) throws IOException {
		ShPromptReader p;
		ShEnvironment v;
		XtraceStream q;
		ShRuntime run;
		ShLexer r;
		ShToken t;
		int a = 0;
		ShTree x;
		String z;

		// initialize
		env.bind("PS1", "$ ");
		env.bind("PS2", "> ");
		filesystem.setCurrentDirectory(filesystem.getHome());
		q   = new XtraceStream(stdout);
		p   = new ShPromptReader(env, q);
		r   = new ShLexer(env, p, q);
		run = new DefaultShRuntime(cmd);
		v   = new ShRootShellEnvironment(env);

		while(true) {
			try {
				while((t = r.nextToken()) == ShToken.NEWLINE);
				x = ShParser.parseCommand(r, t);
				a = x.eval(v, filesystem, cmd, run, stdin, stdout,
						stderr, q);
				if(r.isEof())  return a;
				r.resetPrompt();
			} catch(ShExitException e) {
				return e.getCode();
			} catch(IOException e) {
				if((z = v.getTrap(ShSignal.IO)) != null) {
					try {
						run.eval(v, filesystem, stdin, stdout,
								stderr, q, z);
					} catch(ShCommandNotFoundException e1) {
						stderr.print(ShFacade.NAME);
						stderr.print(": command not found: ");
						stderr.println(e1.getName());
					} catch(IOException e1) {
						//e.printStackTrace(stderr);
						stderr.print(ShFacade.NAME);
						stderr.println(": IO error");
					} catch(ShSyntaxException e1) {
						e.printStackTrace();
						stderr.println("Syntax error");
					}
				}
				e.printStackTrace(stderr);
			} catch(ShCommandNotFoundException e) {
				stderr.print(ShFacade.NAME);
				stderr.print(": command not found: ");
				stderr.println(e.getName());
			} catch(ShSyntaxException e) {
				e.printStackTrace();
				stderr.println("Syntax error");
			} catch(ShRuntimeException e) {
				e.printStackTrace();
				stderr.println("Syntax error");
			}
		}
	}

}

//
class NormalVariable implements SpecialVariable {

	private String value;

	/**
	 * 
	 * @param v
	 */
	public NormalVariable(String v) {
		value = v;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.conf.SpecialVariable#getValue(java.lang.String)
	 */
	public String getValue(String name) {
		return value;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return value;
	}

}

//
class PropertyVariable implements SpecialVariable {

	private String prop;

	/**
	 * 
	 * @param v
	 */
	public PropertyVariable(String p) {
		prop = p;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.conf.SpecialVariable#getValue(java.lang.String)
	 */
	public String getValue(String name) {
		String s;

		s = System.getProperty(prop);
		return s != null ? s : "";
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return getValue(prop);
	}

}
