/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.idlnvr;

import java.util.regex.*;
import java.util.*;
import java.io.*;

public class IdlNvr {
	public static final String VERSION = "1.20";

	public static void checkPassword(String password) throws LoginFailedException {
		final Pattern pat = Pattern.compile("[a-zA-Z0-9]+");

		if(password == null || password.equals("")) throw new LoginFailedException("pX[hł");
		if(password.length() < 3 || password.length() > 15) throw new LoginFailedException("pX[h3~15w肵Ă");

		if(!pat.matcher(password).matches()) {
			throw new LoginFailedException("pX[h͉p݂̂w\ł");
		}
	}

	public static boolean checkPersonName(String personName) {
		final Pattern pat = Pattern.compile("[a-z0-9\\_]+");

		if(personName == null || personName.equals("")) return false;
		return pat.matcher(personName).matches();
	}

	public static void checkUserName(String userName) throws LoginFailedException {
		final Pattern pat = Pattern.compile("[a-z0-9\\_]+");

		if(userName == null || userName.equals("")) throw new LoginFailedException("[Uł");
		if(userName.length() < 3 || userName.length() > 15) throw new LoginFailedException("pX[h3~15w肵Ă");

		if(!pat.matcher(userName).matches()) {
			throw new LoginFailedException("[U͉pEEA_[Ĉ݂w\ł");
		}
	}

	public static int getInt(String s) {
		try {
			return Integer.parseInt(s);
		} catch(NumberFormatException e) {
			return -1;
		}
	}

	public static String getCountryString(int country) {
		if(country == 1) {
			return "<font color=red></font>";
		} else if(country == 2) {
			return "<font color=blue></font>";
		} else {
			return "";
		}
	}

	public static String toHtmlText(String s) {
		if(s == null) return "";

		StringBuffer buf = new StringBuffer(s.length()*2);
		int len = s.length();

		for(int i=0; i<len; i++) {
			char c = s.charAt(i);

			switch(c) {
				case '"': buf.append("&quot;"); break;
				case '&': buf.append("&amp;"); break;
				case '<': buf.append("&lt;"); break;
				case '>': buf.append("&gt;"); break;
				case ' ': buf.append("&nbsp;"); break;
				case '\n': buf.append("<br>"); break;
				default: buf.append(c); 
			}
		}

		return buf.toString();
	}

	private static String reviewToNoBlockHtml(Database db, String s) {
		if(s == null) return "";

		boolean nowEscape = false;
		StringBuffer buf = new StringBuffer(s.length()*2);
		int len = s.length();
		StringBuffer escapedChars = null;

		for(int i=0; i<len; i++) {
			char c = s.charAt(i);

			if(!nowEscape) {
				switch(c) {
					case '"': buf.append("&quot;"); break;
					case '&': buf.append("&amp;"); break;
					case '<': buf.append("&lt;"); break;
					case '>': buf.append("&gt;"); break;
					case ' ': buf.append("&nbsp;"); break;
					case '\n': buf.append("<br>"); break;
					case '[': 
						if(s.substring(i).startsWith("[#")) {
							escapedChars = new StringBuffer();
							nowEscape = true;
						} else {
							buf.append(c);
						}
						break;
					
					default: buf.append(c); 
				}
			} else {
				if(c != ']') {
					escapedChars.append(c);
				} else {
					String cmd = escapedChars.toString();

					if(cmd.startsWith("#detail")) {
						int personId = getInt(getCmdParam(cmd));
						Person person = null;
						
						try {
							 person = db.getPerson(personId);
						} catch(Exception e) {
						}

						if(person != null) {
							buf.append("<a href=\"./detail?n=" + personId + "\">" + person.name + "</a>");
						} else {
							buf.append(toHtmlText("[" + escapedChars + "]"));
						}
					} else if(cmd.startsWith("#right")) {
						int right = getInt(getCmdParam(cmd));
						buf.append("<span style=\"position:absolute; right:" + right + "%\">");
					} else if(cmd.startsWith("#/right")) {
						buf.append("</span>");
					} else if(cmd.startsWith("#link")) {
						String link = getCmdParam(cmd);
						buf.append("<a href=\"" + link + "\">" + link + "</a>");
					} else if(cmd.startsWith("#img")) {
						String src = getCmdParam(cmd);
						buf.append("<img src=\"" + src + "\">");
					} else {
						buf.append(toHtmlText("[" + escapedChars + "]"));
					}
					
					nowEscape = false;
				}
			}
		}
		
		return buf.toString();
	}
	
	public static String reviewToHtml(Database db, String s) {
		return "<br>" + reviewToNoBlockHtml(db, s) + "<br><br>";
	}
	
	private static String getCmdParam(String s) {
		StringTokenizer tokens = new StringTokenizer(s, " ");
		
		if(!tokens.hasMoreTokens()) return "err";
		tokens.nextToken();
		
		if(!tokens.hasMoreTokens()) return "err";
		return tokens.nextToken();
	}

	private static boolean isSkipLine(Vector<String> skipWords, String line) {
		line = line.trim();
		
		if(line.length() == 0) return true;
		
		for(int i=0; i<skipWords.size(); i++) {
			if(line.startsWith(skipWords.elementAt(i))) return true;
		}

		return false;
	}

	public static String reviewToShortHtml(Database db, String s) {
		try {
			Vector<String> skipWords = db.getSkipWords();
			BufferedReader reader = new BufferedReader(new StringReader(s));
			String line;

			while((line = reader.readLine()) != null) {
				if(!isSkipLine(skipWords, line)) {
					reader.close();
					return reviewToNoBlockHtml(db, line);
				}
			}

			reader.close();
			return "";
		} catch(IOException e) {
			return "";
		} catch(java.sql.SQLException e) {
			return "";
		}
	}
}
