/*
 * Copyright (c) 2007, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.enjoybase.conv;

import java.util.*;
import java.text.*;
import java.sql.*;
import java.io.*;

import com.enjoybase.*;
import com.enjoybase.model.*;

public class TopicConverter {
	public String html = "";
	public String text = "";
	private final Storage store;
	private final String editor;

	public TopicConverter(Storage store, String text, String editor) {
		this.text = text.trim();
		this.store = store;
		this.editor = editor;
	}

	private static class LineInfo {
		public final int indent;
		public final String rel;
		
		public LineInfo(int indent, String rel) {
			this.indent = indent;
			this.rel = rel;
		}
	}
	
	private Vector<String> rels = new Vector<String>();
	
	public void convert() throws ConverterException {
		BufferedReader reader = new BufferedReader(new StringReader(text));
		StringWriter htmlStringWriter = new StringWriter();

		PrintWriter htmlWriter = new PrintWriter(htmlStringWriter);
		
		String line;
		int lineno = 1;
		LineInfo curline = new LineInfo(0, "");
		
		rels.addElement("咣");
		
		htmlWriter.println("<div>");
		
		try {
			while((line = reader.readLine()) != null) {
				curline = parseLine(lineno, line, htmlWriter, curline);

				lineno++;
			}
			
			for(int i=0; i<curline.indent; i++) {
				htmlWriter.println("</div></td></tr></table>");
			}

		} catch(IOException e) {
			throw new ConverterException(e);
		}
		
		htmlWriter.println("</div>");
		htmlWriter.close();

		html = htmlStringWriter.toString();
	}
	
	private static final char STAG = 's';
	private static final char ETAG = 't';
	
	private LineInfo parseLine(int lineno, String line, PrintWriter out, LineInfo prevline) throws ConverterException {
		int indent = 0;
		String sindent = "";

		// Cfg̐[߂
		while(indent < line.length() && line.charAt(indent) == '\t') {
			indent++;
			sindent = sindent + " ";
		}

		if(prevline.indent + 1 < indent) {
			throw new ConverterException(
					lineno + "sځFOsCfg" + (indent - prevline.indent) + "iK[ȂĂ܂A"
					+ "Cfg͈sɂ1iK܂ł[邱Ƃł܂");
		}

		line = line.trim();

		if(line.length() == 0) {
			for(int i=0; i<prevline.indent - indent; i++) {
				out.println("</div></td></tr></table>");
			}

			return new LineInfo(indent, "");
		}

		if(line.charAt(0) != STAG) {
			throw new ConverterException(
				lineno + "sڂs" + STAG + ETAG + "ŊJnĂ܂");
		}
		
		int etag = line.indexOf(ETAG);
		
		if(etag == -1) {
			throw new ConverterException(
				lineno + "s: s̕^O܂");
		}
		
		String typestr = line.substring(1, etag).trim();
		
		StringTokenizer tokens = new StringTokenizer(typestr, "F:");
		
		String country = "";
		String rel = "";
		
		while(tokens.hasMoreTokens()) {
			String token = tokens.nextToken().trim();
			
			if(token.equals("") || token.equals("")) {
				country = Enjoybase.toCountryHtml(token) + " ";
			} else {
				rel = token;
			}
		}
		
		if(rel.length() == 0) {
			throw new ConverterException(
					lineno + "sځF̍słB");
		}

		line = line.substring(etag + 1).trim();

		rels.setSize(indent + 1);
		
		if(prevline.indent < indent) {
			out.print("<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\"><tr><td nowrap valign=\"top\">");
			out.print("<span class=\"title\">" + rel + "</span>");
			out.println("</td><td width=\"100%\" class=\"block\">");
		} 
		else if(prevline.indent > indent) {
			for(int i=0; i<prevline.indent - indent; i++) {
				out.println("</div></td></tr></table>");
			}
		}

		if(prevline.indent >= indent) {
			if(!rels.elementAt(indent).equals(rel)) {
				out.println("</div></td></tr></table>");

				out.print("<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\"><tr><td nowrap valign=\"top\">");
				out.print("<span class=\"title\">" + rel + "</span>");
				out.println("</td><td width=\"100%\" class=\"block\">");
			} else {
				out.println("</div>");
			}
		}

		out.print("<div class=\"article\">");		
		out.println(country + toArticleHtml(lineno, line));

		rels.setElementAt(rel, indent);

		return new LineInfo(indent, rel);
	}
	
	private String toArticleHtml(int lineno, String text) throws ConverterException {
		StringWriter buf = new StringWriter();
		PrintWriter out = new PrintWriter(buf);
		int cnt = 0;
		
		for(;;) {
			
			cnt++;
			if(cnt >= 100) {
				throw new ConverterException(
						lineno + "sځF[v̉\m߁A𒆒f܂B" 
						+ "Ǘ҂ɘAĂB");
			}
			
			int stag = text.indexOf(STAG);

			if(stag == -1) {
				out.print(Enjoybase.toArticleHtml(text));
				break;
			}
			
			int etag = text.indexOf(ETAG);
			
			if(etag == -1) {
				throw new ConverterException(lineno + "sځFev[gJnL'"
						+ STAG + "'ɑΉ'" + ETAG + "'܂B");
			}
			
			if(stag > etag) {
				throw new ConverterException(lineno + "sځFev[gJnL'" 
						+ STAG + "'̑O'" + ETAG + "'܂B");
			}

			out.print(Enjoybase.toArticleHtml(text.substring(0, stag)));

			StringTokenizer tokens = new StringTokenizer(text.substring(stag + 1, etag), " @", true);

			if(!tokens.hasMoreTokens()) {
				throw new ConverterException(lineno + "sځF" 
						+ "ev[gw肳Ă܂");
			}
			
			String name = getNextToken(tokens);
			TreeMap<String, String> params = new TreeMap<String, String>();

			String token;

			while(!(token = getNextToken(tokens)).equals("")) {
				String param = token;
				String value = getNextToken(tokens);

				if(value.equals("")) {
					throw new ConverterException(lineno + "sځF" 
							+ "ev[gp[^'" + param + "'̒l܂");
				}
				
				params.put(param, value);
			}

			if(name.equals("web")) {
				String url = params.get("url");
				String title = params.get("title");
				String site = params.get("site");
				String author = params.get("author");
				
				title = (title == null ? url : Enjoybase.toArticleHtml(title));
				site = (site == null ? "" : " " + Enjoybase.toArticleHtml(site));
				author = (author == null ? "" : " " + Enjoybase.toArticleHtml(author));

				if(url == null) {
					throw new ConverterException(lineno + "sځF" 
							+ "webev[gURLw肳Ă܂");
				}

				out.print("y<b><a href=\"" + Enjoybase.toHtmlText(url) + "\">" + title + "</a>" +
						site + author + "</b>z");
			} else if(name.equals("doc")) {
				String author = params.get("author");
				String title = params.get("title");
				String page = params.get("page");
				String date = params.get("date");
				String publisher = params.get("publisher");
				
				author = (author == null ? "" : " " + Enjoybase.toArticleHtml(author));
				title = (title == null ? "" : " u" + Enjoybase.toArticleHtml(title) + "v");
				page = (page == null ? "" : " " + Enjoybase.toArticleHtml(page));
				date = (date == null ? "" : " " + Enjoybase.toArticleHtml(date));
				publisher = (publisher == null ? "" : " " + Enjoybase.toArticleHtml(publisher));
				
				out.print("y<b>" + (author + title + page + date + publisher).trim() + "</b>z");
			} else if(name.equals("jacar")) {
				String title = params.get("title");
				String ref = params.get("ref");
				
				ref = (ref == null ? "" : " Ref." + Enjoybase.toArticleHtml(ref));
				title = (title == null ? "" : " u" + Enjoybase.toArticleHtml(title) + "v");

				out.print("y<b>" + (ref + title + " AWAjZ^[").trim() + "</b>z");
			} else {
				throw new ConverterException(lineno + "sځF" 
						+ "sȃev[g'" + name + "'");
			}

			text = text .substring(etag+1);
		}
		
		out.close();
		return buf.toString();
	}
	
	private String getNextToken(StringTokenizer tokens) {
		String token = " ";
		
		while(token.equals(" ") || token.equals("@")) {
			if(!tokens.hasMoreTokens()) return "";
			token = tokens.nextToken();
		}

		return token;
	}

	public static void main(String args[]) throws Exception {
		String text=
			"sF咣tsweb url http://bbs.enjoykorea.jp/tbbs/read.php?board_id=thistory&nid=1818357t\n";

		TopicConverter conv = new TopicConverter(null, text, "");
		conv.convert();
		
		System.out.println(conv.html);
	}
}
