/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.conv;

import java.util.regex.*;
import java.util.*;

public class StupidConverter implements ContentsConverter {
//	private boolean patternInitialized = false;
	
	private static class Replacer {
		public Pattern pattern;
		public String replace;
		
		public Replacer(String pattern, String replace) {
			this.pattern =  Pattern.compile(pattern, Pattern.CASE_INSENSITIVE);
			this.replace = replace;
		}

		public String exec(String s) {
			Matcher matcher = pattern.matcher(s);
			return matcher.replaceAll(replace);
		}
	}
	
	private Vector<Replacer> replacers;

	public void init() {
		replacers = new Vector<Replacer>();
		
		// S폜
		String[] deleteAll = {
			"<!\\-\\-.*?\\-\\->",
			"<noscript.*?>.*?</noscript>",
			"<script.*?></script>",
		};

		for(int i=0; i<deleteAll.length; i++) {
			replacers.addElement(new Replacer(deleteAll[i], ""));
		}

		// ^O̍Ղcč폜
		String[][] tagOnly = {
			{"<applet.*?>.*?</applet>", "#applet"},
			{"<object.*?>.*?</object>", "#object"},
			{"<map.*>.*?</map>", "#map"},
			{"<embed.*?>", "#embed"},
			{"<img.*?>", "#img"},
		};

		for(int i=0; i<tagOnly.length; i++) {
			replacers.addElement(new Replacer(tagOnly[i][0], tagOnly[i][1]));
		}

		// ^Oƕcč폜
		String[][] tagAndText = {
			{"<a\\s+.*?>(.*?)</a>", "#link($1)"}
		};
		
		for(int i=0; i<tagAndText.length; i++) {
			replacers.addElement(new Replacer(tagAndText[i][0], tagAndText[i][1]));
		}

		// sɒu
		String[] toEnter = {
			"tr", "P", "H\\d+", "UL", "OL", "DIR", "MENU", "PRE", "DL", "DIV", "CENTER", "NOSCRIPT", "NOFRAMES", "BLOCKQUOTE", "FORM", "ISINDEX", "HR", "TABLE", "FIELDSET", "ADDRESS", "LTICOL",
			"O:P"
		};

		for(int i=0; i<toEnter.length; i++) {
			replacers.addElement(new Replacer("<" + toEnter[i] + "\\s+.*?>", "<br>"));
			replacers.addElement(new Replacer("<" + toEnter[i] + ">", "<br>"));
			replacers.addElement(new Replacer("</" + toEnter[i] + ">", "<br>"));
		}

		// 󔒂ɒu
		String[] toSpace = {
			"td", "TT", "I", "B", "U", "S", "STRIKE", "BIG", "SMALL", "EM", "STRONG", "DFN", "CODE", "SAMP", "KBD", "VAR", "CITE", "ABBR", "ACRONYM", "A", "IMG", "APPLET", "OBJECT", "FONT", "BASEFONT", "SCRIPT", "MAP", "Q", "SUB", "SUP", "SPAN", "BDO", "IFRAME", "INPUT", "SELECT", "TEXTAREA", "LABEL", "BUTTON", "BLINK", "EMBED", "LAYER", "ILAYER", "NOLAYER", "NOBR", "WBR", "RUBY", "RB", "RP", "RT", "SPACER", "INS", "DEL"
		};
		
		for(int i=0; i<toSpace.length; i++) {
			replacers.addElement(new Replacer("<" + toSpace[i] + "\\s+.*?>", "&nbsp;"));
			replacers.addElement(new Replacer("<" + toSpace[i] + ">", "&nbsp;"));
			replacers.addElement(new Replacer("</" + toSpace[i] + ">", "&nbsp;"));
		}

		// 폜
		String[] delTag = {
			"a", "abbr", "acronym", "address", "applet", "area", "b", "base", "basefont", "bdo", "bgsound", "big", "", "link", "blockquote", "body", "button", "caption", "center", "cite", "code", "col", "colgroup", "comment", "dd", "del", "dfn", "dir", "div", "dl", "dt", "em", "embed", "fieldset", "font", "form", "fram", "", "frameset", "head", "hr", "html", "i", "iframe", "ilayer", "img", "input", "ins", "isindex", "kbd", "", "eygen", "label", "layer", "legend", "li", "link", "listing", "map", "marquee", "menu", "meta", "mult", "col", "nextid", "nobr", "noembed", "noframes", "nolayer", "noscript", "object", "ol", "optgroup", "option", "p", "param", "plaintext", "pre", "q", "rb", "rp", "rt", "ruby", "s", "samp", "script", "select", "server", "small", "spacer", "span", "strike", "strong", "style", "sub", "sup", "table", "tbody", "td", "textarea", "tfoot", "th", "thead", "title", "tr", "tt", "u", "ul", "var", "wbr", "xmp"
		};
		
		for(int i=0; i<delTag.length; i++) {
			replacers.addElement(new Replacer("<" + delTag[i] + "\\s+.*?>", ""));
			replacers.addElement(new Replacer("<" + delTag[i] + ">", ""));
			replacers.addElement(new Replacer("</" + delTag[i] + ">", ""));
		}
		
		// sE^u͋󔒂ɒu
		replacers.addElement(new Replacer("[\\r\\n\\t]", " "));
		
		// As^O1ɒu
//		replacers.addElement(new Replacer("(:?(:?<br\\s*/?>)|\r|\n)*", "<br>"));

		// s^Osɒu
		replacers.addElement(new Replacer("<br\\s*/?>", "\n"));

		String[][] markup = {
			{"&", "amp"},
			{"\"", "quot"},
			{"<", "lt"},
			{">", "gt"},
			{" ", "nbsp"}
		};

		// }[NAbv𐶂̕ɒu
		for(int i=0; i<markup.length; i++) {
			replacers.addElement(new Replacer("&" + markup[i][1] + ";", markup[i][0]));
		}

		// A󔒂1ɒu
		replacers.addElement(new Replacer(" ++", " "));
	
		// As1ɒu
		replacers.addElement(new Replacer("\\n[ \\n]*", "\n"));
	
		// ̕Ăу}[NAbvɒu
		for(int i=0; i<markup.length; i++) {
			replacers.addElement(new Replacer(markup[i][0], "&" + markup[i][1] + ";"));
		}

		// ss^Oɖ߂
		replacers.addElement(new Replacer("\\n", "<br>"));
	}

	public String convert(String imageUrl, String s) {
		for(int i=0; i<replacers.size(); i++) {
			Replacer replacer = (Replacer)replacers.elementAt(i);
			s = replacer.exec(s);
		}
		
		return s;
	}

	public String getFullContents(String contents) {
		return contents;
	}

	public String getShortContents(
			int desiredLineNum,
			int desiredCharNum,
			String contents,
			String link) {

		if(desiredCharNum > contents.length()) return contents;
		return contents.substring(0,desiredCharNum) + "<br>" + link;
	}

	public static void main(String args[]) {
		StupidConverter conv = new StupidConverter();
		conv.init();

		System.out.println(conv.convert(null, "he\"l&quot;lo<p>w<O:P>or</O:P>ld</p>"));
	}
}
