/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.store;

import java.sql.*;
import java.util.*;

import com.aibonware.inaver.model.*;
import com.aibonware.nvrlib.model.*;

// }X^EobNAbvɗp\ȃT[rX
public abstract class StoreBase {
	public static final String REGISTER_OK = "OK";
	
	SQLSanitizer safer = new SQLSanitizer();

	Connection con;
	Statement sql;
	protected final String storeName;
	protected final String dbname;

	public String getDBName() {
		return dbname;
	}
	
	/**
	 * RXgN^B
	 */
	public StoreBase(String storeName, String dbHost, String dbPass, String dbname) throws StoreException {
		try {
			this.storeName = storeName;
			this.dbname = dbname;

			// f[^x[Xڑ
			con = DriverManager.getConnection(
				"jdbc:mysql://" + dbHost + "/" + dbname + "?useUnicode=true&characterEncoding=UTF-8",
				"inav",
				dbPass);

			sql = con.createStatement();
						
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public void close() throws StoreException {
		try {
			con.close();
			sql.close();
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	/**
	 * ꗗf[^x[X擾
	 */
	public BoardList queryBoardList(boolean mustVisible, boolean mustCrawlable) throws StoreException {
		try {
			BoardList boards = new BoardList();
			
			// ꗗǂݍށB
			ResultSet rs = sql.executeQuery("select * from Boards");

			// eɂāAޔJԂB
			while(rs.next()) {
				// 擾
				int boardId = rs.getInt("BoardID");
				String boardName = rs.getString("BoardName");
				String dispName = rs.getString("DispName");
				int maxPageNum = rs.getInt("MaxPageNum");
				java.util.Date lastCrawlTime = safer.sqlDatetimeToDate(rs.getString("LastCrawlTime"));
				boolean visible = safer.sqlBool(rs.getString("Visible"));
                int interval = rs.getInt("WaitTime");
                boolean mustCrawl = safer.sqlBool(rs.getString("DoCrawl"));
                int boardType = rs.getInt("BoardType");
                int lockNid = rs.getInt("LockNID");

				if(mustVisible && !visible) continue;
				if(mustCrawlable && !mustCrawl) continue;

				// WebSiteCX^Xɔǉ
				boards.addBoard(new Board(
					boardId,
					boardName, 
					dispName,
					maxPageNum,
					lastCrawlTime,
					visible,
                    interval,
                    mustCrawl,
                    boardType,
                    lockNid)); 
			}
			
			rs.close();

			return boards;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	/**
	 * f[^x[X擾
	 */
	public Board queryBoard(int srcBoardId) throws StoreException {
		try {

			// ꗗǂݍށB
			ResultSet rs = sql.executeQuery("select * from Boards where BoardID=" + srcBoardId);
			rs.next();

			// 擾
			int boardId = rs.getInt("BoardID");
			String boardName = rs.getString("BoardName");
			String dispName = rs.getString("DispName");
			int maxPageNum = rs.getInt("MaxPageNum");
			java.util.Date lastCrawlTime = safer.sqlDatetimeToDate(rs.getString("LastCrawlTime"));
			boolean visible = safer.sqlBool(rs.getString("Visible"));
            int interval = rs.getInt("WaitTime");
            boolean mustCrawl = safer.sqlBool(rs.getString("DoCrawl"));
            int boardType = rs.getInt("BoardType");
            int lockNid = rs.getInt("LockNID");

			Board board = new Board(
				boardId,
				boardName, 
				dispName,
				maxPageNum,
				lastCrawlTime,
				visible,
                interval,
	            mustCrawl,
	            boardType,
	            lockNid); 
			
			rs.close();

			return board;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public ThreadList queryThreadList(
			int boardId, 
			int start, 
			int num, 
			String[] topPosters,
			boolean newThreadToTop,
			int minArticleNum,
			Vector<String> delPosters) throws StoreException {
		try {
			final int maxThreadNum = 1000;
			
			if(num > maxThreadNum) {
				throw new StoreException("error: too large query specified.");
			}

			ThreadList board = new ThreadList();

			// SQL̍쐬
			String sqlText = "select * from Threads where boardID=" + boardId + " ";

			if(newThreadToTop && minArticleNum>0) {
				sqlText = sqlText + "and ArticleNum>=" + minArticleNum + " ";
			}
			
			if(delPosters != null) {
				for(int i=0; i<delPosters.size(); i++) {
					sqlText = sqlText + "and Poster != " + safer.text(delPosters.elementAt(i));
				}
			}

			sqlText = sqlText + " order by ";
			
			if(topPosters != null && topPosters.length > 0) {
				sqlText = sqlText + "(";
				String delim = "";

				for(int i=0; i<topPosters.length; i++) {
					sqlText = sqlText + " " + delim + " Poster=\"" + topPosters[i] + "\"";
					delim = "or";
				}

				sqlText = sqlText + ")";
				
				sqlText = sqlText + " and (date_sub(now(), interval 1 hour)<=CreateDate)";
				sqlText = "(" + sqlText + ") desc, ";
			}

//			if(newThreadToTop) {
//				sqlText = sqlText + "LastArticleDate desc ";
//			} else {
				sqlText = sqlText + "NID desc ";
//			}

			sqlText = sqlText + " limit " + start + "," + num;

			// w肳ꂽw肳ꂽXbhe擾
			ResultSet rs = sql.executeQuery(sqlText);

			// 擾ʂ̊eXbhɂāAXbh̓eɒǉ
			while(rs.next()) {
				NThread thread = new NThread(
					rs.getInt("Depth"),
					safer.sqlStr(rs.getString("DispID")),
					safer.sqlBool(rs.getString("IsAdmin")),
					rs.getInt("NID"),
					rs.getInt("Country"),
					safer.sqlStr(rs.getString("Title")),
					rs.getInt("ArticleNum"),
					safer.sqlStr(rs.getString("Poster")),
					safer.sqlDatetimeToDate(rs.getString("CreateDate")),
					safer.sqlDatetimeToDate(rs.getString("ModifiedDate")),
					rs.getInt("ViewNum"),
					safer.sqlStr(rs.getString("Mail")),
					safer.sqlDatetimeToDate(rs.getString("LastCrawlTime")),
                    safer.sqlStr(rs.getString("ImageURL")),
                    safer.sqlBool(rs.getString("IsHot")));
				
				board.addThread(thread);
			}

			rs.close();
			return board;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public int queryThreadListOnLogMode(
			int boardId, 
			int start, 
			int num,
			ThreadList board) throws StoreException {
		try {
			final int maxThreadNum = 1000;
			
			if(num > maxThreadNum) {
				throw new StoreException("error: too large query specified.");
			}

			// ŐVXbhNID擾
			ResultSet rs = sql.executeQuery("select max(NID) from Threads where BoardID=" + boardId);
			rs.next();
			int maxnid = rs.getInt(1);
			rs.close();

			// SQL̍쐬
			String sqlText = "select * from Threads where boardID=" + boardId + " ";

			sqlText = sqlText + "and NID <= " + (maxnid-start) + " ";
			sqlText = sqlText + "order by NID desc limit 0," + num;

			// w肳ꂽw肳ꂽXbhe擾
			rs = sql.executeQuery(sqlText);

			// 擾ʂ̊eXbhɂāAXbh̓eɒǉ
			while(rs.next()) {
				NThread thread = new NThread(
					rs.getInt("Depth"),
					safer.sqlStr(rs.getString("DispID")),
					safer.sqlBool(rs.getString("IsAdmin")),
					rs.getInt("NID"),
					rs.getInt("Country"),
					safer.sqlStr(rs.getString("Title")),
					rs.getInt("ArticleNum"),
					safer.sqlStr(rs.getString("Poster")),
					safer.sqlDatetimeToDate(rs.getString("CreateDate")),
					safer.sqlDatetimeToDate(rs.getString("ModifiedDate")),
					rs.getInt("ViewNum"),
					safer.sqlStr(rs.getString("Mail")),
					safer.sqlDatetimeToDate(rs.getString("LastCrawlTime")),
                    safer.sqlStr(rs.getString("ImageURL")),
                    safer.sqlBool(rs.getString("IsHot")));
				
				board.addThread(thread);
			}

			rs.close();
			return maxnid-start;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public ArticleList queryArticleList(int boardId, int nid, int start, int num,  Vector<String> delPosters) throws StoreException {
		try {
			String sqlText;
			ResultSet rs;
			
/*			// Xe擾
			sqlText = "select Title, Country, Poster, CreateDate, Content from Threads where BoardId = " + boardId + " and NID=" + nid;

			rs = sql.executeQuery(sqlText);
			
			rs.next();
	
			String title = rs.getString("Title");
			int country = rs.getInt("Country");
			String poster = rs.getString("Poster");
			java.util.Date createDate = safer.sqlDatetimeToDate(rs.getString("CreateDate"));
			String content = rs.getString("Content");

			rs.close();		

			// ̃Xbh̃X擾
			rs = sql.executeQuery("select count(*) as total from Articles where BoardID=" + boardId + " and NID=" + nid);
			rs.next();
			int totalArticleCount = rs.getInt("total");
			rs.close();
*/
			ArticleList list = new ArticleList();
		
			// Xꗗ擾
			sqlText = "select * from Articles where BoardID=" + boardId + " and NID=" + nid;

			if(delPosters != null) {
				for(int i=0; i<delPosters.size(); i++) {
					sqlText = sqlText + " and Author != " + safer.text(delPosters.elementAt(i));
				}
			}
			
			sqlText = sqlText + " order by ArticleID ";


				sqlText = sqlText + "asc";


			sqlText = sqlText + " limit " + /*start*/ 0 + "," + (start + num);
			rs = sql.executeQuery(sqlText);

			TreeMap<Integer /* aid */, Integer /* index */> map = new TreeMap<Integer, Integer>();
			
			int aindx = 0;
			
			for(int i=0; i<start; i++) {
				if(!rs.next()) break;
				
				map.put(rs.getInt("ArticleID"), aindx++);
			}

			// eX̓e擾AXbhɒǉ
			while(rs.next()) {
				map.put(rs.getInt("ArticleID"), aindx++);
				
				Article art = new Article(
					rs.getString("Author"),
					rs.getInt("Country"),
					rs.getString("Content"),
					rs.getInt("ArticleID"),
					safer.sqlDatetimeToDate(rs.getString("PostTime")),
					rs.getInt("ParentArticleID")
				);
	
				if(art.parentArticleId != -1) {
					Integer parentIndex = map.get(art.parentArticleId);

					if(parentIndex != null) {
						art.parentArticleIndex = parentIndex;
					}
				}
				
				list.addArticle(art);
			}
	
			rs.close();

			return list;
		
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public ThreadDiff getThreadDiff(int boardId, ThreadTitle threadSummary) throws StoreException {
		try {
			ThreadDiff diff = null;
			// y[W̊eXbhT}̐VKEXV`FbN

			// XbhT}f[^x[Xɂ邩ǂ`FbN
			ResultSet rs = sql.executeQuery(
				"select NID, ArticleNum, isnull(Content) as nullchk from Threads where "
				+ "BoardID=" + boardId + " and NID=" + threadSummary.nid);
				
			if(!rs.next()) { 
				// Xbhf[^x[XɌȂꍇAVKXbhł邱Ƃޔ
				diff = new ThreadDiff(threadSummary.nid, ThreadDiff.NEW);
			}
			else { // f[^f[^x[Xɂꍇ́AXVKvǂ`FbN
				// ŌɊi[ۂ̓eƁAee擾ς݂ǂ擾
				int lastArticleNum = rs.getInt("ArticleNum");
				int nullchk = rs.getInt("nullchk");

				boolean mustUpdateArticle = (lastArticleNum != threadSummary.articleNum); // VKeǂ
				boolean mustUpdateContents = (nullchk == 1); //ee擾ς݂ǂ
				
				// VKeLA܂̓Rec擾̏ꍇA
				if(mustUpdateArticle || mustUpdateContents) {
					diff = new ThreadDiff(threadSummary.nid, ThreadDiff.ARTICLE | ThreadDiff.NO_CONTENTS);
				} else {
					// ύXȂ΁AtO=0ThreadDiffԂ
					diff = new ThreadDiff(threadSummary.nid, 0);
				}
			}

			rs.close();

			return diff;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
	
	public int getLastArticleId(int boardId, int nid) throws StoreException {
		try {
			// Ō̃XID擾
			ResultSet maxResult = sql.executeQuery(
				"select max(ArticleID) as maxid from Articles where BoardID=" + boardId + " and NID=" + nid);

			int max;
	
			// 擾ł΂Ō̃XIDƂA擾łȂ-1Ō̃XIDɂ		
			if(!maxResult.next()) {
				max = -1;
			} else {
				String s = maxResult.getString("maxid");
					
				if(s != null && !s.equals("NULL")) {
					max = Integer.parseInt(s);
				} else {
					max = -1;
				}
			}
			
			maxResult.close();
			return max;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public String queryThreadContents(int boardId, int nid) throws StoreException {
		try {
			// Xe擾
			String sqlText = "select Content from Threads where BoardID=" + boardId + " and NID=" + nid;
			ResultSet rs = sql.executeQuery(sqlText);

			rs.next();

			String content = rs.getString("Content");

			rs.close();
			return content;
		
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

    public String queryRawThreadContents(int boardId, int nid) throws StoreException {
        try {
            String sqlText = "select RawContent from Threads where BoardID=" + boardId + " and NID=" + nid;
            ResultSet rs = sql.executeQuery(sqlText);

            rs.next();
            
            String rawContent = rs.getString("RawContent");

            rs.close();
            return rawContent;
        
        } catch(SQLException e) {
            throw new StoreException(e);
        }
    }

	public NThread queryThread(int boardId, int nid) throws StoreException {
		try {
			ResultSet rs = sql.executeQuery("select * from Threads where BoardID=" + boardId + " and NID=" + nid);
			rs.next();

			NThread thread = new NThread(
				rs.getInt("Depth"),
				safer.sqlStr(rs.getString("DispID")),
				safer.sqlBool(rs.getString("IsAdmin")),
				rs.getInt("NID"),
				rs.getInt("Country"),
				safer.sqlStr(rs.getString("Title")),
				rs.getInt("ArticleNum"),
				safer.sqlStr(rs.getString("Poster")),
				safer.sqlDatetimeToDate(rs.getString("CreateDate")),
				safer.sqlDatetimeToDate(rs.getString("ModifiedDate")),
				rs.getInt("ViewNum"),
				safer.sqlStr(rs.getString("Mail")),
				safer.sqlDatetimeToDate(rs.getString("LastCrawlTime")),
                safer.sqlStr(rs.getString("ImageURL")),
                safer.sqlBool(rs.getString("IsHot")));
	
			rs.close();
			return thread;
		
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public ThreadTitle queryThreadTitle(int boardId, int nid) throws StoreException {
		try {
			ResultSet rs = sql.executeQuery("select * from Threads where BoardID=" + boardId + " and NID=" + nid);
			rs.next();

			ThreadTitle thread = new ThreadTitle(
				rs.getInt("Depth"),
				safer.sqlStr(rs.getString("DispID")),
				safer.sqlBool(rs.getString("IsAdmin")),
				rs.getInt("NID"),
				rs.getInt("Country"),
				safer.sqlStr(rs.getString("Title")),
				rs.getInt("ArticleNum"),
				safer.sqlStr(rs.getString("Poster")),
				safer.sqlDatetimeToDate(rs.getString("CreateDate")),
				safer.sqlDatetimeToDate(rs.getString("ModifiedDate")),
				rs.getInt("ViewNum"),
				safer.sqlStr(rs.getString("Mail")),
				safer.sqlDatetimeToDate(rs.getString("LastCrawlTime")),
				safer.sqlBool(rs.getString("IsHot")));
	
			rs.close();
			return thread;
		
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

	public boolean existsThread(int boardId, int threadId) throws StoreException {
		try {
			ResultSet rs = sql.executeQuery(
				"select count(*) as cnt from Threads where BoardID=" + boardId + " and NID=" + threadId);

			rs.next();
			int count = rs.getInt("cnt");

			rs.close();
			return count > 0;

		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}

    public int queryThreadTraffic(int boardId, int nid) throws StoreException {
        try {
            ResultSet rs;
            rs = sql.executeQuery("select Traffic from Threads where BoardID=" + boardId + " and NID=" + nid);
            rs.next();
            
            int traffic = rs.getInt("Traffic");
            rs.close();
            
            return traffic;
        } 
        catch(SQLException e) {
            throw new StoreException(e);
        }
    }

	public void deleteThread(int boardId, int nid) throws StoreException {
		try {
			sql.executeUpdate("delete from Articles where BoardID=" + boardId + " and NID=" + nid);
			sql.executeUpdate("delete from Threads where BoardID=" + boardId + " and NID=" + nid);
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
	
	public ArticleFullPathList queryNewArticles(int boardId, int num) throws StoreException {
		try {
			ResultSet rs = sql.executeQuery("select * from Articles use index (idx_order2) where BoardID=" + boardId 
				+ " order by ArticleID desc limit 0," + num);
			
			ArticleFullPathList articles = new ArticleFullPathList();
			
			while(rs.next()) {
				articles.addArticle(new ArticleFullPath(
					boardId,
					rs.getInt("NID"),
					rs.getString("Author"),
					rs.getInt("Country"),
					rs.getString("Content"),
					rs.getInt("ArticleID"),
					safer.sqlDatetimeToDate(rs.getString("PostTime")),
					rs.getInt("ParentArticleID")
				));
			}

			rs.close();
			return articles;
		} catch(SQLException e) {
			throw new StoreException(e);
		}
	}
}
