/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.browser.win32;

import java.util.*;
import java.awt.*;
import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.browser.*;

@SuppressWarnings("serial")
public class Win32BrowserImpl extends EditableBrowser {
	int hBrowser = 0;
	private final Win32BrowserThread browserThread;
	boolean firstNotify = true;
	private final Canvas canvas;

	private Vector<Runnable> browserCommands = new Vector<Runnable>();
	
	private void flushCommands() {
		if(browserCommands.size() > 0) {
			for(int i=0; i<browserCommands.size(); i++) {
				browserCommands.elementAt(i).run();
			}
			
			browserCommands.clear();
		}
	}
	
	private void execBrowserCommand(Runnable r) {
		if(hBrowser == 0) {
			browserCommands.addElement(r);
		} else {
			flushCommands();
			r.run();
		}
	}

	private class BrowserCanvas extends Canvas {
		public BrowserCanvas() {
			setFocusable(true);
			setMinimumSize(new Dimension(1, 1));
		}

		@Override public void addNotify() {
			super.addNotify();

			if(!firstNotify) ViewNaver.println(
					"xFBrowser#addNotify()Ă΂܂A"
					+ "Win32BrowserImpl͍ėpɑΉĂ܂B\n"
					+ "sƊ댯ȏԂƂȂ\܂B\n");

			if(hBrowser == 0) {
				hBrowser = browserThread.createWin32Browser(this);
				Win32BrowserThread.addBrowser(Win32BrowserImpl.this);
				firstNotify = false;
				flushCommands();
			}
		}

		@Override public void removeNotify() {
			Win32BrowserThread.removeBrowser(Win32BrowserImpl.this);

			if(hBrowser != 0) {
				browserThread.disposeWin32Browser(hBrowser);
				hBrowser = 0;
			}

			super.removeNotify();
		}


		@Override public void setSize(int width, int height) {
			super.setSize(width, height);
			resizeBrowser(width, height);
		}

		@Override public void setSize(Dimension d) {
			super.setSize(d);
			resizeBrowser(d.width, d.height);
		}

		@Override public void setBounds(int x, int y, int width, int height) {
			super.setBounds(x, y, width, height);
			resizeBrowser(width, height);
		}

		@Override public void setBounds(Rectangle r) {
			super.setBounds(r);
			resizeBrowser(r.width, r.height);
		}
	}
	
	public Win32BrowserImpl(Win32BrowserThread browserThread) {
		canvas = new BrowserCanvas();

		canvas.setFocusable(true);
		this.browserThread = browserThread;
	}

	@Override public Component getComponent() {
		return canvas;
	}


	private void resizeBrowser(final int width, final int height) {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.resizeWin32Browser(hBrowser, width, height);
			}
		});
	}

	@Override public void setBodyHtml(final String html) {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.setWin32BodyHtml(hBrowser, html);
			} 
		});
	}
	
	@Override public void insertBodyHtml(final String tagid, final String html) {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.insertWin32BodyHtml(hBrowser, tagid, html);
			}
		});
	}
	
	@Override public void replaceBodyHtml(final String tagid, final String html) {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.replaceWin32BodyHtml(hBrowser, tagid, html);
			}
		});
	}
	
	@Override public String getBodyHtml() {
		if(hBrowser == 0) return null;
		return browserThread.getWin32BodyHtml(hBrowser);
	}
	
	@Override public void setEditable(boolean editable) {
		if(hBrowser == 0) return;
		browserThread.setWin32Editable(hBrowser, editable);
	}

	@Override public void toggleBold() {
		if(hBrowser == 0) return;
		browserThread.toggleWin32Bold(hBrowser);
	}
	
	@Override public void setColor(String color) {
		if(hBrowser == 0) return;
		browserThread.setWin32Color(hBrowser, color);
	}
	
	@Override public void insertHR() {
		if(hBrowser == 0) return;
		browserThread.insertWin32HR(hBrowser);
	}
	
	@Override public void toggleItalic() {
		if(hBrowser == 0) return;
		browserThread.toggleWin32Italic(hBrowser);
	}
	
	@Override public void toggleUnderLine() {
		if(hBrowser == 0) return;
		browserThread.toggleWin32UnderLine(hBrowser);
	}
	
	@Override public void setFontName(String fontName) {
		if(hBrowser == 0) return;
		browserThread.setWin32FontName(hBrowser, fontName);
	}
	
	@Override public void setFontSize(String fontSize) {
		if(hBrowser == 0) return;
		browserThread.setWin32FontSize(hBrowser, fontSize);
	}

	@Override public void scrollTop(final String tagid) {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.win32scrollTop(hBrowser, tagid);

			}
		});
	}

	/** @deprecated */
	@Override public void refresh() {
		if(hBrowser == 0) return;
		browserThread.win32Refresh(hBrowser);
	}
	
	/** @deprecated */
	@Override public void reset() {
		if(hBrowser == 0) return;
		browserThread.win32Reset(hBrowser);
	}

	@Override public void writeDocHtml(final String html) {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.writeWin32DocHtml(hBrowser, html);
			}
		});
	}
	
	@Override public void clearDocHtml() {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.clearWin32DocHtml(hBrowser);
			}
		});
	}
	
	@Override public void closeDocHtml() {
		execBrowserCommand(new Runnable() {
			public void run() {
				browserThread.closeWin32DocHtml(hBrowser);
			}
		});
	}

}
