/*
 * Copyright (c) 2007, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.server.image;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.*;
import javax.imageio.*;

import Acme.Serve.servlet.*;

import Acme.Serve.servlet.http.*;
import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.net.*;

public class ImageCache {
	private Vector<ImageCacheItem> images = new Vector<ImageCacheItem>();
	private HashMap<String /*url*/, ImageCacheItem> urlMapping = new HashMap<String, ImageCacheItem>();

	public final int maxWidth;
	public final int maxHeight;
	public final int maxSize;
	public final HashMap<RenderingHints.Key, Object> renderingHints;
	
	public static final HashMap<RenderingHints.Key, Object> RENDERING_HINTS_STANDARD =  new HashMap<RenderingHints.Key, Object>() {
		{
			put(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
			put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
			put(RenderingHints.KEY_DITHERING, RenderingHints.VALUE_DITHER_ENABLE);
			put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
			put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
			put(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
			put(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_NORMALIZE);
		}
	};

	public static final HashMap<RenderingHints.Key, Object> RENDERING_HINTS_HISPEED =  new HashMap<RenderingHints.Key, Object>() {
		{
			put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
			put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
			put(RenderingHints.KEY_DITHERING, RenderingHints.VALUE_DITHER_DISABLE);
			put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
			put(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);
			put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);
			put(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
			put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_SPEED);
			put(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_DEFAULT);
		}
	};
	
	public ImageCache(int maxImageWidth, int maxImageHeight, int maxImageSizePerKB, HashMap<RenderingHints.Key, Object> renderingHints) {
		this.maxWidth = maxImageWidth;
		this.maxHeight = maxImageHeight;
		this.maxSize = maxImageSizePerKB * 1024;
		this.renderingHints = renderingHints;
	}

	public synchronized int putImageUrl(String url) {
		ImageCacheItem image = urlMapping.get(url);

		if(image != null) {
			return image.no;
		}

		image = new ImageCacheItem(images.size(), url);
		images.addElement(image);
		urlMapping.put(url, image);
		
		return image.no;
	}

	private byte[] shrinkImage(String url) throws IOException, NetException {
		byte[] bin = ViewNaver.instance.server.outerSession.getImageBinary(url).bin;
	
		BufferedImage bufferedImage = ImageIO.read(new ByteArrayInputStream(bin));

		int width = bufferedImage.getWidth();
		int height = bufferedImage.getHeight();

		if(width > maxWidth || height > maxHeight) {
			if(width > maxWidth) {
				height = height * maxWidth / width;
				width = maxWidth;
			}

			if(height > maxHeight) {
				width = width * maxHeight / height;
				height = maxHeight;
			}

			// O
			if(width > maxWidth) width = maxWidth;

			BufferedImage shrinkImage = new BufferedImage(width,height, bufferedImage.getType());
			Graphics2D g2d = shrinkImage.createGraphics();
			g2d.setRenderingHints(renderingHints);
			g2d.drawImage(bufferedImage, 0, 0, width, height, null);

			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			
			ImageIO.write(shrinkImage, "jpg", bout);
			g2d.dispose();
			bout.close();
			
			bin = bout.toByteArray();
		} 

		if(bin.length > maxSize) throw new IOException("too large image size " + maxSize);

		return bin;
	}
	
	public static final LoginSession.ImageBinary INVALID_IMAGE_BINARY = new LoginSession.ImageBinary(null, null);
	
	public void putImage(HttpServletResponse response, int imageNo, ServletOutputStream out) throws IOException, NetException {
		ImageCacheItem image = null;

		try {
			image = images.elementAt(imageNo);

			synchronized(image) {
				if(image.binary == INVALID_IMAGE_BINARY) {
					putInvalidImage(response, out);
					return;
				}

				if(image.binary == null) {
					byte[] shrinkedImage = shrinkImage(image.url);
	
					image.binary = new LoginSession.ImageBinary(
						"image/jpeg",
						shrinkedImage);
				}
			}

			response.setStatus(HttpServletResponse.SC_OK);
			response.setContentType(image.binary.contentType);

			out.write(image.binary.bin);
		} catch(Exception e) {
			if(image != null) image.binary = INVALID_IMAGE_BINARY;
			putInvalidImage(response, out);
		}
	}

	private void putInvalidImage(HttpServletResponse response, ServletOutputStream out) {
		response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
	}

}
