/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.xml;


import java.text.*;
import java.util.*;
import org.w3c.dom.*;

import com.aibonware.viewnaver.model.*;

public class BaseXMLReader {
    protected SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
    
	public BaseXMLReader() {
	}
	
	public static final Element getAttrElem(Element elem, String name) throws XMLReaderException {
		Node child = elem.getFirstChild();

		while(child != null) {
			if(child instanceof Element && child.getNodeName().equals(name)) return (Element)child;
			
			child = child.getNextSibling();
		}
		
		throw new XMLReaderException("^O" + name + "܂");
	}
	
	public static final Vector<Element> getAttrElems(Element elem, String name) {
		Vector<Element> children = new Vector<Element>();
		
		Node child = elem.getFirstChild();

		while(child != null) {
			if(child instanceof Element && child.getNodeName().equals(name)) {
				children.addElement((Element)child);
			}
			
			child = child.getNextSibling();
		}

		return children;
	}
	
	protected String readStringElem(Element elem, String name) throws XMLReaderException {
		String s;
		
		try {
			s = getAttrElem(elem, name).getTextContent();
		} catch(XMLReaderException e) {
			return null;
		}
	
		return s;
	}

	protected int readIntElem(Element elem, String name) throws XMLReaderException {
		String s = getAttrElem(elem, name).getTextContent();
		
		try {
			return Integer.parseInt(s);
		} catch(NumberFormatException e) {
			throw new XMLReaderException("^O'" + name + "'̒l'" + s + "'͕sȐ\łB", e);
		}
	}

	protected boolean readBoolElem(Element elem, String name) throws XMLReaderException {
		String s = getAttrElem(elem, name).getTextContent();
		
		return Boolean.parseBoolean(s);
	}
	
	protected Date readDateElem(Element elem, String name) throws XMLReaderException {
		String s;
		
		try {
			s = getAttrElem(elem, name).getTextContent();
		} catch(XMLReaderException e) {
			return null;
		}
		
		try {
			return df.parse(s);
		} catch(ParseException e) {
			throw new XMLReaderException("^O'" + name + "'̒l'" + s + "'͕sȓt\łB", e);
		}
	}
	
	protected Country readCountryElem(Element elem, String name) throws XMLReaderException {
		String s = getAttrElem(elem, name).getTextContent();
		
		try {
			return Country.valueOf(s);
		} catch(IllegalArgumentException e) {
			throw new XMLReaderException("^O'" + name + "'̒l'" + s + "'͕sȍʕ\łB", e);
		}
	}
}
