/*
 * Copyright (c) 2007, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.browser.basic;

import java.awt.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.text.html.*;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.browser.*;
import javax.swing.event.*;

public class BasicBrowserImpl extends EditableBrowser {
	public JEditorPane comp;
	public JScrollPane scr;
	
	public BasicBrowserImpl(boolean editable) {
		comp = new JEditorPane();
		scr = new JScrollPane(comp);
		
		comp.setEditable(editable);
		comp.setContentType("text/html");

		comp.addHyperlinkListener(new HyperlinkListener () {
			public void hyperlinkUpdate(HyperlinkEvent e) {
				if(e.getEventType() != HyperlinkEvent.EventType.ACTIVATED) return;
				if(e.getDescription() == null) return;
				
				String url = e.getDescription();
				notifyBeforeNavigate(url);
			}
		});
		
	}

	public Component getComponent() {
		return scr;
	}
	
	private Element getBodyElement(HTMLDocument doc) {
		Element root = doc.getDefaultRootElement();
		
		for(int i=0; i<root.getElementCount(); i++) {
			Element child = root.getElement(i);
			if(child.getName().equalsIgnoreCase("body")) {
				return child;
			}
		}

		return null;
	}
	
	public void setBodyHtml(String html) {
		try {
			HTMLDocument doc = (HTMLDocument)comp.getDocument();
			Element body = getBodyElement(doc);
			doc.setInnerHTML(body, html);
		} catch(Exception e) {
			ViewNaver.err(e);
		}
	}

	public void insertBodyHtml(String tagid, String html) {
		try {
			HTMLDocument doc = (HTMLDocument)comp.getDocument();
			Element elem = doc.getElement(tagid);
			doc.insertAfterEnd(elem, html);
		} catch(Exception e) {
			ViewNaver.err(e);
		}
	}

	public void replaceBodyHtml(String tagid, String html) {
		try {
			HTMLDocument doc = (HTMLDocument)comp.getDocument();
			Element elem = doc.getElement(tagid);
			doc.setInnerHTML(elem, "<div align=\"left\">" + html + "</div>");
		} catch(Exception e) {
			ViewNaver.err(e);
		}
	}
	
	public String getBodyHtml() {
		boolean oldEditable = comp.isEditable();
		
		try {
			if(oldEditable) comp.setEditable(false);
			
			HTMLDocument doc = (HTMLDocument)comp.getDocument();
			Element body = getBodyElement(doc);
	
			int start = body.getStartOffset();
			int end = body.getEndOffset();
		
			String html = doc.getText(start, end - start);
			
			return html;
		} catch(Exception e) {
			ViewNaver.err(e);
			return "";
		} finally {
			comp.setEditable(oldEditable);
		}
	}

	public void scrollTop(String tagid) {
		scr.getVerticalScrollBar().setValue(0);
	}

	public void refresh() {}
	public void reset() {}

	private StringBuffer writingBuf = null;

	public void beginDocWriting() {
		writingBuf = new StringBuffer();
	}
	
	public void writeDocHtml(String html) {
		writingBuf.append(html);
	}

	public void endDocWriting() {
		String html = writingBuf.toString();
		comp.setText(html);
		writingBuf = null;
	}
	
	public void clearDocHtml() {
		comp.setText("");
	}
	
	public void closeDocHtml() {}


	
	// EditableBrowser
	
	public void toggleBold() {}
	public void setColor(String color) {}
	public void insertHR() {}
	public void toggleItalic() {}
	public void toggleUnderLine() {}
	public void setFontName(String fontName) {}
	public void setFontSize(String fontSize) {}

	public void setEditable(boolean editable) {}
}
