/*
 * File: dillocfg-gtk.c
 *
 * Copyright (C) 2003 Kiyo <kiyo@teki.jpn.ph>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#ifndef _WIN32
#include <gdk/gdkx.h>
#endif
#include <gtk/gtk.h>

#include "../config.h"
#include "../src/intl.h"

#define WIDTH  740
#define HEIGHT 340
#define USER_CFG ".dillo/dillorc"

#define GET_STR(obj) obj->get(obj)

typedef struct {
   guint      token;
   GtkWidget *button;
   gboolean   value;
   gboolean   default_value;
   gchar *(*get)();
} CheckOption;

typedef struct {
   guint      token;
   GtkWidget *hbox;
   GtkWidget *label;
   GtkWidget *entry;
   gchar     *value;
   gchar     *default_value;
   gboolean  allow_space;
   gchar *(*get)();
} EntryOption;

typedef struct {
   guint      token;
   GtkWidget *hbox;
   GtkWidget *label;
   GtkWidget *option_menu;
   gchar    **items;
   gint       value;
   gint       default_value;
   gboolean  allow_space;
   gchar *(*get)();
} SelectOption;

typedef struct {
   guint      token;
   GtkWidget *hbox;
   GtkWidget *label;
   GtkWidget *button;
   GtkWidget *button_image;
   GdkImage  *image;
   GtkWidget *dialog;
   GdkColor  *value;
   GdkColor  *default_value;
   gchar *(*get)();
} ColorOption;

typedef struct {
   guint      token;
   GtkWidget *hbox;
   GtkWidget *label;
   GtkWidget *button;
   GtkAdjustment *adj;
   double     value;
   double     default_value;
   gboolean is_integer;
   gchar *(*get)();
} SpinOption;

#ifndef DISABLE_TABS
typedef struct {
   GtkWidget   *vbox;
   GtkWidget   *table;
   GtkWidget   *table2;
   CheckOption *tab_bar_homogeneous;
   CheckOption *tab_bar_scroller;
   CheckOption *tab_bar_show_fullscreen;
   CheckOption *tab_bar_show_single_tab;
   CheckOption *tab_instead_of_window;
   CheckOption *tab_load_in_background;
   EntryOption *compress_vowels;
   EntryOption *compress_common_prefixes;
   CheckOption *tab_title_compress;
} DillocfgTab;
#endif

typedef struct {
   GtkWidget   *vbox;
   GtkWidget   *table;
   CheckOption *use_dicache;
   CheckOption *focus_location_on_new;
   CheckOption *enterpress_forces_submit;
   CheckOption *limit_text_width;
   CheckOption *generate_submit;
} DillocfgOther;

typedef struct {
   GtkWidget   *vbox;
   GtkWidget   *table;
   SpinOption *geometry_x;
   SpinOption *geometry_y;
   SelectOption *panel_size;
   CheckOption *show_back;
   CheckOption *show_bookmarks;
   CheckOption *show_clear_url;
   CheckOption *show_forw;
   CheckOption *show_home;
   CheckOption *show_menubar;
   CheckOption *show_popup_navigation;
   CheckOption *show_progress_box;
   CheckOption *show_reload;
   CheckOption *show_save;
   CheckOption *show_stop;
   CheckOption *show_tooltip;
   CheckOption *show_url;
   CheckOption *show_user_agent;
   CheckOption *small_icons;
   CheckOption *fullwindow_start;
   CheckOption *transient_dialogs;
} DillocfgInterface;

typedef struct {
   GtkWidget   *vbox;
   GtkWidget   *table;
   GtkWidget   *table2;
   EntryOption *font_sizes;
   SpinOption  *font_factor;
   EntryOption *vw_fontname;
   EntryOption *fw_fontname;
   EntryOption *vw_aafontname;
   EntryOption *fw_aafontname;
   CheckOption *use_oblique;
   CheckOption *limit_font_decoration;
} DillocfgFont;

typedef struct {
   GtkWidget   *vbox;
   GtkWidget   *table;
   GtkWidget   *table2;
   CheckOption *force_my_colors;
   CheckOption *force_visited_color;
   CheckOption *allow_white_bg;
   ColorOption *link_color;
   ColorOption *bg_color;
   ColorOption *visited_color;
   ColorOption *text_color;
} DillocfgColor;

typedef struct {
   GtkWidget   *vbox;
   GtkWidget   *table;
   SelectOption *user_agent;
   EntryOption *home;
   EntryOption *startup;
   EntryOption *http_proxy;
   EntryOption *no_proxy;
} DillocfgNetwork;

typedef struct {
   GtkWidget *vbox;
   GtkWidget *dialog_main;
   GtkWidget *notebook;
   GtkWidget *button_hbox;
   GtkWidget *button_reset;
   GtkWidget *button_reload;
   GtkWidget *button_save;
   GtkWidget *button_quit;
   DillocfgInterface *interface;
   DillocfgFont *font;
   DillocfgColor     *color;
   DillocfgNetwork   *network;
   DillocfgTab   *tab;
   DillocfgOther   *other;
} DillocfgWin;

/* define enumeration values to be returned for specific symbols */
typedef enum {
   TOKEN_GEOMETRY = 0,
   TOKEN_GEOMETRY_X,
   TOKEN_GEOMETRY_Y,
   TOKEN_PROXY,
   TOKEN_NOPROXY,
   TOKEN_USER_AGENT,
   TOKEN_LINK_COLOR,
   TOKEN_VISITED_COLOR,
   TOKEN_BG_COLOR,
   TOKEN_ALLOW_WHITE_BG,
   TOKEN_FORCE_MY_COLORS,
   TOKEN_FORCE_VISITED_COLOR,
   TOKEN_TEXT_COLOR,
   TOKEN_USE_OBLIQUE,
   TOKEN_STARTUP,
   TOKEN_HOME,
   TOKEN_PANEL_SIZE,
   TOKEN_SMALL_ICONS,
   TOKEN_FONT_FACTOR,
   TOKEN_FONT_SIZES,
   TOKEN_SHOW_TOOLTIP,
   TOKEN_LIMIT_TEXT_WIDTH,
   TOKEN_LIMIT_FONT_DECORATION,
   TOKEN_USE_DICACHE,
   TOKEN_SHOW_BACK,
   TOKEN_SHOW_FORW,
   TOKEN_SHOW_HOME,
   TOKEN_SHOW_RELOAD,
   TOKEN_SHOW_SAVE,
   TOKEN_SHOW_STOP,
   TOKEN_SHOW_BOOKMARKS,
   TOKEN_SHOW_MENUBAR,
   TOKEN_SHOW_CLEAR_URL,
   TOKEN_SHOW_URL,
   TOKEN_SHOW_PROGRESS_BOX,
   TOKEN_SHOW_POPUP_NAVIGATION,
   TOKEN_SHOW_USER_AGENT,
   TOKEN_FULLWINDOW_START,
   TOKEN_TRANSIENT_DIALOGS,
   TOKEN_FW_FONT,
   TOKEN_VW_FONT,
   TOKEN_FW_AAFONT,
   TOKEN_VW_AAFONT,
   TOKEN_GENERATE_SUBMIT,
   TOKEN_ENTERPRESS_FORCES_SUBMIT,
   TOKEN_FOCUS_LOCATION_ON_NEW,
#ifndef DISABLE_TABS
   TOKEN_TAB_LOAD_IN_BACKGROUND,
   TOKEN_TAB_INSTEAD_OF_WINDOW,
   TOKEN_TAB_BAR_SHOW_FULLSCREEN,
   TOKEN_TAB_BAR_SHOW_SINGLE_TAB,
   TOKEN_TAB_BAR_SCROLLER,
   TOKEN_TAB_BAR_HOMOGENEOUS,
   TOKEN_TAB_TITLE_COMPRESS,
   TOKEN_COMPRESS_VOWELS,
   TOKEN_COMPRESS_COMMON_PREFIXES,
#endif /* !DISABLE_TABS */

   TOKEN_LAST
} Dillo_Rc_TokenType;

/* symbol array */
static const struct {
   gchar *name;
   guint  token;
} symbols[] = {
   { "geometry", TOKEN_GEOMETRY },
   { "http_proxy", TOKEN_PROXY },
   { "no_proxy", TOKEN_NOPROXY },
   { "user_agent", TOKEN_USER_AGENT },
   { "link_color", TOKEN_LINK_COLOR },
   { "visited_color", TOKEN_VISITED_COLOR, },
   { "bg_color", TOKEN_BG_COLOR },
   { "allow_white_bg", TOKEN_ALLOW_WHITE_BG },
   { "force_my_colors", TOKEN_FORCE_MY_COLORS },
   { "force_visited_color", TOKEN_FORCE_VISITED_COLOR },
   { "text_color", TOKEN_TEXT_COLOR },
   { "use_oblique", TOKEN_USE_OBLIQUE },
   { "startup", TOKEN_STARTUP },
   { "home", TOKEN_HOME },
   { "show_tooltip", TOKEN_SHOW_TOOLTIP },
   { "panel_size", TOKEN_PANEL_SIZE },
   { "small_icons", TOKEN_SMALL_ICONS },
   { "limit_text_width", TOKEN_LIMIT_TEXT_WIDTH },
   { "limit_font_decoration", TOKEN_LIMIT_FONT_DECORATION },
   { "font_factor", TOKEN_FONT_FACTOR },
   { "font_sizes", TOKEN_FONT_SIZES },
   { "use_dicache", TOKEN_USE_DICACHE },
   { "show_back", TOKEN_SHOW_BACK },
   { "show_forw", TOKEN_SHOW_FORW },
   { "show_home", TOKEN_SHOW_HOME },
   { "show_reload", TOKEN_SHOW_RELOAD },
   { "show_save", TOKEN_SHOW_SAVE },
   { "show_stop", TOKEN_SHOW_STOP },
   { "show_bookmarks", TOKEN_SHOW_BOOKMARKS },
   { "show_menubar", TOKEN_SHOW_MENUBAR },
   { "show_clear_url", TOKEN_SHOW_CLEAR_URL },
   { "show_url", TOKEN_SHOW_URL },
   { "show_progress_box", TOKEN_SHOW_PROGRESS_BOX },
   { "show_popup_navigation", TOKEN_SHOW_POPUP_NAVIGATION },
   { "show_user_agent", TOKEN_SHOW_USER_AGENT },
   { "fullwindow_start", TOKEN_FULLWINDOW_START },
   { "transient_dialogs", TOKEN_TRANSIENT_DIALOGS },
   { "vw_aafontname", TOKEN_VW_AAFONT },
   { "fw_aafontname", TOKEN_FW_AAFONT },
   { "vw_fontname", TOKEN_VW_FONT },
   { "fw_fontname", TOKEN_FW_FONT },
   { "generate_submit", TOKEN_GENERATE_SUBMIT },
   { "enterpress_forces_submit", TOKEN_ENTERPRESS_FORCES_SUBMIT },
   { "focus_location_on_new", TOKEN_FOCUS_LOCATION_ON_NEW },
#ifndef DISABLE_TABS
   { "tab_load_in_background", TOKEN_TAB_LOAD_IN_BACKGROUND },
   { "tab_instead_of_window", TOKEN_TAB_INSTEAD_OF_WINDOW },
   { "tab_bar_show_fullscreen", TOKEN_TAB_BAR_SHOW_FULLSCREEN },
   { "tab_bar_show_single_tab", TOKEN_TAB_BAR_SHOW_SINGLE_TAB },
   { "tab_bar_scroller", TOKEN_TAB_BAR_SCROLLER },
   { "tab_bar_homogeneous", TOKEN_TAB_BAR_HOMOGENEOUS },
   { "tab_title_compress", TOKEN_TAB_TITLE_COMPRESS },
   { "compress_vowels", TOKEN_COMPRESS_VOWELS },
   { "compress_common_prefixes", TOKEN_COMPRESS_COMMON_PREFIXES },
#endif /* !DISABLE_TABS */
};

#define PANEL_SIZES_NUM 3
static gchar *panel_sizes[PANEL_SIZES_NUM] = {"tiny", "medium", "large"};
#define USER_AGENTS_NUM 7
static gchar *user_agents[USER_AGENTS_NUM]  = {
"Dillo/%v",
"Mozilla/3.0 (compatible;) Dillo/%v",
"Mozilla/4.0 (compatible; MSIE 4.0;) Dillo/%v",
"Mozilla/4.0 (Macintosh; U; PPC Mac OS X; ja-jp) Dillo/%v",
"DoCoMo/1.0/x505i (Dillo/%v)",
"J-PHONE/4.2/J-x53 (Dillo/%v)",
"KDDI-x24 UP.Browser/6.0.x.x (GUI) MMP/1.1 (Dillo/%v)"
};

static const guint n_symbols = sizeof (symbols) / sizeof (symbols[0]);
static gboolean modified_ = FALSE;

const gchar *get_token_name(guint tkn);
CheckOption *CheckOption_new(guint token, const gchar *label, const gboolean defaultvalue,
      GtkWidget *table, gint x, gint y);
void CheckOption_init(CheckOption *c, gboolean use_default);
gchar *CheckOption_get(CheckOption *c);
void CheckOption_free(CheckOption *c);
EntryOption *EntryOption_new(guint token, const gchar *label, const gchar *defaultvalue,
      gboolean allow_space, GtkWidget *table, gint x, gint y);
void EntryOption_init(EntryOption *e, gboolean use_default);
gchar *EntryOption_get(EntryOption *e);
void EntryOption_free(EntryOption *e);
SelectOption *SelectOption_new(guint token, const gchar *label, GtkWidget *menu, const gint defaultvalue, gchar **items, gboolean allow_space, GtkWidget *table, gint x, gint y);
void SelectOption_init(SelectOption *s, gboolean use_default);
gchar *SelectOption_get(SelectOption *s);
void SelectOption_free(SelectOption *s);
ColorOption *ColorOption_new(guint token, const gchar *label, const GdkColor *defaultvalue,
      GtkWidget *table, gint x, gint y);
void ColorOption_init(ColorOption *c, gboolean use_default);
gchar *ColorOption_get(ColorOption *c);
void ColorOption_free(ColorOption *c);
SpinOption *SpinOption_new(guint token, const gchar *label, const gfloat defaultvalue,
      gfloat lower, gfloat upper, gfloat increment, guint digits,
      GtkWidget *table, gint x, gint y);
void SpinOption_init(SpinOption *s, gboolean use_default);
gchar *SpinOption_get(SpinOption *s);
void SpinOption_free(SpinOption *s);


DillocfgWin *DillocfgWin_new();
void DillocfgWin_init(DillocfgWin *d, gboolean use_default);
void DillocfgWin_free(GtkWidget *widget, DillocfgWin *d);
void DillocfgWin_callback_reset(GtkButton *button, gpointer user_data);
void DillocfgWin_callback_reload(GtkButton *button, gpointer user_data);
void DillocfgWin_callback_save(GtkButton *button, gpointer user_data);
DillocfgInterface *DillocfgInterface_new(DillocfgWin *w, const gchar *label);
void DillocfgInterface_init(DillocfgInterface *i, gboolean use_default);
void DillocfgInterface_free(DillocfgInterface *i);
DillocfgFont *DillocfgFont_new(DillocfgWin *w, const gchar *label);
void DillocfgFont_init(DillocfgFont *r, gboolean use_default);
void DillocfgFont_free(DillocfgFont *r);
DillocfgColor *DillocfgColor_new(DillocfgWin *w, const gchar *label);
void DillocfgColor_init(DillocfgColor *c, gboolean use_default);
void DillocfgColor_free(DillocfgColor *c);
DillocfgNetwork *DillocfgNetwork_new(DillocfgWin *w, const gchar *label);
void DillocfgNetwork_init(DillocfgNetwork *n, gboolean use_default);
void DillocfgNetwork_free(DillocfgNetwork *n);
#ifndef DISABLE_TABS
DillocfgTab *DillocfgTab_new(DillocfgWin *w, const gchar *label);
void DillocfgTab_init(DillocfgTab *g, gboolean use_default);
void DillocfgTab_free(DillocfgTab *g);
#endif
DillocfgOther *DillocfgOther_new(DillocfgWin *w, const gchar *label);
void DillocfgOther_init(DillocfgOther *g, gboolean use_default);
void DillocfgOther_free(DillocfgOther *g);
void ReadUserconfig(DillocfgWin *d);
void WriteUserconfig(DillocfgWin *d);

/****************************** M A I N ********************************/
int main(int argc, char **argv) {
   DillocfgWin *dialog;

   /* set locale */
   gtk_set_locale();
   /* Initialize Gettext */
   bindtextdomain (PACKAGE, LOCALEDIR);
   textdomain (PACKAGE);
   /* Initialize GUI and parse GTK related args */
   gtk_init(&argc, &argv);
   gdk_rgb_init();

   dialog = DillocfgWin_new();
   ReadUserconfig(dialog);

   /* Start the GTK+ cycle */
   gtk_main();

   g_print("Dillocfg:normal exit\n");
   gtk_exit (0);
   return 0;
}

const gchar *get_token_name(guint tkn) {
   gint i;
   for(i=0;i<n_symbols&&symbols[i].token != tkn;i++);
   return symbols[i].name;
}

CheckOption *CheckOption_new(guint token, const gchar *label, const gboolean defaultvalue,
      GtkWidget *table, gint x, gint y) {
   CheckOption *c = g_new(CheckOption, 1);

   c->token         = token;
   c->button        = gtk_check_button_new_with_label(label);
   c->value         = defaultvalue;
   c->default_value = defaultvalue;
   c->get           = CheckOption_get;
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(c->button), c->value);
   gtk_table_attach_defaults(GTK_TABLE(table), c->button, x, x + 1, y, y + 1);
   return c;
}

void CheckOption_init(CheckOption *c, gboolean use_default) {
   if(use_default) c->value = c->default_value;
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(c->button), c->value);
}

gchar *CheckOption_get(CheckOption *c) {
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(c->button)))
      return g_strdup_printf("%s=YES", get_token_name(c->token));
   else return g_strdup_printf("%s=NO", get_token_name(c->token));
}

void CheckOption_free(CheckOption *c) {
   g_free(c);
}

EntryOption *EntryOption_new(guint token, const gchar *label, const gchar *defaultvalue, gboolean allow_space,
      GtkWidget *table, gint x, gint y) {
   EntryOption *e = g_new(EntryOption, 1);

   e->token       = token;
   e->hbox        = gtk_hbox_new(FALSE, 0);
   e->label       = gtk_label_new(label);
   e->entry       = gtk_entry_new();
   e->value       = g_strdup(defaultvalue);
   e->default_value = g_strdup(defaultvalue);
   e->get           = EntryOption_get;
   e->allow_space = allow_space;
   //gtk_box_set_homogeneous(GTK_BOX(e->hbox), TRUE);
   gtk_label_set_justify (GTK_LABEL(e->label), GTK_JUSTIFY_LEFT);
   if (e->value) gtk_entry_set_text(GTK_ENTRY(e->entry), e->value);
   gtk_box_pack_start(GTK_BOX(e->hbox), e->label, FALSE, FALSE, 10);
   gtk_box_pack_end(GTK_BOX(e->hbox), e->entry, TRUE, TRUE, 5);
   gtk_table_attach_defaults(GTK_TABLE(table), e->hbox, x, x + 1, y, y + 1);
   return e;
}

void EntryOption_init(EntryOption *e, gboolean use_default) {
   if(use_default) {
      g_free(e->value);
      e->value = g_strdup(e->default_value);
   }
   if (e->value) gtk_entry_set_text(GTK_ENTRY(e->entry), e->value);
}

gchar *EntryOption_get(EntryOption *e) {
   gchar *format;
   if (e->allow_space) format = "%s=\"%s\"";
   else format = "%s=%s";
   return g_strdup_printf(format, get_token_name(e->token),
         gtk_entry_get_text(GTK_ENTRY(e->entry)));
}

void EntryOption_free(EntryOption *e) {
   g_free(e);
}

SelectOption *SelectOption_new(guint token, const gchar *label, GtkWidget *menu, const gint defaultvalue, gchar **items, gboolean allow_space, GtkWidget *table, gint x, gint y) {
   SelectOption *s = g_new(SelectOption, 1);
    
   s->token = token;
   s->hbox = gtk_hbox_new(FALSE, 0);
   s->label = gtk_label_new(label);
   s->option_menu = gtk_option_menu_new ();
   gtk_option_menu_set_menu (GTK_OPTION_MENU(s->option_menu), menu);
   s->value = defaultvalue;
   s->default_value = defaultvalue;
   s->get           = SelectOption_get;
   s->allow_space = allow_space;
   s->items = items;
   gtk_option_menu_set_history (GTK_OPTION_MENU(s->option_menu), s->value);
   gtk_box_pack_start(GTK_BOX(s->hbox), s->label, FALSE, FALSE, 5);
   gtk_box_pack_start(GTK_BOX(s->hbox), s->option_menu, TRUE, TRUE, 5);
   gtk_table_attach_defaults(GTK_TABLE(table), s->hbox, x, x + 1, y, y + 1);
   return s;
}

void SelectOption_init(SelectOption *s, gboolean use_default) {
   if(use_default) s->value = s->default_value;
   gtk_option_menu_set_history (GTK_OPTION_MENU(s->option_menu), s->value);
}

gchar *SelectOption_get(SelectOption *s) {
   gchar *format;
   GtkWidget *menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(s->option_menu));
   /* always null...
   gchar *str = NULL;
   GtkWidget *item = gtk_menu_get_active(GTK_MENU(menu));
   GtkWidget *label = GTK_BIN(GTK_MENU_ITEM(item))->child;
   if(!label) str = g_strdup("");
   else gtk_label_get(GTK_LABEL(label), &str);
   */
   gint i = g_list_index(GTK_MENU_SHELL(menu)->children,
                    GTK_OPTION_MENU(s->option_menu)->menu_item);
   if (s->allow_space) format = "%s=\"%s\"";
   else format = "%s=%s";
   return g_strdup_printf(format, get_token_name(s->token), s->items[i]);
}

void SelectOption_free(SelectOption *s) {
   g_free(s);
}

void change_color_GdkColor(const double *color, GdkColor* c) {
   c->red   = color[0] * 65535;
   c->green = color[1] * 65535;
   c->blue  = color[2] * 65535;
}

void change_color_double(const GdkColor* c, double *color) {
   color[0] = c->red/(double)65535;
   color[1] = c->green/(double)65535;
   color[2] = c->blue/(double)65535;
}

void ColorDialog_ok(GtkButton *button, gpointer user_data) {
   ColorOption *c = (ColorOption *)user_data;
   double color[4];
   gtk_color_selection_get_color(
         GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(c->dialog)->colorsel),
         color);
   change_color_GdkColor(color, c->value);
   ColorOption_init(c, FALSE);
   gtk_widget_destroy(c->dialog);
}

void ColorDialog_cancel(GtkButton *button, gpointer user_data) {
   ColorOption *c = (ColorOption *)user_data;
   gtk_widget_destroy(c->dialog);
}

void ColorOption_clicked(GtkButton *button, gpointer user_data) {
   ColorOption *c = (ColorOption *)user_data;
   gchar *title;
   double color[4];
   gtk_label_get(GTK_LABEL(c->label), &title);
   change_color_double(c->value, color);
   c->dialog = gtk_color_selection_dialog_new(title);
   gtk_color_selection_set_color(
         GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(c->dialog)->colorsel),
         color);
   gtk_signal_connect(
         GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(c->dialog)->ok_button),
         "clicked", GTK_SIGNAL_FUNC(ColorDialog_ok), c);
   gtk_signal_connect(
         GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(c->dialog)->cancel_button),
         "clicked", GTK_SIGNAL_FUNC(ColorDialog_cancel), c);
   gtk_widget_hide(GTK_COLOR_SELECTION_DIALOG(c->dialog)->help_button);
   gtk_widget_show(c->dialog);
}

#define PREVIEW_SIZE 48
ColorOption *ColorOption_new(guint token, const gchar *label, const GdkColor *defaultvalue,
      GtkWidget *table, gint x, gint y) {
   ColorOption *c = g_new(ColorOption, 1);

   c->token = token;
   c->hbox = gtk_hbox_new(FALSE, 0);
   c->label = gtk_label_new(label);
   c->button = gtk_button_new();
   c->image = gdk_image_new(GDK_IMAGE_NORMAL,
         gdk_visual_get_system(), PREVIEW_SIZE, PREVIEW_SIZE);
   c->button_image = gtk_image_new(c->image, NULL);
   c->value = g_new(GdkColor, 1);
   c->value = gdk_color_copy(defaultvalue);
   c->default_value = g_new(GdkColor, 1);
   c->default_value = gdk_color_copy(defaultvalue);
   c->get           = ColorOption_get;
   gtk_signal_connect(GTK_OBJECT(c->button), "clicked",
         GTK_SIGNAL_FUNC(ColorOption_clicked), c);
   gtk_container_add(GTK_CONTAINER(c->button), c->button_image);
   //gtk_box_set_homogeneous(GTK_BOX(c->hbox), TRUE);
   gtk_label_set_justify (GTK_LABEL(c->label), GTK_JUSTIFY_LEFT);
   gtk_box_pack_start(GTK_BOX(c->hbox), c->button, FALSE, FALSE, 10);
   gtk_box_pack_start(GTK_BOX(c->hbox), c->label, FALSE, FALSE, 10);
   gtk_table_attach_defaults(GTK_TABLE(table), c->hbox, x, x + 1, y, y + 1);
   return c;
}

guint32 change_color(GdkColor *c) {
   gdk_colormap_alloc_color(gdk_colormap_get_system(),
         c, TRUE, TRUE);
   return c->pixel;
}

void ColorOption_init(ColorOption *c, gboolean use_default) {
   int i, j;
   if(use_default) c->value = gdk_color_copy(c->default_value);
   for (i = 0; i < PREVIEW_SIZE; i++)
      for (j = 0; j < PREVIEW_SIZE; j++)
         gdk_image_put_pixel(c->image, i, j, change_color(c->value));
}

gchar *ColorOption_get(ColorOption *c) {
   return g_strdup_printf("%s=0x%.2x%.2x%.2x",
         get_token_name(c->token),
         c->value->red>>8,
         c->value->green>>8,
         c->value->blue>>8);
}

void ColorOption_free(ColorOption *c) {
   gdk_image_destroy (c->image);
   g_free(c);
}

SpinOption *SpinOption_new(guint token, const gchar *label, const gfloat defaultvalue,
      gfloat lower, gfloat upper, gfloat increment, guint digits,
      GtkWidget *table, gint x, gint y) {
   SpinOption *s = g_new(SpinOption, 1);

   s->token = token;
   s->adj = (GtkAdjustment *)gtk_adjustment_new(defaultvalue,
         lower, upper, increment, increment, increment);
   s->hbox = gtk_hbox_new(FALSE, 0);
   s->label = gtk_label_new(label);
   s->button = gtk_spin_button_new(s->adj, increment, digits);
   s->value = defaultvalue;
   s->default_value = defaultvalue;
   s->get           = SpinOption_get;
   if (digits == 0) s->is_integer = TRUE;
   else s->is_integer = FALSE;
   gtk_box_pack_start(GTK_BOX(s->hbox), s->label, FALSE, FALSE, 10);
   gtk_box_pack_start(GTK_BOX(s->hbox), s->button, FALSE, FALSE, 10);
   gtk_table_attach_defaults(GTK_TABLE(table), s->hbox, x, x + 1, y, y + 1);
   return s;
}

void SpinOption_init(SpinOption *s, gboolean use_default) {
   if(use_default) s->value = s->default_value;
   gtk_spin_button_set_value(GTK_SPIN_BUTTON(s->button), s->value);
}

gchar *SpinOption_get(SpinOption *s) {
   gboolean noname = FALSE;
   if (s->token == TOKEN_GEOMETRY_X || s->token == TOKEN_GEOMETRY_Y)
      noname = TRUE;
   if (s->is_integer)
      return g_strdup_printf("%s%s%d", (noname)?"":get_token_name(s->token),
            (noname)?"":"=",
            gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(s->button)));
   else
      return g_strdup_printf("%s%s%f", (noname)?"":get_token_name(s->token),
            (noname)?"":"=",
            gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(s->button)));
}

void SpinOption_free(SpinOption *s) {
   g_free(s);
}

DillocfgWin *DillocfgWin_new() {
   DillocfgWin *d = g_new(DillocfgWin, 1);
   d->dialog_main = gtk_window_new(GTK_WINDOW_TOPLEVEL);//DIALOG);
   gtk_window_set_title(GTK_WINDOW(d->dialog_main), _("Dillo Preferences"));
   gtk_window_set_policy(GTK_WINDOW(d->dialog_main), TRUE, TRUE, FALSE);
   gtk_signal_connect(GTK_OBJECT(d->dialog_main), "delete_event",
         GTK_SIGNAL_FUNC(gtk_widget_destroy), d);
   gtk_signal_connect(GTK_OBJECT(d->dialog_main), "destroy",
         GTK_SIGNAL_FUNC(DillocfgWin_free), d);
   gtk_window_set_wmclass(GTK_WINDOW(d->dialog_main), "dillocfg", _("Dillo Preferences"));
   gtk_widget_set_usize(d->dialog_main, WIDTH, HEIGHT);

   d->notebook = gtk_notebook_new();
   gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(d->notebook), TRUE);
   d->vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_add(GTK_CONTAINER(d->dialog_main), d->vbox);
   gtk_container_add(GTK_CONTAINER(d->vbox), d->notebook);
   d->button_hbox = gtk_hbox_new(FALSE, 0);
   d->button_reset = gtk_button_new_with_label(_("Reset"));
   d->button_reload = gtk_button_new_with_label(_("Reload"));
   d->button_save = gtk_button_new_with_label(_("Save"));
   d->button_quit = gtk_button_new_with_label(_("Quit"));
   gtk_signal_connect(GTK_OBJECT(d->button_reset), "clicked",
         GTK_SIGNAL_FUNC(DillocfgWin_callback_reset), d);
   gtk_signal_connect(GTK_OBJECT(d->button_reload), "clicked",
         GTK_SIGNAL_FUNC(DillocfgWin_callback_reload), d);
   gtk_signal_connect(GTK_OBJECT(d->button_save), "clicked",
         GTK_SIGNAL_FUNC(DillocfgWin_callback_save), d);
   gtk_signal_connect(GTK_OBJECT(d->button_quit), "clicked",
         GTK_SIGNAL_FUNC(DillocfgWin_free), d);
   gtk_box_set_spacing(GTK_BOX(d->button_hbox), 5);
   gtk_container_add(GTK_CONTAINER(d->button_hbox), d->button_reset);
   gtk_container_add(GTK_CONTAINER(d->button_hbox), d->button_reload);
   gtk_container_add(GTK_CONTAINER(d->button_hbox), d->button_save);
   gtk_container_add(GTK_CONTAINER(d->button_hbox), d->button_quit);
   gtk_box_pack_end(GTK_BOX(d->vbox), d->button_hbox, FALSE, FALSE, 5);

   /* Interface */
   d->interface = DillocfgInterface_new(d, _("Interface"));

   /* Font */
   d->font = DillocfgFont_new(d, _("Font"));

   /* Color */
   d->color     = DillocfgColor_new(d, _("Color"));

   /* Network */
   d->network   = DillocfgNetwork_new(d, _("Network"));

#ifndef DISABLE_TABS
   /* Tab */
   d->tab   = DillocfgTab_new(d, _("Tab"));
#endif

   /* Other */
   d->other   = DillocfgOther_new(d, _("Other"));

   gtk_widget_show_all(d->dialog_main);
   return d;
}

void DillocfgWin_init(DillocfgWin *d, gboolean use_default) {
   DillocfgInterface_init(d->interface, use_default);
   DillocfgFont_init(d->font, use_default);
   DillocfgColor_init(d->color, use_default);
   DillocfgNetwork_init(d->network, use_default);
#ifndef DISABLE_TABS
   DillocfgTab_init(d->tab, use_default);
#endif
   DillocfgOther_init(d->other, use_default);
}

void DillocfgWin_free(GtkWidget *widget, DillocfgWin *d) {
   DillocfgInterface_free(d->interface);
   DillocfgFont_free(d->font);
   DillocfgColor_free(d->color);
   DillocfgNetwork_free(d->network);
#ifndef DISABLE_TABS
   DillocfgTab_free(d->tab);
#endif
   DillocfgOther_free(d->other);
   g_free(d);
   gtk_main_quit();
}

#ifndef DISABLE_TABS
DillocfgTab *DillocfgTab_new(DillocfgWin *w, const gchar *label) {
   DillocfgTab *t = g_new(DillocfgTab, 1);
   gint x = 0;
   gint y = 0;
   t->vbox            = gtk_vbox_new(FALSE, 0);
   t->table           = gtk_table_new(3, 2, FALSE);
   t->table2           = gtk_table_new(3, 1, FALSE);
   gtk_table_set_homogeneous(GTK_TABLE(t->table), TRUE);
   gtk_table_set_homogeneous(GTK_TABLE(t->table2), TRUE);
   gtk_container_add(GTK_CONTAINER(t->vbox), t->table);
   gtk_container_add(GTK_CONTAINER(t->vbox), t->table2);
   t->tab_bar_homogeneous      = CheckOption_new(TOKEN_TAB_BAR_HOMOGENEOUS, _("Tab bar homogeneous"), TRUE, t->table, x, y++);
   t->tab_bar_scroller         = CheckOption_new(TOKEN_TAB_BAR_SCROLLER, _("Tab bar scroller"), FALSE, t->table, x, y++);
   t->tab_bar_show_fullscreen  = CheckOption_new(TOKEN_TAB_BAR_SHOW_FULLSCREEN, _("Tab bar show fullscreen"), FALSE, t->table, x, y++);
   x++; y=0;
   t->tab_bar_show_single_tab  = CheckOption_new(TOKEN_TAB_BAR_SHOW_SINGLE_TAB, _("Tab bar show single tab"), FALSE, t->table, x, y++);
   t->tab_instead_of_window    = CheckOption_new(TOKEN_TAB_INSTEAD_OF_WINDOW, _("Tab instead of window"), TRUE, t->table, x, y++);
   t->tab_load_in_background   = CheckOption_new(TOKEN_TAB_LOAD_IN_BACKGROUND, _("Tab load in background"), TRUE, t->table, x, y++);
   x = 0; y = 0;
   t->tab_title_compress
      = CheckOption_new(TOKEN_TAB_TITLE_COMPRESS, _("Tab title compress"), FALSE, t->table2, x, y++);
   t->compress_vowels = EntryOption_new(TOKEN_COMPRESS_VOWELS, _("Compress vowels"), "aeiouyAEIOUY", TRUE, t->table2, x, y++);
   t->compress_common_prefixes = EntryOption_new(TOKEN_COMPRESS_COMMON_PREFIXES, _("Compress_common_prefixes"),
         "index of ;re: ;fwd: ;www.;welcome to ;the ", TRUE, t->table2, x, y++);
   gtk_notebook_append_page(GTK_NOTEBOOK(w->notebook), t->vbox, gtk_label_new(label));
   return t;
}

void DillocfgTab_init(DillocfgTab *t, gboolean use_default) {
   CheckOption_init(t->tab_bar_homogeneous, use_default);
   CheckOption_init(t->tab_bar_scroller, use_default);
   CheckOption_init(t->tab_bar_show_fullscreen, use_default);
   CheckOption_init(t->tab_bar_show_single_tab, use_default);
   CheckOption_init(t->tab_instead_of_window, use_default);
   CheckOption_init(t->tab_load_in_background, use_default);
   EntryOption_init(t->compress_vowels, use_default);
   EntryOption_init(t->compress_common_prefixes, use_default);
   CheckOption_init(t->tab_title_compress, use_default);
}

void DillocfgTab_free(DillocfgTab *t) {
   CheckOption_free(t->tab_bar_homogeneous);
   CheckOption_free(t->tab_bar_scroller);
   CheckOption_free(t->tab_bar_show_fullscreen);
   CheckOption_free(t->tab_bar_show_single_tab);
   CheckOption_free(t->tab_instead_of_window);
   CheckOption_free(t->tab_load_in_background);
   EntryOption_free(t->compress_vowels);
   EntryOption_free(t->compress_common_prefixes);
   CheckOption_free(t->tab_title_compress);
   gtk_widget_destroy(t->vbox);
   g_free(t);
}
#endif

DillocfgOther *DillocfgOther_new(DillocfgWin *w, const gchar *label) {
   DillocfgOther *g = g_new(DillocfgOther, 1);
   gint x = 0;
   gint y = 0;
   g->vbox            = gtk_vbox_new(FALSE, 0);
   g->table           = gtk_table_new(9, 1, FALSE);
   gtk_table_set_homogeneous(GTK_TABLE(g->table), TRUE);
   gtk_container_add(GTK_CONTAINER(g->vbox), g->table);
   g->use_dicache     = CheckOption_new(TOKEN_USE_DICACHE, _("Use dicache"), FALSE, g->table, x, y++);
   g->focus_location_on_new
      = CheckOption_new(TOKEN_FOCUS_LOCATION_ON_NEW, _("Focus location on new"), TRUE, g->table, x, y++);
   g->enterpress_forces_submit
      = CheckOption_new(TOKEN_ENTERPRESS_FORCES_SUBMIT, _("Enterpress forces submit"), FALSE, g->table, x, y++);
   g->generate_submit          = CheckOption_new(TOKEN_GENERATE_SUBMIT, _("Generate submit"), TRUE, g->table, x, y++);
   g->limit_text_width         = CheckOption_new(TOKEN_LIMIT_TEXT_WIDTH, _("Limit text width"), FALSE, g->table, x, y++);
   gtk_notebook_append_page(GTK_NOTEBOOK(w->notebook), g->vbox, gtk_label_new(label));
   return g;
}

void DillocfgOther_init(DillocfgOther *g, gboolean use_default) {
   CheckOption_init(g->use_dicache, use_default);
   CheckOption_init(g->generate_submit, use_default);
   CheckOption_init(g->limit_text_width, use_default);
   CheckOption_init(g->focus_location_on_new, use_default);
   CheckOption_init(g->enterpress_forces_submit, use_default);
}

void DillocfgOther_free(DillocfgOther *g) {
   CheckOption_free(g->use_dicache);
   CheckOption_free(g->generate_submit);
   CheckOption_free(g->limit_text_width);
   CheckOption_free(g->focus_location_on_new);
   CheckOption_free(g->enterpress_forces_submit);
   gtk_widget_destroy(g->vbox);
   g_free(g);
}

DillocfgInterface *DillocfgInterface_new(DillocfgWin *w, const gchar *label) {
   DillocfgInterface *i = g_new(DillocfgInterface,1);
   gint x = 0;
   gint y = 0;
   GtkWidget *panel_size_menu = gtk_menu_new ();
   GtkWidget *panel_size_label0 = gtk_menu_item_new_with_label(panel_sizes[0]);
   GtkWidget *panel_size_label1 = gtk_menu_item_new_with_label(panel_sizes[1]);
   GtkWidget *panel_size_label2 = gtk_menu_item_new_with_label(panel_sizes[2]);

   gtk_menu_append (GTK_MENU(panel_size_menu), panel_size_label0);
   gtk_menu_append (GTK_MENU(panel_size_menu), panel_size_label1);
   gtk_menu_append (GTK_MENU(panel_size_menu), panel_size_label2);
   i->vbox  = gtk_vbox_new(FALSE, 0);
   i->table = gtk_table_new(10, 2, FALSE);
   gtk_table_set_homogeneous (GTK_TABLE(i->table), FALSE);
   gtk_container_add(GTK_CONTAINER(i->vbox), i->table);
   i->geometry_x = SpinOption_new(TOKEN_GEOMETRY_X, _("Window Width"), 640, 1, 65535, 1, 0, i->table, x, y++);
   i->geometry_y = SpinOption_new(TOKEN_GEOMETRY_Y, _("Window Height"), 550, 1, 65535, 1, 0, i->table, x, y++);
   i->panel_size               = SelectOption_new(TOKEN_PANEL_SIZE, _("Panel Size"), panel_size_menu, 0,  panel_sizes, FALSE, i->table, x, y++);
   i->show_back                = CheckOption_new(TOKEN_SHOW_BACK, _("Show Back"), TRUE, i->table, x, y++);
   i->show_forw                = CheckOption_new(TOKEN_SHOW_FORW, _("Show Forw"), TRUE, i->table, x, y++);
   i->show_bookmarks           = CheckOption_new(TOKEN_SHOW_BOOKMARKS, _("Show Bookmarks"), TRUE, i->table, x, y++);
   i->show_clear_url           = CheckOption_new(TOKEN_SHOW_CLEAR_URL, _("Show ClearURL"), TRUE, i->table, x, y++);
   i->show_home                = CheckOption_new(TOKEN_SHOW_HOME, _("Show Home"), TRUE, i->table, x, y++);
   i->show_menubar             = CheckOption_new(TOKEN_SHOW_MENUBAR, _("Show Menubar"), TRUE, i->table, x, y++);
   i->show_progress_box        = CheckOption_new(TOKEN_SHOW_PROGRESS_BOX, _("Show Progress box"), TRUE, i->table, x, y++);
   x++; y=0;
   i->show_reload              = CheckOption_new(TOKEN_SHOW_RELOAD, _("Show Reload"), TRUE, i->table, x, y++);
   i->show_save                = CheckOption_new(TOKEN_SHOW_SAVE, _("Show Save"), FALSE, i->table, x, y++);
   i->show_stop                = CheckOption_new(TOKEN_SHOW_STOP, _("Show Stop"), TRUE, i->table, x, y++);
   i->show_tooltip             = CheckOption_new(TOKEN_SHOW_TOOLTIP, _("Show Tooltip"), TRUE, i->table, x, y++);
   i->show_url                 = CheckOption_new(TOKEN_SHOW_URL, _("Show URL box"), TRUE, i->table, x, y++);
   i->small_icons              = CheckOption_new(TOKEN_SMALL_ICONS, _("Small icons"), TRUE, i->table, x, y++);
   i->fullwindow_start         = CheckOption_new(TOKEN_FULLWINDOW_START, _("Fullwindow start"), FALSE, i->table, x, y++);
   i->transient_dialogs        = CheckOption_new(TOKEN_TRANSIENT_DIALOGS, _("Transient dialogs"), FALSE, i->table, x, y++);
   i->show_popup_navigation    = CheckOption_new(TOKEN_SHOW_POPUP_NAVIGATION, _("Show Popup navigation"), TRUE, i->table, x, y++);
   i->show_user_agent          = CheckOption_new(TOKEN_SHOW_USER_AGENT, _("Show User-Agent"), FALSE, i->table, x, y++);
   gtk_notebook_append_page(GTK_NOTEBOOK(w->notebook), i->vbox, gtk_label_new(label));
   return i;
}

void DillocfgInterface_init(DillocfgInterface *i, gboolean use_default) {
   SpinOption_init(i->geometry_x, use_default);
   SpinOption_init(i->geometry_y, use_default);
   SelectOption_init(i->panel_size, use_default);
   CheckOption_init(i->show_back, use_default);
   CheckOption_init(i->show_bookmarks, use_default);
   CheckOption_init(i->show_clear_url, use_default);
   CheckOption_init(i->show_forw, use_default);
   CheckOption_init(i->show_home, use_default);
   CheckOption_init(i->show_menubar, use_default);
   CheckOption_init(i->show_popup_navigation, use_default);
   CheckOption_init(i->show_progress_box, use_default);
   CheckOption_init(i->show_reload, use_default);
   CheckOption_init(i->show_save, use_default);
   CheckOption_init(i->show_stop, use_default);
   CheckOption_init(i->show_tooltip, use_default);
   CheckOption_init(i->show_url, use_default);
   CheckOption_init(i->show_user_agent, use_default);
   CheckOption_init(i->small_icons, use_default);
   CheckOption_init(i->fullwindow_start, use_default);
   CheckOption_init(i->transient_dialogs, use_default);
}

void DillocfgInterface_free(DillocfgInterface *i) {
   SpinOption_free(i->geometry_x);
   SpinOption_free(i->geometry_y);
   SelectOption_free(i->panel_size);
   CheckOption_free(i->show_back);
   CheckOption_free(i->show_bookmarks);
   CheckOption_free(i->show_clear_url);
   CheckOption_free(i->show_forw);
   CheckOption_free(i->show_home);
   CheckOption_free(i->show_menubar);
   CheckOption_free(i->show_popup_navigation);
   CheckOption_free(i->show_progress_box);
   CheckOption_free(i->show_reload);
   CheckOption_free(i->show_save);
   CheckOption_free(i->show_stop);
   CheckOption_free(i->show_tooltip);
   CheckOption_free(i->show_url);
   CheckOption_free(i->show_user_agent);
   CheckOption_free(i->small_icons);
   CheckOption_free(i->fullwindow_start);
   CheckOption_free(i->transient_dialogs);
   g_free(i);
}

DillocfgFont *DillocfgFont_new(DillocfgWin *w, const gchar *label) {
   DillocfgFont *r =g_new(DillocfgFont, 1);
   gint x = 0;
   gint y = 0;
   r->vbox    = gtk_vbox_new(FALSE, 5);
   r->table   = gtk_table_new(4, 1, FALSE);
   gtk_table_set_homogeneous (GTK_TABLE(r->table), TRUE);
   gtk_table_set_row_spacings(GTK_TABLE(r->table), 5);
   r->table2  = gtk_table_new(2, 2, FALSE);
   gtk_table_set_homogeneous (GTK_TABLE(r->table2), TRUE);
   //gtk_box_pack_start(GTK_BOX(r->vbox), r->table, TRUE, FALSE, 5);
   //gtk_box_pack_start(GTK_BOX(r->vbox), r->table2, TRUE, FALSE, 5);
   gtk_container_add(GTK_CONTAINER(r->vbox), r->table);
   gtk_container_add(GTK_CONTAINER(r->vbox), r->table2);
   r->vw_fontname              = EntryOption_new(TOKEN_VW_FONT, _("vw_fontname"), "*-*", TRUE, r->table, x, y++);
   r->vw_aafontname            = EntryOption_new(TOKEN_VW_AAFONT, _("vw_aafontname"), "*", TRUE, r->table, x, y++);
   r->fw_fontname              = EntryOption_new(TOKEN_FW_FONT, _("fw_fontname"), "*-*", TRUE, r->table, x, y++);
   r->fw_aafontname            = EntryOption_new(TOKEN_FW_AAFONT, _("fw_aafontname"), "*", TRUE, r->table, x, y++);
   x=0; y=0;
   r->font_sizes = EntryOption_new(TOKEN_FONT_SIZES, _("Font Sizes"), "10 12 14 16 18 20", TRUE, r->table2, x, y++);
   r->use_oblique              = CheckOption_new(TOKEN_USE_OBLIQUE, _("Use oblique"), FALSE, r->table2, x, y++);
   x++; y=0;
   r->font_factor              = SpinOption_new(TOKEN_FONT_FACTOR, _("Font factor"), 1.0, 0.1, 20, 0.1, 1, r->table2, x, y++);
   r->limit_font_decoration    = CheckOption_new(TOKEN_LIMIT_FONT_DECORATION, _("Limit font decoration"), FALSE, r->table2, x, y++);

   gtk_notebook_append_page(GTK_NOTEBOOK(w->notebook), r->vbox, gtk_label_new(label));
   return r;
}

void DillocfgFont_init(DillocfgFont *r, gboolean use_default) {
   EntryOption_init(r->font_sizes, use_default);
   SpinOption_init(r->font_factor, use_default);
   EntryOption_init(r->vw_fontname, use_default);
   EntryOption_init(r->fw_fontname, use_default);
   EntryOption_init(r->vw_aafontname, use_default);
   EntryOption_init(r->fw_aafontname, use_default);
   CheckOption_init(r->limit_font_decoration, use_default);
   CheckOption_init(r->use_oblique, use_default);
}

void DillocfgFont_free(DillocfgFont *r) {
   EntryOption_free(r->font_sizes);
   SpinOption_free (r->font_factor);
   EntryOption_free(r->vw_fontname);
   EntryOption_free(r->fw_fontname);
   EntryOption_free(r->vw_aafontname);
   EntryOption_free(r->fw_aafontname);
   CheckOption_free(r->limit_font_decoration);
   CheckOption_free(r->use_oblique);
   g_free(r);
}

DillocfgColor *DillocfgColor_new(DillocfgWin *w, const gchar *label) {
   DillocfgColor *c = g_new(DillocfgColor, 1);
   gint x = 0;
   gint y = 0;
   GdkColor link_color = {0,0, 0, 65535};
   GdkColor bg_color = {0,65535, 65535, 65535};
   GdkColor visited_color = {0,40960, 8192, 61440};
   GdkColor text_color = {0,0, 0, 0};
   c->vbox    = gtk_vbox_new(FALSE, 0);
   c->table   = gtk_table_new(3, 1, FALSE);
   gtk_table_set_homogeneous (GTK_TABLE(c->table), TRUE);
   c->table2  = gtk_table_new(2, 2, TRUE);
   gtk_table_set_homogeneous (GTK_TABLE(c->table2), FALSE);
   gtk_box_pack_start(GTK_BOX(c->vbox), c->table, FALSE, FALSE, 5);
   gtk_box_pack_start(GTK_BOX(c->vbox), c->table2, FALSE, FALSE, 5);
   c->force_my_colors = CheckOption_new(TOKEN_FORCE_MY_COLORS, _("Force my colors"), FALSE, c->table, x, y++);
   c->force_visited_color = CheckOption_new(TOKEN_FORCE_VISITED_COLOR, _("Force visited color"), FALSE, c->table, x, y++);
   c->allow_white_bg  = CheckOption_new(TOKEN_ALLOW_WHITE_BG, _("Allow white bg"), TRUE, c->table, x, y++);
   y=0;
   c->link_color = ColorOption_new(TOKEN_LINK_COLOR, _("Link color"), &link_color, c->table2, x, y++);
   c->bg_color = ColorOption_new(TOKEN_BG_COLOR, _("BackGround color"), &bg_color, c->table2, x, y++);
   x++; y=0;
   c->visited_color = ColorOption_new(TOKEN_VISITED_COLOR, _("Visited color"), &visited_color, c->table2, x, y++);
   c->text_color = ColorOption_new(TOKEN_TEXT_COLOR, _("Text color"), &text_color, c->table2, x, y++);

   gtk_notebook_append_page(GTK_NOTEBOOK(w->notebook), c->vbox, gtk_label_new(label));
   return c;
}

void DillocfgColor_init(DillocfgColor *c, gboolean use_default) {
   CheckOption_init(c->force_my_colors, use_default);
   CheckOption_init(c->force_visited_color, use_default);
   CheckOption_init(c->allow_white_bg, use_default);
   ColorOption_init(c->link_color, use_default);
   ColorOption_init(c->bg_color, use_default);
   ColorOption_init(c->visited_color, use_default);
   ColorOption_init(c->text_color, use_default);
}

void DillocfgColor_free(DillocfgColor *c) {
   CheckOption_free(c->force_my_colors);
   CheckOption_free(c->force_visited_color);
   CheckOption_free(c->allow_white_bg);
   ColorOption_free(c->link_color);
   ColorOption_free(c->bg_color);
   ColorOption_free(c->visited_color);
   ColorOption_free(c->text_color);
   g_free(c);
}

DillocfgNetwork *DillocfgNetwork_new(DillocfgWin *w, const gchar *label) {
   DillocfgNetwork *n = g_new(DillocfgNetwork, 1);
   gint x = 0;
   gint y = 0;
   gint i = 0;
   GtkWidget *user_agent_label[7];
   GtkWidget *user_agent_menu = gtk_menu_new();
   for (i = 0; i < 7; i++) {
      user_agent_label[i] = gtk_menu_item_new_with_label(user_agents[i]);
      gtk_menu_append (GTK_MENU(user_agent_menu), user_agent_label[i]);
   }
   n->vbox  = gtk_vbox_new(FALSE, 0);
   n->table = gtk_table_new(10, 2, FALSE);
   gtk_container_add(GTK_CONTAINER(n->vbox), n->table);
   n->user_agent = SelectOption_new(TOKEN_USER_AGENT, _("User-Agent"), user_agent_menu, 0,  user_agents, TRUE, n->table, x, y++);
   n->home       = EntryOption_new(TOKEN_HOME, _("HOMEPAGE"), "http://www.google.com/", FALSE, n->table, x, y++);
   n->startup    = EntryOption_new(TOKEN_STARTUP, _("STARTUP"), "about:splash", FALSE, n->table, x, y++);;
   n->http_proxy = EntryOption_new(TOKEN_PROXY, _("HTTP Proxy"), "", FALSE, n->table, x, y++);;
   n->no_proxy   = EntryOption_new(TOKEN_NOPROXY, _("NO Proxy"), "", TRUE, n->table, x, y++);;
   gtk_notebook_append_page(GTK_NOTEBOOK(w->notebook), n->vbox, gtk_label_new(label));
   return n;
}

void DillocfgNetwork_init(DillocfgNetwork *n, gboolean use_default) {
   SelectOption_init(n->user_agent, use_default);
   EntryOption_init(n->home, use_default);
   EntryOption_init(n->startup, use_default);
   EntryOption_init(n->http_proxy, use_default);
   EntryOption_init(n->no_proxy, use_default);
}

void DillocfgNetwork_free(DillocfgNetwork *n) {
   SelectOption_free(n->user_agent);
   EntryOption_free(n->home);
   EntryOption_free(n->startup);
   EntryOption_free(n->http_proxy);
   EntryOption_free(n->no_proxy);
   g_free(n);
}

#define BUFSIZE 4096
void ReadUserconfig(DillocfgWin *d) {
   FILE *usercfg;
   gchar *filepass = g_strconcat(g_get_home_dir(), "/", USER_CFG, NULL);
   char buf[BUFSIZE], name[BUFSIZE], value[BUFSIZE];

   if (!(usercfg = fopen(filepass, "r"))) {
      g_print(_("Read failed <%s>\n"), filepass);
      return;
   }
   g_free(filepass);
   while (fgets(buf, BUFSIZE, usercfg)) {
      int i = 0, j = 0, k = 0;
      guint token = 0;
#define IS_LINE_END(c) (c  == '\0' || c == '#')
#define SKIPSPACE(buf, i) while (!IS_LINE_END(buf[i]) && isspace(buf[i])) i++
      SKIPSPACE(buf, i);
      if (buf[i] == '#') continue;
      SKIPSPACE(buf, i);
      while (!IS_LINE_END(buf[i]) && buf[i] != '=' && !isspace(buf[i]))
         name[j++] = buf[i++];
      name[j] = '\0';
      SKIPSPACE(buf, i);
      if (buf[i++] != '=') continue;
      SKIPSPACE(buf, i);
      if (buf[i] == '"') {
         i++;
         while (!IS_LINE_END(buf[i]) && buf[i] != '"')
            value[k++] = buf[i++];
      } else {
         while (!IS_LINE_END(buf[i]) && !isspace(buf[i]))
            value[k++] = buf[i++];
      }
      value[k] = '\0';
      for (i = 0; i < n_symbols; i++)
         if (strcmp(symbols[i].name, name) == 0)
            token = symbols[i].token;
      if(strlen(value) == 0) continue;
      /* assign value and exit successfully */
      switch (token) {
         case TOKEN_GEOMETRY: 
            {
               gchar *ptr;
               gint width, height;

               if ( (ptr = strchr(value, 'x')) && (width = strtol(value,NULL,10)) &&
                     (height = strtol(++ptr,NULL,10)) ){
                  d->interface->geometry_x->value = width;
                  d->interface->geometry_y->value = height;
               }
            }
            break;
         case TOKEN_PROXY:
            g_free(d->network->http_proxy->value);
            d->network->http_proxy->value = g_strdup(value);
            break;
         case TOKEN_NOPROXY:
            g_free(d->network->no_proxy->value);
            d->network->no_proxy->value = g_strdup(value);
            break;
         case TOKEN_USER_AGENT:
         {
            gint i;
            for (i = 0; i < USER_AGENTS_NUM; i++) {
               if (strcmp(value,user_agents[i]) == 0)
                  d->network->user_agent->value = i;
            }
            break;
         }
         case TOKEN_LINK_COLOR:
#define SET_COLOR(obj) \
         {\
            guint32 i;\
            if (strlen(value) != 8) break;\
            if (value[0] != '0' || value[1] != 'x') break;\
            sscanf(value, "0x%x", &i);\
            obj->value->red = (i&0x00FF0000) >> 8;\
            obj->value->green = (i&0x0000FF00);\
            obj->value->blue = (i&0x000000FF) << 8;\
         }
            SET_COLOR(d->color->link_color);
            break;
         case TOKEN_VISITED_COLOR:
            SET_COLOR(d->color->visited_color);
            break;
         case TOKEN_TEXT_COLOR:
            SET_COLOR(d->color->text_color);
            break;
         case TOKEN_BG_COLOR:
            SET_COLOR(d->color->bg_color);
            break;
         case TOKEN_ALLOW_WHITE_BG:
#define SET_BOOL(obj) {obj->value = (strcmp(value, "YES") == 0);}
            SET_BOOL(d->color->allow_white_bg);
            break;
         case TOKEN_FORCE_MY_COLORS:
            SET_BOOL(d->color->force_my_colors);
            break;
         case TOKEN_FORCE_VISITED_COLOR:
            SET_BOOL(d->color->force_visited_color);
            break;
         case TOKEN_USE_OBLIQUE:
            SET_BOOL(d->font->use_oblique);
            break;
         case TOKEN_PANEL_SIZE:
            {
               gint i;
               for (i = 0; i < PANEL_SIZES_NUM; i++)
                  if (!g_strcasecmp(value, panel_sizes[i]))
                     d->interface->panel_size->value = i;
            }
            break;
         case TOKEN_SMALL_ICONS:
            SET_BOOL(d->interface->small_icons);
            break;
         case TOKEN_HOME:
            g_free(d->network->home->value);
            d->network->home->value = g_strdup(value);
            break;
         case TOKEN_STARTUP:
            g_free(d->network->startup->value);
            d->network->startup->value = g_strdup(value);
            break;
         case TOKEN_SHOW_TOOLTIP:
            SET_BOOL(d->interface->show_tooltip);
            break;
         case TOKEN_FONT_FACTOR:
            d->font->font_factor->value = strtod(value, NULL);
            break;
         case TOKEN_FONT_SIZES:
            g_free(d->font->font_sizes->value);
            d->font->font_sizes->value = g_strdup(value);
            break;
         case TOKEN_LIMIT_TEXT_WIDTH:
            SET_BOOL(d->other->limit_text_width);
            break;
         case TOKEN_LIMIT_FONT_DECORATION:
            SET_BOOL(d->font->limit_font_decoration);
            break;
         case TOKEN_USE_DICACHE:
            SET_BOOL(d->other->use_dicache);
            break;
         case TOKEN_SHOW_BACK:
            SET_BOOL(d->interface->show_back);
            break;
         case TOKEN_SHOW_FORW:
            SET_BOOL(d->interface->show_forw);
            break;
         case TOKEN_SHOW_HOME:
            SET_BOOL(d->interface->show_home);
            break;
         case TOKEN_SHOW_RELOAD:
            SET_BOOL(d->interface->show_reload);
            break;
         case TOKEN_SHOW_SAVE:
            SET_BOOL(d->interface->show_save);
            break;
         case TOKEN_SHOW_STOP:
            SET_BOOL(d->interface->show_stop);
            break;
         case TOKEN_SHOW_BOOKMARKS:
            SET_BOOL(d->interface->show_bookmarks);
            break;
         case TOKEN_SHOW_MENUBAR:
            SET_BOOL(d->interface->show_menubar);
            break;
         case TOKEN_SHOW_CLEAR_URL:
            SET_BOOL(d->interface->show_clear_url);
            break;
         case TOKEN_SHOW_URL:
            SET_BOOL(d->interface->show_url);
            break;
         case TOKEN_SHOW_PROGRESS_BOX:
            SET_BOOL(d->interface->show_progress_box);
            break;
         case TOKEN_SHOW_POPUP_NAVIGATION:
            SET_BOOL(d->interface->show_popup_navigation);
            break;
         case TOKEN_SHOW_USER_AGENT:
            SET_BOOL(d->interface->show_user_agent);
            break;
         case TOKEN_FULLWINDOW_START:
            SET_BOOL(d->interface->fullwindow_start);
            break;
         case TOKEN_TRANSIENT_DIALOGS:
            SET_BOOL(d->interface->transient_dialogs);
            break;
         case TOKEN_FW_FONT:
            g_free(d->font->fw_fontname->value);
            d->font->fw_fontname->value = g_strdup(value);
            break;
         case TOKEN_FW_AAFONT:
            g_free(d->font->fw_aafontname->value);
            d->font->fw_aafontname->value = g_strdup(value);
            break;
         case TOKEN_VW_FONT:
            g_free(d->font->vw_fontname->value);
            d->font->vw_fontname->value = g_strdup(value);
            break;
         case TOKEN_VW_AAFONT:
            g_free(d->font->vw_aafontname->value);
            d->font->vw_aafontname->value = g_strdup(value);
            break;
         case TOKEN_GENERATE_SUBMIT:
            SET_BOOL(d->other->generate_submit);
            break;
         case TOKEN_ENTERPRESS_FORCES_SUBMIT:
            SET_BOOL(d->other->enterpress_forces_submit);
            break;
         case TOKEN_FOCUS_LOCATION_ON_NEW:
            SET_BOOL(d->other->focus_location_on_new);
            break;
#ifndef DISABLE_TABS
         case TOKEN_TAB_LOAD_IN_BACKGROUND:
            SET_BOOL(d->tab->tab_load_in_background);
            break;
         case TOKEN_TAB_INSTEAD_OF_WINDOW:
            SET_BOOL(d->tab->tab_instead_of_window);
            break;
         case TOKEN_TAB_BAR_SHOW_FULLSCREEN:
            SET_BOOL(d->tab->tab_bar_show_fullscreen);
            break;
         case TOKEN_TAB_BAR_SHOW_SINGLE_TAB:
            SET_BOOL(d->tab->tab_bar_show_single_tab);
            break;
         case TOKEN_TAB_BAR_SCROLLER:
            SET_BOOL(d->tab->tab_bar_scroller);
            break;
         case TOKEN_TAB_BAR_HOMOGENEOUS:
            SET_BOOL(d->tab->tab_bar_homogeneous);
            break;
         case TOKEN_TAB_TITLE_COMPRESS:
            SET_BOOL(d->tab->tab_title_compress);
            break;
         case TOKEN_COMPRESS_VOWELS:
            g_free(d->tab->compress_vowels->value);
            d->tab->compress_vowels->value = g_strdup(value);
            break;
         case TOKEN_COMPRESS_COMMON_PREFIXES:
            g_free(d->tab->compress_common_prefixes->value);
            d->tab->compress_common_prefixes->value = g_strdup(value);
            break;
#endif /*DISABLE_TABS */
         default:
            break;   /* Not reached */
      }
   }
   fclose(usercfg);
   DillocfgWin_init(d, FALSE);
}

void DillocfgWin_callback_reset(GtkButton *button, gpointer user_data) {
   DillocfgWin *d = (DillocfgWin *)user_data;
   DillocfgWin_init(d, TRUE);
   modified_ = TRUE;
}

void DillocfgWin_callback_reload(GtkButton *button, gpointer user_data) {
   DillocfgWin *d = (DillocfgWin *)user_data;
   ReadUserconfig(d);
   DillocfgWin_init(d, FALSE);
   modified_ = FALSE;
}

void WriteUserconfig(DillocfgWin *d) {
   FILE *usercfg;
   gchar *filepass = g_strconcat(g_get_home_dir(), "/", USER_CFG, NULL);

   if (!(usercfg = fopen(filepass, "w"))) {
      g_print(_("can't write <%s>\n"), filepass);
      return;
   }
   g_free(filepass);
   fprintf(usercfg, "#This file is written by Dillocfg\n");
#define WRITE_DATA(obj) fprintf(usercfg, "%s\n", GET_STR(obj));
   fprintf(usercfg, "%s=%sx%s\n", symbols[TOKEN_GEOMETRY].name,
         GET_STR(d->interface->geometry_x), GET_STR(d->interface->geometry_y));
   WRITE_DATA(d->network->http_proxy);
   WRITE_DATA(d->network->no_proxy);
   WRITE_DATA(d->network->user_agent);
   WRITE_DATA(d->color->link_color);
   WRITE_DATA(d->color->visited_color);
   WRITE_DATA(d->color->bg_color);
   WRITE_DATA(d->color->allow_white_bg);
   WRITE_DATA(d->color->force_my_colors);
   WRITE_DATA(d->color->force_visited_color);
   WRITE_DATA(d->color->text_color);
   WRITE_DATA(d->font->use_oblique);
   WRITE_DATA(d->network->startup);
   WRITE_DATA(d->network->home);
   WRITE_DATA(d->interface->panel_size);
   WRITE_DATA(d->interface->small_icons);
   WRITE_DATA(d->font->font_factor);
   WRITE_DATA(d->font->font_sizes);
   WRITE_DATA(d->interface->show_tooltip);
   WRITE_DATA(d->other->limit_text_width);
   WRITE_DATA(d->font->limit_font_decoration);
   WRITE_DATA(d->other->use_dicache);
   WRITE_DATA(d->interface->show_back);
   WRITE_DATA(d->interface->show_forw);
   WRITE_DATA(d->interface->show_home);
   WRITE_DATA(d->interface->show_reload);
   WRITE_DATA(d->interface->show_save);
   WRITE_DATA(d->interface->show_stop);
   WRITE_DATA(d->interface->show_bookmarks);
   WRITE_DATA(d->interface->show_menubar);
   WRITE_DATA(d->interface->show_clear_url);
   WRITE_DATA(d->interface->show_url);
   WRITE_DATA(d->interface->show_progress_box);
   WRITE_DATA(d->interface->show_popup_navigation);
   WRITE_DATA(d->interface->show_user_agent);
   WRITE_DATA(d->interface->fullwindow_start);
   WRITE_DATA(d->interface->transient_dialogs);
   WRITE_DATA(d->font->fw_fontname);
   WRITE_DATA(d->font->vw_fontname);
   WRITE_DATA(d->font->fw_aafontname);
   WRITE_DATA(d->font->vw_aafontname);
   WRITE_DATA(d->other->generate_submit);
   WRITE_DATA(d->other->enterpress_forces_submit);
   WRITE_DATA(d->other->focus_location_on_new);
#ifndef DISABLE_TABS
   WRITE_DATA(d->tab->tab_load_in_background);
   WRITE_DATA(d->tab->tab_instead_of_window);
   WRITE_DATA(d->tab->tab_bar_show_fullscreen);
   WRITE_DATA(d->tab->tab_bar_show_single_tab);
   WRITE_DATA(d->tab->tab_bar_scroller);
   WRITE_DATA(d->tab->tab_bar_homogeneous);
   WRITE_DATA(d->tab->tab_title_compress);
   WRITE_DATA(d->tab->compress_vowels);
   WRITE_DATA(d->tab->compress_common_prefixes);
#endif
   
   fclose(usercfg);
}

void DillocfgWin_callback_save(GtkButton *button, gpointer user_data) {
   DillocfgWin *d = (DillocfgWin *)user_data;
   WriteUserconfig(d);
}

/* vim: set ts=3 sw=3 sts=3 expandtab:*/
