/******************************************************************************
 *
 * Copyright (c) 1999	TOSHIYUKI ARAI. ALL RIGHTS RESERVED. 
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 *  
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 *	DrawableBinaryNode.cpp
 *
 *****************************************************************************/

// SOL++2000

#include <sol\DrawableBinaryNode.h>
#include <sol\StockObject.h>
#include <sol\Brush.h>

int DrawableBinaryNode::LINEAR = 0;
int DrawableBinaryNode::ZIGZAG = 1;


void DrawableBinaryNode::setDefaultColor()
{
	normalBackground = ::GetSysColor(COLOR_BTNFACE);

	normalTextColor  = ::GetSysColor(COLOR_WINDOWTEXT);
	activeBackground = ::GetSysColor(COLOR_HIGHLIGHT);
	activeTextColor  = ::GetSysColor(COLOR_HIGHLIGHTTEXT);
}


DrawableBinaryNode::DrawableBinaryNode(DrawableBinaryNode* parent, 
			const char* name, Object* object)
			:BinaryNode(parent, name, object)
{
	px = 0;
	py = 0;
	status = FALSE;
	setDefaultColor();

}
	
DrawableBinaryNode::DrawableBinaryNode(DrawableBinaryNode* parent, const char*  name, 
			DrawableBinaryNode* left,
			DrawableBinaryNode* right)
	:BinaryNode(parent, name, left, right)
{
	px = 0;
	py = 0;
	status = FALSE;
	setDefaultColor();
}


void DrawableBinaryNode::arrange(int width, int height, 
			   int leftMargin, int topMargin)
{
	px = (int)(getX() * (float)width ) + leftMargin;
	py = (int)(getY() * (float)height) + topMargin;

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		left -> arrange(width, height, leftMargin, topMargin);
	}
	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		right -> arrange(width, height, leftMargin, topMargin);
	}
}


void DrawableBinaryNode::arrange(DC* dc, int xposition, int height, 
			   int leftMargin, int topMargin)
{
	char* name = getName();
	SIZE size;
	dc -> getTextExtent(name, strlen(name), &size);

	int width   = size.cx + size.cy; 

	px = (xposition+leftMargin);
	py = (int)(getY() * (float)height) + topMargin;

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		left -> arrange(dc, px + width, height, leftMargin, topMargin);
	}
	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		right -> arrange(dc, px + width, height, leftMargin, topMargin);
	}
}


void DrawableBinaryNode::drawSelf(DC* dc, int x, int y, SIZE* s)
{
	char* name = getName();
	SIZE size;
	dc -> getTextExtent(name, strlen(name), &size);
	s->cx = size.cx;
	s->cy = size.cy;

	COLORREF backc = normalBackground;
	COLORREF textc = normalTextColor;
	if(status == TRUE) {
		backc = activeBackground;
		textc = activeTextColor;
	}

	HGDIOBJ hpen   = dc -> select(GetStockObject(BLACK_PEN));
	HGDIOBJ hbrush = dc -> select(GetStockObject(NULL_BRUSH));
	dc -> rectangle(x + px, y+py-2, x+px+size.cx+size.cy,
			y + py + size.cy + 2);
	
	Brush brush(backc);
	HGDIOBJ oldBrush= dc -> select(&brush);

	dc -> select(hpen);
	hpen = dc -> select(GetStockObject(NULL_PEN));

	dc -> rectangle(x + px, y+py-2, x+px+size.cx+size.cy,
			y + py + size.cy + 2);

	dc -> select(hpen);
	dc -> select(oldBrush);

	dc -> setBkMode(TRANSPARENT);
	dc -> setTextColor(textc);
	dc -> textOut(x + px+size.cy/2, y + py, name, strlen(name));
}


void DrawableBinaryNode::draw(DC* dc, DrawableBinaryNode* root, int policy, int x, int y)
{
	SIZE size;
	drawSelf(dc, x, y, &size);
	int xext = size.cx;
	int yext = size.cy;

	if(this != root) {
		if(policy == ZIGZAG) {
			POINT pt; 
			dc -> getCurrentPosition(&pt);

			int xx = pt.x;	
			int yy = pt.y;

			dc -> lineTo(x+px - (x+px - xx)/2, yy);
			dc -> lineTo(x+px - (x+px - xx)/2, 
				y+ py + yext/2);
			dc -> lineTo(x+ px,
				y+ py + yext/2);
		}
		else {
			dc -> lineTo(x + px, y + py + yext/2);
		}
	}

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		dc -> moveTo(x + px + xext+yext,
		             y + py + yext/2, NULL);

		left -> draw(dc, root, policy, x, y);
	}
	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		dc -> moveTo(x + px + xext+yext,
		             y + py + yext/2, NULL);

		right -> draw(dc, root, policy, x, y);
	}
}


void DrawableBinaryNode::setBackground(COLORREF normal, COLORREF active)
{
	normalBackground = normal;
	activeBackground = active;

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		left -> setBackground(normal, active);
	}

	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		right -> setBackground(normal, active);

	}
}


void DrawableBinaryNode::setTextColor(COLORREF normal, COLORREF active)
{
	normalTextColor = normal;
	activeTextColor = active;

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		left -> setTextColor(normal, active);
	}

	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		right -> setTextColor(normal, active);
	}
}

void DrawableBinaryNode::getExtent(DC* dc, int *maxx, int* maxy)
{
	char* name = getName();

	SIZE size;
	dc -> getTextExtent(name, strlen(name), &size);

	int nx  = px + size.cx+size.cy; 
	int ny  = py + size.cy/2; 

	if(*maxx < nx) *maxx = nx;
	if(*maxy < ny) *maxy = ny;

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		left -> getExtent(dc, maxx, maxy);
	}

	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		right -> getExtent(dc, maxx, maxy);
	}
}


int DrawableBinaryNode::nodeMaxLength(DC* dc, int len)
{
	int max = len;
	char* name = getName();

	SIZE size;
	dc -> getTextExtent(name, strlen(name), &size);
	int strLen  = size.cx;	
	if(strLen > max) max = strLen;

	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		left -> nodeMaxLength(dc, max);
	}

	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		right -> nodeMaxLength(dc, max);
	}
	
	return max;
}


DrawableBinaryNode* DrawableBinaryNode::search(DC* dc, int x1, int y1)
{
	char* name = getName();
	SIZE size;
	dc -> getTextExtent(name, strlen(name), &size);

	int nx  = px + size.cx+size.cy; 
	int ny  = py + size.cy;			
	if(x1 > px && x1 <nx &&
 	   y1 > py && y1 <ny ) {
		return this;
	}
	DrawableBinaryNode* found = null;
	DrawableBinaryNode* left = (DrawableBinaryNode*)getLeft();
	if (left) {
		found = left -> search(dc, x1, y1);
		if (found) {
			return found;
		}
	}

	DrawableBinaryNode* right = (DrawableBinaryNode*)getRight();
	if (right) {
		found = right -> search(dc, x1, y1);
		if (found) {
			return found;
		}
	}
	return NULL;
}

