/******************************************************************************
 *
 * Copyright (c) 1999	TOSHIYUKI ARAI. ALL RIGHTS RESERVED. 
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 *  
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 *	MemoryMonitor.cpp
 *
 *****************************************************************************/


// SOL++2000

#include <sol\ApplicationView.h>
#include <sol\ScrolledText.h>
#include <sol\Profile.h>
#include <sol\Font.h>
#include <sol\Stdio.h>
#include <sol\Thread.h>
#include <sol\LogFile.h>


class MemoryChecker :public Thread {
	ScrolledText* sctext;
	Boolean looping;
	int		interval;
	LogFile logFile;
  public:
	  MemoryChecker(ScrolledText* text, int mseconds)
		  :logFile(".\\memory.log") {
		sctext = text;
		looping = true;
		this->interval = mseconds;
	}

	~MemoryChecker() {
		terminate();
	}

	void terminate() {
		looping = false;
		wait();
	}

	void run() {
		while (looping) {
			sleep(interval);
			// Get global memory.
			MEMORYSTATUS ms;
			memset(&ms, 0, sizeof(ms));
			ms.dwLength = sizeof(ms);
			GlobalMemoryStatus(&ms);
		
			double ftm = (double)ms.dwAvailPhys/1024.0; 
			double tm  = (double)ms.dwTotalPhys/1024.0; 

			sctext->printf("Avail(KB)/Total(KB) = %.3f / %.3f\r\n", 
				ftm, tm);
			logFile.printf("Avail(KB)/Total(KB) = %.3f / %.3f\r\n", 
				ftm, tm);
			
		}
	}
};


class MemoryMonitor :public ApplicationView {
	Profile  profile;
	Font	 font;
	ScrolledText sctext;
	MemoryChecker memoryChecker;
	
	void	clear(Action& action) {
			sctext.clear();
	}

	long	close(Event& event);
  public:
	MemoryMonitor(Application& applet, const char* name, Args& args);
};


MemoryMonitor::MemoryMonitor(Application& applet, const char* name, Args& args)
	:ApplicationView(applet, name, args),
		memoryChecker(&sctext, 2000)
{
	Args ar;

	int fontSize = -14;
	if(GetSystemMetrics(SM_CXSCREEN) <= 1024 &&
			   GetSystemMetrics(SM_CYSCREEN) <= 768) {
		fontSize = -12;
	}
	ar.reset();
	ar.set(XmNheight, fontSize);
	ar.set(XmNfaceName, "COURIER");
	font.create(ar);

	ar.reset();
	sctext.create(this, "", ar);
	sctext.setReadOnly();

	// Add sctext to the default layout manager.
	add(sctext);

	sctext.limitText(32000); 
	sctext.setFont(font);
	sctext.addCallback(XmNmaxTextCallback, this, (Callback)&MemoryMonitor::clear, null);

	addEventHandler(WM_CLOSE, this, (Handler)&MemoryMonitor::close, NULL);

	// Start the memoryChecker thread.
	memoryChecker.start();

	restorePlacement();
}


long MemoryMonitor::close(Event& event)
{
	savePlacement();
	return defaultProc(event);
}


// MemoryMonitor Main
void	Main(int argc, char** argv)
{
	const char* appClass = "MemoryMonitor - Physical Memory";
	Application applet(appClass, argc, argv);
		
	Args args;
	args.set(XmNclassName, appClass);
	MemoryMonitor monitor(applet, appClass, args);
	monitor.realize();
	applet.run();
}

