/******************************************************************************
 *
 * Copyright (c) 1999	TOSHIYUKI ARAI. ALL RIGHTS RESERVED. 
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 *  
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 *	RichTextEditor.cpp
 *
 *****************************************************************************/

// SOL++2000
// 2000.02.18


#include <sol\ApplicationView.h>
#include <sol\FileDialog.h>
#include <sol\ScrolledRichText.h>
#include <sol\Font.h>
#include <sol\FontDialog.h>
#include <sol\ClientDC.h>
#include <sol\FileStream.h>
#include "resource.h"


class RichTextEditor :public ApplicationView {
	ScrolledRichText	text;
	Font				font;
	FileDialog			filedlg;
	FontDialog			fontdlg;

	void	clear(Action& action) { text.clear();}
	void	openRTF(Action& action);
	void	openText(Action& action);
	void	saveAsRTF(Action& action);
	void	saveAsText(Action& action);
	void	dropped(Action& action);

	void	bold(Action& action);
	void	italic(Action& action);
	void	underline(Action& action);
	void	incSize(Action& action);
	void	decSize(Action& action);
	void	changeFont(Action& action);
	void	setCaption(char* filename);
  public:
	RichTextEditor(Application& applet, const char* name, Args& args);
};


RichTextEditor::RichTextEditor(Application& applet, const char* name, Args& args)
	:ApplicationView(applet, name, args)
{
	Args	ar;
	ar.set(XmNexStyle, (ulong)WS_EX_ACCEPTFILES);
	text.create(this, "", ar);
	add(text);

	text.setEventMask(ENM_DROPFILES);
	text.addCallback(XmNdropFilesCallback, this,
			(Callback)&RichTextEditor::dropped, NULL);

	addCallback(XmNmenuCallback, ID_NEW, this,
			(Callback)&RichTextEditor::clear, NULL);
	addCallback(XmNmenuCallback, ID_OPENRTF, this,
			(Callback)&RichTextEditor::openRTF, NULL);
	addCallback(XmNmenuCallback, ID_OPENTEXT, this,
			(Callback)&RichTextEditor::openText, NULL);
	addCallback(XmNmenuCallback, ID_SAVEASRTF, this,
			(Callback)&RichTextEditor::saveAsRTF, NULL);
	addCallback(XmNmenuCallback, ID_SAVEASTEXT, this,
			(Callback)&RichTextEditor::saveAsText, NULL);
	addCallback(XmNmenuCallback, ID_EXIT, this,
			(Callback)&RichTextEditor::exit, NULL);

	addCallback(XmNmenuCallback, ID_BOLD, this,
			(Callback)&RichTextEditor::bold, NULL);
	addCallback(XmNmenuCallback, ID_ITALIC, this,
			(Callback)&RichTextEditor::italic, NULL);
	addCallback(XmNmenuCallback, ID_UNDERLINE, this,
			(Callback)&RichTextEditor::underline, NULL);

	addCallback(XmNmenuCallback, ID_INCSIZE, this,
			(Callback)&RichTextEditor::incSize, NULL);
	addCallback(XmNmenuCallback, ID_DECSIZE, this,
			(Callback)&RichTextEditor::decSize, NULL);

	addCallback(XmNmenuCallback, ID_CHANGEFONT, this,
			(Callback)&RichTextEditor::changeFont, NULL);

	ar.reset();
	filedlg.create(this, NULL, ar);
	ar.reset();
	fontdlg.create(this, NULL, ar);

	ClientDC dc(this);

	ar.reset();
	ar.set(XmNheight, (-12));
	// 2000.04.28
	//ar.set(XmNfaceName, "FixedSys");
	ar.set(XmNcharSet, (ulong)dc.getTextCharsetInfo(NULL));

	font.create(ar);
	text.setFont(font);
}


void RichTextEditor::setCaption(char* filename)
{
	if(filename) {
		char text[300];
		sprintf(text, "%s - RichTextEditor", filename);
		setText(text);
	}
}


void RichTextEditor::openRTF(Action& action)
{
	filedlg.setCaption("Open as RTF");
	if(filedlg.open()) {
		char* filename  = filedlg.getFileName();
		text.streamIn(filename, SF_RTF);
		setCaption(filename);
	}
}

void RichTextEditor::openText(Action& action)
{
	filedlg.setCaption("Open as Text");
	if(filedlg.open()) {
		char* filename  = filedlg.getFileName();
		text.streamIn(filename, SF_TEXT);
		setCaption(filename);
	}
}


void RichTextEditor::saveAsRTF(Action& action)
{
	filedlg.setCaption("Sava as RTF");
	if(filedlg.save()) {
		char* filename  = filedlg.getFileName();
		text.streamOut(filename, SF_RTF);
		setCaption(filename);
	}
}


void RichTextEditor::saveAsText(Action& action)
{
	filedlg.setCaption("Sava as Text");
	if(filedlg.save()) {
		char* filename  = filedlg.getFileName();
		text.streamOut(filename, SF_TEXT);
		setCaption(filename);
	}
}


void RichTextEditor::dropped(Action& action)
{
	Event& event = action.getEvent();
	ENDROPFILES* endrop = (ENDROPFILES*)event.getLParam();
	HDROP hdrop = (HDROP)endrop->hDrop;
	char filename[_MAX_PATH];
	::DragQueryFile(hdrop, 0, filename, sizeof(filename));

	text.streamIn(filename, SF_TEXT|SFF_SELECTION);

}


void RichTextEditor::bold(Action& action)
{
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);

    text.getCharFormat(TRUE, &cf);
    cf.dwMask      |= CFM_BOLD;
    cf.dwEffects  ^= CFE_BOLD;  
    text.setCharFormat(SCF_SELECTION, &cf);
}


void RichTextEditor::italic(Action& action)
{
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);
    text.getCharFormat(TRUE, &cf);

    cf.dwMask      |= CFM_ITALIC;
    cf.dwEffects  ^= CFE_ITALIC;  
    text.setCharFormat(SCF_SELECTION, &cf);

}

void RichTextEditor::underline(Action& action)
{
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);
    text.getCharFormat(TRUE, &cf);

    cf.dwMask      |= CFM_UNDERLINE;
    cf.dwEffects  ^= CFE_UNDERLINE;  
    text.setCharFormat(SCF_SELECTION, &cf);
}


void RichTextEditor::incSize(Action& action)
{
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);
    text.getCharFormat(TRUE, &cf);
    cf.dwMask      = CFM_SIZE;

	cf.yHeight += 20*2;

    text.setCharFormat(SCF_SELECTION, &cf);
}

void RichTextEditor::decSize(Action& action)
{
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);
    text.getCharFormat(TRUE, &cf);
    cf.dwMask      = CFM_SIZE;

	cf.yHeight += 20*(-2);
    text.setCharFormat(SCF_SELECTION, &cf);
}


void RichTextEditor::changeFont(Action& action)
{
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);
    text.getCharFormat(TRUE, &cf);
	LOGFONT lf;

	ClientDC dc(this);
	int logPixelInY = dc.getDeviceCaps(LOGPIXELSY);

    memset(&lf, 0, sizeof(lf));
    lf.lfHeight = MulDiv(cf.yHeight, logPixelInY, -1440);

    if (cf.dwEffects & CFE_BOLD) {
        lf.lfWeight = FW_BOLD;
	}
    else {
        lf.lfWeight = FW_NORMAL;
	}
    lf.lfItalic = (BOOL)(cf.dwEffects & CFE_ITALIC);
    lf.lfUnderline = (BOOL)(cf.dwEffects & CFE_UNDERLINE);
    lf.lfQuality   = DEFAULT_QUALITY;
    lf.lfPitchAndFamily = cf.bPitchAndFamily;
    strcpy(lf.lfFaceName, cf.szFaceName);

	CHOOSEFONT* chooseFont = fontdlg.get();
	chooseFont->lpLogFont = &lf;
    chooseFont->Flags = CF_BOTH | CF_SCALABLEONLY | CF_EFFECTS 
                       | CF_NOVECTORFONTS | CF_INITTOLOGFONTSTRUCT;
    chooseFont->rgbColors = cf.crTextColor;
    chooseFont->nFontType = 0;

    if (fontdlg.choose()) {
        cf.dwMask = CFM_BOLD | CFM_FACE | CFM_ITALIC | CFM_COLOR |
					CFM_CHARSET| CFM_STRIKEOUT |
                    CFM_OFFSET | CFM_SIZE | CFM_UNDERLINE;
        cf.yHeight = 2 * chooseFont->iPointSize;
	    cf.crTextColor = chooseFont->rgbColors;
        cf.dwEffects = 0;
	 
		ClientDC dc(this);

		cf.bCharSet = dc.getTextCharsetInfo(NULL);

        if (lf.lfWeight >= FW_BOLD) {
            cf.dwEffects |= CFE_BOLD;
		}
        if (lf.lfItalic) {
            cf.dwEffects |= CFE_ITALIC;
		}
        if (lf.lfUnderline) {
            cf.dwEffects |= CFE_UNDERLINE;
		}
        cf.bPitchAndFamily = lf.lfPitchAndFamily;
        strcpy(cf.szFaceName, lf.lfFaceName);
		//Printf("%s\r\n", cf.szFaceName);

		text.setCharFormat(SCF_SELECTION, &cf);
    }
}



void	Main(int argc, char** argv)
{
	const char* name = "RichTextEditor";
	Application applet(name, argc, argv);
	Args args;
	RichTextEditor richTextEditor(applet, name, args);
	richTextEditor.realize();
	applet.run();
}

