/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2234 $
|                     $Date: 2009-03-31 14:56:48 +0200 (mar, 31 mar 2009) $
\*============================================================================*/

/** \mainpage DPWSCore API Reference
 * This document exposes detailed information about the DPWS toolkit user API.
 * The reference is organized according main stack features:
 * <ul>
 * <li>\subpage core "Core stack features"</li>
 * <li>\subpage xml "XML utilities"</li>
 * <li>\subpage dyndepl "Dynamic deployment"</li>
 * <li>\subpage advanced "Advanced features"</li>
 * </ul>
 *
 * \page core Core stack features
 * All DPWS features related to:
 * <ul>
 * <li>device discovery,</li>
 * <li>metadata retrieval,</li>
 * <li>service invocation (using gSOAP),</li>
 * <li>event subscription.</li>
 * </ul>
 * are available to the user by including a single file: dc_Dpws.h.
 *
 * The dc_Error.h file should also be considered for stack error codes.
 */

/***************************************************************************//**
* \file
* Core toolkit API Reference.
* Most application using the standard DPWS features should only include this
* file.
*******************************************************************************/

#ifndef DPWS_H
# define DPWS_H


#ifndef DC_RT_FMAC1	/* runtime functions declaration prefix */
# define DC_RT_FMAC1
#elif !defined(SOAP_FMAC1)
# define SOAP_FMAC1 DC_RT_FMAC1
#endif

#include "dc/dc_Types.h"	// includes stdsoap2.h
#include "dc/dc_Error.h"

#ifdef __cplusplus
extern "C" {
#endif


/*----------------------------------------------------------------------------*\
 *                                   API                                      *
\*----------------------------------------------------------------------------*/

/***************************************************************************//**
* \name Initialization API
* For DPWS stack startup/shutdown.
* @{
*******************************************************************************/

/**
 * DPWS tookit initialization function (IPv4 mode).
 * This function must be called before any other API call. Using this API makes
 * the toolkit select one IP address among the one eligible (IPv4, not loopback).
 * For multiple interface & protocol support, please use \a dpws_init_ex.
 * @return a potential API error code.
 */
DC_RT_FMAC1 int dpws_init();

/**
 * DPWS tookit initialization function (IPv6 mode).
 * This function must be called before any other API call. Using this API makes
 * the toolkit select one IP address among the one eligible (IPv6, not loopback).
 * For multiple interface & protocol support, please use \a dpws_init_ex.
 * @return a potential API error code.
 */
DC_RT_FMAC1 int dpws_init6();

/**
 * Full-feature DPWS tookit initialization function.
 * This function must be called before any other API call and allows multiple
 * interface & protocol support.
 * @param selector A structure containing filtering information to select
 * network interfaces and IP addresses for the toolkit.
 * @param hostname If supplied (nullable), the DNS name will be used in
 * published transport addresses instead of IP addresses.
 * @param versions a bit set of supported DPWS versions (DPWS10_VERSION,
 * DPWS11_VERSION ...). DPWS_DEFAULT_VERSION designates an
 * implementation-specific default value.
 * @return a potential API error code.
 */
DC_RT_FMAC1 int dpws_init_ex(dc_ip_filter_t * selector, const char * hostname, int versions);

/**
 * Shutdowns the DPWS toolkit.
 * After this call, all DPWS toolkit resources are freed and a call to
 * dpws_init() must be performed to be able to operate again.
 * This api is called in the Windows DLL version delivery when a process
 * stops to use the DLL.
 * @return a potential API error code.
 */
DC_RT_FMAC1 int dpws_shutdown();

// @}

/*******************************************************************************
*                               Configuration APIs                             *
*******************************************************************************/

/***************************************************************************//**
* \name Configuration API
* These APIs are used to configure the toolkit, especially devices.
* @{
*******************************************************************************/

#define RANDOM_DEVICE_ID	0x7FFF	/**< If passed to dpws_create_device or dpws_create_custom_device as a local ID will make the default UUID randomly generated. */

/***************************************************************************//**
* \name Configuration parameters setters macros
* The following macro call configuration API using a syntax close to the
* parameter syntax.
* @{
*******************************************************************************/

#define DPWS_SET_PTR_ATT(h,p,v) dpws_set_ptr_att(h,p,v)	/**< Macro to set the value of configurable object attribute of pointer type. */
#define DPWS_SET_STR_ATT(h,p,v) dpws_set_ptr_att(h,p,v)	/**< Macro to set the value of configurable object attribute of string type. */
#define DPWS_SET_INT_ATT(h,p,v) dpws_set_int_att(h,p,v)	/**< Macro to set the value of configurable object attribute of integer type. */
#define DPWS_SET_BOOL_ATT(h,p,v) dpws_set_int_att(h,p,v)	/**< Macro to set the value of configurable object attribute of boolean type. */
#define DPWS_ADD_PTR_ATT(h,p,v) dpws_add_ptr_att(h,p,v)	/**< Macro to add a value to a configurable object multi-valued attribute of pointer type. */
#define DPWS_ADD_STR_ATT(h,p,v) dpws_add_ptr_att(h,p,v)	/**< Macro to add a value to a configurable object multi-valued attribute of string type. */
#define DPWS_ADD_INT_ATT(h,p,v) dpws_add_int_att(h,p,v)	/**< Macro to add a value to a configurable object multi-valued attribute of integer type. */
#define DPWS_ADD_BOOL_ATT(h,p,v) dpws_add_int_att(h,p,v)	/**< Macro to add a value to a configurable object multi-valued attribute of boolean type. */

// @}

/***************************************************************************//**
* \name Configuration parameters
* The following token in configuration parameters are useful to get an hint
* on the type of the value and then on the macro to use.
* <ul>
* <li>PTR are supposed to point to custom structures</li>
* <li>STR are string that are copied</li>
* <li>INT are integer parameters</li>
* </ul>
* @{
*******************************************************************************/

/** \name registry/toolkit configuration parameters
 * @{
 */
#define DC_REGISTRY_HANDLE	-1	/**< The reserved handle for the registry. */
#define DC_TOOLKIT_HANDLE	-1	/**< The reserved handle for the "toolkit". Supplied for backward compatibility. */

#define DPWS_INT_BOOT_SEQ					1	/**< Host boot sequence. Mandatory. */
#define DPWS_PTR_CALLBACK_DEVICE_METADATA	2	/**< \a metadata_cbk called when discovery device metadata version changes. Optional. */
#define DPWS_INT_HTTP_PORT					3	/**< Toolkit HTTP server port. Optional. Default is 80. */
#define DPWS_BOOL_TCP_LISTEN				4	/**< If DC_FALSE, the HTTP listener is not started. Optional (DC_TRUE if missing). @deprecated This parameter is kept for API backwards compatibility and defines the default setting for \a dpws_server_init. */
#define DPWS_INT_HTTP_BACKLOG				5	/**< The number of TCP pending connection requests for the HTTP listener. Optional (default is 10). @deprecated This parameter is kept for API backwards compatibility and defines the default setting for \a dpws_server_init.*/
// @}

/** \name toolkit configuration parameters
 * @{
 */
#define DPWS_STR_PREFERRED_LANG				10	/**< The "xml:lang format" value that will help the toolkit to select the device metadata language when several are retrieved. Optional. */
#define DPWS_BOOL_BP1_1_COMPATIBILITY		11	/**< When DC_TRUE, WS-Addressing headers are not required which ensures the main runtime compatibility feature for WSI Basic Profile 1.1 support. Optional. Default is false. */
// @}

/** \name device instance, service class parameters
 * @{
 */
#define DPWS_PTR_TYPE						20	/**< Device or service class types. qname struct C type. Multivalued and mandatory. */
#define DPWS_PTR_PREFIXED_TYPE				21	/**< Device or service class types. struct prefixed_qname C type. Multivalued and mandatory. */
#define DPWS_PTR_WSDL						22	/**< Device or service class WSDL references. struct wsdl_info C type. Multivalued and mandatory. */
// @}

/** \name device model, service class parameters
 * @{
 */
#define DPWS_STR_ID							30	/**< String ID for device model and service class objects. Optional. default: string form of the object handle reference */
// @}

/** \name device, service class and endpoint parameters
 * @{
 */
#define DPWS_PTR_FAULT_FUNCTION				40	/**< \a dispatch_cbk that can be set on device, service class or endpoint and will be called when a fault is received. Optional. */
#define DPWS_PTR_CALLBACK_EVENT_END			41	/**< \a event_end_cbk that can be set on device, service class or endpoint and will be called when a WS-Eventing subscription expires in unexpected manner. Optional. */
#define DPWS_PTR_HANDLING_FUNCTION			42	/**< \a dispatch_cbk that can be set on device, service class or endpoint and will be called when a SOAP message is received. Multivalued and optional. */
// @}

/** \name device model and instance parameters
 * @{
 */
#define DPWS_PTR_MODEL_NAME					50	/**< Mandatory device model name. Multivalued. struct localized_string. */
#define DPWS_STR_MODEL_NUMBER				51	/**< Model number as a string. Optional. */
#define DPWS_STR_MODEL_URL					52	/**< Model URL as a string. Optional. */
#define DPWS_STR_PRESENTATION_URL			53	/**< Presentation URL as a string. Optional but often used for device HTTP page for instance by Windows Vista. */
#define DPWS_PTR_MANUFACTURER				54	/**< Mandatory device manufacturer. Multivalued. struct localized_string. */
#define DPWS_STR_MANUFACTURER_URL			55	/**< Manufacturer URL as a string. Optional. */
// @}

/** \name device or service port parameters
 * @{
 */
#define DPWS_STR_ADDRESS					60	/**< Optional relative or absolute URL. If absolute : service may not be performed by the built-in HTTP server. Default: URL made of device IP + generated UUID */
// @}

/** \name service instance parameters
 * @{
 */
#define DPWS_STR_SERVICE_ID					70	/**< Service ID. Optional. Default: service class ID */
#define DPWS_INT_CLASS_HREF					100	/**< Read-only class handle reference for service instances. */
// @}

/** \name device instance parameters
 * @{
 */
#define DPWS_INT_METADATA_VERSION			80	/**< Mandatory device metadata version. Should be incremented every time device metadata changes. */
#define DPWS_STR_DEVICE_ID					81	/**< Device UUID. Optional (generated if missing). */
#define DPWS_STR_SCOPE						82	/**< Optional. Multivalued. Default is "http://schemas.xmlsoap.org/ws/2004/02/discovery/adhoc". */
#define DPWS_PTR_FRIENDLY_NAME				83	/**< Mandatory and potentially multivalued \a struct localized_string for device friendly name. */
#define DPWS_STR_FIRMWARE_VERSION			84	/**< Device firmware version as a string. Optional. */
#define DPWS_STR_SERIAL_NUMBER				85	/**< Device serial number as a string. Optional. */
// @}

/** \name device, service instance and class parameters
 * @{
 */
#define DPWS_PTR_USER_DATA					90	/**< Optional pointer on free user data that can be retrieved in service implementation function. */
// @}

/***************************************************************************//**
* \name Discovery cache configuration parameters
* @{
*******************************************************************************/

#define DC_CACHE_HANDLE		-2	/**< The reserved handle for the cache. */

/**
 * Specifies what devices should be stored in the cache.
 * Also affects the hello callback.
 * Optional, multivalued. Contains discovery_filter_t structures.
 */
#define DPWS_PTR_CACHE_FILTER				200

/**
 * A discovery_cbk function called when a device matching the cache filter has
 * joined the network and was added to the cache.
 * Optional.
 */
#define DPWS_PTR_CALLBACK_HELLO				201

/**
 * A discovery_cbk function called when a device leaves the cache.
 * Optional.
 */
#define DPWS_PTR_CALLBACK_BYE				202

/**
 * Maximum number of "unused" device proxies that the cache can contain.
 * Unused proxies are devices that are not hold by the user through the API.
 * When the maximum number of proxies is reached, a LRU algorithm removes the
 * oldest one.
 * Optional. Default value is unlimited (2e16).
 */
#define DPWS_INT_MAX_DEVICE_PROXIES			203

// @}

/***************************************************************************//**
* \name Subscription management parameters
* @{
*******************************************************************************/

#define DC_SUBSC_MANAGER_HANDLE	-3	/**< The reserved handle for the WS-Eventing. */


#define DPWS_INT_MAX_SUBSC_NB				301	/**< The maximum number of subscriptions. If exceeded, subscription is refused. Optional. Default is almost unlimited (0xFFFFFFFF) */
#define DPWS_STR_MAX_SUBSC_DURATION			302	/**< Maximum duration for WS-Eventing subscriptions as an XML duration. Optional. Default is "P1D" (1 day). Cannot be retrieved: use the integer representation. */
#define DPWS_INT_MAX_SUBSC_DURATION			303	/**< Maximum duration for WS-Eventing subscriptions in seconds. Optional. Default is 86400 (1 day). */

// @}
// @}

/*
 * Server object creation APIs
 */

/**
 * Creates an empty device model object.
 * @return A handle reference, if no error occurred. -1 otherwise.
 */
DC_RT_FMAC1 short dpws_create_device_model();

/**
 * Creates an empty service class object.
 * A service class is immutable when the server loop is running and
 * it is used by a hosted service.
 * @return A handle reference, if no error occurred. -1 otherwise.
 */
DC_RT_FMAC1 short dpws_create_service_class();

/** Associates a service class to a device model.
 * Devices that will be created from the device model will host one service
 * by service class registered to the device model. If the registration is not
 * performed, hosted services will have to be created manually using
 * dpws_create_hosted_service.
 * @param device_model_href The device class model reference.
 * @param service_class_href The service class handle reference.
 * @return DPWS_OK, if no error occurred. A DPWS or gSOAP error code otherwise.
 */
DC_RT_FMAC1 int dpws_register_service_class(short device_model_href, short service_class_href);

/** Creates a device instance object with all associated hosted services in
 * one API call.
 * This function creates an operational device object that will host a set of
 * services corresponding to the service classes registered to the model of
 * the device. Each service created will also have its own HTTP service port
 * to which is will be bound. Note that the model is used as a template that
 * will not be used anymore after device creation.
 * @param local_id A numeric identifier for the device instance that, with the
 * listening port and the MAC address, will be used to generate the stable
 * device logical address (i.e., the UUID). Note that passing RANDOM_DEVICE_ID
 * means that the UUID will be generated and that the UUID should be set to the
 * same value thanks to the API during the next device configuration stage.
 * @param model_href The device model handle reference.
 * @return A handle reference, if no error occurred. a negative error code
 * otherwise. Note that the handle reference is only hold by the caller
 * until device activation, so a release would lead to its destruction.
 */
DC_RT_FMAC1 short dpws_create_device(unsigned short local_id, short model_href);

/** Creates a device instance object without hosted services.
 * This function creates an operational device object that host no services.
 * The hosted services can be added later with the dpws_create_hosted_service
 * API. Note that the model is used as a template that
 * will not be used anymore after device creation.
 * @param local_id A numeric identifier for the device instance that, with the
 * listening port and the MAC address, will be used to generate the stable
 * device logical address (i.e., the UUID). Note that passing RANDOM_DEVICE_ID
 * means that the UUID will be generated and that the UUID should be set to the
 * same value thanks to the API during the next device configuration stage.
 * @param model_href The device model handle reference. -1 (no model), is
 * accepted.
 * @return A handle reference, if no error occurred. a negative error code
 * otherwise. Note that the handle reference is only hold by the caller
 * until device activation, so a release would lead to its destruction.
 */
DC_RT_FMAC1 short dpws_create_custom_device(unsigned short local_id, short model_href);

/** Creates a hosted service (and its associated HTTP service port) on a device
 * instance.
 * The hosted service can only be added if that service class is registered to
 * the class of the device. The service is created with the service class
 * identifier as identifier. The service id attribute is an attribute that can
 * be overwritten later.
 * @param device_href The device instance handle reference.
 * @param service_class_href The service class handle reference.
 * @return A handle reference, if no error occurred. a negative error code
 * otherwise. Note that the handle reference is hold by its device and not by
 * the API caller so there is no need to release it. It will be released with
 * the device.
 */
DC_RT_FMAC1 short dpws_create_hosted_service(short device_href, short service_class_href);

/** Creates a new service port on the default HTTP listener. A default, HTTP
 * address is created for the port with the toolkit IP address and port plus
 * a generated UUID which can be superseded by any absolute or relative URL
 * by setting the DPWS_STR_ADDRESS parameter on the server object.
 * @return A handle reference, if no error occurred. a negative error code
 * otherwise.
 */
DC_RT_FMAC1 short dpws_create_service_port();

/** Publishes a service endpoint on a specific address (service port).
 * @param hrefService The service endpoint handle reference.
 * @param hrefServicePort The service port handle reference.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_bind_service(short hrefService, short hrefServicePort);

/** Creates an endpoint and a bound service port. Services (and then devices)
 * are endpoints, from the DPWS specification point of vue. An endpoint can be
 * considered as a service which is not hosted, i.e., which is not advertized
 * in device's metadata.
 * The endpoint is a kind of service which can be configured like a service
 * class since its main use is to register handlers on it in order to receive
 * events or asynchronous responses.
 * An endpoint reference can be retrieved using dpws_get_local_endpoint_ref
 * @return A handle reference, if no error occurred. a negative error code
 * otherwise.
 */
DC_RT_FMAC1 short dpws_create_endpoint();


/*
 * Retrieval APIs
 */

/** Retrieves a device model handle by its id attribute.
 * @param id The device model id attribute.
 * @return A handle reference that should be released when not used anymore by
 * the caller. -1 is returned if the model was not found.
 */
DC_RT_FMAC1 short dpws_get_device_model_handle(char * id);

/** Retrieves a service class handle by its id attribute.
 * @param id The service class id attribute.
 * @return A handle reference that should be released when not used anymore by
 * the caller. -1 is returned if the service class was not found.
 */
DC_RT_FMAC1 short dpws_get_service_class_handle(char * id);

/** Retrieves all service class handle references.
 * @param[in,out] hrefs A short array provided by the caller to receive results.
 * Note that returned handles should be released after use.
 * @param[in,out] len Must contain initially the length of the array hrefs.
 * When the function returns it contains the number of results actually retrieved.
 * @return DPWS_OK if the array supplied was large enough to receive results.
 * DPWS_ERR_MORE_RESULTS, if extra-results are available. Note that even if
 * DPWS_ERR_MORE_RESULTS was returned, handle references should be released after
 * use.
 */
DC_RT_FMAC1 int dpws_get_service_class_handles(short* hrefs, int* len);

/** Retrieves a device handle by its uuid.
 * This method may be used when a discovery event is received using the user
 * callback and device data has to be retrieved. Looks only for devices being
 * configured or enabled. Disabled or cloned devices cannot be retrieved with
 * this API.
 * @param uuid the device uuid.
 * @return A handle reference that should be released when not used anymore by
 * the caller. -1 is returned if the device was not found.
 */
DC_RT_FMAC1 short dpws_get_device_handle(char * uuid);

/** Retrieves all configured device handle references.
 * The retrieved devices must have been enabled once. Clones are not retrieved.
 * @param[in,out] hrefs A short array provided by the caller to receive results.
 * Note that returned handles should be released after use.
 * @param[in,out] len Must contain initially the length of the array hrefs.
 * When the function returns it contains the number of results actually retrieved.
 * @return DPWS_OK if the array supplied was large enough to receive results.
 * DPWS_ERR_MORE_RESULTS, if extra-results are available. Note that even if
 * DPWS_ERR_MORE_RESULTS was returned, handle references should be released after
 * use.
 */
DC_RT_FMAC1 int dpws_get_device_handles(short* hrefs, int* len);

/** Retrieves a hosted service handle by its service identifier.
 * This function may be used to retrieve a hosted service when the
 * dpws_create_device opaque function was used to create a device.
 * @param device_href The device instance handle reference.
 * @param service_id the service identifier used as a key.
 * @return A handle reference that should be released when not used anymore by
 * the caller. -1 is returned if the service was not found.
 */
DC_RT_FMAC1 short dpws_get_service_handle(short device_href, char * service_id);

/** Retrieves a hosted service handle by its service class.
 * This function may be used to retrieve a hosted service when the
 * dpws_create_device opaque function was used to create a device.
 * @param device_href The device instance handle reference.
 * @param service_class_href The service class handle reference used as a key.
 * @return A handle reference that should be released when not used anymore by
 * the caller. -1 is returned if the service class was not found.
 */
DC_RT_FMAC1 short dpws_get_service_by_class(short device_href, short service_class_href);

/**
 * Retrieves all service implementations of a class.
 * @param hrefClass The service class handle reference.
 * @param hrefs An array to receive the result handle references. Note that
 * returned handles should be released after use.
 * @param[in,out] len Initially contains the size of the results array. When
 * the function returns, it contains the actual number of results found.
 * @return DPWS_OK if the array supplied was large enough to receive results.
 * DPWS_ERR_MORE_RESULTS, if extra-results are available. Note that even if
 * DPWS_ERR_MORE_RESULTS was returned, handle references should be released after
 * use.
 */
DC_RT_FMAC1 int dpws_get_services_by_class(short hrefClass, short* hrefs, int* len);

/**
 * Retrieves the device hosting a service.
 * @param href The handle reference of the hosted service.
 * @return The handle reference of the hosting device that should be released
 * when not used anymore by the caller.
 */
DC_RT_FMAC1 short dpws_get_service_device(short href);

/** Retrieves service handles attached to a device instance.
 * This function may be used to retrieve a hosted services when the
 * dpws_create_device opaque function was used to create a device.
 * @param device_href The device instance handle reference.
 * @param[in,out] hrefs A short array provided by the caller to receive results.
 * Note that returned handles should be released after use.
 * @param[in,out] len Must contain initially the length of the array hrefs.
 * When the function returns it contains the number of results actually retrieved.
 * @return DPWS_OK if the array supplied was large enough to receive results.
 * DPWS_ERR_MORE_RESULTS, if extra-results are available. Note that even if
 * DPWS_ERR_MORE_RESULTS was returned, handle references should be released after
 * use.
 */
DC_RT_FMAC1 int dpws_get_service_handles(short device_href, short* hrefs, int* len);

/** Retrieves service port handle references bound to a local device or service
 * endpoint. This function may be used to retrieve service ports when the
 * dpws_create_device opaque function was used to create a device.
 * @param endpoint_href The (service) endpoint handle reference.
 * @param[in,out] hrefs A short array provided by the caller to receive results.
 * Note that returned handles should be released after use.
 * @param[in,out] len Must contain initially the length of the array hrefs.
 * When the function returns it contains the number of results actually retrieved.
 * @return DPWS_OK if the array supplied was large enough to receive results.
 * DPWS_ERR_MORE_RESULTS, if extra-results are available. Note that even if
 * DPWS_ERR_MORE_RESULTS was returned, handle references should be released after
 * use.
 */
DC_RT_FMAC1 int dpws_get_service_port_handles(short endpoint_href, short* hrefs, int* len);

/** Get the default service port handle reference bound to a device or service
 * endpoint. Practically the first HTTP service port.
 * @param endpoint_href The (service) endpoint handle reference.
 * @return The handle reference of the default service port that should be
 * released when not used anymore by the caller.
 */
DC_RT_FMAC1 short dpws_get_default_service_port(short endpoint_href);


/*
 * Server object handle APIs
 */

/** Releases a toolkit configuration object handle when not needed anymore by
 * the user. This is good practice to release object handles in order to avoid
 * memory exhaustion. Practically, top-level object like device or service
 * classes should be released but also object retrieved by API other than
 * creation APIs.
 * @param href The handle reference on the object to release.
 * @return DPWS_OK, if no error occurred. A DPWS error code otherwise.
 */
DC_RT_FMAC1 int dpws_release_handle(short href);

/** Releases a toolkit configuration objects handle array when not needed anymore
 * by the user. This is good practice to release object handles in order to avoid
 * memory exhaustion. Practically, top-level object like device or service
 * classes should be released but also object retrieved by API other than
 * creation APIs.
 * @param hrefs A short array that contains handle references.
 * @param len Must contain the length of the array hrefs.
 * @return DPWS_OK, if no error occurred. A DPWS error code otherwise.
 */
DC_RT_FMAC1 int dpws_release_handles(short * hrefs, int len);

/** Retrieves the number of user of a toolkit configuration object.
 * @param href The handle reference on the configuration object.
 * @return The use count. Should be > 0. -1 in case of error.
 */
DC_RT_FMAC1 short dpws_get_handle_use_count(short href);


/*
 * Other server APIs
 */

/** Allocates and initialize a wsa_endpoint_ref structure for a service port
 * attached to a device, an hosted service or any local endpoint.
 * Must be called after the call to \a dpws_server_init().
 * NOTE: This API was designed to be used with \a dpws_event_subscribe which
 * should be replaced with \a dpws_event_subscribe_ex and is then kept for
 * backwards compatibility but is not suited to multiple interface & protocol
 * support.
 * @param service_port_href A service port handle reference.
 * @return A heap allocated endpoint reference. The caller is responsible for
 * liberation when no more needed (using dpws_endpoint_ref_free for instance).
 * NULL if the DPWS_ERR_INIT_SERVER is raised (dpws_server_init() not called).
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_get_local_endpoint_ref(short service_port_href);

/** Builds the default physical address for a local service port.
 * Must be called after the call to dpws_server_init().
 * NOTE: The new \a dpws_get_transport_addresses API should be preferred to this
 * API when the multiple network interface or protocol support is used. Kept for
 * backwards compatibility.
 * @param href_service_port The handle for a local service port.
 * @return A heap allocated URL. The caller is responsible for liberation
 * when not needed anymore using dpws_free(). NULL if the DPWS_ERR_INIT_SERVER
 * is raised (dpws_server_init() not called).
 */
DC_RT_FMAC1 char * dpws_get_transport_address(short href_service_port);

/** Retrieves the physical addresses for a local service port corresponding to
 * all the IP addresses selected for the toolkit.
 * These addresses should normally be used for information purposes.
 * Must be called after the call to one API of the dpws_server_init() family.
 * @param href_service_port The handle for a local service port.
 * @param[out] res_size The number of addresses retrieved.
 * @return A heap allocated array of URLs which size is given. The caller is
 * responsible for liberation of both the array and its content using
 * dpws_free(). NULL if the DPWS_ERR_INIT_SERVER is raised (dpws_server_init()
 * not called).
 */
DC_RT_FMAC1 char ** dpws_get_transport_addresses(short href_service_port, int * res_size);


/*
 * Hot configuration APIs
 */

/**	Deletes a hosted service.
 * Undeploys a hosted service. Must ne performed on an "offline" device.
 * @param service_href The service handle reference.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_delete_hosted_service(short service_href);

/** Make a device available on the network.
 * This API schedules a Hello message that is sent if the server is running.
 * Note that the device handle held by the DPWS registry in addition to the
 * device creator. A hold on the device handle will be taken so the hold
 * of the creator can be released (making it disposable if the device is
 * disabled).
 * @param device_href The device handle reference.
 * @return DPWS_OK, if no error occurred. A DPWS error code otherwise.
 */
DC_RT_FMAC1 int dpws_enable_device(short device_href);

/** Removes a device from the network.
 * This API schedules a Bye message that is sent if the server is running.
 * Note that the device handle is released by the DPWS registry making it held
 * only by the device creator.
 * @param device_href The device handle reference.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_disable_device(short device_href);

/**
 * Clones an online device in order to be able to perform "hot-configuration"
 * on the device.
 * The device created is a "deep" copy of the original in the sense that all
 * attached objects are also cloned (hosted services, service ports) except
 * service classes. Once configuration operations is performed on the offline
 * cloned device, the original one can be replaced with the cloned one using
 * dpws_clone_device.
 * @param href The handle reference of the device to be cloned.
 * @return The handle reference of the clone device.
 */
DC_RT_FMAC1 short dpws_clone_device(short href);

/**
 * Substitutes an online device with an offline one.
 * All device online service ports are replaced by the new device ones.
 * Note that the offline device UUID and metadata version will be overwritten
 * with the online ones and a hello message sent (with an incremented metadata
 * version) if necessary. A hold on the replacement device handle will be taken
 * so the hold of the creator can be released (making it disposable if the
 * device is disabled).
 * @param href_online The handle reference of the online device that will
 * be replaced.
 * @param href_offline The substitution device handle reference.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_replace_device(short href_online, short href_offline);

#define dpws_restore_cloned_device dpws_replace_device

/** Removes all the subscriptions attached to a service endpoint for a given
 * notification address. This function can be used in conjunction with the
 * fnotification_failure callback to get rid of invalid subscribers.
 * @param event_source The handle reference on the service acting as the event source.
 * @param notify_to The endpoint reference of the subscriber to be removed.
 * @return The number of removed subscriptions.
 */
DC_RT_FMAC1 int SOAP_FMAC2 dpws_remove_subscriber(short event_source, struct wsa_endpoint_ref * notify_to);

#define dpws_remove_subscriptor dpws_remove_subscriber

/*
 * Server object configuration APIs
 */

/**
 * Used to set a server object attribute value for pointer parameter types
 * (strings included). If the attribute is multivalued, replace the existing
 * attributes.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @param value The parameter value. Must be of the appropriate type, depending
 * on the attribute.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_set_ptr_att(short href, int att, const void * value);

/**
 * Used to set a server object attribute value for integer types (boolean
 * included). If the attribute is multivalued, replace the existing attributes.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @param value The integer parameter value.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_set_int_att(short href, int att, unsigned long value);

/**
 * Used to add a server object attribute value for pointer parameter types
 * (strings included) when this attribute is multivalued. If the attribute is
 * monovalued, replace the existing attribute.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @param value The parameter value. Must be of the appropriate type, depending
 * on the attribute.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_add_ptr_att(short href, int att, const void * value);

/**
 * Used to add a server object attribute value for integer types (boolean
 * included) when this attribute is multivalued. If the attribute is
 * monovalued, replace the existing attribute.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @param value The integer parameter value.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_add_int_att(short href, int att, unsigned long value);

/** Returns server object attributes for pointer types (string or not) attributes.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @return A pointer to the asked attribute value. Must not be freed since no
 * copy is performed. Multivalued attributes will be returned as NULL
 * terminated arrays.
 */
DC_RT_FMAC1 void * dpws_get_ptr_att(short href, int att);

/** Returns the number of attributes attached to a server object.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @return The number of attributes.
 */
DC_RT_FMAC1 int dpws_get_att_count(short href, int att);

/** Returns a specific instance of a multivalued server object attributes for
 * pointer types (string or not) attributes.
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @param index The index of the multivalued attribute.
 * @return A pointer to the asked attribute value. Must not be freed since no
 * copy is performed.
 */
DC_RT_FMAC1 void * dpws_get_ptr_att_item(short href, int att, int index);

/** Returns server object attributes of integer type (or boolean).
 * @param href The server object handle reference.
 * @param att The attribute identifier.
 * @return The asked attribute value.
 */
DC_RT_FMAC1 unsigned long dpws_get_int_att(short href, int att);

// @}

/***************************************************************************//**
* \name Server Runtime API
* These APIs allow to:
* <ul>
* <li>Run a listener,</li>
* <li>Stop a listener,</li>
* <li>Implement services (along with XML gSOAP data binding).</li>
* </ul>
* @{
*******************************************************************************/

/** Enumeration to use with \a dpws_fault */
typedef enum {
	SENDER_FAULT_VALUE = 0,	/**< The fault is due to the sender of the message (client); */
	RECEIVER_FAULT_VALUE = 1	/**< The fault is due to the receiver of the message (server); */
} fault_value_t;

/** HTTP listener */
#define DC_LISTENER_HTTP		0x1

/** WS-Discovery listener */
#define DC_LISTENER_DISCOVERY	0x2


/** Initialize the dpws structure and perform server-side initialization.
 * The initialization includes:
 * - the creation of the listening sockets,
 * - the scheduling of hello messages for all enabled devices,
 * - the mandatory configuration parameters check (according to the server mode),
 * - default parameters setting. Must be called before starting the main server
 *   loop.
 * This function sets automatically the server mode according to the
 * configuration calls performed before:
 * 	- Level 1: WS-Discovery listener (no endpoint have been created but at least
 *    one discovery handler has been registered),
 * 	- Level 2: WS-Eventing listener (one endpoint at least has been created but
 *    no device),
 * 	- Level 3: Device (one device at least has been created).
 * @param[in,out] dpws A pointer on the main DPWS API runtime data structure.
 * @param source This structure will be useed in every SOAP message as the WSA
 * source endpoint. The structure passed by the user will not be copied and must
 * be kept allocated by the caller.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise. For instance
 * if a needed parameter is missing and no default value is available.
 */
DC_RT_FMAC1 int dpws_server_init(struct dpws* dpws, struct wsa_endpoint_ref* source);

/** Initialize the dpws structure and perform server-side initialization.
 * The initialization includes:
 * - the creation of the listening sockets,
 * - the scheduling of hello messages for all enabled devices,
 * - the mandatory configuration parameters check (according to the server mode),
 * - default parameters setting. Must be called before starting the main server
 *   loop.
 * This function sets automatically the server mode according to the
 * configuration calls performed before:
 * 	- Level 1: WS-Discovery listener (no endpoint have been created but at least
 *    one discovery handler has been registered),
 * 	- Level 2: WS-Eventing listener (one endpoint at least has been created but
 *    no device),
 * 	- Level 3: Device (one device at least has been created).
 * @param[in,out] dpws A pointer on the main DPWS API runtime data structure.
 * @param source This structure will be useed in every SOAP message as the WSA
 * source endpoint. The structure passed by the user will not be copied and must
 * be kept allocated by the caller.
 * @param listeners Combination of DC_LISTENER_XXX flags to name listeners that
 * will be started.
 * @param bind_flags setsockopt flags to be applied to the TCP listening socket.
 * @param backlog The connection backlog to use for TCP listeners.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise. For instance
 * if a needed parameter is missing and no default value is available.
 */
DC_RT_FMAC1 int dpws_server_init_ex(struct dpws* dpws, struct wsa_endpoint_ref* source, int listeners, int bind_flags, int backlog);

/** Blocks and wait for an incoming SOAP message or expired timer.
 * @param dpws A pointer on the main DPWS API runtime data structure. When returning,
 * this structure is initialized and ready to process the new request (using dpws_serve).
 * @return DPWS_OK, if no error occurred. A DPWS error code otherwise. Recoverable
 * error codes include DPWS_ERR_SERVER_STOPPED (to exit the server loop after a
 * dpws_stop_server has been called) and DPWS_ERR_SERVER_TIMEOUT (a user-specified
 * timer has expired).
 */
DC_RT_FMAC1 int dpws_accept(struct dpws* dpws);

/** Blocks and wait for an incoming SOAP message or expired timer.
 * This alternate version of dpws_accept is particularly suitable for multi-thread
 * implementations since it makes performing the dpws_copy_context call unnecessary.
 * @param dpws A pointer on the main DPWS API runtime data structure. When returning,
 * this structure is not altered.
 * @param dpws_thr A pointer on the service thread target DPWS API runtime data
 * structure. When returning, this structure is initialized and ready to process
 * the new request (using dpws_serve). The content of the structure is undefined
 * if an error occured.
 * @return DPWS_OK, if no error occurred. A DPWS error code otherwise. Recoverable
 * error codes include DPWS_ERR_SERVER_STOPPED (to exit the server loop after a
 * dpws_stop_server has been called) and DPWS_ERR_SERVER_TIMEOUT (a user-specified
 * timer has expired).
 */
DC_RT_FMAC1 int dpws_accept_thr(struct dpws* dpws, struct dpws* dpws_thr);

/** Handles a request message.
 * This function starts reading the SOAP envelope, the SOAP headers and the
 * SOAP body start tag before delegating further processing to the message
 * handling function passed by a previous call to the server configuration
 * functions.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @return DPWS_OK, if no error occurred. A DPWS or gSOAP error code otherwise.
 */
DC_RT_FMAC1 int dpws_serve(struct dpws * dpws);

/**	Stops smoothly the locally running server for multi-thread servers.
 * This function schedules the Eventing end notifications and Bye WS-Discovery
 * messages for all the active devices, unblocks the main loop blocking on dpws_accept()
 * and closes all open listening sockets. This functions blocks first until the scheduled
 * messages are all being processed by server threads and lets an additional amount
 * of time to let the server threads to send the messages. The dpws_stop_server_no_wait()
 * function should be preferred if the developper can supply thread synchronization to
 * avoid server threads to be stopped before having performed their tasks or in mono-thread
 * servers where the call is performed in the server thread.
 * A new call to dpws_server_init() must be then performed to restart the server.
 * @param delay_ms the number of milliseconds to wait after the last "bye" task has been
 * passed to a server thread and that must allow the thread to send pending messages. Must
 * be tuned, especially if the number of threads is high. This is why one should use
 * dpws_stop_server_no_wait() whenever possible.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_stop_server(long delay_ms);

/**	Stops smoothly the locally running server.
 * This function schedules the Eventing end notifications and Bye WS-Discovery
 * messages for all the active devices, unblocks the main loop blocking on dpws_accept(
 * and closes all open listening sockets. This functions returns immediately so either
 * the developper can supply thread synchronization to avoid server threads to be
 * stopped before having performed their tasks or use it in mono-thread servers.
 * A new call to dpws_server_init() must be then performed to restart the server.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_stop_server_no_wait();

/** This API must be called after the network has been stopped (unplugged cable
 * for instance) while the DPWS server was running to send hello messages for
 * enabled devices.
 * Note that if network configuration changed, this API is not enough. The stack
 * must be restarted (\a dpws_shutdown and \a dpws_init must be called).
 * @param boot_seq A new boot number that must be supplied by the caller. It
 * should be strictly superior to the current value.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise, especially
 * DPWS_ERR_INVALID_PARAMETER if the provided boot number is not superior to
 * the current.
 */
DC_RT_FMAC1 int dpws_join_network(unsigned long boot_seq);

/** Retrieves the user data associated to the endpoint that received the
 * message.
 * @param dpws A pointer on the main DPWS API runtime data structure.
 * @return A pointer on the data set on the endpoint by the user.
 */
DC_RT_FMAC1 void * dpws_get_endpoint_user_data(struct dpws* dpws);

/** Retrieves the user data associated to the device instance that received the
 * message.
 * @param dpws A pointer on the main DPWS API runtime data structure.
 * @return A pointer on the data set on the device by the user.
 */
DC_RT_FMAC1 void * dpws_get_device_user_data(struct dpws* dpws);

/** Retrieves the user data associated to the service class of the endpoint
 * that received the message.
 * @param dpws A pointer on the main DPWS API runtime data structure.
 * @return A pointer on the data set on the service class by the user.
 */
DC_RT_FMAC1 void * dpws_get_service_class_user_data(struct dpws* dpws);

/** Allocate message transient memory.
 * This function is generally used in service implementations to allocate data
 * structures for response messages.
 * @param dpws A pointer on the service dpws data structure.
 * @param size The amount of memory required.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 void * dpws_malloc(struct dpws* dpws, size_t size);

/** Prepares a SOAP 1.2 fault.
 * To be called in a service function when the service invocation fails.
 * @param dpws A pointer on the service dpws data structure.
 * @param value The origin of the fault.
 * @param faultstring SOAP 1.2 fault reason.
 * @param subcode Optional SOAP 1.2 fault subcode. Potentially NULL.
 * @param detail Optional SOAP 1.2 fault detail. Potentially NULL.
 * @return An error code that should be used as return value for the service
 * function.
 */
DC_RT_FMAC1 int dpws_fault(struct dpws *dpws, fault_value_t value, const char *faultstring, const char *subcode, const char *detail);

/** Resets server configuration.
 * Should be called after the listener has been stopped using dpws_stop_server_XXX
 * to clean the toolkit registry.
 * @param dpws A optional (nullable) pointer on the main DPWS API runtime data
 * structure. Kept for backwards compatibility.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_server_shutdown(struct dpws* dpws);

// @}

/***************************************************************************//**
* \name Client Runtime API
* These APIs allow:
* <ul>
* <li>Device discovery using WS-Discovery,</li>
* <li>Device metadata retrieval using WS-Transfer/WS-MetadataExchange,</li>
* <li>Service information retrieval using WS-Transfer/WS-MetadataExchange,</li>
* <li>Event subscription management using WS-Eventing.</li>
* </ul>
* @{
*******************************************************************************/

#define WSD_DEFAULT_SCOPE	"http://schemas.xmlsoap.org/ws/2005/04/discovery/adhoc"	/**< WS-Discovery implicit default scope */

/** Client proxy type enumeration used with \a dpws_get_proxy_type */
typedef enum {
	UNKNOWN_PROXY_TYPE = 0,	/**< Proxy type could not be found. */
	DEVICE_PROXY_TYPE = 1,	/**< The proxy features a device. */
	SERVICE_PROXY_TYPE = 2	/**< The proxy features a service. */
} proxy_type_t;

/*
 *  WS-Discovery APIs
 */

/** Initialize the dpws structure and perform one-time client-side
 * initialisation.
 * Must be called once before using a dpws structure.
 * @param[in,out] dpws A pointer on a DPWS API runtime data structure.
 * @param source This optional (nullable) structure will be used in every SOAP
 * message as the WSA source endpoint. The structure passed by the user will not
 * be copied and must be kept allocated by the caller.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_client_init(struct dpws* dpws, struct wsa_endpoint_ref* source);

/** Retrieves one or several device proxies basing on a type and a scope.
 * This function will first look for proxies into the local cache, and then
 * generate a WS-Discovery Probe message if the number of requested proxies
 * is not available in the cache. This function implements the semantics of the
 * WS-Discovery Probe message, for a single device type. Note that this API
 * uses a timeout which value is fixed according to WS-Discovery protocol
 * metrics used by the DPWS specification and guarantees that all devices on the
 * local network should have time to reply (currently 2600 ms). To specify a
 * custom value, use the dpws_lookup_ex API.
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param ns The target namespace of the device type. May be NULL only if the
 * next parameter (type) is also NULL.
 * @param type The local name of the device type. May be null, in which case no
 * filtering on type is performed.
 * @param scope An URI specifying the scope of the search. May be NULL, in
 * which case the scope is unconstrained. The matching rule used for scope is
 * the default one (RFC2396).
 * @param[in,out] res_size The maximum number of proxies to retrieve. When
 * the function returns, it contains the actual number of retrieved device
 * proxies. WARNING : if 0 or less is supplied, the number of devices to
 * be retrieved is not constrained.
 * @return An array of device proxy handle references that will be released
 * during the next call to dpws_end in order to make associated resources
 * available for liberation. To avoid the proxy to be freed and recycled, one
 * should use the dpws_pin_proxy api. Note that between the moment it is
 * retrieved and the one it is used, the proxy may become invalid. The size of
 * the returned array can be retrieved using the output value of the res_size
 * parameter.
 * Note: the returned array is dynamically allocated on the gSOAP heap. This
 * data is available until dpws_end is called to free memory. If an error
 * occurs or no device is found, NULL is returned. To distinguish between the
 * two, use the res_size fields that is 0 is no device were retrieved and -1
 * in case of error.
 */
DC_RT_FMAC1 short * dpws_lookup(struct dpws* dpws, char * ns, char * type, char * scope, int *res_size);

/** Retrieves one or several device endpoints basing on types and scopes.
 * This function implements the semantics of the WS-Discovery Probe message
 * but providing the highest level of control. Furthermore several probe
 * messages can be triggered according to the filter specified.
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param filter The WS-Discovery filter array used for probe matches. Each
 * individual filter has the "intersection" semantic (AND) but specifying
 * several filters allows the caller to add a "union" semantic (OR) to the
 * protocol by performing several consecutive probe operations. If NULL is
 * provided, no filtering is performed at all.
 * @param f_size The number of filters provided in the \a filter parameter.
 * @param[in,out] res_size The maximum number of proxies to retrieve. When
 * the function returns, it contains the actual number of retrieved device
 * endpoints. WARNING : if 0 or less is supplied, the number of devices to
 * be retrieved is not constrained which could lead to memory exhaustion.
 * @param time_out A timeout in milliseconds after which the search should
 * return, when the maximum number of requested proxies has not been reached.
 * This must be a positive value. Note that this parameter is bounded by the
 * DPWS constant MATCH_TIMEOUT which is 10 seconds.
 * @param mode A specification on how the API should use the local cache.
 * @return An array of device proxy handle references that will be released
 * during the next call to dpws_end in order to make associated resources
 * available for liberation. To avoid the proxy to be freed and recycled, one
 * should use the dpws_pin_proxy api. Note that between the moment it is
 * retrieved and the one it is used, the proxy may become invalid. The size of
 * the returned array can be retrieved using the output value of the res_size
 * parameter.
 * Note: the returned array is dynamically allocated on the gSOAP heap. This
 * data is available until dpws_end is called to free memory. If an error
 * occurs or no device is found, NULL is returned. To distinguish between the
 * two, use the res_size fields that is 0 is no device were retrieved and -1
 * in case of error.
 */
#ifndef WITH_LEGACY_DC_API
DC_RT_FMAC1 short * dpws_lookup_ex(struct dpws* dpws, discovery_filter_t * filter, int f_size, int * res_size, unsigned int time_out, cache_mode_t mode);
#else
DC_RT_FMAC1 short * dpws_lookup_ex(struct dpws* dpws, struct qname *types, char ** scopes, scope_match_rule_t match_rule, int * res_size, long time_out, int mode);
#endif

/** Checks a DPWS device is available on the local network and retrieve its
 * proxy.
 * This function will first look for a device proxy into the local cache, and
 * then generate a WS-Discovery Resolve message if the requested endpoint is
 * not available in the cache. This function implements the semantics of the
 * WS-Discovery Resolve message. Note that this API uses a timeout which value
 * is the WSD_MATCH_TIMEOUT constant (currently 2600 ms).
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param uuid The device logical ID.
 * @return The device proxy handle reference if found. DPWS_ERR_NO_HANDLE_FOUND,
 * if not. The handle will be released during the next call to dpws_end in
 * order to make associated resources available for liberation. To avoid the
 * proxy to be freed and recycled, one should use the dpws_pin_proxy api. Note
 * that between the moment it is retrieved and the one it is used, the proxy
 * may become invalid.
 */
DC_RT_FMAC1 short dpws_lookup_by_id(struct dpws* dpws, const char * uuid);

/** Retrieves a device proxy checking a physical address matches a type and
 * a scope.
 * This function issues a directed WS-Discovery Probe message to the supplied
 * physical address and implements the semantics of this message.
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param address The physical address to which the directed probe is sent.
 * @param ns The target namespace of the type, as defined in the WSDL document.
 * May be NULL only if the next parameter (type) is also NULL.
 * @param type The local name of the device type. May be null, in which case no
 * filtering on type is performed.
 * @param scope An URI specifying the scope of the search. May be NULL, in
 * which case the scope is unconstrained. The matching rule used for scope is
 * the default one (RFC2396).
 * @return The device proxy handle reference if the device matched the
 * requirements at the supplied address. -1, if not. The handle
 * will be released during the next call to dpws_end in order to make associated
 * resources available for liberation. To avoid the proxy to be freed and
 * recycled, one should use the dpws_pin_proxy api. Note that between the
 * moment it is retrieved and the one it is used, the proxy may become invalid.
 */
DC_RT_FMAC1 short dpws_probe_address(struct dpws* dpws, char* address, char * ns, char * type, char * scope);

/** Retrieves a device endpoint checking a physical address matches a set of
 * types and scopes.
 * This function issues a directed WS-Discovery Probe message to the supplied
 * physical address and implements the semantics of this message.
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param address The physical address to which the directed probe is sent.
 * @param filter The WS-Discovery filter used for probe match. Mandatory even
 * if content is empty (no scope, no type).
 * @return The device proxy handle reference if the device matched the
 * requirements at the supplied address. -1, if not. The handle
 * will be released during the next call to dpws_end in order to make associated
 * resources available for liberation. To avoid the proxy to be freed and
 * recycled, one should use the dpws_pin_proxy api. Note that between the
 * moment it is retrieved and the one it is used, the proxy may become invalid.
 */
#ifndef WITH_LEGACY_DC_API
DC_RT_FMAC1 short dpws_probe_address_ex(struct dpws* dpws, char* address, discovery_filter_t * filter);
#else
DC_RT_FMAC1 short dpws_probe_address_ex(struct dpws* dpws, char* address, struct qname *types, char ** scopes, scope_match_rule_t match_rule);
#endif

/** Retrieve one or several service proxies basing on a device endpoint and a
 * type.
 * This function will look for endpoints into the local cache or generate a
 * WS-MetadataExchange GetMetadata message if necessary.
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param href_device A valid device proxy handle reference that may have been
 * retrieved using dpws_lookup.
 * @param ns The target namespace of the port type, as defined in the WSDL
 * document.
 * @param port_type The name of the port type. May be null, in which case no
 * filtering on type is performed.
 * @param[out] res_size The number of service proxies retrieved.
 * @return An array of service proxy handle references that will be released
 * during the next call to dpws_end in order to make associated resources
 * available for liberation. To avoid the proxy to be freed and recycled, one
 * should use the dpws_pin_proxy api. Note that between the moment it is
 * retrieved and the one it is used, the proxy may become invalid.
 * The size of the returned array can be retrieved using the output value of
 * the res_size parameter.
 * Note: the returned array is dynamically allocated
 * on the gSOAP heap. This data is available until dpws_end is called to free
 * memory. If an error occurs or no service is found, NULL is returned. To
 * distinguish between the two, use the res_size fields that is 0 is no
 * device were retrieved and -1 in case of error.
 */
DC_RT_FMAC1 short * dpws_get_services(struct dpws* dpws, short href_device, char * ns, char * port_type, int *res_size);

#define dpws_get_service dpws_get_services	// for compatibility

/** Retrieve a service proxy based on its service ID.
 * This function will first look for the service into the local cache, and
 * then generate a WS-MetadataExchange GetMetadata message if the requested
 * service is not available in the cache. This function implements the
 * semantics of the WS-MetadataExchange GetMetadata message.
 * @param dpws A pointer on a dpws runtime data structure used to process the
 * request.
 * @param href_device A valid device proxy handle reference that may have been
 * retrieved using dpws_lookup.
 * @param id The service ID.
 * @return The service proxy handle reference if found. -1, if not. The handle
 * will be released during the next call to dpws_end in order to make associated
 * resources available for liberation. To avoid the proxy to be freed and
 * recycled, one should use the dpws_pin_proxy api. Note that between the
 * moment it is retrieved and the one it is used, the proxy may become invalid.
 */
DC_RT_FMAC1 short dpws_get_service_by_id(struct dpws* dpws, short href_device, char * id);

/*
 *  Discovery Cache local APIs
 */

/** Increase the use counter of a cached device or service so that is cannot
 * be freed by the toolkit even if invalid from the cache point of view. When the
 * proxy is not used anymore, the user should use dpws_release_proxy to let it be
 * freed.
 * @param href_proxy The proxy handle reference to be pin.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_pin_proxy(short href_proxy);

/** Decrease the use counter of a cached device or service so that it can be
 * freed by the toolkit if invalid from the cache point of view.
 * @param href_proxy The proxy handle reference to be released.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_release_proxy(short href_proxy);

/** Return the type of proxy returned by the dpws_get_services. This
 * API is useful to set if dpws_get_device_metadata or dpws_get_service_metadata
 * must be used to retrieve proxy metadata.
 * @param href_proxy The service proxy handle reference (a device is also a
 * service).
 * @return An enum value representing the proxy type.
 */
DC_RT_FMAC1 proxy_type_t dpws_get_proxy_type(short href_proxy);

/**	Retrieves a cached proxy handle reference basing on an id. For device
 * proxies, the id considered is the device UUID. For service proxies, the
 * service ID is defined by the DPWS specifications.
 * @param id the proxy id to be searched.
 * @return the handle reference if retrieved. -1 else.
 */
DC_RT_FMAC1 short dpws_get_proxy_by_id(char * id);

/**	Retrieves a cached proxy handle reference based on a physical address.
 * @param address the proxy physical address to be searched.
 * @return the handle reference if retrieved. -1 else.
 */
DC_RT_FMAC1 short dpws_get_proxy_by_address(char * address);

/** Check if a cached device or service is still valid from the cache point of
 * view.
 * @param href_proxy The proxy handle reference to be checked.
 * @return DC_TRUE if the handle exist and is valid, DC_FALSE otherwise.
 */
DC_RT_FMAC1 DC_BOOL dpws_check_proxy(short href_proxy);

/** Removes a device or service proxy from the cache.
 * This function should be used when a call to a service using a given endpoint
 * is unsuccessful because the endpoint cannot be reached. Note that even if
 * the proxy status is changed and if it is removed from the cache, the proxy
 * object remains (or should remain) hold by the caller that may release it
 * using \a dpws_release_proxy for memory retrieval.
 * @param href The device or hosted service proxy href.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_invalidate_proxy(short href);

/** Checks a device matches a given WS-Discovery match criterion.
 * This function may be used when a discovery callback has been triggered.
 * @param href_device The device proxy handle reference.
 * @param criterion a WS-Discovery filter as used for probe match. Mandatory
 * even if content is empty (no scope, no type).
 * @return DC_TRUE if the device matches.
 */
DC_RT_FMAC1 DC_BOOL dpws_device_proxy_match(short href_device, discovery_filter_t * criterion);

/** Retrieves the metadata version of a cached device proxy.
 * @param href_device The device proxy.
 * @return The metadata version or 0 if an error occured.
 */
DC_RT_FMAC1 unsigned long dpws_cache_get_device_metadata_version(short href_device);

/** Retrieves the scopes of a device in the cache.
 * @param dpws A pointer on a dpws data structure used to process the API call.
 * @param href_device A valid device proxy handle reference that may have been
 * retrieved using dpws_lookup.
 * @return a null-terminated array of strings featuring the scopes. Data is
 * allocated on the dpws structure and will be freed during the next call to
 * dpws_end. An empty result may result from an API error.
 */
DC_RT_FMAC1 char ** dpws_cache_get_scopes(struct dpws* dpws, short href_device);

/** Retrieves the uuid of a device in the cache.
 * @param dpws A pointer on a dpws data structure used to process the API call or
 * NULL if the result must be allocated on the heap and should be freed using
 * dpws_free().
 * @param href_device A valid device proxy handle reference that may have been
 * retrieved using dpws_lookup.
 * @return a string featuring the device uuid. Data is allocated on the dpws
 * structure, if provided, and will be freed during the next call to dpws_end.
 * An empty result may result from an API error.
 */
DC_RT_FMAC1 char * dpws_cache_get_uuid(struct dpws* dpws, short href_device);

/** Retrieves the port types for a device or service proxy lying in the cache.
 * @param dpws A pointer on a dpws data structure used to process the API call.
 * @param href_proxy The device or hosted service proxy.
 * @return a null-terminated array of pointers on qname structures featuring
 * the port types. Data is allocated on the dpws structure and will be freed
 * during the next call to dpws_end. An empty result may result from an API
 * error.
 */
DC_RT_FMAC1 struct qname ** dpws_cache_get_types(struct dpws* dpws, short href_proxy);

/** Retrieves in the cache the URLs for a device.
 * @param dpws A pointer on a dpws data structure used to allocate the result.
 * @param href_device The device proxy handle reference.
 * @param[out] res_size The number of URLs retrieved.
 * @return An array of strings containing the transport addresses of the device.
 * If the device handle reference is invalid, an DPWS_ERR_NO_HANDLE_FOUND error
 * will be raised.
 */
DC_RT_FMAC1 char ** dpws_cache_get_device_transport_addresses(struct dpws* dpws, short href_device, int * res_size);

/** Gets the default endpoint reference for a device or service proxy.
 * In this version, the first endpoint is considered. If a specific endpoint
 * reference is required, use the dpws_get_endpoint_refs.
 * @param dpws A pointer on a dpws data structure used to allocate the result.
 * If NULL, the result is allocated on the process heap.
 * @param href_proxy The device or service proxy which endpoint reference must
 * be retrieved for.
 * @return An endpoint reference structure. Must be freed by the user using
 * dpws_end if a dpws structure was provided, dpws_endpoint_ref_free else.
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_get_default_endpoint_ref(struct dpws* dpws, short href_proxy);

#define dpws_get_default_endpoint_reference dpws_get_default_endpoint_ref	// for compatibility

/** Gets all the endpoint references retrieved for a device or service proxy.
 * @param dpws A pointer on a dpws data structure used to allocate the result.
 * If NULL, the result is allocated on the process heap.
 * @param href_proxy The device or service proxy which endpoint references must
 * be retrieved for.
 * @param[out] res_size The number of endpoints retrieved.
 * @return An array of endpoint references.
 * The returned array and endpoint structures are dynamically allocated
 * on the gSOAP or process heap. This data must be freed using dpws_end or
 * the  dpws_endpoint_ref_array_free utility according to the fact that a dpws
 * structure has been provided or not. If an error occurs or no proxy is found,
 * NULL is returned. To distinguish between the two, use the res_size
 * fields that is 0 is no endpoint were retrieved and -1 in case of error.
 */
DC_RT_FMAC1 struct wsa_endpoint_ref ** dpws_get_endpoint_refs(struct dpws* dpws, short href_proxy, int * res_size);

#define dpws_get_endpoint_references dpws_get_endpoint_refs	// for compatibility

// WS-Metadata Exchange APIs

/**	Retrieves the whole DPWS metadata for a device.
 * This API uses the WS-Transfer Get request and the WS-MetadataExchange message
 * format. This API, like all the functions WS-Metadata Exchange APIs, does not
 * cache the results of network queries performed so that a query is performed
 * every time.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param href_device The device proxy handle reference that may have been
 * retrieved using dpws_lookup.
 * @return A structure containing metadata allocated on the dpws structure (
 * freed during the next call to dpws_end).
 */
DC_RT_FMAC1 struct device_metadata * dpws_get_device_metadata(struct dpws* dpws, short href_device);

/**	Retrieves the service DPWS metadata for a hosted service.
 * This function uses no dialect in the GetMetadata request. This API, like
 * all the functions WS-Metadata Exchange APIs, does not cache the results of
 * network queries performed so that a query is performed every time.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param href_service A hosted service proxy handle reference.
 * @return A structure containing metadata allocated on the dpws structure (
 * freed during the next call to dpws_end).
 */
DC_RT_FMAC1 struct service_metadata * dpws_get_service_metadata(struct dpws* dpws, short href_service);

/**	Retrieves in the cache the hosting device proxy of a hosted service.
 * This function only uses the cache.
 * @param href_service A hosted service proxy handle reference.
 * @return The host device proxy handle reference. -1 if not found.
 */
DC_RT_FMAC1 short dpws_get_host_device_proxy(short href_service);


// WS-Eventing APIs

/** Subscribes to a WS-Eventing event source.
 * This version of the API is supposed to be used:
 * <ul><li>When the \a notify_to &amp; \a end_to refer to network endpoints
 * located outside the current instance of the toolkit. These should be either
 * built "manually" or retrieved using WS-Discovery.</li>
 *<li>When legagy APIs like \a dpws_get_local_endpoint_ref are used even if not
 * suited to multiple interface or protocol support.</li></ul>
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param event_source A valid endpoint reference (that may have been retrieved
 * using dpws_lookup or dpws_get_services) of an WS-Eventing event source.
 * @param notify_to A valid device endpoint reference (that may have been
 * retrieved using dpws_lookup) that will receive the events.
 * @param end_to A valid device endpoint reference (that may have been
 * retrieved using dpws_lookup) that will receive abnormal subscription
 * termination. Optional (if NULL no termination event is sent).
 * @param filter A NULL-terminated string array of DPWS actions to limit events
 * to receive from the event source.
 * @param[in,out] expiration An XML Schema dateTime or duration, to specify the
 * subscription duration requested. This parameter will be modified with the
 * value granted by the subscription manager with a string allocated on the
 * dpws runtime allocator. If no expiration is provided, a default value will
 * be provided by the device.
 * @return The subscription manager endpoint that shall be used as a ticket to
 * manage the subscription. NULL if an error occurred.
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_event_subscribe(struct dpws* dpws, struct wsa_endpoint_ref* event_source, struct wsa_endpoint_ref* notify_to, struct wsa_endpoint_ref* end_to, char ** filter, char ** expiration);

/** Subscribes to a WS-Eventing event source for a local event sink.
 * This version of the API is supposed to be used when the \a notify_to &amp;
 * \a end_to are local service ports (i.e. network endpoints) and  performs
 * automatic EPR selection.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param event_source A valid endpoint (that may have been retrieved using
 * dpws_lookup or dpws_get_services) of an WS-Eventing event source.
 * @param href_notify_to The handle of a local service port that will receive
 * the events.
 * @param href_end_to The handle of a local service port that will receive
 * abnormal subscription termination. Optional (if -1 no termination event is
 * sent).
 * @param filter A NULL-terminated string array of DPWS actions to limit events
 * to receive from the event source.
 * @param[in,out] expiration An XML Schema dateTime or duration, to specify the
 * subscription duration requested. This parameter will be modified with the
 * value granted by the subscription manager with a string allocated on the
 * dpws runtime allocator. If no expiration is provided, a default value will
 * be provided by the device.
 * @return The subscription manager endpoint that shall be used as a ticket to
 * manage the subscription. NULL if an error occurred.
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_event_subscribe_ex(struct dpws* dpws, struct wsa_endpoint_ref* event_source, short href_notify_to, short href_end_to, char ** filter, char ** expiration);

/** Renews a subscription to a WS-Eventing event source.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param subscription_manager The subscription ticket gained during initial
 * call to dpws_event_subscribe.
 * @param expiration XML Schema dateTime or duration, to specify the new
 * subscription duration requested. If NULL is provided, the renew duration is
 * provided by the device.
 * @return The subscription duration (XML Schema dateTime or duration) granted
 * by the manager. NULL if an error occurred.
 */
DC_RT_FMAC1 char * dpws_event_renew(struct dpws* dpws, struct wsa_endpoint_ref* subscription_manager, char* expiration);

/** Gets the remaining duration for a subscription to a WS-Eventing event source.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param subscription_manager The subscription ticket gained during initial
 * call to dpws_event_subscribe.
 * @return The subscription duration (XML Schema dateTime or duration) still
 * running. NULL if an error occurred.
 */
DC_RT_FMAC1 char * dpws_event_status(struct dpws* dpws, struct wsa_endpoint_ref* subscription_manager);

/** Cancel a running subscription to a WS-Eventing event source.
 * @param dpws A pointer on a dpws data structure used to process the request.
 * @param subscription_manager The subscription ticket gained during initial
 * call to dpws_event_subscribe.
 * @return DPWS_OK, if no error occurred. A DPWS or gSOAP error code otherwise.
 */
DC_RT_FMAC1 int dpws_event_unsubscribe(struct dpws* dpws, struct wsa_endpoint_ref* subscription_manager);

// @}

/***************************************************************************//**
* \name General usage utilities
* These APIs can be used both for client & server operations.
* @{
*******************************************************************************/

/** Clears structures and free dynamically allocated memory used for request
 * processing.
 * Must be called after processing a message on a client, a server or a
 * handler.
 * @param dpws A pointer on a dpws data structure that was used to process the
 * request.
 * @return DPWS_OK, if no error occurred. A DPWS code otherwise.
 */
DC_RT_FMAC1 int dpws_end(struct dpws* dpws);

/** Returns the last DPWS or gSOAP runtime error.
 * @param dpws A pointer on a dpws data structure used when the error occurred.
 * @return A DPWS or gSOAP api error code.
 */
DC_RT_FMAC1 int dpws_get_error(struct dpws* dpws);

/** Gets a message for a SOAP fault or a DPWS/gSOAP error that has
 * previously occurred.
 * @param dpws A pointer on a dpws runtime data structure used when the error
 * occured.
 * @return A readonly error message.
 */
DC_RT_FMAC1 const char * dpws_get_error_msg(struct dpws* dpws);

/** Initialize a wsa_endpoint_ref structure that the user may use for a
 * subsequent API call.
 * @param endpoint_ref his structure will be reset.
 */
DC_RT_FMAC1 void dpws_default_wsa_endpoint_ref(struct wsa_endpoint_ref *endpoint_ref);

/** Allocates and initialize a wsa_endpoint_ref structure for a device.
 * No verification is performed for device existence (this could be done thanks
 * to the dpws_lookup_by_id API).
 * @param uuid A device uuid.
 * @return A DPWS library heap allocated endpoint reference. The caller is
 * responsible for liberation when no more needed (using dpws_endpoint_ref_free).
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_create_device_endpoint_ref(char * uuid);

/** Creates a deep copy of a wsa_endpoint_ref structure on DPWS library heap.
 * May be used for instance to keep a WS-Eventing subscription endpoint after
 * dpws_end has been called.
 * @param endpoint The structure to copy.
 * @return A heap allocated endpoint reference. The caller is responsible for
 * liberation when no more needed (using dpws_endpoint_ref_free).
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_endpoint_ref_dup(struct wsa_endpoint_ref * endpoint);

/** Copy of a wsa_endpoint_ref structure to a provided destination.
 * This function uses the standard malloc function to allocate the structure
 * contents.
 * @param dest he destination structure.
 * @param src The structure to copy.
 * @return The dest param. The API user will have the responsibility to free
 * the allocated contents of the structure using dpws_dpws_endpoint_ref_content_free.
 */
DC_RT_FMAC1 struct wsa_endpoint_ref * dpws_endpoint_ref_copy(struct wsa_endpoint_ref * dest, struct wsa_endpoint_ref * src);

/** Frees a string or any buffer allocated on the DPWS library heap.
 * @param buffer The memory block to be freed.
 */
DC_RT_FMAC1 void dpws_free(void * buffer);

/** Frees the contents of an endpoint reference.
 * Liberates the contents of a wsa_endpoint_ref structure allocated on the
 * DPWS library heap. This function can be used along with dpws_endpoint_ref_copy.
 * @param endpoint The structure which contents must be freed.
 */
DC_RT_FMAC1 void dpws_endpoint_ref_content_free(struct wsa_endpoint_ref * endpoint);

/** Frees an endpoint reference and its contents.
 * This function can be used along with dpws_endpoint_ref_dup.
 * @param endpoint The structure which must be freed.
 */
DC_RT_FMAC1 void dpws_endpoint_ref_free(struct wsa_endpoint_ref * endpoint);

/** Frees an endpoint reference array and its contents.
 * This function can be used along with dpws_endpoint_ref_dup that allows for
 * instance the user to keep the selected endpoint reference before array
 * deletion.
 * @param endpoints A array of endpoint reference pointers.
 * @param nb_endpoints The size of the array to free.
 */
DC_RT_FMAC1 void dpws_endpoint_ref_array_free(struct wsa_endpoint_ref ** endpoints, int nb_endpoints);

// @}

#ifdef __cplusplus
}
#endif

#endif
