/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 827 $
|                     $Date: 2008-03-13 16:34:05 +0100 (jeu., 13 mars 2008) $
\*============================================================================*/

/***************************************************************************//**
* \file
* WS-Management dispatcher API.
* This dispatcher is used by the dynamic deployment feature but allow it to
* coexist with other WS-Management implementations.
*******************************************************************************/

#ifndef DC_WSMAN_H
#define DC_WSMAN_H

#include "dc/dc_GenericInvocation.h"	// includes DC API an then stack configuration
#include "dc/dc_WsmanTypes.h"

#ifndef DC_RT_FMAC1	/* runtime functions declaration prefix */
# define DC_RT_FMAC1
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------------*\
 *                               API Error codes                              *
\*----------------------------------------------------------------------------*/

#define DPWS_ERR_XPATH_FRAGMENT_PARSING		-300	/**< The WS-Management XPath Fragment is not syntactically correct */

/** \name Resource related fault error codes
 * @{
 */
#define WSMAN_ERR_INVALID_RESOURCE				1	// subcode : wsa:DestinationUnreachable
// @}

/** \name Selector related fault error codes
 * @{
 */
#define WSMAN_ERR_INSUFFICIENT_SELECTORS		2	// subcode : wsman:InvalidSelectors
#define WSMAN_ERR_SELECTOR_TYPE_MISMATCH		3	// subcode : wsman:InvalidSelectors
#define WSMAN_ERR_INVALID_SELECTOR_VALUE		4	// subcode : wsman:InvalidSelectors
#define WSMAN_ERR_UNEXPECTED_SELECTORS			5	// subcode : wsman:InvalidSelectors
#define WSMAN_ERR_DUPLICATE_SELECTORS			6	// subcode : wsman:InvalidSelectors
// @}

/** \name Other error codes
 * @{
 */
#define WSMAN_ERR_INTERNAL_ERROR				7	// subcode : wsman:InternalError
#define WSMAN_ERR_SCHEMA_VALIDATION_ERROR		8	// subcode : wsman:InternalError
#define WSMAN_ERR_CONCURRENCY					9	// subcode : wsman:InternalError
// @}

/** \name Create & put related fault error codes
 * @{
 */
#define WSMAN_ERR_INVALID_VALUES				10	// subcode : wst:InvalidRepresentation
#define WSMAN_ERR_MISSING_VALUES				11	// subcode : wst:InvalidRepresentation
#define WSMAN_ERR_INVALID_NAMESPACES			12	// subcode : wst:InvalidRepresentation
// @}

/** \name Create related fault error codes
 * @{
 */
#define WSMAN_ERR_ALREADY_EXISTS				13	// subcode : wsman:AlreadyExists
// @}

/** \name Fragment-level related fault error codes
 * @{
 */
#define WSMAN_ERR_UNSUPPORTED_FRAGMENT_ACCESS	14	// subcode : wsman:UnsupportedFeature
#define WSMAN_ERR_INVALID_FRAGMENT				15	// subcode : wst:InvalidRepresentation
// @}

/*----------------------------------------------------------------------------*\
 *                               Type definitions                             *
\*----------------------------------------------------------------------------*/

/***************************************************************************//**
* \name Configuration API
* @{
*******************************************************************************/

/**
 * A structure used to hold a set of handling functions for each
 * of the four WS-Management operations, and the resource URI
 * corresponding to the type of resource to be managed.
 */
typedef struct wsman_resource_handler {
	char* resource_uri;	/**< The WS-Management resource URI. */
	serve_cbk get_func;	/**< The get callback. */
	serve_cbk put_func;	/**< The put callback. */
	serve_cbk create_func;	/**< The create callback. */
	serve_cbk delete_func;	/**< The delete callback. */
} wsman_resource_handler_t;

/*----------------------------------------------------------------------------*\
 *                                    APIs                                    *
\*----------------------------------------------------------------------------*/

/**
 * Create a service class object used to handle WS-Management messages.
 * The service class is initialized with a set of handling functions structures,
 * corresponding to the different types of resources to be managed.
 * @param handlers An array of pointers to the set of handling functions.
 * The array will be copied, but not the structures pointed to (they must be long-lived).
 * @param handlers_count The number of handlers passed in the array.
 * @param default_handler the set of handling functions to be used when other handlers
 * do not match. Can be NULL.
 * @param class_id the name of the newly created class. Can be NULL.
 * @return a handle to the created service class, or -1 otherwise.
 */
DC_RT_FMAC1 short wsman_create_management_class(struct wsman_resource_handler ** handlers, int handlers_count, struct wsman_resource_handler * default_handler, char * class_id);

/**
 * Initialize a management service class with a set of handling functions structures,
 * corresponding to the different types of resources to be managed.
 * @param href_class A handle to the management service class to be initialized.
 * @param handlers An array of pointers to the set of handling functions.
 * The array will be copied, but not the structures pointed to (they must be long-lived).
 * @param handlers_count The number of handlers passed in the array.
 * @param default_handler the set of handling functions to be used when other handlers
 * do not match. Can be NULL.
 */
DC_RT_FMAC1 void wsman_set_resource_handlers(short href_class, struct wsman_resource_handler ** handlers, int handlers_count, struct wsman_resource_handler * default_handler);

/**
 * Create a service object for the associated device and management service class.
 * @param href_device The device that hosts the service.
 * @param href_class The management class of the service.
 * @param context_path The context_path of the service URL.
 * @return a handle to the created service, or -1 otherwise.
 */
DC_RT_FMAC1 short wsman_create_management_service(short href_device, short href_class, char* context_path);

/**
 * Configure the management service for a given resource type. Neither the resource URI string
 * nor the configuration data will be copied (they must be long-lived).
 * @param href_service The management service handle.
 * @param resource_uri The resource type to which the configuration data is attached.
 * @param configuration_data The data to be attached to the resource type.
 * @return DPWS_OK if successful, an error value otherwise.
 */
DC_RT_FMAC1 int wsman_configure_management_service(short href_service, char* resource_uri, void* configuration_data);

// @}

/***************************************************************************//**
* \name Runtime API
* @{
*******************************************************************************/

/**
 * WS-MAN dispatch function.
 * @param dpws the DPWS runtime context.
 * @return a gSOAP error code.
 */
DC_RT_FMAC1 int wsman_serve_request(struct dpws *dpws);

/**
 * Retrieves the resource URI for the current request.
 * @param dpws the DPWS runtime context.
 * @return a string containing the resource URI.
 */
DC_RT_FMAC1 char * wsman_get_resource_uri(struct dpws * dpws);

/**
 * Retrieves the selectors for the current request. The returned values are only valid
 * during the processing of the request.
 * @param dpws the DPWS runtime context.
 * @param selectors a pointer to the array of selectors, filled on output
 * @param selectors_count a pointer to an int containing the number of returned selectors,
 * filled on output.
 * @return DPWS_OK if successful, an error value otherwise.
 */
DC_RT_FMAC1 int wsman_get_selectors(struct dpws * dpws, struct wsman_selector ** selectors, int * selectors_count);

/**
 * Returns true is a fragment transer header is used for the current request.
 * @param dpws the DPWS runtime context.
 * @return 0 or 1.
 */
DC_RT_FMAC1 int wsman_use_fragment_transfer(struct dpws * dpws);

/**
 * Retrieves the XPath fragment as an array of QNames for the current request.
 * The returned values are only valid during the processing of the request.
 * @param dpws the DPWS runtime context.
 * @param xpath a pointer to the array of QNames, filled on output
 * @param segment_count a pointer to an int containing the size of the QName array,
 * filled on output.
 * @return DPWS_OK if successful, an error value otherwise.
 */
DC_RT_FMAC1 int wsman_get_xpath_level_1(struct dpws * dpws, struct qname ** xpath, int * segment_count);

/**
 * Retrieves the configuration information associated with the current service and
 * specified resource URI.
 * @param dpws the DPWS runtime context.
 * @param resource_uri the resource URI used in the request
 * @return the configuration data, or NULL if none is set.
 */
DC_RT_FMAC1 void * wsman_get_configuration_data(struct dpws * dpws, char* resource_uri);

/**
 * Sets the approapriate SOAP fault parameters in the dpws context.
 * @param dpws the DPWS runtime context.
 * @param err the WS-Man error code (see above)
 * @param underlying the internal error code (EPX, gSOAP, etc.)
 * @return DPWS_OK, or an error code.
 */
DC_RT_FMAC1 int wsman_set_fault(struct dpws * dpws, int err, int underlying);

/**
 * Callback function to be used when returning from a WS-Man creation handler
 * to serialize the created endpoint reference.
 * @param sctx the serialization context of the response.
 * @param user_data the endpoint reference to be serialized
 * @return DPWS_OK, or an error code.
 */
DC_RT_FMAC1 int wsman_serialize_created_epr_callback(void * sctx, void* user_data);

// @}

#ifdef __cplusplus
}
#endif

#endif
