/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2153 $
|                     $Date: 2009-03-05 17:36:55 +0100 (jeu, 05 mar 2009) $
\*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.
#include "dc/dc_WsManConstants.h"	// Defines WS-Management constants used to retrieve dynamic deployment service.
#include "dc/dc_DyndeplClient.h"	// Dynamic deployment client functions include file.
#include "dc/dc_FileX.h"	// XML file utilities.

#include "fileio.h"	// File utilities.

#define DYNDEPL_NS_URI	"http://www.soa4d.org/dpwscore/2007/08/dyndepl"	// The namespace used for the dynamic deployment bootstrap device.
#define BOOTSTRAP_DEVICE_TYPE "DPWSCore"	// The type used for the bootstrap device hosting the WS-Management dynamic deployment service.

/*
 *  Utilities
 */

/* Generic stub callback that uses EPX XML processing API to produce
 * WS-Management request bodies from an XML file.
 */
int request_cbk (void * szr_ctx, char * file_name)
{
	int ret = xml_file_parse(szr_ctx, file_name);
	if (ret)
		printf("Could not load file.\n");
	return ret;
}

/* Generic stub callback that uses EPX XML processing API to read
 * WS-Management response bodies that are printed on the standard
 * output.
 */
int response_cbk (void * parser_ctx, void * dummy)
{
	int ret = SOAP_OK;
	printf("\nResponse content:\n\n");
	ret = xml_file_serialize(parser_ctx, NULL);
	if (ret)
		printf("Error receiving response.\n");
	return ret;
}

/* Function printing error message. */
static void print_error(char * msg, struct dpws * dpws) {
	fprintf(stderr, "\nError occured : %s.\n%s\n\n", msg, dpws_get_error_msg(dpws));
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: manager <action> <resource> [<id1>] [<id2>] [<request_file>]\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "Usage: manager <action> <resource> [<id1>] [<id2>] [<request_file>]\n");
	fprintf(stderr, "\nParameters : \n");
	fprintf(stderr, "\n  <action>\n");
	fprintf(stderr, "    The action to be performed, should be either get, create, delete or put.\n");
	fprintf(stderr, "\n  <resource>\n");
	fprintf(stderr, "    The type of registry element on which the action should be executed,\n");
	fprintf(stderr, "    should be either device, types, scopes, this_device, this_model, serv_class or service.\n");
	fprintf(stderr, "\n  Note :\n");
	fprintf(stderr, "  The only supported combinations are : \n");
	fprintf(stderr, "    manager get [device/types/scopes/this_device/this_model/serv_class] dev_uuid\n");
	fprintf(stderr, "    manager get service dev_uuid serv_uuid\n");
	fprintf(stderr, "    manager put [device/types/scopes/this_device/this_model] dev_uuid file\n");
	fprintf(stderr, "    manager create [device/serv_class] file\n");
	fprintf(stderr, "    manager create service dev_uuid serv_uuid file\n");
	fprintf(stderr, "    manager delete [device/serv_class] dev_uuid\n");
	fprintf(stderr, "    manager delete service dev_uuid serv_uuid\n");
	fprintf(stderr, "\nOptions : \n");
	fprintf(stderr, "\n  <id1>\n");
	fprintf(stderr, "    The device identifier.\n");
	fprintf(stderr, "  <id2>\n");
	fprintf(stderr, "    The service identifier.\n");
	fprintf(stderr, "  <request_file>\n");
	fprintf(stderr, "    The file containing necessary information for the command.\n");
	exit(1);
}

/*
 *  Main function performing dynamic deployment requests.
 *	Usage: manager <action> <resource> [<id1>] [<id2>] [<request_file>]
 */
int main(int argc, char* argv[]) {

	int nb_devices = 1,  nb_services, status = DPWS_OK;
	short device_proxy = -1, * device_proxies, service_proxy = -1, * service_proxies;
	struct dpws dpws;
	char * file_name;
	char * action;
	char * resource;

	// Command line parsing
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help")))
		usage();

	if (argc < 4 || argc > 6)
		simple_usage();

	action = argv[1];
	resource = argv[2];
	file_name = argv[argc - 1];
	if (!strcasecmp(action, "get")) {
		// verify resource.
		if (strcasecmp(resource, "device") && strcasecmp(resource, "types") && strcasecmp(resource, "scopes") && strcasecmp(resource, "this_device") && strcasecmp(resource, "this_model") && strcasecmp(resource, "serv_class") && strcasecmp(resource, "service"))
			usage();
		// verify argument number.
		if (!strcasecmp(resource, "service")) {
			if (argc != 5)
				usage();
		} else if (argc != 4)
			usage();
	}
	else if (!strcasecmp(action, "put")) {
		// verify resource.
		if (strcasecmp(resource, "device") && strcasecmp(resource, "types") && strcasecmp(resource, "scopes") && strcasecmp(resource, "this_device") && strcasecmp(resource, "this_model"))
			usage();
		// verify argument number.
		if (argc != 5)
			usage();
	}
	else if (!strcasecmp(action, "create")) {
		// verify resource.
		if (strcasecmp(resource, "device") && strcasecmp(resource, "serv_class") && strcasecmp(resource, "service"))
			usage();
		// verify argument number.
		if (!strcasecmp(resource, "service")) {
			if (argc != 5)
				usage();
		} else if (argc != 4)
			usage();
	}
	else if (!strcasecmp(action, "delete")) {
		// verify resource.
		if (strcasecmp(resource, "device") && strcasecmp(resource, "serv_class") && strcasecmp(resource, "service"))
			usage();
		// verify argument number.
		if (!strcasecmp(resource, "service")) {
			if (argc != 5)
				usage();
		} else if (argc != 4)
			usage();
	}
	else
		usage();

	/* Manager program description (client initialization and dynamic deployment) */
	// 1. Initialize the DPWS stack.
	// 2. Initialize the dpws structure for client operations.
	// 3. Find device proxy implementing the service to be invoked.
	// 4. Find service proxy of the service to be invoked.
	// 5. Operation selection and invocation

	/* 1. Initialize the DPWS stack. */

	// This is the first function to be called when using the DPWS stack.
	// This version selects one single local IPv4 address.
	if ((status = dpws_init())) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Initialize the dpws structure for client operations. */
	// Initializes the dpws structure and performs one-time client-side initialisation.
	if ((status = dpws_client_init(&dpws, NULL) != DPWS_OK)) {
		fprintf(stderr, "Could not initialize client side of the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 3. Find device proxy implementing the service to be invoked. */

	// The dpws_lookup function retrieves device proxies using the WS-Discovery
	// feature of the DPWSCore stack.
	// It will first look for proxies into the local cache, and then generate a
	// WS-Discovery Probe message if the number of requested proxies is not
	// available in the cache.
	// Note that a type and/or a scope filter can be used but in a very simple
	// form: dpws_lookup_ex function should be used to specify more parameters.

	printf("<-- Looking for device :\n");
	printf("\tWith type : {%s}%s\n", DYNDEPL_NS_URI, BOOTSTRAP_DEVICE_TYPE);

	nb_devices = 1;	// The program searches a single device.
	device_proxies = dpws_lookup(&dpws, DYNDEPL_NS_URI, BOOTSTRAP_DEVICE_TYPE, NULL, &nb_devices);

	/* Handling of the dpws_lookup function return values. */
	if (nb_devices != 1) {
		if (nb_devices == 0)
			printf("\nNo Device found, please check probe parameters.\n");
		else
			print_error("Could not execute dpws_lookup", &dpws);
		goto error;
	}
	printf("Bootstrap device found.\n");

	device_proxy = device_proxies[0];

	// Make sure the device proxy will not be erased from the cache using
	// dpws_pin_proxy because next dpws_end call will release the returned array
	// reference so the potential reception of BYE message (if a discovery
	// listener was running, which is not the case however in this sample) may
	// erase the proxy.
	if (dpws_pin_proxy(device_proxy) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", &dpws);
		goto error;
	}
	// Free dynamically allocated memory used for request processing and releases
	// proxies retrieved using a lookup API. This is good practice to call it
	// after every call to avoid memory peak.
	if (dpws_end(&dpws) != DPWS_OK) {
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 4. Find service proxy of the service to be invoked. */

	// The dpws_get_services retrieves one or several service proxies hosted by
	// a remote device using a type filtering (other APIs are available).
	// It will look for proxy available into the local cache or generate a
	// WS_Transfer/WS-MetadataExchange message if necessary.

	printf("Looking for service :\n\tOn device proxy : %d\n", device_proxy);
	printf("\tWith type : {%s}%s\n", WST_URI, WST_RESOURCE_PORT_TYPE);

	nb_services = 1;	// The program searches a single service.
	service_proxies = dpws_get_services(&dpws, device_proxy, WST_URI, WST_RESOURCE_PORT_TYPE, &nb_services);

	if (nb_services != 1)
	{
		if (nb_services == 0)
			printf("No service endpoint found, please check the port type specifications.\n");
		else
			print_error("Could not execute dpws_get_services", &dpws);
		goto error;
	}

	service_proxy = service_proxies[0];

	// Make sure the service proxy will not be erased from the cache for the same
	// reasons than for the hosting device.
	if (dpws_pin_proxy(service_proxy) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", &dpws);
		goto error;
	}
	if (dpws_end(&dpws) != DPWS_OK) {	// Good practice to call it after every call to avoid memory peak.
		print_error("Could not execute dpws_end", &dpws);
		goto error;
	}

	/* 5. Operation selection and invokation */
	if (!strcasecmp(action, "get"))
	{
		if (!strcasecmp(resource, "device"))
			status = dyndepl_get_device(&dpws, service_proxy, argv[3], response_cbk, NULL);
		else if (!strcasecmp(resource, "types"))
			status = dyndepl_get_device_types(&dpws, service_proxy, argv[3], response_cbk, NULL);
		else if (!strcasecmp(resource, "scopes"))
			status = dyndepl_get_device_scopes(&dpws, service_proxy, argv[3], response_cbk, NULL);
		else if (!strcasecmp(resource, "this_device"))
			status = dyndepl_get_this_device(&dpws, service_proxy, argv[3], response_cbk, NULL);
		else if (!strcasecmp(resource, "this_model"))
			status = dyndepl_get_this_model(&dpws, service_proxy, argv[3], response_cbk, NULL);
		else if (!strcasecmp(resource, "serv_class"))
			status = dyndepl_get_service_class(&dpws, service_proxy, argv[3], response_cbk, NULL);
		else if (!strcasecmp(resource, "service"))
			status = dyndepl_get_service(&dpws, service_proxy, argv[3], argv[4], response_cbk, NULL);
	}
	else if (!strcasecmp(action, "put"))
	{
		if (!strcasecmp(resource, "device"))
			status = dyndepl_put_device(&dpws, service_proxy, argv[3], (parser_cbk)request_cbk, response_cbk, file_name);
		else if (!strcasecmp(resource, "types"))
			status = dyndepl_put_device_types(&dpws, service_proxy, argv[3], (parser_cbk)request_cbk, response_cbk, file_name);
		else if (!strcasecmp(resource, "scopes"))
			status = dyndepl_put_device_scopes(&dpws, service_proxy, argv[3], (parser_cbk)request_cbk, response_cbk, file_name);
		else if (!strcasecmp(resource, "this_device"))
			status = dyndepl_put_this_device(&dpws, service_proxy, argv[3], (parser_cbk)request_cbk, response_cbk, file_name);
		else if (!strcasecmp(resource, "this_model"))
			status = dyndepl_put_this_model(&dpws, service_proxy, argv[3], (parser_cbk)request_cbk, response_cbk, file_name);
	}
	else if (!strcasecmp(action, "create"))
	{
		if (!strcasecmp(resource, "device"))
			status = dyndepl_create_device(&dpws, service_proxy, (parser_cbk)request_cbk, response_cbk, file_name);
		else if (!strcasecmp(resource, "serv_class"))
			status = dyndepl_create_service_class(&dpws, service_proxy, (parser_cbk)request_cbk, response_cbk, file_name);
		else if (!strcasecmp(resource, "service"))
			status = dyndepl_create_service(&dpws, service_proxy, argv[3], (parser_cbk)request_cbk, response_cbk, file_name);
	}
	else if (!strcasecmp(action, "delete"))
	{
		if (!strcasecmp(resource, "device"))
			status = dyndepl_delete_device(&dpws, service_proxy, argv[3]);
		else if (!strcasecmp(resource, "serv_class"))
			status = dyndepl_delete_service_class(&dpws, service_proxy, argv[3]);
		else if (!strcasecmp(resource, "service"))
			status = dyndepl_delete_service(&dpws, service_proxy, argv[3], argv[4]);
		if (!status)
			printf("Resource deleted.\n");
	}
	if (status)
		print_error("Could not execute request", &dpws);

error:
	if (dpws_end(&dpws) != DPWS_OK)	// Resets structure especially memory allocated for request processing.
		print_error("Could not execute dpws_end", &dpws);
	if (device_proxy >=0) {	// Releasing handle on proxy kept using dpws_pin_proxy
		if (dpws_release_proxy(device_proxy) != DPWS_OK)
			print_error("Could not execute dpws_release_proxy", &dpws);
	}
	if (service_proxy >= 0) {	// Releasing handle on proxy kept using dpws_pin_proxy
		if (dpws_release_proxy(service_proxy) != DPWS_OK)
			print_error("Could not execute dpws_release_proxy", &dpws);
	}
	if ((status = dpws_shutdown()))	// cleans all stack data and resources.
		fprintf(stderr, "Could not shut down the DPWSCore stack (err %d)\n\n", status);

	return 0;
}


