/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|             ->> Copyright 2004-2009 Schneider Electric SA <<-                |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2196 $
|                     $Date: 2009-03-11 18:53:17 +0100 (mer, 11 mar 2009) $
\*============================================================================*/
/******************************************************************************\
 *                       Washer Static Service Class Loader                   *
\******************************************************************************/

#include "sscl_wsh.h"

#include "wshStub.h" // Generated stub & skeleton include file for washer device implementation.
#include "wsh.nsmap" // Generated namespace table include file for washer device implementation.

#include "litStub.h" // Generated stub & skeleton include file for light device reference client use.

#include "dc/dc_Epx.h"	// XML streaming processing API.
#include "dc/dc_XMLUtils.h"	// XML duration utilities.
#include "serverDPWS.h"	// OS-dependent code (especially time functions).

/* Service class loader callback declarations. */
static int wsh_new_service(short href_endpoint);
static int wsh_free_service(short href_endpoint);
static int wsh_serialize_impl(short href_sclass, void * szr_ctx);

/*
 *	Definitions
 */

struct washerData {	// Device instance runtime data for the washing machine.
	int cycle;	// Will receive the running cycle enumeration value. -1 means no cycle.
	int32_t scheduledEnd;	// time in seconds for the end of the cycle
	struct dpws dpws;	// runtime structure for event notification & client invocation
	short hEventSource;	// event source handle reference used for event notification
};

/*
 *	Runtime data
 */

// Identifier for this service implementation (service class loader) used in the
// configuration file.
struct qname wsh_iqn = { WSH_NS, WSH_ELT_SSCL_IMPL };

struct wsa_endpoint_ref * warn_light_epr = NULL;	// Light EPR for cycle end invocation

/* Callback associated to the previously defined qualified name.
   It will be called when the washer service class loader is loaded in order to
   initialize the various functions called when:
	+ a service of this class is created,
	+ a service of this class is deleted,
	+ the service class is to be serialized in XML for instance for configuration
	  backup,
	+ a request for a service is to be processed.
	See load_cbk reference for more information.
*/
int wsh_load_cbk(short href_sclass, void * psr_ctx, dispatch_cbk * p_dispatch_cbk, struct scl_callbacks * p_cbks)
{
	/* Achieves implementation tag parsing with EPX (here very simple). Note that
	 * the current event is the 'implementation' start tag
	 */
	if (epx_next(psr_ctx) != EPX_EVT_END_ELEMENT || QNAME_NOT_EQUALS_WILDCARD(epx_get_ns_uri(psr_ctx), epx_get_lname(psr_ctx), WSH_NS, WSH_ELT_SSCL_IMPL))
		return DPWS_ERR_INCORRECT_IMPL_TAG;

	*p_dispatch_cbk = wsh_serve_request;	// generated skeleton dispatch function.
	p_cbks->new_service = wsh_new_service;	// service instance creation callback.
	p_cbks->free_service = wsh_free_service;	// service instance deletion callback.
	p_cbks->serialize_impl = wsh_serialize_impl;	// serialization callback
	return DPWS_OK;
}

/* Allows to initialize washer service runtime data on service creation. */
int wsh_new_service(short href_endpoint)
{
	struct washerData * washer = malloc(sizeof(struct washerData));	// service data allocated on the heap.
	/* Initialize washer state. */
	washer->cycle = -1; // -1 means no cycle
	washer->scheduledEnd = 0;
	dpws_client_init(&washer->dpws, NULL);
	washer->hEventSource = href_endpoint;
	return dpws_set_ptr_att(href_endpoint, DPWS_PTR_USER_DATA, washer);	// store it in service user data
}

/* On service deletion the user data is freed using this callback. */
int wsh_free_service(short href_endpoint)
{
	free(dpws_get_ptr_att(href_endpoint, DPWS_PTR_USER_DATA));
	return DPWS_OK;
}

/* Service class implementation tag serialization callback. */
int wsh_serialize_impl(short href_sclass, void * szr_ctx)
{
	epx_start_element(szr_ctx, WSH_NS, WSH_ELT_SSCL_IMPL);	// Here the whole tag must be generated using EPX
	epx_define_prefix(szr_ctx, WSH_PREFIX, WSH_NS);
	epx_end_element(szr_ctx, WSH_NS, WSH_ELT_SSCL_IMPL);
	return DPWS_OK;
}

/*
 *                   Service operations
 */

/* Starts a washing cycle. */
int __wsh__LaunchCycle(struct dpws* dpws, struct _wsh__LaunchCycle *wsh__LaunchCycle)
{
	struct washerData * pState;
	struct _wsh__CycleEnd cycleEnd;
	int cycleDuration = 0, status = DPWS_OK, prop_nb = 1;
	char * prop_value, * prop_name;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_endpoint_user_data.
	pState = (struct washerData *)dpws_get_endpoint_user_data(dpws);

	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	printf("--> Requesting cycle launch\n");

	if (pState->cycle != -1) {
		printf("<-- Error : Cycle is already running!\n");
		return dpws_fault(dpws, RECEIVER_FAULT_VALUE, "Cycle running", NULL, NULL);
	}
	printf("Starting cycle...") ;
	switch (wsh__LaunchCycle->Name)	// cycle type switch
	{
		case wsh__Cycle__Gentle:
			prop_name = "GentleCycleDuration";
			break;
		case wsh__Cycle__Regular:
			prop_name = "RegularCycleDuration";
			break;
		case wsh__Cycle__Heavy:
			prop_name = "HeavyCycleDuration";
			break;
	}
	/* Retrieves duration from the configuration file using properties. */
	status = dpws_get_property_values(dpws->href_endpoint, prop_name, &prop_value, &prop_nb);
	if (status) {
		printf("Error while retrieving washing properties (%d).\n", status);
		goto exit;
	}
	cycleDuration = atoi(prop_value);
	printf(" (duration %ds)\n", cycleDuration);
	pState->scheduledEnd = portableTime() + cycleDuration;
	pState->cycle = wsh__LaunchCycle->Name;
	portableSleep(cycleDuration);	// NOTE: server must be multithread or won't be able to answer any request
	pState->cycle = -1;

	/* Cycle end WS-Eventing notification */
	printf("<-- Notifying end of cycle\n") ;
	cycleEnd.CycleName = wsh__LaunchCycle->Name;
	if ((status = dpws_notify___wsh__CycleEnded(&pState->dpws, pState->hEventSource, &cycleEnd)))	// send an event to subscribers using a specific dpws structure
		fprintf(stderr, "Could not send 'cycle end' event (err %d)\n", status);
	dpws_end(&pState->dpws);	// free memory allocated for event message

	/* Light on kitchen light */
	if (!warn_light_epr)
	{
		status = dpws_lookup_binding_ref(&pState->dpws, dpws->href_endpoint, WSH_BREF_WARNLIGHT, &warn_light_epr);
		if (status) {
			printf("Error while binding warn light reference (%d).\n", status);
			goto exit;
		}
	}

	printf("<-- Switching kitchen light ON\n") ;
	status = dpws_send___lit__Switch(&pState->dpws, warn_light_epr, lit__PowerState__ON);
	if (status)
		status = dpws_bind_reference(&pState->dpws, dpws->href_endpoint, WSH_BREF_WARNLIGHT, &warn_light_epr);

exit:
	dpws_end(&pState->dpws);	// free memory allocated for invocation
	return status;
}

/* Retrieves the current washing machine status. */
int __wsh__GetCycleStatus(struct dpws* dpws, struct _wsh__CycleStatus *wsh__CycleStatus)
{
	struct washerData * pState;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_endpoint_user_data.
	pState = (struct washerData *)dpws_get_endpoint_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	printf("--> Requesting cycle status\n");
	if (pState->cycle == -1) {
		wsh__CycleStatus->TimeLeft = NULL;
		printf("<-- No cycle running\n");
	} else {
		/*
		NOTE about result allocation:
			Even if the top level structure is allocated by the skeleton,
			contents are under service implementor's responsibility.
		*/
		char * duration = dpws_malloc(dpws, DURATION_MAX_SIZE + 1);	// Constant defined in the DPWS API
		wsh__CycleStatus->TimeLeft = dpws_malloc(dpws, sizeof(struct struct_1));	// message transient allocation
		wsh__CycleStatus->TimeLeft->__item = dcxml_duration2xmlduration(duration, pState->scheduledEnd - portableTime());
		wsh__CycleStatus->TimeLeft->CycleName = pState->cycle;
		switch (wsh__CycleStatus->TimeLeft->CycleName) {
		case wsh__Cycle__Gentle:
			printf("<-- Cycle name GENTLE , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		case wsh__Cycle__Regular:
			printf("<-- Cycle name REGULAR , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		case wsh__Cycle__Heavy:
			printf("<-- Cycle name HEAVY , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		}
	}
	return DPWS_OK;
}
