/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2137 $
|                     $Date: 2009-03-02 19:10:12 +0100 (lun, 02 mar 2009) $
\*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.
#include "dc/dc_XMLUtils.h"	// included for XML duration conversion utilities

#include "wshStub.h" // Generated stub & skeleton include file.
#include "wsh.nsmap" // Generated namespace table include file.

#include "fileio.h"	// File utility used for boot sequence persistency.
#include "serverDPWS.h"	// OS-dependent code (especially server loop).

/*
 * IMPORTANT NOTE: This multi-thread server is supplied as an example for studying purpose
 * and does not meet the usual standards of industrial software. For instance, device data
 * should be protected against concurrent access from server threads.
 */

/*
 *	Device Configuration data
 */

#define MAX_SCOPE_LENGTH 255	// Maximum length for computed scope.
#define SAMPLE_SCOPE "http://www.soa4d.org/DPWS/Samples/WashingMachine/Scope"	// The scope for the washing machine device

#define BOOT_SEQ_FILE "bootseq.log"	// The name of the file holding the boot sequence number.

#define SAMPLE_NS "http://www.soa4d.org/DPWS/Samples/WashingMachine"	// The namespace used to qualify device port types.
#define LAUNDRY_NS "http://www.soa4d.org/DPWS/Samples/Home/Laundry"	// The namespace used to qualify service port types.

struct qname WashingMachineType = {SAMPLE_NS, "WashingMachine"};	// Qualified name used to specify the type supported by the device (could have been prefixed).

struct prefixed_qname WashPortType	// Prefixed qualified name used to define the service port type (from WSDL). Using
	= {{LAUNDRY_NS, "Wash"}, "wsh"};	// a prefix will ensure it will be used in messages.

struct wsdl_info WashingMachineWsdl	// Structure containing the WSDL namespace and location for the hosted service.
	= {LAUNDRY_NS, "http://www.soa4d.org/DPWS/Samples/WashingMachine/WashingMachine.wsdl"};

/*
 *	Runtime data
 */

struct washerData {	// Device instance runtime data
	int cycle;	// Will receive the running cycle enumeration value. -1 means no cycle.
	int32_t scheduledEnd;	// time in seconds for the end of the cycle
	struct dpws dpws;	// runtime structure for event notification & client invocation
	short hEventSource;	// event source handle reference used for event notification
} washer;

static struct dpws master_dpws;	// The dpws structure used for message monitoring (main thread of the multi-thread server)


/*
 *  Utilities
 */

static char diversified_scope[MAX_SCOPE_LENGTH];	// computed scope buffer

/* Function adding the suffix to default scope (if suffix was defined).
 * Returns 0 if ok 1 if error (string too long).
 */
static int compute_diversified_scope(char *scope_suffix)
{
	strcpy(diversified_scope, SAMPLE_SCOPE);
	if (scope_suffix) {
		if (sizeof(SAMPLE_SCOPE) + strlen(scope_suffix) > MAX_SCOPE_LENGTH)
			return 1;
		if (scope_suffix[0] != '/')
			strcat(diversified_scope, "/");
		strcat(diversified_scope, scope_suffix);
	}
	return 0;
}

/* Function printing information related to a device. */
static void print_device_info(short hrefDevice)
{
	int len = 10, sz, i, j;
	short hrefs[10];
	char ** transportAddrs = NULL;

	// One service port (HTTP binding) created by default but potentially multiple
	// local addresses (multiple network interfaces & IP versions)
	transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefDevice), &sz);

	printf("\n+ Device [%s] started\n\nAddresses:\n", dpws_get_ptr_att(hrefDevice, DPWS_STR_DEVICE_ID));
	for (i = 0; i < sz; i++) {
		printf("[%d] %s\n", i, transportAddrs[i]);
		dpws_free(transportAddrs[i]);	// because of dpws_get_transport_addresses
	}
	dpws_free(transportAddrs);	// because of dpws_get_transport_addresses

	if (!dpws_get_service_handles(hrefDevice, hrefs, &len)) {
		printf("\n++ Hosted services:\n\n");
		for (i = 0; i < len; i++)
		{
			transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefs[i]), &sz);	// same as for device
			printf("Service ID: %s\nAddresses:\n", dpws_get_ptr_att(hrefs[i], DPWS_STR_SERVICE_ID));
			for (j = 0; j < sz; j++) {
				printf("[%d] %s\n", j, transportAddrs[j]);
				dpws_free(transportAddrs[j]);	// because of dpws_get_transport_addresses
			}
			dpws_free(transportAddrs);	// because of dpws_get_transport_addresses
		}
	}
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: server [-s <scopeDiversifier>] [-p <httpPort>] [-i <ipMode>]\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "\nUsage: server [-s <scopeDiversifier>] [-p <httpPort>] [-i <ipMode>]\n");
	fprintf(stderr, "\nOptions : \n");
	fprintf(stderr, "\n -s <scopeDiversifier>\n");
	fprintf(stderr, "\n\tThe <scopeDiversifier> is used to avoid conflicts on the local\n");
	fprintf(stderr, "\t\tnetwork by participating in building the scope of the device.\n");
	fprintf(stderr, "\n -p <httpPort>\n");
	fprintf(stderr, "\n\tThe <httpPort> is the http port used to listen to soap messages.\n");
	fprintf(stderr, "\n -i <ipMode>\n");
	fprintf(stderr, "\n\tThe <ipMode> should be one of '4', '6' or 'dual' according you want\n");
	fprintf(stderr, "\tthe server to listen on all IPv4, IPv6 or both IP addresses. Default is 'dual'.\n");
	fprintf(stderr, "\nNotes : \n");
	fprintf(stderr, "\n\tThe server should be started before the client.\n\n");
	exit(1);
}

/*
 *  Main function performing server initialization.
 *	Usage: server [-s <scopeDiversifier>] [-p <httpPort>] [-i <ipMode>]
 */
int main (int argc, char **argv)
{
	/* Local variables */
	int port = 9876, c, bootSeq, status, i;
	short hServClass, hWasher, hWServPort;
	char * scope_suffix = NULL;
	dc_ip_filter_t ip_selector = {
			NULL,	// consider all interfaces
			DC_FALSE,	// but not loopback
			DC_PROTO_ANY,	// default program setting : dual IPv4/v6
			0,	// no address filtering
			NULL	// no address filtering
		};
	struct localized_string ls = {NULL, NULL};

	/* Command line parsing */
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help"))) {
		usage();
	}
	if ((argc < 1) || (argc > 6))
		simple_usage();

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-s")) {
			if (++i == argc)
				simple_usage();
			scope_suffix = argv[i];
		} else if (!strcmp(argv[i], "-p")) {
			if (++i == argc)
				simple_usage();
			port = atoi(argv[i]);
			if (port == 0)
				simple_usage();
		} else if (!strcmp(argv[i], "-i")) {
			if (++i == argc)
				simple_usage();
			if (argv[i][0] == '4')
				ip_selector.proto = DC_PROTO_INET;
			else if(argv[i][0] == '6')
				ip_selector.proto = DC_PROTO_INET6;
			// else DUAL
		} else
			simple_usage();
	}

	// Compute diversified scope, the scope will be used to discover devices on the network.
	// If a suffix has been used only the clients started using the same suffix will detect
	// the current device.
	if (compute_diversified_scope(scope_suffix)) {
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		simple_usage();
	}

	/* Simple server program description (single device with single service) */
	// 1. Initialize the DPWS stack.
	// 2. Configure server.
	// 	2.1 Set the boot sequence number & others toolkit parameters.
	//	2.2 Create service class.
	//	2.3 Create & configure device.
	//	2.4 Create & configure hosted service.
	//	2.5 Enable device.
	// 3. Initialize server.
	// 4. Start server loop.
	// 5. Stop the server

	printf("Configuring DPWS device...\n\n");

	/* 1. Initialize the dpws stack. */

	// This is the first function to be called when using the DPWS stack.
	// This version allows multiple address selection.
	if ((status = dpws_init_ex(&ip_selector, NULL, DPWS_DEFAULT_VERSION))) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Configure server. */

	// 	2.1 Set the boot sequence number & others toolkit parameters.

	// The boot sequence number is used is WS-Discovery messages and should be stored
	// and incremented each time the server is started.
	// Failing to do so may result in messages ignored by peers.
	bootSeq = readintfromfile(BOOT_SEQ_FILE);	// Reading the boot sequence number from a file
	writeinttofile(bootSeq + 1, BOOT_SEQ_FILE);	// Writing the next boot sequence number to the same file

	/*
	NOTE:
		All configuration APIs may return an error code. It has not been tested
		in this sample for clarity's sake and because the only errors that can
		occur are memory exhaustion (bad omen at initialization time) or due to
		invalid parameters (and we hope this sample has been correctly written :-)).
	*/
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT, port);	// Configure the HTTP listen port.
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ, bootSeq);	// Configure the boot sequence number to use in this session (mandatory).

	printf("Initializing DPWS washing machine device...\n");

	// 2.2 Create service class
	printf("Creating service class...\n");
	hServClass = dpws_create_service_class();

	// Configure the service class attributes.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &WashPortType);	// The service implements the 'wash" port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_WSDL, &WashingMachineWsdl);	// The WSDL information for the previous port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_HANDLING_FUNCTION, &wsh_serve_request);	// The generated dispatch function for the port type (from wshStub.h).
	DPWS_SET_STR_ATT(hServClass, DPWS_STR_ID, "http://www.soa4d.org/DPWS/Samples/DPWS/WashingMachine/Washer1");	// A string ID that will used as default for mandatory service ID

	// 2.3 Create & configure device
	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "WashingMachine");
	hWasher = dpws_create_custom_device(0, -1);	// the id (0) must be unique in the physical local device

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hWasher, DPWS_PTR_TYPE, &WashingMachineType);
	/*
	NOTE:
		Should change everytime a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hWasher, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Washing machine";
	DPWS_SET_STR_ATT(hWasher, DPWS_PTR_FRIENDLY_NAME, &ls);
	ls.s = "Salsa";
	DPWS_SET_STR_ATT(hWasher, DPWS_PTR_MODEL_NAME, &ls);
	ls.s = "Whurlpool";
	DPWS_SET_STR_ATT(hWasher, DPWS_PTR_MANUFACTURER, &ls);

	// Configure the optional device attributes.
	printf(" with scope %s\n\n", diversified_scope);
	DPWS_ADD_STR_ATT(hWasher, DPWS_STR_SCOPE, diversified_scope);
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_FIRMWARE_VERSION, "1.0");
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_SERIAL_NUMBER, "85374012");
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_MODEL_NUMBER, "2.0");
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_MODEL_URL, "http://www.whurlpool.com/Salsa.html");
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_PRESENTATION_URL, "WashersOverview.html");	// Any relative URI will use the default presentation server
	DPWS_SET_STR_ATT(hWasher, DPWS_STR_MANUFACTURER_URL, "http://www.whurlpool.com");

	// Device instance data that will be accessible in the service implementation
	washer.cycle = -1;	// -1 means no cycle
	washer.scheduledEnd = 0;
	dpws_client_init(&washer.dpws, NULL);
	DPWS_SET_PTR_ATT(hWasher, DPWS_PTR_USER_DATA, &washer);

	// 2.4 Create & configure hosted service.
	hWServPort = dpws_create_service_port();	// creates a service port (network endpoint)
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	DPWS_SET_STR_ATT(hWServPort, DPWS_STR_ADDRESS, "MyWashingService");	// sets the context path for the service physical address.
	washer.hEventSource = dpws_create_hosted_service(hWasher, hServClass);	// creates the service & store it in the device "user data"
	dpws_release_handle(hServClass);	// Now used by the service and not needed anymore
	dpws_bind_service(washer.hEventSource, hWServPort);	// associates it with a "network endpoint"
	dpws_release_handle(hWServPort);	// Not needed anymore

	// 2.5 Enable device.
	printf("<-- Enabling device...\n");
	if ((status = dpws_enable_device(hWasher)))	// Enables device (especially schedule hello messages).
	{
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	/* 3. Initialize server. */
	if ((status = dpws_server_init(&master_dpws, NULL)))	// initializes the dpws structure for server-side operations.
	{
		fprintf(stderr, "Could not initialize DPWS server (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// Print devices and services information
	print_device_info(hWasher);	// Print device and services information.
	dpws_release_handle(hWasher);	// Enabling makes it now hold by the registry

	/* 4. Start server loop. */
	// Creates a new thread to serve incoming messages. In the current sample,
	// a thread is created for each received message.
	if (bootServer(&master_dpws)) {
		fprintf(stderr, "Could not boot server...\n");
		dpws_shutdown();
		exit(1);
	}

	/* 5. Stop the server */

	printf("\n\nDPWS server ready. Type 'q' or 'Q' to stop it smoothly.\n\n");
	do {
		c = getchar();
	}
	while (c !='q' && c != 'Q');

	stopServer();	// 1. Calls the dpws_stop_server function that closes all
	// open listening sockets, schedules the 'Eventing end' notifications
	// and Bye WS-Discovery messages for all the active devices and blocks
	// until all server processing is finished and the main loop blocking
	// on dpws_accept() can exit.
	// 2. Uses platform-specific thread APIs to wait for the main server
	// loop thread dies.
}

/*
 *                   Service operations
 */

/* Starts a washing cycle. */
int __wsh__LaunchCycle(struct dpws* dpws, struct _wsh__LaunchCycle *wsh__LaunchCycle)
{
	struct washerData * pState;
	struct _wsh__CycleEnd cycleEnd;
	int32_t cycleDuration = 0;
	int nbEndpt = 1, status;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (struct washerData *)dpws_get_device_user_data(dpws);

	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	printf("--> Requesting cycle launch\n");

	/* Fault sending in case of incorrect parameters */
	if (pState->cycle != -1) {
		printf("<-- Error : Cycle is already running!\n");
		return dpws_fault(dpws, RECEIVER_FAULT_VALUE, "Cycle running", NULL, NULL);	// triggers a fault
	}
	switch (wsh__LaunchCycle->Name)	// cycle type switch
	{
		case wsh__Cycle__Gentle:
			cycleDuration = 10;
			printf("GENTLE (duration 10s)\n");
			break;
		case wsh__Cycle__Regular:
			cycleDuration = 20;
			printf("REGULAR (duration 20s)\n");
			break;
		case wsh__Cycle__Heavy:
			cycleDuration = 30;
			printf("HEAVY (duration 30s)\n");
			break;
	}
	pState->scheduledEnd = portableTime() + cycleDuration;	// set the time of the cycle end.
	pState->cycle = wsh__LaunchCycle->Name;	// keep trace of the type of cycle running

	printf("Starting cycle...\n");
	portableSleep(cycleDuration);	// sleep for the cycle duration
	pState->cycle = -1;	// cycle ended

	/* Cycle end WS-Eventing notification */
	printf("<-- Notifying end of cycle\n") ;
	cycleEnd.CycleName = wsh__LaunchCycle->Name;
	if ((status = dpws_notify___wsh__CycleEnded(&pState->dpws, pState->hEventSource, &cycleEnd)))	// send an event to subscribers using a specific dpws structure
		fprintf(stderr, "Could not send 'cycle end' event (err %d)\n", status);
	dpws_end(&pState->dpws);	// free memory allocated for event message

	return DPWS_OK;
}

/* Retrieves the current washing machine status. */
int __wsh__GetCycleStatus(struct dpws* dpws, struct _wsh__CycleStatus *wsh__CycleStatus)
{
	struct washerData * pState;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (struct washerData *)dpws_get_device_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	printf("--> Requesting cycle status\n");
	if (pState->cycle == -1) {
		wsh__CycleStatus->TimeLeft = NULL;
		printf("<-- No cycle running\n");
	} else {
		/*
		NOTE about result allocation:
			Even if the top level structure is allocated by the skeleton,
			contents are under service implementor's responsibility.
		*/
		char * duration = dpws_malloc(dpws, DURATION_MAX_SIZE + 1);	// Constant defined in the DPWS API
		wsh__CycleStatus->TimeLeft = dpws_malloc(dpws, sizeof(struct struct_1));	// message transient allocation
		wsh__CycleStatus->TimeLeft->__item = dcxml_duration2xmlduration(duration, pState->scheduledEnd - portableTime());
		wsh__CycleStatus->TimeLeft->CycleName = pState->cycle;
		switch (wsh__CycleStatus->TimeLeft->CycleName) {
		case wsh__Cycle__Gentle:
			printf("<-- Cycle name GENTLE , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		case wsh__Cycle__Regular:
			printf("<-- Cycle name REGULAR , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		case wsh__Cycle__Heavy:
			printf("<-- Cycle name HEAVY , time left is %s\n", wsh__CycleStatus->TimeLeft->__item);
			break;
		}
	}
	return DPWS_OK;
}
