/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.8 $
|                     $Date: 2008/02/01 18:03:04 $
\*============================================================================*/
/*******************************************************************************
*                                RTOS features                                 *
*******************************************************************************/
#ifndef AL_RTOS_H_
#define AL_RTOS_H_

/** \file
    Abstract Layer (AL): RTOS basic features.
    <em>Note</em>: This API does not intend to provide extensive services for a
    virtual RTOS, but aims at providing a simple set of features common to many 
    implementations through a set of functions defined hereafter. Parameters
    semantics may especially be different from one RTOS to another making it
    more a programming facility than a full-fledged abstraction layer.
*/

#include "al_common.h"

#if defined AL_RTOS_FREE_RTOS
#include "al_rtos_free_rtos.h" 
#elif defined AL_RTOS_POSIX
#include "alRTOS_Posix.h" 
#elif defined AL_RTOS_WIN32 || defined AL_RTOS_WINCE
#include "alRTOS_Win32.h" 
#elif defined AL_RTOS_ANDERSEN
#include "alRTOS_Andersen.h"
#elif defined AL_RTOS_EMBOS
#include "alRTOS_Embos.h"
#elif defined AL_RTOS_VXWORKS
#include "alRTOS_VxWorks.h"
#else
#error "No OS defined ..."
#endif


/*----------------------------------------------------------------------------*\
 *                            RTOS error management                           *
\*----------------------------------------------------------------------------*/

/** \name RTOS error management.
 */
/*@{*/

/**
 * Returns the last RTOS error status.
 * Error codes returned are native (no translation), so they can only be used
 * for informative purposes. Note that separating error code retrieval from the
 * API responsible for the error is not thread-safe which may not be a problem
 * in a RTOS context (depending on the preemption scheme).
 * @return A native RTOS error code.
 */
int al_rtos_error (void);

/*@}*/

/*----------------------------------------------------------------------------*\
 *                                Task management                             *
\*----------------------------------------------------------------------------*/

/** \name Task management.
 */
/*@{*/

/**
 * Creates a new task.
 * Tasks (i.e. threads for OS with processes) are created by executing the \a
 * task routine with the priority \a prio. Note that the priority semantics
 * (range, order) are not defined by this API.
 * @param th A pointer to the task handle that is allocated by the caller.
 * @param name A friendly name for the task. Should be static (not copied by
 * the callee). May be ignored by the implementation.
 * @task task The task routine.
 * @param data A pointer to user data passed as an argument to the task routine.
 * @param prio The task priority.<b>Nor the semantics of this priority, nor the
 * multitasking scheme is defined by this API.</b>
 * @param stack_ptr A pointer to a buffer used by the stack and allocated by the caller.
 * @param stack_size the size of the stack buffer passed in \a stack_ptr
 * @return AL_SUCCESS or any AL general error code.
 */
int al_task_create(	al_task_t 	*th, 
					char 		*name, 
					void 	   (*task)( void *), 
					void 		*data, 
					int 		 prio,
                    void 		*stack_ptr, 
                    int 		 stack_size);

/**
 * Deletes a task.
 * This feature is optional and may not be implemented for a given RTOS. The
 * caller is then responsible to free the task handle and the stack buffer if
 * it was not statically allocated.
 * @param th A pointer to the task handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_task_delete(al_task_t *th);

/**
 * Terminates the current task.
 * This feature is optional and may not be implemented for a given RTOS.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_task_exit(void);

/**
 * Return the current process ID.
 * RTOS that do not implement processes but only tasks (threads) should return
 * any value.
 * @return The process ID.
 */
uint32_t al_get_pid(void);

/*@}*/

/*----------------------------------------------------------------------------*\
 *                                Time features                               *
\*----------------------------------------------------------------------------*/

/** \name Time features.
 */
/*@{*/

/**
 * Multi-purpose fine-grained time structure.
 */
typedef struct {
	uint32_t sec;	/**< Number of seconds. */
	uint32_t nsec;	/**< Number of nanoseconds. */
} al_timestruct_t;

/**
 * Gregorian calendar datetime structure.
 */
typedef struct {
	short year;	/**< The year in the gregorian calendar. */
	unsigned char month;	/**< The month index: 1=January, 2=February, ..., 12=December. */
	unsigned char day;	/**< The day index: 0=Sunday, 1=Monday, ..., 6=Thursday. */
	unsigned char hour;	/**< The hour: from 0 to 23. */
	unsigned char min;	/**< The minute: from 0 to 59. */
	unsigned char sec;	/**< The second: from 0 to 59. */
	unsigned short msec;	/**< The millisecond: from 0 to 999. */
} al_datetime_t;

/**
 * Gets the time elapsed since boot or any absolute time with any origin.
 * @param[out] t A time structure for receiving the result.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_get_clock(al_timestruct_t * t);

/**
 * Makes the current task to freeze for a given time.
 * The implementation may have to divide the supplied delay if it is too
 * fine-grained for the available timers.
 * @param dly Sleep duration.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_sleep(al_timestruct_t * dly);


/** Absolute time is not always available on embedded systems.
 */
#ifdef AL_HAVE_DATETIME

/**
 * Gets the UTC datetime in a calendar form.
 * @param[out] dt Will contain the returned time.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_get_utc_time(al_datetime_t * dt);

/**
 * Returns the delay since 00:00:00 GMT,January 1,1970.
 * @param[out] t Will contain the returned time.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_get_posix_time(al_timestruct_t * t);	// time since 00:00:00 GMT,January 1,1970.

#endif

/*@}*/

/*----------------------------------------------------------------------------*\
 *                             Mutexes management                             *
\*----------------------------------------------------------------------------*/

/** \name Concurrency management features.
 */
/*@{*/

/**
 * Creates and initialize a mutex.
 * Must be called before any other call to a mutex API. The created mutex is
 * expected not to be reentrant and local (not for IPC).
 * @param mutex A pointer to the mutex handle that is allocated by the caller.
 * @param name A friendly name for the mutex. Should be static (not copied by
 * the callee). May be ignored by the implementation.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_mutex_init	(al_mutex_t *mutex, const char* name);

/**
 * Deletes a mutex.
 * This feature is optional and may not be implemented for a given RTOS.
 * @param mutex A pointer to the mutex handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_mutex_delete	(al_mutex_t *mutex);

/**
 * Locks a mutex.
 * This function locks a mutex, previously initialised by al_mutex_init(). If
 * the mutex is already locked the calling task is blocked until the mutex is
 * unlocked.
 * @param mutex A pointer to the mutex handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_mutex_lock	(al_mutex_t *mutex);

/**
 * Unlocks a mutex.
 * This function unlocks a previously locked mutex. If several tasks are
 * blocked on the mutex the highest priority task is unblocked.
 * @param mutex A pointer to the mutex handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_mutex_unlock	(al_mutex_t *mutex);

/*@}*/

/*----------------------------------------------------------------------------*\
 *                            Semaphores management                           *
\*----------------------------------------------------------------------------*/

/** \name Semaphores features.
 */
/*@{*/

extern const al_timestruct_t 		al_dly_zero;

/**
 * Creates and initialize a counting semaphore.
 * Must be called before any other call to a counting semaphore API.
 * @param sem A pointer to the semaphore handle that is allocated by the caller.
 * @param max value of the semaphore count.
 * @param value Initial value of the semaphore count.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_sem_init			(al_sem_t *sem,int max_count, int value);

/**
 * Deletes a semaphore.
 * This feature is optional and may not be implemented for a given RTOS.
 * @param sem A pointer to the semaphore handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_sem_delete		(al_sem_t *sem);

/**
 * Increments the count value of a semaphore.
 * This function increments the count value of a specified semaphore. If
 * several tasks are blocked on the semaphore, the task with the highest
 * priority is unblocked.
 * @param sem A pointer to the semaphore handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_sem_post			(al_sem_t *sem);

/**
 * Decrements the value of a semaphore.
 * This function decrements the value of the semaphore or, if the count value
 * of the semaphore is 0, blocks the calling task for a given duration of
 * indefinitely.
 * @param sem A pointer to the semaphore handle.
 * @param dly The delay to wait for or NULL if infinite.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_sem_wait			(al_sem_t *sem, const al_timestruct_t * dly);

/**
 * Clears the count value of a semaphore.
 * This function clears the count value of the specified semaphore. The
 * semaphore must have been initialised by trOsSemInit() before this function
 * is called.
 * @param sem A pointer to the semaphore handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_sem_clear		(al_sem_t *sem);

/*@}*/

/*----------------------------------------------------------------------------*\
 *                            Queue management                                *
\*----------------------------------------------------------------------------*/

/** \name Message queues features.
 */
/*@{*/

/**
 * Creates and initialize a message queue.
 * Must be called before any other call to a message queue API.
 * @param queue A pointer to the queue handle, allocated by the caller.
 * @param name A friendly name for the mutex. Should be static (not copied by
 * the callee). May be ignored by the implementation.
 * @param max_msg_length Size of each message in the queue, in bytes
 * @param max_msgs Maximum number of messages that can be queued
 * @param work_buffer Queue workspace buffer for the queue, allocated by 
 * the caller.The size of the buffer must be at 
 * least max_msg_length * max_msgs bytes
 * @param work_buffer_size Size (in bytes) of the workspace buffer
 * @return AL_SUCCESS or any AL general error code.
 */
int al_queue_create	(al_queue_t 	*queue, 
					 char 			*name, 
					 unsigned long   max_msg_length, 
					 unsigned long   max_msgs,
        			 void 			*work_buffer,
        			 unsigned long   work_buffer_size); 

        			 		 
/**
 * Deletes a message queue.
 * @param queue A pointer to the queue handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_queue_delete	(al_queue_t *queue);

/**
 * Receive a message from a message queue.
 * @param queue A pointer to the queue handle.
 * @param buffer A pointer to a buffer to store the received message. 
 * Note that the buffer must be allocated by the caller, 
 * and have a size of max_msg_length (see al_queue_create)
 * @param dly The maximum delay to wait for a message or NULL if infinite. 
 * @return AL_SUCCESS on message reception, AL_ETIMEDOUT for a timeout, 
 * or any AL general error code.
 */
int al_queue_receive(al_queue_t *queue, void *buffer, const al_timestruct_t * dly);

/**
 * Send a message into a message queue.
 * @param queue A pointer to the queue handle.
 * @param buffer A pointer to the message. Note that the buffer must 
 * have a size of max_msg_length (see al_queue_create)
 * @param dly The maximum delay to wait trying to send the message or AL_NO_WAIT if infinite. 
 * @return AL_SUCCESS on message reception, AL_ETIMEDOUT for a timeout, 
 * or any AL general error code.
 */
int al_queue_send	(al_queue_t 	 		*queue, 
					 void 			 	 	*buffer, 
					 const al_timestruct_t 	*dly);


/*@}*/

#endif /*AL_RTOS_H_*/
