/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.4 $
|                     $Date: 2008/02/04 13:45:15 $
\*============================================================================*/
/*******************************************************************************
*                            Persistence features                              *
*******************************************************************************/
#ifndef AL_STOR_H_
#define AL_STOR_H_

/** \file
    Abstract Layer (AL): Persistence API.
    This API is designed for platforms that do not have filesystems to be able
    to provide persistence (with streaming or not).
*/

#include "al_common.h"
#if defined AL_RTOS_ANDERSEN
#include "alSTOR_Andersen.h"
#else
#include "alSTOR_Stdio.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*----------------------------------------------------------------------------*\
 *                             Stream persistence                             *
\*----------------------------------------------------------------------------*/

/** \name Streaming API.
 * 	The <tt>al_stream_t</tt> type is to be defined by the implementation.
 */
/*@{*/

/* Define the position seek */
#define AL_SEEK_SET    0	/* Beginning of file */
#define AL_SEEK_CUR    1	/* Current position of the file pointer */
#define AL_SEEK_END    2	/* End of file */

/**
 * The stream is open for reading.
 */
#define AL_STREAM_READ				1

/**
 * The stream is open for writing starting at the beginning of the stream.
 */
#define AL_STREAM_WRITE_OVERWRITE	2

/**
 * The stream is open for writing starting at the end of the stream.
 */
#define AL_STREAM_WRITE_APPEND		3

/**
 * Open a stream.
 * If the id is not known by the implementation, a new storage unit is
 * created.
 * @param stream A pointer to the stream handle that is allocated by the caller.
 * @param id Stream implementation-dependent unique id.
 * @param flag Select the mode for opening the stream. One of \a AL_STREAM_READ,
 * \a AL_STREAM_WRITE_OVERWRITE or \a AL_STREAM_WRITE_APPEND should be used.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_init(al_stream_t * stream, const char * id, int flag);

/**
 * Read data from an opened stream.
 * @param in_stream A stream handle opened for reading.
 * @param[out] buf A buffer for receiving read data.
 * @param[in,out] size Initially contains the size of \a buf or the length of
 * the data to be read. Contains the length of data read when the call returns.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_read(al_stream_t * in_stream, char * buf, unsigned long * size);

/**
 * Write data to an opened stream.
 * @param out_stream A stream handle opened for writing.
 * @param buf A buffer containing data to be sent.
 * @param[in,out] size Initially contains the size of \a buf or the length of
 * the data to be sent. Contains the length of data really sent when the call
 * returns.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_write(al_stream_t * out_stream, const char * buf, unsigned long * size);

/**
 * Close an opened stream.
 * Resources used by the stream may be released.
 * @param stream An opened stream handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_close(al_stream_t * stream);

/**
 * Return an implementation error code for the streaming API.
 * @param stream A opened stream handle.
 * @return An implementation-specific error code.
 */
int al_stream_error(al_stream_t * stream);

/**
 * Set the position indicator associated with the stream to a new position defined by adding offset to a reference position specified by origin.
 * @param stream A opened stream handle.
 * @param offset Number of bytes to offset from origin.
 * @param origin Position from where offset is added. It is specified by one of the following constants (SEEK_SET, SEEEK_CUR, SEEK_END).
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_seek(al_stream_t * stream, unsigned long offset, int origin);

/**
 * Deletes the file whose name is specified in filename.
 * @param filename C string containing the name of the file to be deleted.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_remove(char * filename);

/**
 * Return the current value of the position indicator of the stream.
 * @param stream A opened stream handle.
 * @return The current value of position indicator is returned.
 */
unsigned long al_stream_tell(al_stream_t * stream);

/**
 * Set the position indicator associated with stream to the beginning of the file.
 * @param stream A opened stream handle.
 * @return none.
 */
void al_stream_rewind(al_stream_t * stream);

/**
 * Return the size in fileSize of the file whose name is specified in fullPathFileName.
 * @param fullPathFileName C string containing the complete path and the name of the file to be calculated its size.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_stream_file_length(char* fullPathFileName, unsigned long *fileSize);


/*@}*/

/*----------------------------------------------------------------------------*\
 *                             Block persistence                              *
\*----------------------------------------------------------------------------*/

/**
 * The block is open for reading.
 */
#define AL_BLOCK_READ_ONLY			1

/**
 * The block is open for writing.
 */
#define AL_BLOCK_WRITE				2

/**
 * Open a data block for further operations.
 * If the id is not known by the implementation, a new storage unit is
 * created.
 * @param block A pointer to the block handle that is allocated by the caller.
 * @param id Block implementation-dependent unique id.
 * @param flag Select the mode for opening the block. One of \a AL_BLOCK_READ_ONLY,
 * or \a AL_BLOCK_WRITE should be used.
 * @param size The size of the block.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_block_init(al_block_t * block, const char * id, int flag, unsigned long size);

/**
 * Read data from an opened block.
 * @param block A block handle opened for reading.
 * @param offset The block position to start reading from.
 * @param size Contains the number of bytes to read.
 * @param[out] buf A buffer for receiving read data.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_block_read(al_block_t * block, unsigned long offset, unsigned long size, char * buf);

/**
 * Write data to an opened block.
 * @param block A block handle opened for writing.
 * @param data A buffer containing data to write.
 * @param size Contains the number of bytes to write.
 * @param offset The block position to start writing from.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_block_write(al_block_t * block, const char * data, unsigned long size, unsigned long offset);

/**
 * Close an opened block.
 * Resources used by the block may be released.
 * @param block An opened block handle.
 * @return AL_SUCCESS or any AL general error code.
 */
int al_block_close(al_block_t * block);

/**
 * Return an implementation error code for the block API.
 * @param stream A opened block handle.
 * @return An implementation-specific error code.
 */
int al_block_error(al_block_t * block);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /*AL_STOR_H_*/
