/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.5 $
|                     $Date: 2008/02/05 18:06:16 $
\*============================================================================*/
/*******************************************************************************
*                   Network configuration access for Linux                     *
*******************************************************************************/
#include "al_net.h"
#include "al_mem.h"
#include <stdlib.h>
#include <memory.h>
#include <unistd.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <netinet/in.h>

// WARNING assertion is made that AL_SOCKADDR_FULL_OPAQUE is set

static al_netif_info_t * nifs = NULL;
static int nb_nifs = 0;

#ifdef AL_HAVE_IPV6
#include <ifaddrs.h>

#define IS_VALID_NETIF(nif) ((nif)->ifa_addr->sa_family == AF_INET || (nif)->ifa_addr->sa_family == AF_INET6)

int al_get_netif_info(al_netif_info_t * netifs[], int * nb_netifs)
{
	struct ifaddrs * ifaddrs = NULL, * w_addr, * w_addr2;
    int ret = AL_SUCCESS, w_nbif = 0, s = -1;

	if (nifs)
		goto exit;

	/* Retrieve the IP netif info */
	if (getifaddrs(&ifaddrs))
		return AL_ERROR;

    /* Count interfaces (count only IP and not doubles) & allocate result */
    for (w_addr = ifaddrs; w_addr; w_addr = w_addr->ifa_next) {
    	if (IS_VALID_NETIF(w_addr)) {
		    for (
			    w_addr2 = ifaddrs;
			    w_addr2 != w_addr && (!IS_VALID_NETIF(w_addr2) || strcmp(w_addr2->ifa_name, w_addr->ifa_name));
			    w_addr2 = w_addr2->ifa_next
		    );
		    if (w_addr2 == w_addr)	// not found
		    	nb_nifs++;
    	}
    }

    if (nb_nifs == 0)
    	goto exit;

    nifs = AL_MALLOC(nb_nifs * sizeof(al_netif_info_t));

    /* Fill results */
    for (w_addr = ifaddrs; w_addr; w_addr = w_addr->ifa_next) {
    	if (IS_VALID_NETIF(w_addr)) {
			int i;
    		struct ifreq buffer;

			// Check this is not an already collected one
		    for (i = 0; i < w_nbif && strcmp(w_addr->ifa_name, nifs[i].name); i++);
		    if (i == w_nbif) // new
		    {
		    	w_nbif++;

	    		// Count the number of entries with the same name
	    		nifs[i].addr_nb = 0;
	 		    for (w_addr2 = w_addr; w_addr2; w_addr2 = w_addr2->ifa_next) {
	 		    	 if (IS_VALID_NETIF(w_addr2) && !strcmp (w_addr2->ifa_name, w_addr->ifa_name))
	 		    	 	nifs[i].addr_nb++;
	 		    }
	 		    // Fill Interface info
				nifs[i].name = strdup(w_addr->ifa_name);

	 		    // allocate addresses space
	    		nifs[i].addrs = AL_MALLOC(sizeof(al_sockaddr_t) * nifs[i].addr_nb);
	    		memset(nifs[i].addrs, 0, sizeof(al_sockaddr_t) * nifs[i].addr_nb);

	    		nifs[i].addr_nb = 0;	// used as the position counter
	    	}
	    	// else: already retrieved

	    	memcpy(&nifs[i].addrs[nifs[i].addr_nb++], w_addr->ifa_addr, sizeof(al_sockaddr_t)); // opaque is assumed

	    	// Retrieve MAC address
		    strcpy(buffer.ifr_name,  w_addr->ifa_name);
		    s = socket(PF_INET, SOCK_DGRAM, 0);
		    if (s < 0 || ioctl(s, SIOCGIFHWADDR, &buffer) < 0) {
		        ret = AL_ERROR;
		        goto exit;
		    }
			memcpy(nifs[i].mac_address, buffer.ifr_hwaddr.sa_data, 6);
    		nifs[i].interface_selector = if_nametoindex(w_addr->ifa_name);
    	}
    }

exit:
	if (ifaddrs)
		freeifaddrs(ifaddrs);
	if (s >=0)
		close(s);
	*netifs = nifs;
	*nb_netifs = nb_nifs;
	return ret;
}

#else
// Retrieves only IPv4 addresses even with PF_INET6 socket
int al_get_netif_info(al_netif_info_t * netifs[], int * nb_netifs)
{
	al_netif_info_t * netif_info;
    int s, ret = AL_SUCCESS, i;
    struct ifconf ifcfg = {0, NULL};

	if (nifs)
		goto exit;

    s = socket(PF_INET, SOCK_DGRAM, 0);
    if (s < 0)
        return AL_ERROR;

    /* Retrieve the IP netif info */
    while (ifcfg.ifc_len == (sizeof(struct ifreq) * nb_nifs))
    {
    	nb_nifs += 2;
		ifcfg.ifc_len = sizeof(struct ifreq) * nb_nifs;
		AL_FREE(ifcfg.ifc_buf);	// Should tolerate NULL
		ifcfg.ifc_buf = AL_MALLOC(ifcfg.ifc_len);
		if (ioctl(s, SIOCGIFCONF, &ifcfg) < 0)
			return AL_ERROR;
    }
    nb_nifs = ifcfg.ifc_len / sizeof(struct ifreq);

    /* Allocate result */
    nifs = AL_MALLOC(nb_nifs * sizeof(al_netif_info_t));

    /* Fill results */
    for (i = 0; i < nb_nifs; i++)
    {
    	memcpy(nifs[i].mac_address, ifcfg.ifc_req[i].ifr_hwaddr.sa_data, 6);
    	nifs[i].name = strdup(ifcfg.ifc_req[i].ifr_name);
   		nifs[i].interface_selector = 0;
	    nifs[i].addr_nb = 1;
	    nifs[i].addrs = AL_MALLOC(sizeof(al_sockaddr_t));
	    memcpy(nifs[i].addrs, &ifcfg.ifc_req[i].ifr_addr, sizeof(al_sockaddr_t)); // opaque is assumed
    }

exit:
	close(s);
	AL_FREE(ifcfg.ifc_buf);
	*netifs = nifs;
	*nb_netifs = nb_nifs;
	return ret;
}
#endif

void al_release_netif_info()
{
	int i;
	for(i = 0; i < nb_nifs; i++) {
		AL_FREE(nifs[i].name);
		AL_FREE(nifs[i].addrs);
	}
	AL_FREE(nifs);
	nifs = NULL;
	nb_nifs = 0;
}
