/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.4 $
|                     $Date: 2008/02/05 18:06:16 $
\*============================================================================*/
/*******************************************************************************
*             AL RTOS features implementation for POSIX platforms              *
*******************************************************************************/
#include "al_rtos.h"
#include <time.h>
#include <sys/time.h>
#include <sys/times.h>
#include <unistd.h>
#include <errno.h>

/*----------------------------------------------------------------------------*\
 *                              RTOS error management                         *
\*----------------------------------------------------------------------------*/

extern int errno;

int al_rtos_error(void) { return errno;}

/*----------------------------------------------------------------------------*\
 *                              Thread management                             *
\*----------------------------------------------------------------------------*/

int al_task_create(al_task_t 	*th, 
					 char 		*name, 
					 void 	   (*task)( void *), 
					 void 		*data, 
					 int 		 prio,
                     void 		*stack_ptr, 
                     int 		 stack_size)
{
	// NOTE: copied from TR documentation
	pthread_attr_t attr;
	struct sched_param param;
	pthread_attr_init(&attr);
	pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
	param.sched_priority = prio;	// TODO: What semantics ?
	pthread_attr_setschedparam(&attr, &param);
	pthread_attr_setschedpolicy(&attr, SCHED_OTHER);
	if (stack_ptr && stack_size > 0)
	{
#ifdef _POSIX_THREAD_ATTR_STACKADDR

		pthread_attr_setstack(&attr, stack_ptr, stack_size);
#else
		pthread_attr_setstacksize(&attr, stack_size);
#endif
	}
	return pthread_create(th, &attr, (void *(*)(void *))task, data);	// dirty cast: hope the return value will be ignored
}

int al_task_delete(al_task_t *th)
{
	if (pthread_cancel(*th) == 0)
		return AL_SUCCESS;
	return AL_EINVAL;
}

int al_task_exit(void)
{
	pthread_exit(NULL);
	return AL_SUCCESS;
}

uint32_t al_get_pid(void) { return getpid();}

/*----------------------------------------------------------------------------*\
 *                              Clock management                              *
\*----------------------------------------------------------------------------*/
static long NB_TICKS_PER_S = -1;

int al_get_clock(al_timestruct_t * t)
{
	clock_t ck = 0;
	struct tms tms;
	if (NB_TICKS_PER_S < 0)
		NB_TICKS_PER_S = sysconf(_SC_CLK_TCK);
	ck = times(&tms);
	t->sec = ck / NB_TICKS_PER_S;
	t->nsec = (ck % NB_TICKS_PER_S) * (1000000000 / NB_TICKS_PER_S);
	return AL_SUCCESS;
}

int al_sleep(al_timestruct_t * dly)
{
	if (dly->sec > 4292)
		sleep(dly->sec);
	else
		usleep(dly->sec * 1000000 + dly->nsec / 1000);
	return AL_SUCCESS;
}

int al_get_utc_time(al_datetime_t * dt)
{
	time_t t;
	struct tm *tm;
	time(&t);
	tm = gmtime(&t);
    dt->year = 1900 + tm->tm_year;
	dt->month = tm->tm_mon;
	dt->day = tm->tm_mday;
	dt->hour = tm->tm_hour;
	dt->min = tm->tm_min;
	dt->sec = tm->tm_sec;
	dt->msec = 0;
	return AL_SUCCESS;
}

int al_get_posix_time(al_timestruct_t * time)
{
    struct timeval tv;
    gettimeofday(&tv, (struct timezone *)0);
    time->sec = tv.tv_sec;
    time->nsec = tv.tv_usec * 1000;
	return AL_SUCCESS;
}


/*----------------------------------------------------------------------------*\
 *                             Mutexes management                             *
\*----------------------------------------------------------------------------*/

int al_mutex_init(al_mutex_t *mutex, const char* name)
{
	pthread_mutex_t mx = PTHREAD_MUTEX_INITIALIZER;	// a struct static initializer on Linux
	*mutex = mx;	// struct copy because of linux
    return pthread_mutex_init(mutex, NULL) ? AL_ERROR : AL_SUCCESS;
}

int al_mutex_delete(al_mutex_t *mutex)
{
    return pthread_mutex_destroy(mutex) ? AL_ERROR : AL_SUCCESS;
}

int al_mutex_lock(al_mutex_t *mutex)
{
    return pthread_mutex_lock(mutex) ? AL_ERROR : AL_SUCCESS;
}

int al_mutex_unlock(al_mutex_t *mutex)
{
    return pthread_mutex_unlock(mutex) ? AL_ERROR : AL_SUCCESS;
}


/*----------------------------------------------------------------------------*\
 *                            Semaphores management                           *
\*----------------------------------------------------------------------------*/

int al_sem_init(al_sem_t *sem, int max_count, int value)
{
	return sem_init(sem, 0, value) ? AL_ERROR : AL_SUCCESS;	
}

int al_sem_delete(al_sem_t *sem)
{
	return sem_destroy(sem) ? AL_ERROR : AL_SUCCESS;	
}

int al_sem_post(al_sem_t *sem)
{
	return sem_post(sem) ? AL_ERROR : AL_SUCCESS;	
}

int al_sem_wait(al_sem_t *sem, const al_timestruct_t * dly)
{
 int Result, ReturnValue;

 struct timespec timeout = {0};
 al_timestruct_t currenttime = {0};

	// Compute timeout with specific cases (no wait and blocking call)
	if(dly == AL_NO_WAIT)
		// Immediate return
		Result = sem_trywait(sem);
	else if(dly == AL_WAIT_FOREVER)
		// Blocking call
		Result = sem_wait(sem);
	else
	{
		// Timed wait
		al_get_posix_time(&currenttime);
		timeout.tv_sec = currenttime.sec + dly->sec;
		timeout.tv_nsec = currenttime.nsec + dly->nsec;
		while ((Result = sem_timedwait(sem, &timeout)) == -1 && errno == EINTR)
           continue;       // Restart if interrupted by handler 
	}
	
	if(Result == 0)
	{
		ReturnValue = AL_SUCCESS;
	}
	else
	{
		switch(errno)
		{
		case ETIMEDOUT:
			/* usleep(1); Spurious wakeup for GDB bug uncomment to make debuf usable*/
			ReturnValue = AL_ETIMEDOUT;
			break;
			
		default:
			ReturnValue = AL_ERROR;
			break;
		}
	}

	return ReturnValue;
}

int al_sem_clear(al_sem_t *sem)
{
	int ret;
	
	while (1) {
		if ((ret = sem_trywait(sem)))
			break;
	}
	return ret == EAGAIN ? AL_SUCCESS : AL_ERROR;
}
