/*============================================================================*\
|                                                                              |
|                          SOA4D Abstraction Layer                             |
|                                                                              |
|               ->>  Copyright 2008 Schneider Electric SA <<-                  |
|                                                                              |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1.4 $
|                     $Date: 2008/02/11 10:17:17 $
\*============================================================================*/
/*******************************************************************************
*             AL RTOS features implementation for Win32 platforms              *
*******************************************************************************/
#include "alRTOS_Win32.h"
#include "al_rtos.h"

/*----------------------------------------------------------------------------*\
 *                              RTOS error management                         *
\*----------------------------------------------------------------------------*/

int al_rtos_error (void) { return GetLastError();}

/*----------------------------------------------------------------------------*\
 *                              Thread management                             *
\*----------------------------------------------------------------------------*/

int al_task_create(al_task_t 	*th, 
					 char 		*name, 
					 void 	   (*task)( void *), 
					 void 		*data, 
					 int 		 prio,
                     void 		*stack_ptr, 
                     int 		 stack_size)
{
	*th = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)task, data, 0, NULL);
	return *th == NULL ? AL_ERROR : AL_SUCCESS;
}

int al_task_delete(al_task_t *th)
{
	return TerminateThread(*th, 0) ? AL_SUCCESS : AL_ERROR;
}

int al_task_exit(void)
{
	ExitThread(0);
	return AL_SUCCESS;
}

uint32_t al_get_pid(void) { return GetCurrentProcessId();}

/*----------------------------------------------------------------------------*\
 *                              Clock management                              *
\*----------------------------------------------------------------------------*/

int al_get_clock(al_timestruct_t * t)
{
	DWORD ticks = GetTickCount();
	t->sec = ticks / 1000;
	t->nsec = (ticks % 1000) * 1000000;
	return AL_SUCCESS;
}

int al_sleep(al_timestruct_t * dly)
{
	Sleep(dly->sec * 1000 + dly->nsec / 1000000);
	return AL_SUCCESS;
}

int al_get_utc_time(al_datetime_t * dt)
{
    SYSTEMTIME systime;
    GetSystemTime(&systime);
    dt->year = systime.wYear;
	dt->month = (unsigned char)systime.wMonth;
	dt->day = (unsigned char)systime.wDay;
	dt->hour = (unsigned char)systime.wHour;
	dt->min = (unsigned char)systime.wMinute;
	dt->sec = (unsigned char)systime.wSecond;
	dt->msec = systime.wMilliseconds;
	return AL_SUCCESS;
}

int al_get_posix_time(al_timestruct_t * time)
{
    FILETIME        ft;
    LARGE_INTEGER   li;
    __int64         t;	// 100ns unit
    SYSTEMTIME systime;
    GetSystemTime(&systime);
    SystemTimeToFileTime(&systime,&ft);	// GetSystemTimeAsFileTime does not exit on CE
    li.LowPart  = ft.dwLowDateTime;
    li.HighPart = ft.dwHighDateTime;
    t  = li.QuadPart;           /* In 100-nanosecond intervals */
    t -= 116444736000000000LL; /* Offset to the Epoch time */
    time->sec = (long)(t / 10000000);
    time->nsec = (long)(t % 10000000) * 100;
    return AL_SUCCESS;
}

/*----------------------------------------------------------------------------*\
 *                             Mutexes management                             *
\*----------------------------------------------------------------------------*/

int al_mutex_init(al_mutex_t *mutex, const char* name)
{
    InitializeCriticalSection(mutex);
    return AL_SUCCESS;
}

int al_mutex_delete(al_mutex_t *mutex)
{
    DeleteCriticalSection(mutex);
    return AL_SUCCESS;
}

int al_mutex_lock(al_mutex_t *mutex)
{
    EnterCriticalSection(mutex);
    return AL_SUCCESS;
}

int al_mutex_unlock(al_mutex_t *mutex)
{
    LeaveCriticalSection(mutex);
    return AL_SUCCESS;
}


/*----------------------------------------------------------------------------*\
 *                            Semaphores management                           *
\*----------------------------------------------------------------------------*/

int al_sem_init(al_sem_t *sem, int max_count, int value)
{
	*sem = CreateSemaphore(NULL, value, max_count, NULL);

	if(*sem == NULL)
		return AL_ERROR;
	else
		return AL_SUCCESS;	
}

int al_sem_delete(al_sem_t *sem)
{
	CloseHandle(*sem);
	return AL_SUCCESS;	
}

int al_sem_post(al_sem_t *sem)
{
 BOOL Result;

	Result = ReleaseSemaphore(*sem, 1, NULL);

	if(Result)
		return AL_SUCCESS; 
	else
		return AL_ERROR;
}

int al_sem_wait(al_sem_t *sem, const al_timestruct_t * dly)
{
 int Result, ReturnValue;

 long timeout;

	// Compute timeout with specific cases (no wait and blocking call)
	if(dly == AL_NO_WAIT)
		// Immediate return
		timeout = 0;
	else if(dly == AL_WAIT_FOREVER)
		// Blocking call
		timeout = INFINITE;
	else
		// Timed wait
		timeout = dly->sec * 1000 + dly->nsec / 1000;


	Result = WaitForSingleObject(*sem, timeout);

	switch(Result)
	{
	case WAIT_OBJECT_0:
		ReturnValue = AL_SUCCESS;
		break;

	case WAIT_TIMEOUT:
		ReturnValue = AL_ETIMEDOUT;
		break;

	case WAIT_ABANDONED:
		ReturnValue= AL_ERROR;
		break;

	default:
		ReturnValue = AL_EINVAL;
		break;
	}

	return ReturnValue;


}

int al_sem_clear(al_sem_t *sem)
{
	return AL_SUCCESS;
}


/*----------------------------------------------------------------------------*\
 *                            Queues management			                      *
\*----------------------------------------------------------------------------*/
int al_queue_create( al_queue_t *queue, char *namePtr, unsigned long maxMsgLength, 
					unsigned long maxMsgs, void *queueStart, unsigned long queueSize)
{
 BOOL RetValue;

	RetValue = CreatePipe(&queue->ReadPipe, &queue->WritePipe, NULL, 0);
	
	if(RetValue)
	{
		queue->maxMsgLength = maxMsgLength;
		return AL_SUCCESS;
	}
	else
		return AL_ERROR;
}


/*****************************************************************************
 * trOsQueueDelete()
 *****************************************************************************
 * This function terminate and free a message queue.
 *****************************************************************************
 * Parameters:
 *  queue           Pointer to queue object.
 *
 * Return value: TR_SUCCESS if successful, otherwise a negative error code.
 */
int al_queue_delete( al_queue_t *queue)
{
	CloseHandle(queue->ReadPipe);
	CloseHandle(queue->WritePipe);

	return AL_SUCCESS;
}


/*****************************************************************************
 * al_queue_receive()
 *****************************************************************************
 * This function receive a message from a message queue.
 *****************************************************************************
 * Parameters:
 *  queue           Pointer to queue object.
 *  buffer          Pointer of buffer to receive message
 *  timeout         ticks to wait
 *
 * Return value: TR_SUCCESS if successful, otherwise a negative error code.
 */
int al_queue_receive( al_queue_t *queue, void *buffer, const al_timestruct_t * dly)
{
 BOOL Result;
 int ReturnValue = AL_SUCCESS;
 long Dummy;
 int NbBytesAvail; 

	// Only two type of timeout are allowed:
	// TR_NO_WAIT and TR_WAIT_FOREVER

	// Compute timeout with specific cases (no wait and blocking call)
	if(dly == AL_NO_WAIT)
	{
		// Immediate return
		// First, check to see if there is something to read
		PeekNamedPipe(queue->ReadPipe, NULL, 0, NULL, &NbBytesAvail, NULL);
		if(NbBytesAvail > 0)
		{
			Result = ReadFile(queue->ReadPipe, buffer, queue->maxMsgLength, &Dummy, NULL);

			if(Result)
				ReturnValue = AL_SUCCESS;
			else
				ReturnValue = AL_ERROR;
		}
		else
			ReturnValue = AL_ENOMSG;
	}
	else if(dly == AL_WAIT_FOREVER)
	{
		// Blocking call
		// Normal condition of operation (blocking call)
		Result = ReadFile(queue->ReadPipe, buffer, queue->maxMsgLength, &Dummy, NULL);

		if(Result)
			ReturnValue = AL_SUCCESS;
		else
			ReturnValue = AL_ERROR;
	}
	else
	{
		// Timed wait
		// Not implemented for windows
		ReturnValue = AL_ERROR;
	}

	return ReturnValue;
}


/*****************************************************************************
 * al_queue_send()
 *****************************************************************************
 * This function send a message to a message queue.
 *****************************************************************************
 * Parameters:
 *  queue           Pointer to queue object.
 *  buffer          Pointer of message to send
 *  timeout         ticks to wait
 *  priority        priority
 *
 * Return value: TR_SUCCESS if successful, otherwise a negative error code.
 */
int al_queue_send( al_queue_t *queue, void *buffer, const al_timestruct_t * dly)
{
 BOOL RetValue;
 long NbWritten;

	RetValue = WriteFile(queue->WritePipe, buffer, queue->maxMsgLength, &NbWritten, NULL);
	
	if(RetValue)
		return AL_SUCCESS;
	else
		return AL_ERROR;
}

