/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2135 $
|                     $Date: 2009-02-28 23:53:25 +0100 (sam, 28 fév 2009) $
\*============================================================================*/

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.

#include "wshStub.h"	// Generated stub & skeleton include file.
#include "wsh.nsmap"	// Generated namespace table include file.

#define LAUNDRY_NS "http://www.soa4d.org/DPWS/Samples/DynHome/Laundry"	// The namespace used to qualify port types (copy from wsdl file).
#define WASH_TYPE "Wash"	// The local name of the port type supported by the service (defined in the wsdl file).


/*
 *  Utilities
 */

// Variables for menu processing.
static char command[32], param1[32];
static int status, nbParams;

/* Function printing error message. */
static void print_error(char * msg, struct dpws * dpws) {
	fprintf(stderr, "\nError occured : %s.\n%s\n\n", msg, dpws_get_error_msg(dpws));
}

/* Function to get the washing machine service proxy for a given device proxy. */
static short getServiceProxy(struct dpws *dpws, short device_proxy)
{
	int nb_services = 1;	// One service required.
	short hService = -1;

	// The dpws_get_services retrieves one or several service proxies hosted by
	// a remote device using a type filtering (other APIs are available).
	// It will look for proxy available into the local cache or generate a
	// WS_Transfer/WS-MetadataExchange message if necessary.
	short * service_proxies = dpws_get_services(dpws, device_proxy, LAUNDRY_NS, WASH_TYPE, &nb_services);

	/* The program searches a single service. */
	if (nb_services != 1) {
		if (nb_services == 0)
			printf("No service endpoint found, please check the port type specifications.\n");
		else
			print_error("Could not execute dpws_get_services", dpws);
		goto exit;
	}

	// Make sure the service proxy will not be erased from the cache for instance
	// because of a bye message.
	if (dpws_pin_proxy(service_proxies[0]) != DPWS_OK) {
		print_error("Could not execute dpws_pin_proxy", dpws);
		goto exit;
	}
	hService = service_proxies[0];

exit:
	// Clears structures and free dynamically allocated memory used for request processing.
	if (dpws_end(dpws) != DPWS_OK)
		print_error("Could not execute dpws_end", dpws);

	return hService;
}

void washerClient(struct dpws *dpws, short device_proxy, short event_endpoint)
{
	short service_proxy, event_sink;
	struct wsa_endpoint_ref * subscManager = NULL, * servEndPt;	// event subscription variables
	char *duration = "PT1H";	// 1 hour in 'XML'

	/* Washer client program description (service invocation && event subscription) */
	// 1. Find service proxy of the service to be invoked.
	// 2. Get the default invocation endpoint reference.
	// 3. Subscribe to the end of cycle event.
	// 4. Invocation loop
	//	4.1 Get EPR of the service to be invoked.
	//	4.2 Invoke the service.
	//  4.3 Free message processing memory.
	// 5. Release service proxy.
	// 6. Unsubscribe to the end of cycle event.

	/* 1. Find service proxy of the service to be invoked. */
	service_proxy = getServiceProxy(dpws, device_proxy);
	if (service_proxy < 0)
		return;

	/* 2. Get the default invocation endpoint reference. */

	// Retrieves the service proxy default endpoint reference (EPR) used for
	// invocation.
	// Several endpoints may be available and dpws_get_default_endpoint_ref
	// returns the first one. This is why one may have to use dpws_get_endpoint_refs
	// especially if the 1st EPR is not reachable.
	servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);
	if (!servEndPt) {
		print_error("Could not find default endpoint reference", dpws);
		return;
	}

	/* 3. Subscribe to the end of cycle event. */
 	printf("<-- Subscribe to Wash events for %s\n", duration);
 	event_sink = dpws_get_default_service_port(event_endpoint);

	// Subscribes to a WS-Eventing event source.
	// notify_to and end_to are the same local service port (i.e. network
 	// endpoints). Note that no action filter is used here.
 	subscManager = dpws_event_subscribe_ex(dpws, servEndPt, event_sink, event_sink, NULL, &duration);
	if (!subscManager) {
		print_error("Could not subscribe to Wash events", dpws);
		return;
	}

	// The endpoint reference (EPR) of the subscription manager is copied for
	// later reuse (to avoid release by next dpws_end). The subscription
	// manager EPR can be considered as a subscription ticket.
	subscManager = dpws_endpoint_ref_dup(subscManager);
	if (!subscManager) {
		print_error("Could not execute dpws_endpoint_ref_dup", dpws);
		return;
	}
	printf("--> Successful subscription to Wash events for %s\n", duration);

	if (dpws_end(dpws) != DPWS_OK) {	// Good practice to call it after every call to avoid memory peak.
		print_error("Could not execute dpws_end", dpws);
		return;
	}

	/* 4. Invocation loop */

	printf("\nWasher device type commands (Q to quit this menu):\n\n");
	printf("\n Launch (GENTLE|REGULAR|HEAVY) <temp> <spin speed>\n");
	printf("\n\tLaunches a specified cycle.\n");
	printf("\tGENTLE  -> small duration.\n");
	printf("\tREGULAR -> medium duration.\n");
	printf("\tHEAVY   -> long duration.\n");
	printf("\t<temp> indicates of temperature for the washing (0-90).\n");
	printf("\t<spin speed> indicates of drying spin speed for the washing (0-1500).\n");
	printf("\n GetCycleStatus\n");
	printf("\n\tRetrieves the status of the current cycle, if any.\n");

	for (;;)
	{
		printf("\nEnter command:\n");
		nbParams = scanf("%32s", command);
		if (nbParams == 0) {
			printf("Type mismatch !\n");
			continue;
		}

		/* 4.1 Get EPR of the service to be invoked. */

		// This is done again because dpws_end has been called.
		servEndPt = dpws_get_default_endpoint_reference(dpws, service_proxy);

		/* 4.2 Invoke the service. */

		// Command parsing & switch
		if (!strcasecmp(command, "Launch"))	// Launch cycle
		{
			int temp, spin;
			nbParams = scanf("%32s %d %d", param1, &temp, &spin);
			if (nbParams != 3)
				printf("Bad parameter count !\n");
			else {
				struct _wsh__LaunchCycle cycle;
				cycle.Temperature = temp;
				cycle.SpinDryingSpeed = spin;
				if (!strcasecmp(param1, "GENTLE"))
					cycle.Name = wsh__Cycle__Gentle;
				else if (!strcasecmp(param1, "HEAVY"))
					cycle.Name = wsh__Cycle__Heavy;
				else
					cycle.Name = wsh__Cycle__Regular;
				printf("<-- Launching cycle %s %d %d\n", param1, temp, spin);
				status = dpws_send___wsh__LaunchCycle(dpws, servEndPt, &cycle);
				if (status)
					printf("Could not start cycle : %s.\n",dpws_get_error_msg(dpws));
			}
		}
		else if (!strcasecmp(command, "GetCycleStatus"))	// Get cycle information
		{
			struct _wsh__CycleStatus cycleStatus;
			printf("<-- Getting cycle status ...\n");
			status = dpws_call___wsh__GetCycleStatus(dpws, servEndPt, NULL, &cycleStatus);
			if (!status)
			{
				printf("--> Received cycle status ...\n");
				if (cycleStatus.TimeLeft)
				{
					printf("--> Cycle ");
					switch (cycleStatus.TimeLeft->CycleName) {
					case wsh__Cycle__Gentle:
						printf("GENTLE running for %s.\n", cycleStatus.TimeLeft->__item);
						break;
					case wsh__Cycle__Regular:
						printf("REGULAR running for %s.\n", cycleStatus.TimeLeft->__item);
						break;
					case wsh__Cycle__Heavy:
						printf("HEAVY running for %s.\n", cycleStatus.TimeLeft->__item);
						break;
					}
				}
				else
					printf("No cycle running\n");
			}
			else
				printf("Could not retrieve cycle status : %s.\n",dpws_get_error_msg(dpws));
		}
		else if (!strcasecmp(command, "Q"))	// Exit menu
		{
			printf("Exiting washer menu...\n");
			break;
		}
		else
			printf("Unknown command.\n");

		if (status)
			fprintf(stderr, dpws_get_error_msg(dpws));

		/* 4.3 Free message processing memory. */
		if (dpws_end(dpws) != DPWS_OK)
			print_error("Could not execute dpws_end", dpws);
	}

	/* 5. Release service proxy. */
	if (dpws_release_proxy(service_proxy) != DPWS_OK)
		print_error("Could not execute dpws_release_proxy", dpws);

	/* 6. Unsubscribe to the end of cycle event. */
	printf("<-- Unsubscribe to Wash events.\n");

	// The dpws_event_unsubscribe function cancels a running subscription to a
	// WS-Eventing event source.
	if (dpws_event_unsubscribe(dpws, subscManager) == DPWS_OK)
		printf("--> Successful unsubscription to Wash events.\n");
	else
		print_error("Could not execute dpws_event_unsubscribe (may be normal if server has been stopped first)", dpws);

	dpws_endpoint_ref_free(subscManager);	// Free subscription manager EPR.
}

/*
 *                    WS-Eventing handlers implementation
 */

/* WS-Eventing handler */
int __wsh__CycleEnded(struct dpws* dpws, struct _wsh__CycleEnd *wsh__CycleEnd)
{
	printf("--> Event 'CycleEnded' received ! Cycle: ");
	switch (wsh__CycleEnd->CycleName)
	{
	case wsh__Cycle__Gentle:
		printf("GENTLE\n");
		break;
	case wsh__Cycle__Regular:
		printf("REGULAR\n");
		break;
	case wsh__Cycle__Heavy:
		printf("HEAVY\n");
		break;
	}
	return DPWS_OK;
}

/*
IMPORTANT NOTE:
	The following handler is not a WS-Eventing handler but a handler for asynchronous
	responses to request-reply invocations. It must however  be implemented even if
	not used since referenced by the same dispatch function.
 */
int __wsh__GetCycleStatus_handler(struct dpws* dpws, struct _wsh__CycleStatus *a)
{
	return DPWS_OK;
}
