/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|             ->> Copyright 2004-2009 Schneider Electric SA <<-                |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2125 $
|                     $Date: 2009-02-26 23:52:06 +0100 (jeu, 26 fév 2009) $
\*============================================================================*/
/******************************************************************************\
 *                        Static Service Class Loader                         *
\******************************************************************************/

#include "sscl_dim.h"

#include "litStub.h" // Generated stub & skeleton include file for light device implementation.
#include "lit.nsmap" // Generated namespace table include file for light device implementation.

#include "dc/dc_Epx.h"	// XML streaming processing API
#include "serverDPWS.h"	// OS-dependent code (especially time functions).

/* Service class loader callback declarations. */
static int dim_new_service(short href_endpoint);
static int dim_free_service(short href_endpoint);
static int dim_serialize_impl(short href_sclass, void * szr_ctx);

/*
 *	Definitions
 */

struct dimmerData {	// Device instance runtime data for dimmer
	enum lit__PowerState state; // light state: must be first to be usable by 'sscl_lit'
	short level;	//  instantaneous current dimming level
	short target;	// user set dimming level
	struct dpws dpws;	// runtime structure for event notification & client invocation
	short hEventSource; // for event notification
};

/*
 *	Runtime data
 */

// Identifier for this service implementation (service class loader) used in the
// configuration file.
struct qname dim_iqn = { DIM_NS, DIM_ELT_SSCL_IMPL };

/* Callback associated to the previously defined qualified name.
   It will be called when the washer service class loader is loaded in order to
   initialize the various functions called when:
	+ a service of this class is created,
	+ a service of this class is deleted,
	+ the service class is to be serialized in XML for instance for configuration
	  backup,
	+ a request for a service is to be processed.
	See load_cbk reference for more information.
*/
int dim_load_cbk(short href_sclass, void * psr_ctx, dispatch_cbk * p_dispatch_cbk, struct scl_callbacks * p_cbks)
{
	/* Achieves implementation tag parsing with EPX (here very simple). Note that
	 * the current event is the 'implementation' start tag
	 */
	if (epx_next(psr_ctx) != EPX_EVT_END_ELEMENT || QNAME_NOT_EQUALS_WILDCARD(epx_get_ns_uri(psr_ctx), epx_get_lname(psr_ctx), DIM_NS, DIM_ELT_SSCL_IMPL))
		return DPWS_ERR_INCORRECT_IMPL_TAG;

	*p_dispatch_cbk = lit_serve_request;	// generated skeleton dispatch function.
	p_cbks->new_service = dim_new_service;	// service instance creation callback.
	p_cbks->free_service = dim_free_service;	// service instance deletion callback.
	p_cbks->serialize_impl = dim_serialize_impl;	// serialization callback
	return DPWS_OK;
}

/* Allows to initialize dimmer service runtime data on service creation. */
int dim_new_service(short href_endpoint)
{
	struct dimmerData * dimmer = malloc(sizeof(struct dimmerData));	// service data allocated on the heap.
	/* Initialize dimmer state. */
	dimmer->state = lit__PowerState__OFF;
	dimmer->level = 100;
	dimmer->target = 100;
	dpws_client_init(&dimmer->dpws, NULL);
	dimmer->hEventSource = href_endpoint;
	return dpws_set_ptr_att(href_endpoint, DPWS_PTR_USER_DATA, dimmer);
}

/* On service deletion the user data is freed using this callback. */
int dim_free_service(short href_endpoint)
{
	free(dpws_get_ptr_att(href_endpoint, DPWS_PTR_USER_DATA));
	return DPWS_OK;
}

/* Service class implementation tag serialization callback. */
int dim_serialize_impl(short href_sclass, void * szr_ctx)
{
	epx_start_element(szr_ctx, DIM_NS, DIM_ELT_SSCL_IMPL);	// Here the whole tag must be generated using EPX
	epx_define_prefix(szr_ctx, DIM_PREFIX, DIM_NS);
	epx_end_element(szr_ctx, DIM_NS, DIM_ELT_SSCL_IMPL);
	return DPWS_OK;
}

/*
 *                   Service operations
 */

/*
IMPORTANT NOTE:
	The 'switch power' port type cannot be implemented twice so implementation
	is shared with 'sscl_lit'. This is why user data has to be polymorphic.
*/

/* Sets the level for dimmer light. */
int __lit__SetLevel(struct dpws* dpws, struct _lit__LightLevelTarget *lit__LightLevelTarget)
{
	short inc;
	struct dimmerData * pState;
	int status = DPWS_OK;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_endpoint_user_data.
	pState = (struct dimmerData *)dpws_get_endpoint_user_data(dpws);

	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. This sample is especially concerned since
		multi-thread.
	*/

	/* Fault sending in case of incorrect parameters */
	if (lit__LightLevelTarget->__item < 0 || lit__LightLevelTarget->__item> 100)
		return dpws_fault(dpws, SENDER_FAULT_VALUE, "Level out of bounds.", NULL, NULL);

	pState->target = lit__LightLevelTarget->__item;
	if (lit__LightLevelTarget->TransitionDuration)
	{
		printf("--> Setting light level to %d%% in %d seconds\n", pState->target, *lit__LightLevelTarget->TransitionDuration);
		inc = (pState->target - pState->level)/(*lit__LightLevelTarget->TransitionDuration);

		/* Shifting implementation. */
		printf("Light level becomes ");
		if (inc> 0) {	/* Up shifting */
			for (; (pState->level + inc) <= pState->target; pState->level += inc) {
				portableSleep(1);
				printf(" %d%%", pState->level);
			}
		}
		else {	/* Down shifting */
			for (; (pState->level + inc) >= pState->target; pState->level += inc) {
				portableSleep(1);
				printf(" %d%%", pState->level);
			}
		}
	}
	else
		printf("--> Setting light level to %d%%\n", pState->target);

	pState->level = lit__LightLevelTarget->__item;
	printf("Light level reached %d%% .\n", pState->level);

	/* Target reached WS-Eventing notification */
	printf("<-- Notifying Light level target reached\n");
	if ((status = dpws_notify___lit__TargetReached(&pState->dpws, pState->hEventSource, pState->level)))
		fprintf(stderr, "Could not send 'target reached' event (err %d)\n", status);
	dpws_end(&pState->dpws);

	return status;
}

/* Retrieves the current dimmer target level. */
int __lit__GetLevelTarget(struct dpws* dpws, short *lit__LightLevel)
{
	struct dimmerData * pState = (struct dimmerData *)dpws_get_endpoint_user_data(dpws);
	printf("--> Requesting light level target\n");
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	*lit__LightLevel = pState->target;
	printf("<-- Light level target is %d%%\n", pState->target);
	return DPWS_OK;
}

/* Retrieves the current dimmer instant level. */
int __lit__GetLevel(struct dpws* dpws, short *lit__LightLevel)
{
	struct dimmerData * pState = (struct dimmerData *)dpws_get_endpoint_user_data(dpws);
	printf("--> Requesting light level\n");
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	*lit__LightLevel = pState->level;
	printf("<-- Light level is %d%%\n", pState->level);
	return DPWS_OK;
}
