/*============================================================================*\
|                                                                              |
|                          SOA4D DPWSCore Samples                              |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|                                                                              |
|        + File info:                                                          |
|                     $Revision: 2207 $
|                     $Date: 2009-03-16 14:47:13 +0100 (lun, 16 mar 2009) $
\*============================================================================*/
#include <stdio.h>

#include "dc/dc_Dpws.h"	// Main DPWSCore API include file.

#include "litStub.h" // Generated stub & skeleton include file.
#include "lit.nsmap" // Generated namespace table include file.

#include "fileio.h"	// File utility used for boot sequence persistency.
#include "serverDPWS.h"// OS-dependent code (especially server loop).


/*
 *	Device Configuration data
 */

#define MAX_SCOPE_LENGTH 255	// Maximum length for computed scope.
#define SAMPLE_SCOPE "http://www.soa4d.org/DPWS/Samples/Lights/Scope"	// The scope for the light device

#define BOOT_SEQ_FILE "bootseq.log"	// The name of the file holding the boot sequence number.

/*
IMPORTANT NOTE about types:
	The namespace does not need to be the same for device types and service port
	types as done in this sample. Indeed the qualified names of the device
	supported types are not necessarily related to a wsdl file and can feature
	an abstract device type.
*/
#define SAMPLE_NS "http://www.soa4d.org/DPWS/Samples/Lights"	// The namespace used to qualify device and service port types.

struct qname SimpleLightType = {SAMPLE_NS, "SimpleLight"};	// Qualified name used to specify the type supported by the device (could have been prefixed).

struct prefixed_qname SwitchPowerPortType	// Prefixed qualified name used to define the service port type (from WSDL). Using
	= {{SAMPLE_NS, "SwitchPower"}, "lit"};	// a prefix will ensure it will be used in messages.

struct wsdl_info LightingWsdl	// Structure containing the WSDL namespace and location for the hosted service.
	= {SAMPLE_NS, "http://www.soa4d.org/DPWS/Samples/Lights/Lighting.wsdl"};


/*
 *	Runtime data
 */

enum lit__PowerState simpleLightState = lit__PowerState__OFF;	// Device instance runtime data (the type is defined in litStub.h).
static struct dpws master_dpws;	// The dpws structure used for message processing (mono-thread server)


/*
 *  Utilities
 */

static char diversified_scope[MAX_SCOPE_LENGTH];	// computed scope buffer

/* Function adding the suffix to default scope (if suffix was defined).
 * Returns 0 if ok 1 if error (string too long).
 */
static int compute_diversified_scope(char *scope_suffix)
{
	strcpy(diversified_scope, SAMPLE_SCOPE);
	if (scope_suffix) {
		if (sizeof(SAMPLE_SCOPE) + strlen(scope_suffix) > MAX_SCOPE_LENGTH)
			return 1;
		if (scope_suffix[0] != '/')
			strcat(diversified_scope, "/");
		strcat(diversified_scope, scope_suffix);
	}
	return 0;
}

/* Function printing information related to a device. */
static void print_device_info(short hrefDevice)
{
	int len = 10, sz, i, j;
	short hrefs[10];
	char ** transportAddrs = NULL;

	// One service port (HTTP binding) created by default but potentially multiple
	// local addresses (multiple network interfaces & IP versions)
	transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefDevice), &sz);

	printf("\n+ Device [%s] started\n\nAddresses:\n", dpws_get_ptr_att(hrefDevice, DPWS_STR_DEVICE_ID));
	for (i = 0; i < sz; i++) {
		printf("[%d] %s\n", i, transportAddrs[i]);
		dpws_free(transportAddrs[i]);	// because of dpws_get_transport_addresses
	}
	dpws_free(transportAddrs);	// because of dpws_get_transport_addresses

	if (!dpws_get_service_handles(hrefDevice, hrefs, &len)) {
		printf("\n++ Hosted services:\n\n");
		for (i = 0; i < len; i++)
		{
			transportAddrs = dpws_get_transport_addresses(dpws_get_default_service_port(hrefs[i]), &sz);	// same as for device
			printf("Service ID: %s\nAddresses:\n", dpws_get_ptr_att(hrefs[i], DPWS_STR_SERVICE_ID));
			for (j = 0; j < sz; j++) {
				printf("[%d] %s\n", j, transportAddrs[j]);
				dpws_free(transportAddrs[j]);	// because of dpws_get_transport_addresses
			}
			dpws_free(transportAddrs);	// because of dpws_get_transport_addresses
		}
	}
}

/* Function displaying usage summary. */
static void simple_usage() {
	fprintf(stderr, "\nUsage: server [-s <scopeDiversifier>] [-p <httpPort>]\n");
	fprintf(stderr, "Use -?, -h or -help for more information\n\n");
	exit(1);
}

/* Function displaying complete program usage. */
static void usage() {
	fprintf(stderr, "\nUsage: server [-s <scopeDiversifier>] [-p <httpPort>]\n");
	fprintf(stderr, "\nOptions : \n");
	fprintf(stderr, "\n -s <scopeDiversifier>\n");
	fprintf(stderr, "\n\tThe <scopeDiversifier> is used to avoid conflicts on the local\n");
	fprintf(stderr, "\tnetwork by participating in building the scope of the device.\n");
	fprintf(stderr, "\n -p <httpPort>\n");
	fprintf(stderr, "\n\tThe <httpPort> is the http port used to listen to soap messages.\n");
	fprintf(stderr, "\nNotes : \n");
	fprintf(stderr, "\n\tThe server should be started before the client.\n\n");
	exit(1);
}

/*
 *  Main function performing server initialization.
 *	Usage: server [-s <scopeDiversifier>] [-p <httpPort>]
 */
int main (int argc, char **argv)
{
	/* Local variables */
	short hServClass, hKitchenLight, hSPServ, hSPServPort;
	int port = 9867, c, bootSeq, status, i;
	char * scope_suffix = NULL;
	struct localized_string ls = {NULL, NULL};

	/* Command line parsing */
	if (argc > 1 && (!strcmp(argv[1], "-?") || !strcmp(argv[1], "-h") || !strcmp(argv[1], "-help"))) {
		usage();
	}

	if ((argc < 1) || (argc > 5))
		simple_usage();

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-s")) {
			if (++i == argc)
				simple_usage();
			scope_suffix = argv[i];
		} else if (!strcmp(argv[i], "-p")) {
			if (++i == argc)
				simple_usage();
			port = atoi(argv[i]);
			if (port == 0)
				simple_usage();
		} else
			simple_usage();
	}

	// Compute diversified scope, the scope will be used to discover devices on the network.
	// If a suffix has been used only the clients started using the same suffix will detect
	// the current device.
	if (compute_diversified_scope(scope_suffix)) {
		fprintf(stderr, "scope_suffix is too long, should not exceed %d characters\n", MAX_SCOPE_LENGTH - sizeof(SAMPLE_SCOPE));
		simple_usage();
	}

	/* Simple server program description (single device with single service) */
	// 1. Initialize the DPWS stack.
	// 2. Configure server.
	// 	2.1 Set the boot sequence number & others toolkit parameters.
	//	2.2 Create service class.
	//	2.3 Create & configure device.
	//	2.4 Create & configure hosted service.
	//	2.5 Enable device.
	// 3. Initialize server.
	// 4. Start server loop.
	// 5. Stop the server

	printf("Configuring DPWS device...\n\n");

	/* 1. Initialize the dpws stack. */

	// This is the first function to be called when using the DPWS stack.
	// This version selects one single local IPv4 address.
	if ((status = dpws_init())) {
		fprintf(stderr, "Could not initialize the DPWSCore stack (err %d)\n\n", status);
		exit(1);
	}

	/* 2. Configure server. */

	// 	2.1 Set the boot sequence number & others toolkit parameters.

	// The boot sequence number is used is WS-Discovery messages and should be stored
	// and incremented each time the server is started.
	// Failing to do so may result in messages ignored by peers.
	bootSeq = readintfromfile(BOOT_SEQ_FILE);	// Reading the boot sequence number from a file
	writeinttofile(bootSeq + 1, BOOT_SEQ_FILE);	// Writing the next boot sequence number to the same file

	/*
	NOTE:
		All configuration APIs may return an error code. It has not been tested
		in this sample for clarity's sake and because the only errors that can
		occur are memory exhaustion (bad omen at initialization time) or due to
		invalid parameters (and we hope this sample has been correctly written :-)).
	*/
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_HTTP_PORT, port);	// Configure the HTTP listen port.
	DPWS_SET_INT_ATT(DC_REGISTRY_HANDLE, DPWS_INT_BOOT_SEQ, bootSeq);	// Configure the boot sequence number to use in this session (mandatory).

	printf("Initializing DPWS light devices...\n");

	// 2.2 Create service class
	printf("Creating service class...\n");
	hServClass = dpws_create_service_class();

	// Configure the service class attributes.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_PREFIXED_TYPE, &SwitchPowerPortType);	// The service implements the 'switch power' port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_WSDL, &LightingWsdl);	// The WSDL information for the previous port type.
	DPWS_ADD_PTR_ATT(hServClass, DPWS_PTR_HANDLING_FUNCTION, &lit_serve_request);	// The generated dispatch function for the port type (from litStub.h).
	DPWS_SET_STR_ATT(hServClass, DPWS_STR_ID, "http://www.soa4d.org/DPWS/Samples/Lights/Light1");	// A string ID that will used as default for mandatory service ID

	// 2.3 Create & configure device
	printf("Creating device...\n with type {%s}%s\n", SAMPLE_NS, "SimpleLight");
	hKitchenLight = dpws_create_custom_device(0, -1);	// the id (0) must be unique in the physical local device

	// Configure the mandatory device attributes.
	DPWS_ADD_PTR_ATT(hKitchenLight, DPWS_PTR_TYPE, &SimpleLightType);
	/*
	NOTE:
		Should change everytime a change on the device impacts DPWS metadata.
	*/
	DPWS_SET_INT_ATT(hKitchenLight, DPWS_INT_METADATA_VERSION, 1);
	ls.s = "Kitchen light";
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_PTR_FRIENDLY_NAME, &ls);
	ls.s = "BrightBulb";
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_PTR_MODEL_NAME, &ls);
	ls.s = "Electrical SA";
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_PTR_MANUFACTURER, &ls);

	// Configure the optional device attributes.
	printf(" with scope %s\n\n", diversified_scope);
	DPWS_ADD_STR_ATT(hKitchenLight, DPWS_STR_SCOPE, diversified_scope);
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_FIRMWARE_VERSION, "1.0");
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_SERIAL_NUMBER, "56080001");
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_MODEL_NUMBER, "1.0");
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_MODEL_URL, "http://www.electrical.com/BrightBulb.html");
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_PRESENTATION_URL, "LightsOverview.html");	// Any relative URI will use the default presentation server
	DPWS_SET_STR_ATT(hKitchenLight, DPWS_STR_MANUFACTURER_URL, "http://www.electrical.com");

	// Device instance data that will be accessible in the service implementation
	DPWS_SET_PTR_ATT(hKitchenLight, DPWS_PTR_USER_DATA, &simpleLightState);

	// 2.4 Create & configure hosted service.
	hSPServPort = dpws_create_service_port();	// creates a service port (network endpoint)
	DPWS_SET_STR_ATT(hSPServPort, DPWS_STR_ADDRESS, "MySwichPowerService");	// sets the context path for the service physical address.
	/*
	IMPORTANT NOTE:
		Overriding the default value (a generated UUID) for the service address
		is important since transport addresses are in the the device metadata
		version scope so letting the random default would require an update of
		the metadata version at every server start like for the boot sequence.
	*/
	hSPServ = dpws_create_hosted_service(hKitchenLight, hServClass);	// creates the service
	dpws_release_handle(hServClass);	// Now used by the service and not needed anymore
	dpws_bind_service(hSPServ, hSPServPort);	// associates it with a "network endpoint"
	dpws_release_handle(hSPServPort);	// Not needed anymore

	// 2.5 Enable device.
	printf("<-- Enabling device...\n");
	if ((status = dpws_enable_device(hKitchenLight)))	// Enables device (especially schedule hello messages).
	{
		fprintf(stderr, "Could not execute dpws_enable_device (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	/* 3. Initialize server. */
	if ((status = dpws_server_init(&master_dpws, NULL)))	// initializes the dpws structure for server-side operations.
	{
		fprintf(stderr, "Could not initialize DPWS server (err %d)\n", status);
		dpws_shutdown();
		exit(1);
	}

	// Print devices and services information
	print_device_info(hKitchenLight);	// Print device and services information.
	dpws_release_handle(hKitchenLight);	// Enabling makes it now hold by the registry

	/* 4. Start server loop. */
	// Creates a new thread to serve incoming messages. In the current sample,
	// only one thread is created that will process all received messages.
	if (bootServer(&master_dpws)) {	// see listener.c
		fprintf(stderr, "Could not boot server...\n");
		dpws_shutdown();
		exit(1);
	}

	/* 5. Stop the server */

	printf("\n\nDPWS server ready. Type 'q' or 'Q' to stop it smoothly.\n\n");
	do {	// Wait for the 'user exit'
		c = getchar();
	}
	while (c !='q' && c != 'Q');

	stopServer();	// 1. Calls the dpws_stop_server function that closes all
		// open listening sockets, schedules the 'Eventing end' notifications
		// and Bye WS-Discovery messages for all the active devices and blocks
		// until all server processing is finished and the main loop blocking
		// on dpws_accept() can exit.
		// 2. Uses platform-specific thread APIs to wait for the main server
		// loop thread dies.
}

/*
 *                   Service operations
 */

/* Change the light ON/OFF state */
int __lit__Switch(struct dpws* dpws, enum lit__PowerState lit__Power)
{
	enum lit__PowerState * pState;

	// The hosting device or service user data can be retrieved in the context
	// using a set of APIs among which dpws_get_device_user_data.
	pState = (enum lit__PowerState *)dpws_get_device_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	*pState = lit__Power;
	printf("--> Setting light %s\n", lit__Power == lit__PowerState__ON ? "ON" : "OFF");
	return DPWS_OK;
}

/* Retrieves the light ON/OFF status */
int __lit__GetStatus(struct dpws* dpws, enum lit__PowerState *lit__Power)
{
	printf("--> Requesting light status\n");
	*lit__Power = *(enum lit__PowerState *)dpws_get_device_user_data(dpws);
	/*
	IMPORTANT NOTE about user data:
		The protection of user data is not performed by the DPWS stack so user
		should normally provide it. Since the 'light' sample is mono-thread, no
		concurrency issue can happen.
	*/
	printf("<-- Light status is %s\n", *lit__Power == lit__PowerState__ON ? "ON" : "OFF");
	return DPWS_OK;
}
