/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                   ->>  Copyright 2008-2009 Odonata <<-                       |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

/*******************************************************************************
*                        DPWSCore Reactor support services                     *
*******************************************************************************/
#include "al_ip.h"
#include "dcDCPL_Mem.h"
#include "dcDCPL_Reactor.h"
#include "dcDCPL_Socket.h"
#include "dcCOMN_Tools.h"

struct interrupt_data {
	DCPL_SOCKET socket;
	int family;
	short port;
};

#define WAKE_UP_DATAGRAM			"<WAKE/>"

/*------------------------- Static Functions prototypes ----------------------*/

static int init_fd_set(al_fd_set * fdset, struct reactor_item * item_list, int maxsock);
static int check_fd_set(al_fd_set * fdset, struct reactor_item ** item_list, int nresults, struct reactor_item ** result_list);

/*----------------------------------------------------------------------------*/

int dcpl_reactor_init(struct reactor * reactor, int family, uint16_t port, struct dcpl_error * error)
{
#ifdef AL_HAVE_IPV6
	int af = family == DCPL_AF_INET6 ? AF_INET6 : AF_INET;
#else
	int af = AF_INET;
#endif
	if (reactor->lock) { //shared reactor
	    DCPL_SOCKET wake_sock;
		struct interrupt_data * interrupt_data;

		interrupt_data = dcpl_malloc(sizeof(struct interrupt_data));
		if (!interrupt_data)
			return DCPL_EOM_ERROR;
		wake_sock = dcpl_udp_bind(family, NULL, 0, port, 0, error);
		if (wake_sock < 0) {
			dcpl_free(interrupt_data);
			return error ? error->error : DCPL_SOCKET_BIND_ERROR;
		}
	    interrupt_data->socket = wake_sock;
	    interrupt_data->family = af;
	    interrupt_data->port = port;
	    reactor->interrupt_data = interrupt_data;
	}
    return DCPL_OK;
}

int dcpl_reactor_free(struct reactor * reactor, struct dcpl_error * error)
{
	if (reactor->interrupt_data) {
		DCPL_SOCKET wake_sock = ((struct interrupt_data *)reactor->interrupt_data)->socket;
		dcpl_free(reactor->interrupt_data);
		if (wake_sock != DCPL_INVALID_SOCKET)
			dcpl_closesocket(wake_sock, NULL); // No need to handle possible error
	}
    return DCPL_OK;
}

int dcpl_reactor_interrupt(struct reactor * reactor, struct dcpl_error * error)
{
    int ret;
	struct interrupt_data * interrupt_data = (struct interrupt_data *)reactor->interrupt_data;
	DCPL_SOCKET s = DCPL_INVALID_SOCKET;

    if (!interrupt_data)
    	return DCPL_OK;

    /* create socket */
    s = al_socket(interrupt_data->family, SOCK_DGRAM, 0);
    if (s < 0)
		return dcpl_set_error(error, DCPL_SOCKET_CREATE_ERROR, NULL);
	if ((ret = dcpl_udp_send_loopback(s, WAKE_UP_DATAGRAM, sizeof(WAKE_UP_DATAGRAM), 0, interrupt_data->family, interrupt_data->port, error)) < 0)
		return ret;

	return dcpl_closesocket(s, error);
}

static int init_fd_set(al_fd_set * fdset, struct reactor_item * item_list, int maxsock)
{
	struct reactor_item * item = NULL;
	AL_FD_ZERO(fdset);
	for (item = item_list; item != NULL; item = item->next)
	{
		AL_FD_SET(item->socket, fdset);
		if ((int)item->socket > maxsock) maxsock = (int)item->socket;
		DPWSLOG2(DC_TRANSPORT, "Socket added -> %d Max socks: %d\n", (int)item->socket, maxsock);
	}
	return maxsock;
}

static int check_fd_set(al_fd_set * fdset, struct reactor_item ** item_list, int nresults, struct reactor_item ** result_list)
{
	struct reactor_item * item = *item_list;
	struct reactor_item * previous = NULL;
	while (item != NULL && nresults > 0)
	{
		if (AL_FD_ISSET(item->socket, fdset)) {
			struct reactor_item * next = item->next;
			if (previous) previous->next = next;
			else *item_list = next;
			item->next = *result_list;
			*result_list = item;
			item = next;
			nresults--;
		} else {
			previous = item;
			item = item->next;
		}
	}
	return nresults;
}

int dcpl_reactor_wait(struct reactor * reactor, uint64_t next_date, struct reactor_item ** result_list, struct dcpl_error * error)
{
	al_fd_set readfds, writefds, excfds;
    al_timeval_t timeout, *ptimeout = NULL;
	uint64_t to = 0;
	int maxsock = -1;
	struct interrupt_data * interrupt_data = (struct interrupt_data *)reactor->interrupt_data;
	DCPL_SOCKET wake_sock = DCPL_INVALID_SOCKET;
	int nresults;

	maxsock = init_fd_set(&readfds, reactor->read_list, maxsock);
	maxsock = init_fd_set(&writefds, reactor->write_list, maxsock);
	maxsock = init_fd_set(&excfds, reactor->exception_list, maxsock);

	if (interrupt_data) {
		wake_sock = interrupt_data->socket;
		AL_FD_SET(wake_sock, &readfds);
		if ((int)wake_sock > maxsock) maxsock = (int)wake_sock;
		DPWSLOG2(DC_TRANSPORT, "Wake socket added -> %d Max socks: %d\n", (int)wake_sock, maxsock);
	}

	if (next_date != 0)
	{
		uint64_t clock = dcpl_get_clock();
		to = next_date > clock ? next_date - clock : 0;
		timeout.sec = (uint32_t)(to / 1000);
		timeout.usec = (uint32_t)((to % 1000)*1000);
		ptimeout = &timeout;
		DPWSLOG2(DC_TRANSPORT, "Entering Select -> Timeout: %d Max socks: %d\n", (int)to, maxsock);
	}
//	else {
//		DPWSLOG1(TRANSPORT, "Entering Select -> Timeout: NULL Max socks: %d\n", maxsock);
//	}

	if (maxsock < 0 && next_date != 0)
		nresults = dcpl_sleep(to);
	else
		nresults = al_select(maxsock+1, &readfds, &writefds, &excfds, ptimeout);
	DPWSLOG1(DC_TRANSPORT, "Exiting Select <- N results: %d\n", nresults);

	if (nresults > 0)
	{
		int remaining = nresults;
		if (wake_sock != DCPL_INVALID_SOCKET && AL_FD_ISSET(wake_sock, &readfds)) {
			char buf[sizeof(WAKE_UP_DATAGRAM)];
			int ret;
			if ((ret = dcpl_udp_recv(wake_sock, buf, sizeof(WAKE_UP_DATAGRAM), 0, NULL, 0, NULL, NULL, error)) < 0)
				return ret;
			remaining--;
		}
		remaining = check_fd_set(&readfds, &reactor->read_list, remaining, result_list);
		remaining = check_fd_set(&writefds, &reactor->write_list, remaining, result_list);
		remaining = check_fd_set(&excfds, &reactor->exception_list, remaining, result_list);
	}
	else if (nresults == 0) {
		// processed at higher level
	}
	else if (al_ip_error() == AL_EINTR) {
		return DCPL_REACTOR_INTR;
	}
	else
		return dcpl_set_error(error, DCPL_SOCKET_SELECT_ERROR, NULL);

	return nresults;
}

