/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                   ->>  Copyright 2008-2009 Odonata <<-                       |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

#include "al_ip.h"
#include "dcDCPL_Error.h"
#include "dcDCPL_Socket.h"
#include "dcDCPL_Os.h"

#define DCPL_SOCKET_READ  1
#define DCPL_SOCKET_WRITE 2

/*------------------------- Static Functions prototypes ----------------------*/

static int set_common_socket_options(DCPL_SOCKET fd, socket_data_t * data, struct dcpl_error * error);
static int socket_timeout(DCPL_SOCKET socket, int *timeout, int socket_type, struct dcpl_error * error);
static int set_nonblocking_mode(DCPL_SOCKET fd, int nonblocking);

/*----------------------------------------------------------------------------*/

/* Common socket section */

int dcpl_closesocket(DCPL_SOCKET socket, struct dcpl_error * error)
{
	if (al_sock_close(socket)) {
		return dcpl_set_error(error, DCPL_SOCKET_CLOSE_ERROR, NULL);
	}
	return DCPL_OK;
}

/* UDP socket section */

DCPL_SOCKET dcpl_udp_open(int family, struct dcpl_error * error)
{
    DCPL_SOCKET fd = DCPL_INVALID_SOCKET;

	/* Change family to AL value */
#ifdef AL_HAVE_IPV6
	family = family == DCPL_AF_INET6 ? AF_INET6 : AF_INET;
#else
	family = AF_INET;
#endif

    /* create sending UDP socket */
    fd = al_socket(family, SOCK_DGRAM, 0);
	if (fd < 0) {
		dcpl_set_error(error, DCPL_SOCKET_CREATE_ERROR, NULL);
		return DCPL_INVALID_SOCKET;
	}
	return fd;
}

int dcpl_set_multicast_netif(DCPL_SOCKET s, int family, uint32_t itf_selector, struct dcpl_error * error)
{
#ifdef AL_HAVE_IPV6
	/* Change family to AL value */
	family = (family == DCPL_AF_INET6 ? AF_INET6 : AF_INET);
	if (family == AF_INET6) {
	    int ttl = 1;
		if (al_setsockopt(s, IPPROTO_IPV6, IPV6_MULTICAST_HOPS, (char*)&ttl, sizeof(ttl))) {
			al_sock_close(s);
			return dcpl_set_error(error, DCPL_SOCKET_MULTICAST_HOPS_ERROR, NULL);
		}
		if (al_setsockopt(s, IPPROTO_IPV6, IPV6_MULTICAST_IF, (char*)&itf_selector, sizeof(uint32_t))) {
			al_sock_close(s);
			return dcpl_set_error(error, DCPL_SOCKET_MULTICAST_IF_ERROR, NULL);
		}
	}
	else
#endif
	{
		unsigned char ttl = 1;
		al_in_addr_t in_addr;
		in_addr.addr = itf_selector;
		if (al_setsockopt(s, IPPROTO_IP, IP_MULTICAST_TTL, (char*)&ttl, sizeof(ttl))) {
			al_sock_close(s);
			return dcpl_set_error(error, DCPL_SOCKET_MULTICAST_HOPS_ERROR, NULL);
		}
		if (al_setsockopt(s, IPPROTO_IP, IP_MULTICAST_IF, (char*)&in_addr, sizeof(in_addr))) {
			al_sock_close(s);
			return dcpl_set_error(error, DCPL_SOCKET_MULTICAST_IF_ERROR, NULL);
		}
	}
    return DCPL_OK;
}

DCPL_SOCKET dcpl_udp_bind(int family, const char* itf_addr, uint32_t itf_selector, uint16_t port, int bind_flags, struct dcpl_error * error)
{
	DCPL_SOCKET fd = DCPL_INVALID_SOCKET;
	al_sockaddr_t servAddr;
	int socklen = sizeof(servAddr);
    int reuse = 1;

	/* Change family to AL value */
#ifdef AL_HAVE_IPV6
	family = (family == DCPL_AF_INET6 ? AF_INET6 : AF_INET);
#else
	family = AF_INET;
#endif

    /* create multicast listening UDP socket */
	fd = al_socket(family, SOCK_DGRAM, 0);
    if (fd < 0) {
		dcpl_set_error(error, DCPL_SOCKET_CREATE_ERROR, NULL);
		return DCPL_INVALID_SOCKET;
    }
	if ((bind_flags & SO_REUSEADDR) && al_setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char*)&reuse, sizeof(reuse)))
    {
		dcpl_set_error(error, DCPL_SOCKET_REUSEADDR_ERROR, NULL);
		al_sock_close(fd);
        return DCPL_INVALID_SOCKET;
    }
    /* bind listening UDP socket */
	if ((socklen = al_sockaddr_init(family, &servAddr, socklen)) < 0
		|| al_sockaddr_setPort(&servAddr, port)
		|| (itf_addr != NULL && (al_sockaddr_setNetAddrFromAddrStr(&servAddr, itf_addr) || al_sockaddr_setScope(&servAddr, itf_selector)))
		|| al_bind(fd, &servAddr, socklen))
	{
		dcpl_set_error(error, DCPL_SOCKET_BIND_ERROR, NULL);
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}
	return fd;
}

DCPL_SOCKET dcpl_multicast_bind(int family, const char* itf_addr, uint32_t itf_selector, uint16_t port, const char* multicast_addr, struct dcpl_error * error)
{
	DCPL_SOCKET fd = DCPL_INVALID_SOCKET;
#ifdef AL_HAVE_IPV6
	int af = family == DCPL_AF_INET6 ? AF_INET6 : AF_INET;
#else
	int af = AF_INET;
#endif

    /* create multicast listening UDP socket */
	fd = dcpl_udp_bind(family, NULL, itf_selector, port, SO_REUSEADDR, error);
    if (fd < 0)
		return DCPL_INVALID_SOCKET;
	/* Join multicast group */
	if (af == AF_INET) {
	    struct ip_mreq mreq;
		if (al_inet_pton(af, multicast_addr, &mreq.imr_multiaddr.s_addr) <= 0)
		{
			dcpl_set_error(error, DCPL_SOCKET_JOIN_GROUP_ERROR, "Incorrect multicast address");
			al_sock_close(fd);
			return DCPL_INVALID_SOCKET;
		}
		if (itf_addr)
		{
			if (al_inet_pton(af, itf_addr, &mreq.imr_interface.s_addr) <= 0)
			{
				dcpl_set_error(error, DCPL_SOCKET_JOIN_GROUP_ERROR, "Incorrect local address");
				al_sock_close(fd);
				return DCPL_INVALID_SOCKET;
			}
		}
		else
			mreq.imr_interface.s_addr = 0;

		if (al_setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, (void *)&mreq, sizeof(mreq)))
		{
			dcpl_set_error(error, DCPL_SOCKET_JOIN_GROUP_ERROR, NULL);
			al_sock_close(fd);
			return DCPL_INVALID_SOCKET;
		}
# ifdef DCPL_HAVE_PKTINFO
		// If listen on any socket, may need to retrieve the receiving interface
		if (!itf_addr)
		{
			int set = 1;
			if (al_setsockopt(fd, IPPROTO_IP, IP_PKTINFO, (void *)&set, sizeof(set)))
			{
				dcpl_set_error(error, DCPL_SOCKET_PACKET_INFO, NULL);
				al_sock_close(fd);
				return DCPL_INVALID_SOCKET;
			}
		}
# endif
	}
#ifdef AL_HAVE_IPV6
	else if (af == AF_INET6) {
	    struct ipv6_mreq mreq;
		if (al_inet_pton(af, multicast_addr, &mreq.ipv6mr_multiaddr.s6_addr) <= 0)
		{
			dcpl_set_error(error, DCPL_SOCKET_JOIN_GROUP_ERROR, "Incorrect multicast address");
			al_sock_close(fd);
			return DCPL_INVALID_SOCKET;
		}
		mreq.ipv6mr_interface = itf_selector;

		if (al_setsockopt(fd, IPPROTO_IPV6, IPV6_JOIN_GROUP, (void *)&mreq, sizeof(mreq)))
		{
			dcpl_set_error(error, DCPL_SOCKET_JOIN_GROUP_ERROR, NULL);
			al_sock_close(fd);
			return DCPL_INVALID_SOCKET;
		}
# ifdef DCPL_HAVE_PKTINFO
		// If listen on any socket, may need to retrieve the receiving interface
		if (!itf_addr)
		{
			int set = 1;
			if (al_setsockopt(fd, IPPROTO_IPV6, IPV6_PKTINFO, (void *)&set, sizeof(set)))
			{
				dcpl_set_error(error, DCPL_SOCKET_PACKET_INFO, NULL);
				al_sock_close(fd);
				return DCPL_INVALID_SOCKET;
			}
		}
# endif
	}
#endif
    return fd;
}

int dcpl_udp_send(DCPL_SOCKET fd, const char *s, size_t n, int flags, const char *host, uint32_t itf_selector, uint16_t port, struct dcpl_error * error)
{
	int nwritten;
	al_sockaddr_t * addrs;
	int nbAddrs;
#ifdef AL_HAVE_IPV6
	int af = is_ipv6_addr(host) ? AF_INET6 : AF_INET;
#else
	int af = AF_INET;
#endif

	if (al_getaddrinfo(af, host, port, &addrs, &nbAddrs))
		return dcpl_set_error(error, DCPL_SOCKET_RESOLVE_ERROR, NULL);

	al_sockaddr_setScope(addrs, itf_selector);

	if ((nwritten = al_sendto(fd, (char *)s, (int)n, flags, addrs, (sizeof(al_sockaddr_t)))) < 0)
		nwritten = dcpl_set_error(error, DCPL_SOCKET_SEND_ERROR, NULL);

	al_freeaddrinfo(addrs);

	return nwritten;
}

int dcpl_udp_send_loopback(DCPL_SOCKET fd, const char *s, size_t n, int flags, int af, unsigned short port, struct dcpl_error * error)
{
	int nwritten;
	al_sockaddr_t sockaddr;

	al_sockaddr_init(af, &sockaddr, sizeof(sockaddr));
	al_sockaddr_setInAddrLoopBack(&sockaddr);
	al_sockaddr_setPort(&sockaddr, port);

	if ((nwritten = al_sendto(fd, (char *)s, (int)n, flags, (const al_sockaddr_t*)&sockaddr, (int)sizeof(sockaddr))) < 0) {
		return dcpl_set_error(error, DCPL_SOCKET_SEND_ERROR, NULL);
	}
	return nwritten;
}

int dcpl_udp_recv(DCPL_SOCKET fd, char *s, size_t n, int flags, char* peer_addr, int addr_len, uint16_t * peer_port, uint32_t * itf, struct dcpl_error * error)
{
	al_sockaddr_t sockaddr;
	int socklen = sizeof(sockaddr);
	int nread;

#ifdef DCPL_HAVE_PKTINFO
	if (itf)
	{
		if ((nread = al_recvon(fd, s, (int)n, flags, (al_sockaddr_t *)&sockaddr, &socklen, itf)) < 0)
			return dcpl_set_error(error, DCPL_SOCKET_RECV_ERROR, NULL);
	}
	else
#endif
	{
		if ((nread = al_recvfrom(fd, s, (int)n, flags, (al_sockaddr_t *)&sockaddr, &socklen)) < 0)
			return dcpl_set_error(error, DCPL_SOCKET_RECV_ERROR, NULL);
	}
	if (peer_addr)
		al_sockaddr_getNetAddrToAddrStr(&sockaddr, peer_addr, addr_len, 1);
	if (peer_port)
		al_sockaddr_getPort(&sockaddr, peer_port);
    return nread;
}

/* TCP socket section */

static int set_common_socket_options(DCPL_SOCKET fd, socket_data_t * data, struct dcpl_error * error)
{
	int set = 1;

	if (!data)
		return DCPL_OK;
	if ((data->flags & SO_LINGER)) {
		struct linger linger;
		memset(&linger, 0, sizeof(struct linger));
		linger.l_onoff = 1;
		linger.l_linger = data->linger_time;
		if (al_setsockopt(fd, SOL_SOCKET, SO_LINGER, (char*)&linger, sizeof(struct linger)))
			goto error;
    }
    if ((data->flags & ~SO_LINGER) && al_setsockopt(fd, SOL_SOCKET, data->flags & ~SO_LINGER, (char*)&set, sizeof(int)))
		goto error;
	if (data->keep_alive && al_setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (char*)&set, sizeof(int)))
		goto error;
	if (data->buflen && al_setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char*)&data->buflen, sizeof(int)))
		goto error;
	if (data->buflen && al_setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char*)&data->buflen, sizeof(int)))
		goto error;
#ifdef TCP_NODELAY
    if (data->tcp_nodelay && al_setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, (char*)&set, sizeof(int)))
		goto error;
#endif
#ifdef SO_NOSIGPIPE
    if (al_setsockopt(fd, SOL_SOCKET, SO_NOSIGPIPE, (char*)&set, sizeof(int)))
		goto error;
#endif
	return DCPL_OK;
error:
	return dcpl_set_error(error, DCPL_SOCKET_SO_ERROR, NULL);
}

static int socket_timeout(DCPL_SOCKET socket, int *timeout, int socket_type, struct dcpl_error * error)
{
	uint64_t clock = dcpl_get_clock();
	al_fd_set fds, *rfds = NULL, *wfds = NULL;

	AL_FD_ZERO(&fds);
	AL_FD_SET(socket, &fds);
	if (socket_type == DCPL_SOCKET_READ)
		rfds = &fds;
	else
		wfds = &fds;
	for (;;)
	{
		al_timeval_t tv;
		int r;
		tv.sec = (uint32_t)(*timeout / 1000);
		tv.usec = (uint32_t)((*timeout % 1000)*1000);
		r = al_select(socket + 1, rfds, wfds, &fds, &tv);
		*timeout = (int)((clock + *timeout) - dcpl_get_clock());
		if (*timeout < 0)
			*timeout = 0;
		if (r > 0)
			break;
		if (r == 0)
		{
			dcpl_set_error(error, DCPL_SOCKET_TIMEOUT_ERROR, NULL);
			error->syserr = 0; // Not a system error
			return DCPL_SOCKET_TIMEOUT_ERROR;
		}
		if (al_ip_error() != AL_EINTR)
			return dcpl_set_error(error, DCPL_SOCKET_SELECT_ERROR, NULL);
	}
	return DCPL_OK;
}

static int set_nonblocking_mode(DCPL_SOCKET fd, int nonblocking)
{
	return al_ioctl(fd, AL_FIONBIO, &nonblocking);
}

DCPL_SOCKET dcpl_tcp_bind_listener(int family, const char* itf_addr, uint32_t itf_selector, uint16_t port, int backlog, socket_data_t * bind_data, struct dcpl_error * error)
{
	register DCPL_SOCKET fd;
	al_sockaddr_t servAddr;
	int socklen = sizeof(servAddr);
#ifdef AL_HAVE_IPV6
	int af = family == DCPL_AF_INET6 ? AF_INET6 : AF_INET;
#else
	int af = AF_INET;
#endif

	fd = al_socket(af, SOCK_STREAM, 0);
    if (fd < 0) {
		dcpl_set_error(error, DCPL_SOCKET_CREATE_ERROR, NULL);
		return DCPL_INVALID_SOCKET;
    }
	if (set_common_socket_options(fd, bind_data, error)) {
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}
	if ((socklen = al_sockaddr_init(af, &servAddr, socklen)) < 0
		|| al_sockaddr_setPort(&servAddr, port)
		|| (itf_addr != NULL && (al_sockaddr_setNetAddrFromAddrStr(&servAddr, itf_addr) || al_sockaddr_setScope(&servAddr, itf_selector)))
		|| al_bind(fd, &servAddr, socklen))
	{
		dcpl_set_error(error, DCPL_SOCKET_BIND_ERROR, NULL);
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}

	if (al_listen(fd, backlog)) {
		dcpl_set_error(error, DCPL_SOCKET_LISTEN_ERROR, NULL);
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}
	return fd;
}

DCPL_SOCKET dcpl_tcp_accept(DCPL_SOCKET listener, socket_data_t * data, char* peer_addr, int addr_len, uint16_t * peer_port, uint32_t * itf, struct dcpl_error * error)
{
	DCPL_SOCKET fd;
	al_sockaddr_t sockaddr;
	int socklen = sizeof(sockaddr);

	fd = al_accept(listener, &sockaddr, &socklen);
	if (fd < 0) {
		dcpl_set_error(error, DCPL_SOCKET_ACCEPT_ERROR, NULL);
        return DCPL_INVALID_SOCKET;
	}
	if (set_common_socket_options(fd, data, error)) {
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}
	if (peer_addr)
		al_sockaddr_getNetAddrToAddrStr(&sockaddr, peer_addr, addr_len, 1);
	if (peer_port)
		al_sockaddr_getPort(&sockaddr, peer_port);
	if (itf) {
		al_ipNetAddr_t netAddr;
		if (al_getsockname(fd, &sockaddr, &socklen))
		{
			dcpl_set_error(error, DCPL_SOCKET_GETSOCKNAME_ERROR, NULL);
	        return DCPL_INVALID_SOCKET;
		}
		switch (al_sockaddr_getDomain(&sockaddr))
		{
		case AF_INET:
			al_sockaddr_getNetAddr(&sockaddr, &netAddr);
#ifdef AL_HAVE_IPV6
			*itf = netAddr.addr.netAddr4.addr;
			break;
		case AF_INET6:
			al_sockaddr_getScope(&sockaddr, itf);
#else
			*itf = netAddr;
#endif
			break;
		}
	}
	return fd;
}

DCPL_SOCKET dcpl_tcp_connect(const char *host, uint32_t itf_selector, uint16_t port, int timeout, socket_data_t * data, struct dcpl_error * error)
{
	DCPL_SOCKET fd;
	al_sockaddr_t * addrs;
	int nbAddrs, i = 0, t_out = timeout;
#ifdef AL_HAVE_IPV6
	int af = is_ipv6_addr(host) ? AF_INET6 : AF_INET;
#else
	int af = AF_INET;
#endif

	fd = al_socket(af, SOCK_STREAM, 0);
    if (fd < 0) {
		dcpl_set_error(error, DCPL_SOCKET_CREATE_ERROR, NULL);
		return DCPL_INVALID_SOCKET;
    }
	if (set_common_socket_options(fd, data, error)) {
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}
	if (al_getaddrinfo(af, host, port, &addrs, &nbAddrs))
	{
		dcpl_set_error(error, DCPL_SOCKET_RESOLVE_ERROR, NULL);
		al_sock_close(fd);
		return DCPL_INVALID_SOCKET;
	}

	if (timeout)
		set_nonblocking_mode(fd, 1);
	else
		set_nonblocking_mode(fd, 0);
	for (;;) // This loop takes care of interrupts while waiting for connection & multiple addresses match
	{
		al_sockaddr_setScope(&addrs[i], itf_selector);
		if (al_connect(fd, &addrs[i], sizeof(al_sockaddr_t)))
		{
			int syserr = al_ip_error();
			if (timeout && (syserr == AL_EINPROGRESS || syserr == AL_EAGAIN))
			{
				int ret;
				if ((ret = socket_timeout(fd, &t_out, DCPL_SOCKET_WRITE, error)))
				{
					if ((ret == DCPL_SOCKET_TIMEOUT_ERROR) || (i == nbAddrs))
					{
						al_sock_close(fd);
						fd = DCPL_INVALID_SOCKET;
						goto exit;
					}
					else	// connection error: try next
					{
						i++;
						continue;
					}
				}
				break;
			}
			else if (syserr != AL_EINTR)
			{
				dcpl_set_error(error, DCPL_SOCKET_CONNECT_ERROR, NULL);
				al_sock_close(fd);
				fd = DCPL_INVALID_SOCKET;
				goto exit;
			}
		}
		else
			break;
	}
	if (timeout)
		set_nonblocking_mode(fd, 0);

exit:
	al_freeaddrinfo(addrs);
	return fd;
}

int dcpl_tcp_get_local_address(DCPL_SOCKET s, char * local_addr_buf, int buf_size, struct dcpl_error * error)
{
	al_sockaddr_t sockaddr;
	int socklen = sizeof(sockaddr);
	if (al_getsockname(s, &sockaddr, &socklen) || al_sockaddr_getNetAddrToAddrStr(&sockaddr, local_addr_buf, buf_size, 0))
	{
		dcpl_set_error(error, DCPL_SOCKET_GETSOCKNAME_ERROR, NULL);
		s = DCPL_INVALID_SOCKET;
		return DCPL_SOCKET_GETSOCKNAME_ERROR;
	}
	return DCPL_OK;
}

int dcpl_tcp_shutdown(DCPL_SOCKET fd, int how, struct dcpl_error * error)
{
	if (al_shutdown(fd, how)) {
		return dcpl_set_error(error, DCPL_SOCKET_SHUTDOWN_ERROR, NULL);
	}
	return DCPL_OK;
}

int dcpl_tcp_send(DCPL_SOCKET fd, const char *s, size_t n, int flags, int timeout, struct dcpl_error * error)
{
	int nwritten;
	int ret;
	while (n)
	{
		if (timeout && (ret = socket_timeout(fd, &timeout, DCPL_SOCKET_WRITE, error))) {
			return ret;
		}
		if ((nwritten = al_send(fd, (char *)s, (int)n, flags)) < 0) {
			return dcpl_set_error(error, DCPL_SOCKET_SEND_ERROR, NULL);
		}
		n -= nwritten;
		s += nwritten;
	}
	return nwritten;
}

int dcpl_tcp_recv(DCPL_SOCKET fd, char *s, size_t n, int flags, int timeout, struct dcpl_error * error)
{
	int nread;
	int ret;
	if (timeout && (ret = socket_timeout(fd, &timeout, DCPL_SOCKET_READ, error))) {
		return ret;
	}
	if ((nread = al_recv(fd, s, (int)n, flags)) < 0) {
		return dcpl_set_error(error, DCPL_SOCKET_RECV_ERROR, NULL);
	}
	return nread;
}

int dcpl_tcp_peek(DCPL_SOCKET fd, char *s, size_t n, int flags, int timeout, struct dcpl_error * error)
{
	return dcpl_tcp_recv(fd, s, n, flags | MSG_PEEK, timeout, error);
}

int dcpl_tcp_socket_valid(DCPL_SOCKET s)
{
	if (s != DCPL_INVALID_SOCKET) {
		char buf[1];
		al_fd_set readset;
		al_fd_set writeset;
		al_timeval_t timeout;

		timeout.sec = 0;
		timeout.usec = 0;
		AL_FD_ZERO(&readset);
		AL_FD_ZERO(&writeset);
		AL_FD_SET(s, &readset);
		AL_FD_SET(s, &writeset);
		if (al_select(s+1, &readset, &writeset, NULL, &timeout) > 0
			&& AL_FD_ISSET(s, &writeset)
			&& (!AL_FD_ISSET(s, &readset) || al_recv(s, buf, 1, MSG_PEEK) > 0))
			return 1;
	}
	return 0;
}

