/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2215 $
|                     $Date: 2009-03-19 15:44:46 +0100 (jeu, 19 mar 2009) $
\*============================================================================*/
/******************************************************************************\
 *                            Dynamic array utilities                         *
\******************************************************************************/
#ifndef DYN_ARRAY_H_
#define DYN_ARRAY_H_

#include "dc/dc_Ctypes.h"


#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------------*\
 *                              Private definitions                           *
\*----------------------------------------------------------------------------*/

/** Allocation callback.
 * @return A pointer on the allocated buffer or NULL if failure.
 */
typedef void * (*da_alloc_cbk) (
		int mod,	/**< The software module for which the allocation is performed. */
		void * param,	/**< The allocator parameter for instance for allocation context. */
		size_t len	/**< Required space. */
		);
/** Memory release callback.
 */
typedef void (*da_free_cbk) (
		int mod,	/**< The software module for which the free is performed. */
		void * param,	/**< The allocator parameter for instance for allocation context. */
		void * buf	/**< The memory buffer to release. */
		);

/** Simple allocator plugin definition used for dynamic array expansion. */
typedef struct da_allocator {
	da_alloc_cbk alloc_cbk;	/**< Allocation callback. */
	da_free_cbk free_cbk;	/**< Free callback. */
	void * param;	/**< The allocator parameter for instance for allocation context. */
} da_allocator_t;

extern const da_allocator_t * p_default_allocator;	/**< The default allocator usin DC_MALLOC & DC_FREE macros. */

/** Definition for standard untyped dynamic array. */
typedef struct dynamic_array {
	void * tab;	/**< The buffer containing array values. Note that this array always contain an extra value filled with 0 as a boundary */
	int nb;	/**< The user size of the array. */
	int size;	/**< The actual allocated size of \a tab. */
	size_t f_size;	/**< The size of the array entry in bytes. */
	int mod;	/**< The software module for which the allocation is performed. */
	const da_allocator_t * allocator;	/**< The allocator that will be used to increase the array as required. */
	int inc;	/**< The number of entries the array is increased every time more space is required. */
} dyn_array_t;

/** Macro initializing an existing dynamic array structure.
 */
#define INIT_DYN_ARRAY(da, fsize, m, dalloc, incr)	(da)->tab = NULL,(da)->nb = 0,(da)->size = 0,(da)->f_size = fsize,(da)->mod = m,(da)->allocator = dalloc,(da)->inc = incr

/** Add a new entry in the end of the dynamic array, increasing its size if
 * necessary.
 * @param tab A pointer on a dynamic array structure.
 * @return 0 if the entry could be created or -1 in case of failure.
 */
int da_new_entry(dyn_array_t * tab);

/** Creates a new entry at a specified position of the dynamic array, increasing
 * its size if necessary.
 * @param tab A pointer on a dynamic array structure.
 * @param index The position where the entry should be inserted.
 * @return 0 if the entry could be created or -1 in case of failure.
 */
int da_insert_entry(dyn_array_t * tab, int index);

/** Fills an empty dynamic array with an existing simple array.
 * The both arrays are expected to contain the same kind of entries.
 * @param tab A pointer to an empty dynamic array to fill.
 * @param entries The array which values will be copied.
 * @param nentries The number of entries of the supplied array \a entries.
 * @return 0 if no allocation error occured.
 */
int da_init_copy(dyn_array_t * tab, void* entries, int nentries);

/** Callback that may be called when an entry is removed from a dynamic array.
 * @param mod The software module for which the free is performed.
 * @param p_alloc The allocator that can be used to free contents.
 * @param p_entry A pointer on the removed entry.
 */
typedef void (*da_free_entry_cbk) (int mod, const da_allocator_t * p_alloc, void * p_entry);

/** Removes a dynamic array entry at a given index.
 * @param tab A pointer on a dynamic array structure.
 * @param index The position where the entry should be removed.
 * @param free_hook An optional callback called for the removed entry.
 */
void da_remove_entry(dyn_array_t * tab, int index, da_free_entry_cbk free_hook);

/** Macro that returns a pointer of a given index of a dynamic array.
 * @param da A pointer on a dynamic array structure.
 * @param index The requested position.
 * @return A pointer on the requested entry value.s
 */
#define GET_ENTRY(da, index) (void*)((char*)(da)->tab + ((index) * (da)->f_size))

/** Removes all entries dynamic array.
 * @param da A pointer on a dynamic array structure.
 * @param free_hook An optional callback called for the removed entry.
 */
void da_empty(dyn_array_t * da, da_free_entry_cbk free_hook);

/** Removes all entries dynamic array and frees the array buffer itself.
 * @param da A pointer on a dynamic array structure.
 * @param free_hook An optional callback called for the removed entry.
 */
void da_free(dyn_array_t * da, da_free_entry_cbk free_hook);

/*----------------------------------------------------------------------------*\
 *                          Dynamic Arrays Algorithms                         *
\*----------------------------------------------------------------------------*/

/** Callback that may be called when copy a dynamic array to another.
 * @param mod The software module for which the new entry is created.
 * @param dalloc Dynamic array allocator for potential entry related content
 * allocation .
 * @param dest_entry A pointer on the new entry.
 * @param src_entry A pointer on the original entry.
 * @param param Callback user data.
 * @return DC_TRUE when the copy went well and the process can continue.
 */
typedef DC_BOOL (*da_deep_copy_cbk) (int mod, const struct da_allocator * dalloc, void *dest_entry, void *src_entry, void * param);

/** Copies a dynamic array into another potentially deeply using a callback.
 * @param dest A pointer on the destination dynamic array structure. This array
 * must have the same type than \a src.
 * @param src A pointer on the original dynamic array structure.
 * @param dup_hook An optional callback that may be used to copy contents when
 * the entries contain pointers.
 * @param param Callback user data.
 * @return 0 when the copy went well.
 */
int da_copy(dyn_array_t * dest, dyn_array_t * src, da_deep_copy_cbk dup_hook, void * param);

/** Dynamic array browsing callback.
 * @param p_entry A pointer on the current entry.
 * @param param Callback user data.
 * @return DC_TRUE when the browsing must stop (found). DC_FALSE makes it
 * continue.
 */
typedef DC_BOOL (*da_browse_cbk) (void * p_entry, void * param);

/** Browses a dynamic array for any processing or search.
 * @param da A pointer on a dynamic array structure.
 * @param hook A callback that will be called for every entry or until one
 * returns DC_TRUE (found).
 * @return The index where the browsing stopped. Is equal to the number of
 * entries in the array if the browsing did not stop because of a matching.
 */
int da_browse(dyn_array_t * da, da_browse_cbk hook, void * param);

/** Entries comparison callback.
 * @param p_entry A pointer on the current entry.
 * @param key The key value of the new potential entry.
 * @return strcmp() semantics < 0 if entry < key and > 0 if entry > key.
 */
typedef int (*da_cmp_cbk) (void *p_entry, const void *key);

/** Insert a new entry in a sorted dynamic array.
 * @param da A pointer on a sorted dynamic array structure.
 * @param cbk A comparison callback that defines the order.
 * @param key The key value that will decide for new entry position.
 * @return The insertion index for the new entry or -1 if EOM.
 */
int da_add_entry_sorted(dyn_array_t * da, da_cmp_cbk cbk, void * key);

/** Retrieves an entry in a sorted dynamic array using dichotomy.
 * @param da A pointer on a sorted dynamic array structure.
 * @param cbk A comparison callback that defines the order.
 * @param key The searched key value.
 * @return The found index for the new entry or -1 if not found.
 */
int da_dichotomy_find(dyn_array_t * da, da_cmp_cbk cbk, const void * key);

/** Test is a dynamic array is included in another from a set point of view.
 * @param a A pointer to the first dynamic array structure to be tested for
 * inclusion.
 * @param b A pointer to the second dynamic array structure to be tested for
 * inclusion.
 * @param cbk A comparison callback used for strict equality only.
 * @return DC_TRUE if a is include in b. DC_FALSE else.
 */
DC_BOOL da_is_included(dyn_array_t *a, dyn_array_t *b, da_cmp_cbk compare);

/*----------------------------------------------------------------------------*\
 *                               Typed Dynamic Arrays                         *
\*----------------------------------------------------------------------------*/

/** Declaration macro for a typed dynamic array. This type of dynamic array is
 * preferred whenever possible for debug reasons.
 * @param name The short type identifier used to diversify the C type.
 * @param type The C type that the dynamic array will contain.
 */
#define DA_TYPED_DECL(name, type) struct da_##name{ \
	type * tab; \
	int nb; \
	int size; \
	size_t f_size; \
	int mod; \
	const struct da_allocator * allocator; \
	int inc; \
}

/** Usage macro for a typed dynamic array.
 * @param name The short type identifier used to diversify the C type and use
 * during typed dynamic array declaration.
 */
#define DA_TYPED(name) struct da_##name


/* Typed array macros */

/** Macro for static initialization of a typed dynamic array structure.
 * @param type The C type that the dynamic array will contain.
 */
#define DA_INITIALIZER(type, mod, inc)  {NULL, 0, 0, sizeof(type), mod, NULL, inc}

/** Macro initializing an existing dynamic array structure.
 * @param type The C type that the dynamic array will contain.
 */
#define DA_INIT(type, da, mod, dalloc, inc)	INIT_DYN_ARRAY(da, sizeof(type), mod, dalloc, inc)

/** Accessor macro for typed arrays (no need to cast result).
 * @param da A pointer on a typed dynamic array structure.
 * @param index The requested position.
 */
#define DA_GET(da, index) ((da)->tab + (index))

// Some compiler like gcc don't like cast of more generic function pointers.
#ifdef DC_NO_CBK_CAST
# define DA_FREE_CBK_CAST
# define DA_CMP_CBK_CAST
# define DA_DEEP_COPY_CBK_CAST
# define DA_BROWSE_CBK_CAST
#else
# define DA_FREE_CBK_CAST (da_free_entry_cbk)
# define DA_CMP_CBK_CAST (da_cmp_cbk)
# define DA_DEEP_COPY_CBK_CAST (da_deep_copy_cbk)
# define DA_BROWSE_CBK_CAST (da_browse_cbk)
#endif

/** Add an entry in a typed dynamic array and returns a pointer on the added entry. */
#define DA_ADD(da)  (da_new_entry((dyn_array_t *)da) ? NULL : DA_GET(da,(da)->nb - 1))

/** Insert an entry in a typed dynamic array and returns a pointer on the added entry. */
#define DA_INSERT(index)  (da_insert_entry((dyn_array_t *)da, index) ? NULL : DA_GET(da,index))

/** Removes an entry in a typed dynamic array. */
#define DA_REMOVE(da, index, fhook)  da_remove_entry((dyn_array_t *)da, index, DA_FREE_CBK_CAST fhook)

/** Add an entry in a sorted typed dynamic array. */
#define DA_ADD_SORTED(da, cbk, key)	(da_add_entry_sorted((dyn_array_t *)da, DA_CMP_CBK_CAST cbk, key))

/** Initialize a typed dynamic array with an existing array. */
#define DA_INIT_COPY(da, tab, nb)  da_init_copy((dyn_array_t *)da, tab, nb)

/** Releases totally a typed dynamic array. */
#define DA_FREE(da, fhook)  da_free((dyn_array_t *)da, DA_FREE_CBK_CAST fhook)

/** Empties a typed dynamic array. */
#define DA_EMPTY(da, fhook)  da_empty((dyn_array_t *)da, DA_FREE_CBK_CAST fhook)

/** Copy a typed dynamic array to another */
#define DA_COPY(dest, da, dhook, p)  da_copy((dyn_array_t *)dest, (dyn_array_t *)da, DA_DEEP_COPY_CBK_CAST dhook, p)

/** Browses a typed dynamic array. */
#define DA_BROWSE(da, hook, param)	da_browse((dyn_array_t *)da, DA_BROWSE_CBK_CAST hook, param)

/** Searches in a sorted typed dynamic array using dichotomy. */
#define DA_SORTED_FIND(da, cbk, key) (da_dichotomy_find((dyn_array_t *)da, DA_CMP_CBK_CAST cbk, key))

/** Test is a typed dynamic array is included in another from a set point of view. */
#define DA_IS_INCLUDED(a, b, cbk)	da_is_included((dyn_array_t *)a, (dyn_array_t *)b, DA_CMP_CBK_CAST cbk)


/*----------------------------------------------------------------------------*\
 *                         Specific Dynamic Arrays tools                      *
\*----------------------------------------------------------------------------*/
/* Usual DA types */

DA_TYPED_DECL(href,short);	/**< A predefined type of dynamic array for handle references */
DA_TYPED_DECL(str, char*);	/**< A predefined type of dynamic array for C strings */


/** Duplicates (strings contents included) a string dynamic array.
 * @param dest The destination string dynamic array.
 * @param src The original string dynamic array.
 * @return A pointer on \a dest if the copy went well. NULL else.
 */
DA_TYPED(str) * da_string_array_dup(DA_TYPED(str) * dest, DA_TYPED(str) * src);

/** Duplicates (strings contents included) a string dynamic array
 * (type array macro style).
 *
 */
#define DA_STRDUP(dest,src) da_string_array_dup((DA_TYPED(str) *)dest, (DA_TYPED(str) *)src)

#ifdef __cplusplus
}
#endif

#endif /*DYN_ARRAY_H_*/
