/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2091 $
|                     $Date: 2009-02-17 15:38:44 +0100 (mar, 17 fév 2009) $
\*============================================================================*/
/******************************************************************************\
 *                          Generic list utilities                            *
\******************************************************************************/
#ifndef LIST_UTILS_H_
#define LIST_UTILS_H_


/*----------------------------------------------------------------------------*\
 *                               Type definitions                             *
\*----------------------------------------------------------------------------*/

/** Generic node for simply-chained lists to include as 1st member in the
 * operational node so it can be retrieved using a cast.
 */
typedef struct scl_node {
	struct scl_node * next;
} scl_node_t, *p_scl_node_t;

/** Simply-chained list type definition. */
typedef scl_node_t scl_list_t;

/** Simply-chained list pointer type definition. */
typedef scl_node_t *p_scl_list_t;

/** Generic node for doubly-chained lists to include as 1st member in the
 * operational node so it can be retrieved using a cast.
 */
typedef struct dcl_node {
	struct dcl_node * next;
	struct dcl_node * previous;
} dcl_node_t, *p_dcl_node_t;

/** Doubly-chained list type definitions. */
typedef struct dcl_list {
	struct dcl_node * first;
	struct dcl_node * last;
} dcl_list_t, *p_dcl_list_t;

/** Callback type definition for the \a scl_find function
 * @param node The current node to test for matching.
 * @param cbk_data Callback data containing search information.
 * @return True if the node matches.
 */
typedef int (*scl_match_cbk_t)(p_scl_node_t node, void * cbk_data);

/** Callback type definition for the \a dcl_find function
 * @param node The current node to test for matching.
 * @param cbk_data Callback data containing search information.
 * @return True if the node matches.
 */
typedef int (*dcl_match_cbk_t)(p_dcl_node_t node, void * cbk_data);

/** Callback type definition for the \a dcl_insert function
 * @param node1 First node to test for comparison.
 * @param node2 Second node to test for comparison.
 * @return A positive value if node1 < node2, 0 if node1 = node2, negative if node1 > node2
 */
typedef int (*dcl_cmp_cbk_t)(p_dcl_node_t node1, p_dcl_node_t node2);

/*----------------------------------------------------------------------------*\
 *                         Simple chained list utilities                      *
\*----------------------------------------------------------------------------*/
/** Adds a node at the beginning of a simply-chained list
 * @param list A pointer on a simply-chained list.
 * @param node the node to add.
 */
void scl_add(p_scl_list_t list, p_scl_node_t node);

/** Gets the first node of a simply-chained list.
 * @param list A pointer on a simply-chained list.
 * @return A pointer to the first node of the list.
 */
#define scl_get_first(list) ((list)->next)

/** Gets a node of a simply-chained list at a specified position.
 * @param list A pointer on a simply-chained list.
 * @param index The index of the required node starting at 0.
 * @return A pointer to the required node or NULL if "out-of-bound".
 */
p_scl_node_t scl_get(p_scl_list_t list, int index);

/** Retrieves a node in a simply-chained list using an intrinsic criterion.
 * @param list A pointer on a simply-chained list.
 * @param comparator The callback function that will decide if a node matches.
 * @param cbk_data Data required by the \a comparator callback to process the
 * node.
 * @return A pointer to the required node or NULL if not found.
 */
p_scl_node_t scl_find(p_scl_list_t list, scl_match_cbk_t comparator, void * cbk_data);

/** Removes the first node of a simply-chained list.
 * @param list A pointer on a simply-chained list.
 * @return A pointer to the removed node.
 */
p_scl_node_t scl_remove(p_scl_list_t list);	// removes first

/** Test the simply-chained list for emptiness.
 * @param list A pointer on a simply-chained list.
 * @return A boolean value.
 */
#define scl_is_empty(list) ((list)->next == NULL)

/*----------------------------------------------------------------------------*\
 *                         Double chained list utilities                      *
\*----------------------------------------------------------------------------*/

/** Adds a node at the beginning of a doubly-chained list
 * @param list A pointer on a doubly-chained list.
 * @param node the node to add.
 */
void dcl_add_first(p_dcl_list_t list, p_dcl_node_t node);

/** Adds a node at the end of a doubly-chained list
 * @param list A pointer on a doubly-chained list.
 * @param node the node to add.
 */
void dcl_add_last(p_dcl_list_t list, p_dcl_node_t node);

/** Gets the first node of a doubly-chained list.
 * @param list A pointer on a doubly-chained list.
 * @return A pointer to the first node of the list.
 */
#define dcl_get_first(list) ((list)->first)

/** Gets the last node of a doubly-chained list.
 * @param list A pointer on a doubly-chained list.
 * @return A pointer to the first node of the list.
 */
#define dcl_get_last(list) ((list)->last)

/** Gets a node of a doubly-chained list at a specified position.
 * @param list A pointer on a doubly-chained list.
 * @param index The index of the required node starting at 0.
 * @return A pointer to the required node or NULL if "out-of-bound".
 */
#define dcl_get(list, pos) ((p_dcl_node_t)scl_get((p_scl_list_t)(list), pos))

/** Retrieves a node in a doubly-chained list using an intrinsic criterion.
 * @param list A pointer on a doubly-chained list.
 * @param comparator The callback function that will decide if a node matches.
 * @param cbk_data Data required by the \a comparator callback to process the
 * node.
 * @return A pointer to the required node or NULL if not found.
 */
#define dcl_find(list, comparator, cbk_data) ((p_dcl_node_t)scl_find((p_scl_list_t)(list), (scl_match_cbk_t)comparator, cbk_data))

/** Removes the specified node in a doubly-chained list.
 * @param list A pointer on a doubly-chained list.
 * @param node A pointer on the node to remove.
 */
void dcl_remove(p_dcl_list_t list, p_dcl_node_t node);

/** Inserts a node in a sorted doubly-chained list.
 * @param list A pointer on a doubly-chained list.
 * @param node The node to insert.
 * @param comparator The callback function that compare contiguous nodes to
 * decide where to insert the new node.
 */
void dcl_insert(p_dcl_list_t list, p_dcl_node_t node, dcl_cmp_cbk_t comparator);

/** Test the doubly-chained list for emptiness.
 * @param list A pointer on a simply-chained list.
 * @return A boolean value.
 */
#define dcl_is_empty(list) ((list)->first == NULL)


#endif /*LIST_UTILS_H_*/
