/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2155 $
|                     $Date: 2009-03-05 18:47:59 +0100 (jeu, 05 mar 2009) $
\*============================================================================*/

/** \mainpage DPWSCore Portability Layer Reference
 * The documented functions should be implemented for a specific target platform
 * so that to provide the following features required by the DPWS stack:
 * <ul>
 * <li>OS facilities (dcDCPL_Os.h),</li>
 * <li>Network adapter information access (dcDCPL_Net.h),</li>
 * <li>High-level IP Sockets (dcDCPL_Socket.h),</li>
 * <li>Memory allocator (dcDCPL_Mem.h),</li>
 * <li>Reactor support functions (sockets monitoring in dcDCPL_Reactor.h).</li>
 * </ul>
 */

/***************************************************************************//**
* \file
* DPWSCore OS portability layer.
*******************************************************************************/

#ifndef DCPL_OS_H_
#define DCPL_OS_H_

#include "stdint.h"

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************//**
* \name Clock management
* @{
*******************************************************************************/

/** Return a system clock value.
 * @return A time value in milliseconds that is not necessarily absolute (since
 * the origin is generally the boot time). Note that of course, the
 * implementation is responsible to provide the best resolution possible without
 * damaging the system performance, knowing that the minimum resolution expected
 * is 10ms.
 */
uint64_t dcpl_get_clock(void);

/** Retrieves the time since 00:00:00 GMT, January 1, 1970.
 * If no absolute clock is available on the platform this API should return
 * a boot relative time value. Note that of course, the implementation is
 * responsible to provide the best resolution possible without damaging the
 * system performance, knowing that the minimum resolution expected is 10ms.
 * @param[out] s The number of seconds since 00:00:00 GMT, January 1, 1970.
 * @param[out] ns The remainder in nanoseconds for the returned time.
 * @return A DCPL error code.
 */
int dcpl_get_time(uint32_t * s, uint32_t * ns);


/** Makes the current thread sleep for a given duration.
 * @param dly The number of milliseconds for which the thread must be stopped.
 * @return DCPL_OK or a DCPL error code.
 */
int dcpl_sleep(uint64_t dly);

// @}

/***************************************************************************//**
* \name Mutexes management
* @{
*******************************************************************************/

/**
 * An implementation dependent type forward declaration for the mutex (can be
 * replaced by a structure by the implementation).
 */
typedef void* dcpl_mutex_t;

/** The number of mutexes used by the dpws stack that may be used to size
 * static memory for the mutexes.
 */
#define DC_LOCK_NB 8	// 7 for dpws + 1 for dyndepl

/**
 * Creates a new mutex for thread synchronization.
 * Note that the mutex is not expected to be reentrant.
 * @return A pointer to the implementation type for the "thread" mutex.
 */
dcpl_mutex_t * dcpl_mutex_init(void);

/**
 * Deletes a mutex created using \a dcpl_mutex_init.
 * @param mutex A pointer to the "thread" mutex.
 * @return DCPL_OK or a DCPL error code.
 */
int dcpl_mutex_delete(dcpl_mutex_t * mutex);

/**
 * Takes a mutex lock.
 * @param mutex A pointer to the "thread" mutex.
 * @return DCPL_OK or a DCPL error code. Since the mutex is not expected to be
 * reentrant, it may return DCPL_ILLEGAL_OPERATION if it is called twice in a
 * raw.
 */
int dcpl_mutex_lock(dcpl_mutex_t * mutex);

/**
 * Releases a mutex lock.
 * @param mutex A pointer to the "thread" mutex.
 * @return DCPL_OK or a DCPL error code.
 */
int dcpl_mutex_unlock	(dcpl_mutex_t * mutex);

// @}

/***************************************************************************//**
* \name Process management
* @{
*******************************************************************************/

/** Return a process ID used for UUID generation diversification.
 * @return A numeric process ID. On a platform without processes, return any
 * value.
 */
uint32_t dcpl_process_id(void);

// @}

#ifdef __cplusplus
}
#endif

#endif /*DCPL_OS_H_*/
