/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|                      ->>  Copyright 2008 Odonata <<-                         |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 1772 $
|                     $Date: 2008-10-09 11:57:03 +0200 (jeu., 09 oct. 2008) $
\*============================================================================*/

/***************************************************************************//**
* \file
* DPWSCore Reactor support services.
*******************************************************************************/

#ifndef DCDCPL_REACTOR_H_
#define DCDCPL_REACTOR_H_

#include "dcpl_target.h"
#include "dcDCPL_Error.h"

#include "dcDPWS_Reactor.h"

#ifdef __cplusplus
extern "C" {
#endif


/** Initialize platform-specific features (only) for a reactor, especially
 * interruption specific data.
 * @param reactor The reactor structure.
 * @param family The DCPL address family for the wake up socket if this method
 * is used for interrupting the reactor.
 * @param port The (UDP) port number for the wake up socket if this method is
 * used for interrupting the reactor.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_reactor_init(struct reactor * reactor, int family, uint16_t port, struct dcpl_error * error);

/** Uninitialize platform-specific features (only) for a reactor initialized
 * with \a dcpl_reactor_init, especially interruption specific data.
 * @param reactor The reactor structure.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_reactor_free(struct reactor * reactor, struct dcpl_error * error);

/** Platform-specific function for multiplexing I/O and timer. Corresponds to the
 * BSD select semantics, with interrupt capabilities.
 * @param reactor The reactor structure containing all sockets.
 * @param next_date the latest absolute date in ms (compared to the platform RT
 * clock) when the call must return. A 0 value means no limit.
 * @param[out] active_list Contains all active sockets.
 * @param[out] error A structure to get detailed error information.
 * @return A positive value indicating the number of active sockets, 0 when a
 * timeout occurs, and a negative value containing the error number on failure.
 */
int dcpl_reactor_wait(struct reactor * reactor, uint64_t next_date, struct reactor_item ** active_list, struct dcpl_error * error);

/** Platform-specific function for interrupting the wait function, when changes
 * in the multiplexed objects set must be performed.
 * The default technique is to use a wake up socket sending a datagram to
 * the \a dcpl_reactor_wait implementation. However implementors are encouraged
 * to used other platform-specific techniques...
 * @param reactor The reactor structure.
 * @param[out] error A structure to get detailed error information.
 * @return DCPL_OK or any DCPL error code.
 */
int dcpl_reactor_interrupt(struct reactor * reactor, struct dcpl_error * error);

#ifdef __cplusplus
}
#endif

#endif /*DCDCPL_REACTOR_H_*/
