/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2229 $
|                     $Date: 2009-03-30 15:16:20 +0200 (lun, 30 mar 2009) $
\*============================================================================*/

/***************************************************************************//**
* \file
* Data types used by the DPWSCore API.
* Contains especially definitions for the runtime structures used by the API.
*******************************************************************************/

#ifndef DCTYPES_H_
#define DCTYPES_H_

#include "dc/dc_Ctypes.h"


/***************************************************************************//**
* \name XML Types
* @{
*******************************************************************************/

#ifndef QNAME_EQUALS
# define QNAME_EQUALS(ns1, ln1, ns2, ln2) (!strcmp(ln1, ln2) && (ns1 ? (ns2 ? !strcmp(ns1, ns2) : 0) : ns2 == NULL))	/**< Compares to QNames given as strings. */
#endif
#ifndef QNAME_STRUCT_EQUALS
#define QNAME_STRUCT_EQUALS(qn1, qn2) QNAME_EQUALS(qn1->ns, qn1->lname, qn2->ns, qn2->lname)	/**< Compares to QNames given as qname_t structs. */
#endif
#ifndef QNAME_EQUALS_WILDCARD
# define QNAME_EQUALS_WILDCARD(ns, ln, ref_ns, ref_ln) (!ref_ln || (!strcmp(ln, ref_ln) && (ns ? (ref_ns ? !strcmp(ns, ref_ns) : 0) : ref_ns == NULL)))	/**< Compares to QNames given as strings but returns true when the reference is NULL. */
#endif
#ifndef QNAME_STRUCT_EQUALS_WILDCARD
# define QNAME_STRUCT_EQUALS_WILDCARD(qn, ref_qn) QNAME_EQUALS_WILDCARD(qn->ns, qn->lname, ref_qn->ns, ref_qn->lname)	/**< Compares to QNames given as qname_t structs but returns true when the reference is NULL. */
#endif

/**
 * An XML qualified name used for instance for WS-Discovery port types.
 * <em>Note</em>: Already defined in EPX. Double waiting that dpws relies
 * on EPX.
 */
typedef struct qname
{
	char * ns;	/**< Namespace. */
	char * lname;	/**< Local name. */
} qname_t;

/**
 * An XML qualified name with a prefix specification used for instance
 * for WS-Discovery port types.
 */
typedef struct prefixed_qname
{
	struct qname qname;	/**< Qualified name. */
	char *prefix;	/**< Prefix to use for the qname namespace. */
} prefixed_qname_t;

/**
 * An association between a prefix and an URI for XML namespaces.
 */
typedef struct prefix_def {
	char * ns_prefix;	/**< A short namespace alias */
	char * ns_uri;	/**< The identifying namespace URI */
} prefix_def_t;

/**
 * Used for API parameters that are localized using the xml:lang attribute.
 */
typedef struct localized_string
{
	char * s;	/**< A string in the language specified by the lang attribute. */
	char * lang;	/**< The values used for this field are defined in the IETF RFC 3066. */
} localized_string_t;

// forward declaration
struct dynamic_array;

// @}

/***************************************************************************//**
* \name WS-Addressing Types
* @{
*******************************************************************************/
#ifdef WITH_WSMAN
# include "dc/dc_WsmanTypes.h"
#endif

/**
 * WS-Addressing version constants
 */
struct wsa_version {
	char* wsa_uri;  /**< WS-Addressing namespace URI */
	char* wsa_anonymous_uri; /**< WS-Addressing anonymous URI */
	char* wsa_fault_uri; /**< WS-Addressing fault URI */
};

struct dpws_protocols;
/**
 * WS-Adressing endpoint reference
 * Runtime structure that contains WS-addressing endpoint reference data.
 * Used for the source, reply to and fault to WSA header fields.
 */
struct wsa_endpoint_ref {
	char* address;	/**< WSA endpoint Address element */
	char* subscription_id;	/**< WS-Eventing subscription identifier reference parameter */
#ifdef WITH_WSMAN
	struct wsman_ref_params wsman_params;	/**<  WS-Management reference parameters */
#endif
	struct dpws_protocols *  dpws_version; /**< DPWS version */
};

// @}

/***************************************************************************//**
* \name WS-Discovery Types
* @{
*******************************************************************************/

/**
 * WS-Discovery Probe scope match rule.
 */
typedef enum {
	WSD_MATCH_BY_UNSPECIFIED = 0,	/**< Do not specify a match rule meaning RFC2396. */
	WSD_MATCH_BY_RFC2396 = 1,		/**< URI based comparison defined in the WS-Discovery specification. */
	WSD_MATCH_BY_UUID = 2,			/**< For UUID URIs. Binary comparison. Optional in DPWS */
	WSD_MATCH_BY_LDAP = 3,			/**< For LDAP URIs. The comparison based on RFC2253. Optional in DPWS. Unsupported in the current version of the toolkit.  */
	WSD_MATCH_BY_STRCMP = 4			/**< Simple case sensitive string comparison. */
} scope_match_rule_t;

/**
 * WS-Discovery filter used for probe match.
 */
typedef struct discovery_filter {
	qname_t* types;		/**< QNames array that a device should implement at least to match. */
	int nb_types;		/**< Number of types in the filter. */
	char **	scopes;		/**< URI array that a device should have in its metadata at least to match. */
	int nb_scopes;		/**< Number of types in the filter. */
	scope_match_rule_t match_rule;		/**< The match rule that the device should use to match scopes. */
} discovery_filter_t;

/**
 * A specification on how the API should use the local cache.
 */
typedef enum {
	/**
	 * The defaut mode where the cache will be checked first before sending a
	 * probe on the network if retrieved endpoints are not as many as asked.
	 */
	WSD_LOOKUP_MODE_CACHE_FIRST = 0,
	/**
	 * Only look in the local cache whatever be the number of endpoints
	 * retrieved.
	 */
	WSD_LOOKUP_MODE_CACHE_ONLY = 1,
	/**
	 * Forces a cache update with a network probe.
	 * Note that invalid endpoint lying in the cache (because no Bye message was
	 * received) are not discarded an may be retrieved.
	 */
	WSD_LOOKUP_MODE_FORCE_NETWORK = 2
} cache_mode_t;


// @}

/***************************************************************************//**
* \name DC Runtime structures
* @{
*******************************************************************************/

/* The following definitions removes non-portable (mainly HTTP) of gSOAP runtime */
#define WITH_NOIO
#define WITH_NOHTTP
#define WITH_IPV6
#define HAVE_CONFIG_H
#include "stdsoap2.h"

/*
 * A structure holding all the constants related to the DPWS protocols. This is used to
 * support multiple versions of the protocols
 */
struct dpws_protocols {
	struct wsa_version * wsa_version;
	char* wdp_uri;
	char* wdp_model_uri;
	char* wdp_device_uri;
	char* wdp_hosting_uri;
	char* wdp_relationship_uri;
	char* wdp_wse_filtering_dialect;
	char* wsd_urn;
	char* wsd_uri;
	char* wsd_hello_action;
	char* wsd_bye_action;
	char* wsd_probe_action;
	char* wsd_resolve_action;
	char* wsd_probematches_action;
	char* wsd_resolvematches_action;
	char* wsd_match_by_rfc2396_uri;
	char* wsd_match_by_uuid_uri;
	char* wsd_match_by_ldap_uri;
	char* wsd_match_by_strcmp_uri;
	char* wsm_uri;
	char* wst_uri;
	char* wst_get_action;
	char* wst_get_response_action;
	char* wse_uri;
	char* wse_subscribe_action;
	char* wse_subscribe_response_action;
	char* wse_subscriptionend_action;
	char* wse_renew_action;
	char* wse_renew_response_action;
	char* wse_unsubscribe_action;
	char* wse_unsubscribe_response_action;
	char* wse_getstatus_action;
	char* wse_getstatus_response_action;
	struct Namespace * default_namespaces;
	struct Namespace * discovery_snd_namespaces;
	struct Namespace * transfer_snd_namespaces;
	struct Namespace * eventing_snd_namespaces;
};

/* forward declarations */
struct wse_sm_plugin;
struct na_info;

struct reactor;
struct reactor_item;
struct transport_fns;

/** DPWS toolkit runtime structure
 * Main DPWS API runtime data structure.
 * The fields in the dpws structure provide support for the WS-Addressing
 * specification. Most of those fields are set automatically or through API
 * calls by the DPWS toolkit, and should therefore be accessed only for
 * reading in user-defined client or server code. The notable exception is
 * the fault_to field, which should be set by the user before calling a proxy
 * function, if required. A future version of the toolkit may add parameters
 * to the proxy function signatures to pass this optional information to the
 * toolkit runtime.
 * As mentioned previously, it is always possible to retrieve the soap
 * structure embedded within the dpws structure, and use it to call gSOAP
 * runtime functions. It is also possible to retrieve the dpws structure
 * corresponding to a soap structure. The dpws_dpws2soap(dpws) and
 * dpws_soap2dpws(soap) macros are available to cast from one structure to
 * the other.
 */
struct dpws {
	struct soap soap;	/**< The gSOAP runtime data structure */
	char* to;	/**< WSA header To element */
	struct wsa_endpoint_ref *source;	/**< This fields stores the source endpoint reference information set through the API and that is used to fill the WSA header source tag (from field) before every sending. */
	char* action;	/**< WSA header Action element */
	char* message_id;	/**< WSA header MessageID element */
	char* relates_to;	/**< WSA header RelatesTo element */
	struct wsa_endpoint_ref from;	/**< WSA header From element */
	struct wsa_endpoint_ref reply_to;	/**< WSA header ReplyTo element */
	struct wsa_endpoint_ref fault_to;	/**< WSA header FaultTo element */
	unsigned long instance_id;	/**< WS-Discovery application sequence instance id */
	unsigned short msg_nb;	/**< WS-Discovery application sequence message number */
	char* subscription_id;	/**< WS-Eventing subscription identifier */
	unsigned short status;	/**< internal DPWS processing status */
	int err;	/**< API error code */
#ifdef WITH_WSMAN
	struct wsman_headers wsman_headers;	/**< WS-Management SOAP headers */
	struct wse_sm_plugin * eventing_sm; /**< WS-Eventing subscription manager plugin */
#endif
	size_t tmp_buf_count;	/**< internal use */
	int (*backup_send_f)(struct soap *, const char *, size_t);	/**< internal use */
	int err_detail_type;	/**< internal use. Contains the XML type to pass to gSOAP-generated marshalling code so that to produce an error detail in a fault. */
	void * err_detail_data;	/**< internal use. Contains the pointer on a serializable C structure to pass to gSOAP-generated marshalling code so that to produce an error detail in a fault. */
	char * err_detail_data_elt;	/**< internal use. Contains the name of the tag to pass to gSOAP-generated marshalling code so that to produce an error detail in a fault. */
	short href_endpoint;	/**< internal use */
  	struct Namespace *namespaces;	/**< internal use */
  	struct dynamic_array * proxyHandles;	/**< internal use */
	void (*fnotification_failure)(struct dpws *, short, struct wsa_endpoint_ref *); /**< callback for event notification failures */
	struct reactor * reactor;   /**< Global reactor used for listening to incoming requests and scheduling global actions */
	struct reactor_item * active_item;  /**< Current active reactor items, used to encapsulate the reactor API in existing DPWS API */
	void * transport_data;      /**< Pointer to the transport data used for the current request. This is considered as opaque when transport_fns is used.*/
	struct transport_fns * transport_fns;  /**<  Pointer to callback functions used for external Web server integration. NULL when the internal Web server is used. */
	soap_mode user_mode;    /**< Used to restore gSOAP user mode after request handling */
	struct dpws_protocols * protocols;  /**< Pointer to the structure holding the constants for the current version of the DPWS protocol */
	struct wsa_version * wsa_version;  /**< Pointer to the structure holding the constants for the current version of WS-Adressing */
};

/**
 * Macro for transforming a DPWS runtime structure into a gSOAP runtime
 * structure.
 * @param dpws a DPWS runtime structure
 */
#define dpws_dpws2soap(dpws) (&dpws->soap)

/**
 * Macro for transforming a gSOAP runtime structure into a DPWS runtime
 * structure (provided the gSOAP structure is in fact embedded in a DPWS
 * structure).
 * @param soap a gSOAP runtime structure
 */
#define dpws_soap2dpws(soap) ((struct dpws *)(soap))

// @}

/***************************************************************************//**
* \name WS-MetadataExchange structures
* @{
*******************************************************************************/

/**
 * Container for WS-MedataExchange APIs results. This particular one
 * relates to the http://schemas.xmlsoap.org/ws/2005/05/devprof/ThisModel
 * dialect.
 */
struct model_info
{
	char * manufacturer;	/**< See [DPWS]. Only one is provided. If the metadata message returns several, the one with default language or in english is assumed. */
	char * manufacturer_url;	/**< Cf. [DPWS]. */
	char * model_name;	/**< See [DPWS]. Only one is provided. If the metadata message returns several, the one with default language or in english is assumed. */
	char * model_number;	/**< See [DPWS]. */
	char * model_url;	/**< See [DPWS]. */
	char * presentation_url;	/**< See [DPWS]. */
};

/**
 * Container for WS-MedataExchange APIs results. This particular one
 * relates to the http://schemas.xmlsoap.org/ws/2005/05/devprof/ThisDevice
 * dialect.
 */
struct device_info
{
	char * friendly_name;	/**< See [DPWS]. Only one is provided. If the metadata message returns several, the one with default language or in english is assumed. */
	char * firmware_version;	/**< See [DPWS]. */
	char * serial_number;	/**< See [DPWS]. */
};

/**
 * Container for WS-MedataExchange APIs results. This particular one relates
 * to the http://schemas.xmlsoap.org/wsdl dialect.
 */
struct wsdl_info
{
	char * target_ns;	/**< The target namespace for the web service endpoint. */
	char * location;	/**< An URL that enables WDSL location for the service. Note that the current version of the toolkit does not support the inline retrieval of WSDL. */
};

/**
 * Container for WS-MedataExchange APIs results. This particular one relates
 * to the http://schemas.xmlsoap.org/ws/2005/05/devprof/Relationship dialect.
 */
struct service_info
{
	char * service_id;	/**< The service identifier (URI). */
	struct wsa_endpoint_ref** endpoints;	/**< A list of endpoint references for the service. */
};

/**
 * Container for WS-MedataExchange APIs results. This particular one relates
 * to all dialects supported by the toolkit.
 */
struct device_metadata {
	struct wsa_endpoint_ref** host_endpoints;	/**< A list of endpoint references for the device. */
	struct model_info model_info;	/**< The device model DPWS information. */
	struct device_info device_info;	/**< The device own information. */
	struct service_info** services;	/**< A list of the services hosted by the device. */
	struct wsdl_info** wsdls;	/**< List of WSDL files for the device. */
};

/**
 * Container for WS-MedataExchange APIs results. This particular one relates
 * to all dialects supported by the toolkit.
 */
struct service_metadata {
	struct wsa_endpoint_ref** host_endpoints;	/**< A list of endpoint references for the device. */
	struct service_info* service_info;	/**< The service information. */
	struct wsdl_info** wsdls;	/**< List of WSDL files for the device. */
};

/**
 * Container for WS-Eventing subscription end event. This event is sent when a
 * subscription manager terminates abnormally a subscription.
 */
struct event_end_info
{
	struct wsa_endpoint_ref * subscription_manager;	/**< The endpoint reference of the subscription manager that cancels the subscription. */
	char * status;	/**< See [WS-Eventing]. */
	char * reason;	/**< See [WS-Eventing]. */
};

// @}

/***************************************************************************//**
* \name Configuration API types
* @{
*******************************************************************************/

/**
 * The prototype of generated function that must be associated to a service class
 * so that it can process web services invocations.
 */
typedef int (*dispatch_cbk)(
	struct dpws * dpws	/**< A DPWS runtime structure for message processing. */
);

/**
 * The prototype of callback that can be set on the toolkit to receive Hello or
 * Bye events.
 */
typedef void (*discovery_cbk) (
	/**
	 * This DPWS runtime structure is passed to be able to perform cache local API
	 * call.
	 * <b>Warning !</b> API that can generate a network message are prohibited.
	 * For instance, one can retrieve the UUID, the types or the scopes of the device
	 * since they must be in the local cache whereas retrieving information like
	 * metadata (model, etc) should use a distinct dpws runtime structure.
	 */
	struct dpws * dpws,
	/**
	 * The handle of the new or leaving device proxy.
	 */
	short href_dev_proxy
);

/**
 * The prototype of callback that can be set on the toolkit to be warned that
 * someone has changed a discovery device metadata version using hot-configuration
 * features.
 */
typedef void (*metadata_cbk) (
	short href_device	/**< The handle of the device proxy which metadata has changed. */
);

/**
 * The prototype of callback that can be set on the toolkit to receive faults that
 * have not been provoked by a local stub use.
 */
typedef int (*fault_cbk)(
		struct dpws * dpws	/**< The DPWS runtime structure that processes the fault */
);

/**
 * The prototype of callback that can be set on the toolkit to be warned that
 * a WS-Eventing subscription has expired in unexpected manner.
 */
typedef void (*event_end_cbk) (struct event_end_info *);

/**
 * The stack will select the interface using the adapter friendly name.
 */
#define DC_NETIF_SELECT_NAME	0

/**
 * The stack will select the interface using its MAC address.
 * The format used for the filter is an hexadecimal strings separated by '-' or
 * ':'.
 */
#define DC_NETIF_SELECT_MAC		1

/**
 * The stack will select the IPv6 interface number.
 */
#define DC_NETIF_SELECT_NUMBER	2


/**
 * Structure used as a filter criteria for network interfaces.
 */
typedef struct dc_netif_filter {
	unsigned char filter_type;	/**< One of DC_NETIF_SELECT_XXX values. */
	union {
		uint32_t itf_nb;	/**< The IPv6 interface index */
		char * buf;	/**< The name or MAC address of the interface */
	} value;
} dc_netif_filter_t;

/**
 * Any protocol.
 */
#define DC_PROTO_ANY	0

/**
 * IP v4 protocol.
 */
#define DC_PROTO_INET	1

/**
 * IP v6 protocol.
 */
#define DC_PROTO_INET6	2

/**
 * Structure used as a filter criteria for IP addresses.
 * The criteria are cumulative.
 */
typedef struct dc_ip_filter {
	dc_netif_filter_t * netif;	/**< Interface filter. If NULL, consider all. */
	DC_BOOL include_loopback;	/**< If true, the loopback interface is considered by the filtering */
	int proto;		/**< Protocol (DC_PROTO_ANY, DC_PROTO_INET, DC_PROTO_INET6) */
	int nb_addrs;		/**< The numbers of addresses on \a addrs */
	char ** addrs;		/**< An array of explicit addresses in their textual form to be retrieved */
} dc_ip_filter_t;

// @}

#endif /*DCTYPES_H_*/
