/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2215 $
|                     $Date: 2009-03-19 15:44:46 +0100 (jeu, 19 mar 2009) $
\*============================================================================*/

/******************************************************************************\
 *                              COMMON DYNAMIC DATA                           *
\******************************************************************************/
#include <string.h>
#include "dcDPWS_Cache.h"
#include "dcDPWS_Metadata.h"
#include "dcCOMN_Handle.h"
#include "dcDCPL_Os.h"
#include "dcDPWS_Utils.h"
#include "dcDPWS_Memory.h"
#include "dcDPWS_Network.h"
#include <stdlib.h>

/*----------------------------------- Data -----------------------------------*/

dcpl_mutex_t * cache_lock = NULL;

handlePool_t cachePool;

struct dpws_cache_config cache_cfg = {DA_INITIALIZER(discovery_filter_t, DC_MEM_CACHE, 2), 0xFFFF, NULL, NULL};

/*----------------------------------- Types ----------------------------------*/

struct lookup_info {
	discovery_filter_t * filter;
	int f_size;
	struct dpws * dpws;
	int max_result;
	int count;
	DA_TYPED(href) result;
};

struct grab_service_info {
	struct dpws * dpws;
	DA_TYPED(href) * grabbed;
	struct qname type;
};

DA_TYPED_DECL(qnp, qname_t *);

/*------------------------- Static Functions prototypes ----------------------*/

static int invalidate_device_proxy(struct device_proxy * device);
static void free_qname_cbk(int mod, da_allocator_t * p_alloc, qname_t * qn);
static void string_free_cbk(int mod, da_allocator_t * p_alloc, char **s);
static int service_free(struct service_proxy *service);
static void service_release(int mod, da_allocator_t * p_alloc, short * href);
static void clean_hosted_services(struct device_proxy * device);
static int device_content_free(struct device_proxy * device);
static int device_free(struct device_proxy * device);
static DC_BOOL clone_qname (int mod, struct da_allocator * allocator, struct qname *dest, struct qname *src, void * dummy);
static DC_BOOL url_match_protocol(int protocols, char * url);
static DC_BOOL filter_XAddrs_cbk (char ** pXAddr, DA_TYPED(str) * collected_addrs);
static int overwrite_entry(struct device_proxy * dest, struct dpws * dpws, struct wsd__HelloType *source);
static int qname_struct_equals(struct qname *type1, struct qname *type2);
static DC_BOOL device_match(DA_TYPED(qn) * types, DA_TYPED(str) * scopes,
		int match_rule, DA_TYPED(qn) *dev_types, DA_TYPED(str) *dev_scopes);
static int releaseHandleHook(handle_s *handleInfo);
static DC_BOOL is_matching_proxy(DA_TYPED(qn) *types, struct qname * type);
static DC_BOOL match_device_by_discovery(short type, struct device_proxy *device, struct lookup_info *info);
static short * lookup(struct dpws * dpws, discovery_filter_t * filter, int f_size, int *res_size);
static struct wsa_endpoint_ref * create_EPR(struct dpws * dpws, char * Xaddr);
static DC_BOOL grab_service(short * href, struct grab_service_info * grabInfo);
static DC_BOOL find_service_hook(short * href, char * id);
static DC_BOOL compare_proxies_id(short type, void * p_object, char * id);
static DC_BOOL string_equals(char ** s1, char * s2);
static DC_BOOL lookup_proxy_addresses(short type, void * p_object, char * id);
static void discovery_filter_free_cbk(int mod, const da_allocator_t * p_alloc, discovery_filter_t *filter);
static int dup_disc_filter(discovery_filter_t * dest, discovery_filter_t *src);
static DC_BOOL match_device_by_id(short type, struct device_proxy *device, char *id);
static struct wsa_endpoint_ref ** get_EPRs(struct dpws * dpws, short href, int * res_size);
static DC_BOOL release_proxy(short * href, void * dummy);
static int invalidate_device_proxy(struct device_proxy * device);

/*----------------------------------------------------------------------------*\
 *                               PRIVATE STUFF                                *
\*----------------------------------------------------------------------------*/
static void free_qname_cbk(int mod, da_allocator_t * p_alloc, qname_t * qn)
{
	p_alloc->free_cbk(mod, p_alloc->param, qn->ns);	// Allocator not used for trace ?
	p_alloc->free_cbk(mod, p_alloc->param, qn->lname);
}

static void string_free_cbk(int mod, da_allocator_t * p_alloc, char **s)
{
	p_alloc->free_cbk(mod, p_alloc->param, *s);
}

static int service_free(struct service_proxy *service)
{
	DA_FREE(&service->super.types, free_qname_cbk);
	DC_FREE(DC_MEM_CACHE, service->id);
	DA_FREE(&service->endpoint_references, string_free_cbk);
	DC_FREE(DC_MEM_CACHE, service);
	return DPWS_OK;
}

static void service_release(int mod, da_allocator_t * p_alloc, short * href)
{
	struct service_proxy * service = (struct service_proxy *)getObject(&cachePool, *href);
	if (service) {
		service->host = NULL;
		releaseHandle(&cachePool, *href);
	}
}

static void clean_hosted_services(struct device_proxy * device)
{
	DA_FREE(&device->hosted_services, service_release);
	device->hosted_retrieved = DC_FALSE;
}

static int device_content_free(struct device_proxy * device)
{
	DA_FREE(&device->super.types, free_qname_cbk);
	dpws_endpoint_ref_content_free(&device->superEPR.super);
	DA_FREE(&device->scopes, string_free_cbk);
	DA_FREE(&device->transport_addresses, string_free_cbk);
	clean_hosted_services(device);
	return DPWS_OK;
}

static int device_free(struct device_proxy * device)
{
	device_content_free(device);
	DC_FREE(DC_MEM_CACHE, device);
	return DPWS_OK;
}

static DC_BOOL clone_qname (int mod, struct da_allocator * allocator, struct qname *dest, struct qname *src, void * dummy)
{
	if (!(dest->ns = DC_STRDUP(mod, src->ns)) && src->ns)
		return DC_FALSE;
	if (!(dest->lname = DC_STRDUP(mod, src->lname)) && src->lname)
		return DC_FALSE;
	return DC_TRUE;
}

/* Keep in the cache only addresses that mach the toolkit mode.*/
static DC_BOOL url_match_protocol(int protocols, char * url)
{
	char *p = url, *p2;

	for(; *p && *p == '/'; p++);

	if (*p)
	{
		p++;
		if (*p != '/')
			return DC_TRUE;	// Not an URL: consider it matches

		// look for potential @
		p++;
		p2 = p;
		for(; *p2 && *p2 == '@'; p2++);
		if (*p2)
			p = p2 + 1;

		if ((*p = '['))	// IPv6
			return (protocols & DCPL_AF_INET6) ? DC_TRUE : DC_FALSE;
		else	// IPv4 or hostname
		{
			int segment_count = 1, car_count = 0;

			for(; *p && *p != '/' && *p != '?'; p++)
			{
				car_count++;
				if (car_count > 3 || !IS_DIGIT(*p))
					return DC_TRUE;	// Not an IPv4 address: consider it matches
				if (*p == '.')
				{
					segment_count++;
					car_count = 0;
				}
			}

			// IPv4
			if (segment_count != 4)
				return DC_TRUE;	// Not an IPv4 address: consider it matches

			return (protocols & DCPL_AF_INET) ? DC_TRUE : DC_FALSE;
		}
	}
	else
		return DC_TRUE;	// Not an URL: consider it matches
}

static DC_BOOL filter_XAddrs_cbk (char ** pXAddr, DA_TYPED(str) * collected_addrs)
{
	if (url_match_protocol(dc_network_mode, *pXAddr))
	{
		char ** pstr = DA_ADD(collected_addrs);
		if (!pstr || !(*pstr = DC_STRDUP(DC_MEM_CACHE, *pXAddr)))
			return DC_TRUE;
	}
	return DC_FALSE;
}

static int overwrite_entry(struct device_proxy * dest, struct dpws * dpws, struct wsd__HelloType *source)
{
	int ret = DPWS_OK;
	DC_ERROR_ASSERT(!DA_COPY(&dest->super.types, source->Types, clone_qname, NULL), DPWS_ERR_EOM);

	dest->superEPR.status = CACHED;
	dpws_endpoint_ref_copy(&dest->superEPR.super, source->wsa__EndpointReference);
	dest->superEPR.super.dpws_version = dpws->protocols;
	dest->metadata_version = source->MetadataVersion;
	dest->message_number = dpws->msg_nb;
	dest->instance_id = dpws->instance_id;

	DA_INIT(char *, &dest->scopes, DC_MEM_CACHE, p_default_allocator, 1);
	if (source->Scopes && source->Scopes->__item->nb > 0)
		DC_ERROR_ASSERT_ALLOC(DA_STRDUP(&dest->scopes, source->Scopes->__item));

	DA_INIT(char *, &dest->transport_addresses, DC_MEM_CACHE, p_default_allocator, 1);
	if (source->XAddrs && source->XAddrs->nb > 0) {
		// only keep URLs for the toolkit mode
		DC_ERROR_ASSERT_ALLOC(DA_BROWSE(source->XAddrs, filter_XAddrs_cbk, &dest->transport_addresses) == source->XAddrs->nb);
	}

	dest->hosted_retrieved = DC_FALSE;
	DA_INIT(short, &dest->hosted_services, DC_MEM_CACHE, p_default_allocator, 2);

	return DPWS_OK;

DC_FUNC_ERROR
	device_content_free(dest);
	return ret;
}

static int qname_struct_equals(struct qname *type1, struct qname *type2)
{
	return QNAME_STRUCT_EQUALS(type1, type2);
}

static DC_BOOL device_match(DA_TYPED(qn) * types, DA_TYPED(str) * scopes,
		int match_rule, DA_TYPED(qn) *dev_types, DA_TYPED(str) *dev_scopes)
{
	return probe_match((dyn_array_t *)types, scopes, match_rule,
			(dyn_array_t *)dev_types, dev_scopes, (da_cmp_cbk)qname_struct_equals);
}

static int releaseHandleHook(handle_s *handleInfo)
{
	int ret = DPWS_OK;
	switch (handleInfo->type)
	{
		case DEVICE_PROXY_TYPE:
			ret = device_free((struct device_proxy *)handleInfo->pObject);
			break;
		case SERVICE_PROXY_TYPE:
			ret = service_free((struct service_proxy *)handleInfo->pObject);
			break;
		default:
			break;
	}
	return ret;
}

static DC_BOOL is_matching_proxy(DA_TYPED(qn) *types, struct qname * type)
{
	int j;
	struct qname * ptype;

	for (j = 0; j < types->nb; j++)
	{
		ptype = DA_GET(types, j);
		if (QNAME_STRUCT_EQUALS_WILDCARD(ptype, type))
			return DC_TRUE;
	}
	return DC_FALSE;
}

static DC_BOOL match_device_by_discovery(short type, struct device_proxy *device, struct lookup_info *info)
{
	if (device->superEPR.status == CACHED)
	{
		DA_TYPED(qn)	types_array = DA_INITIALIZER(qname_t, DC_MEM_CACHE, 1);	// wrapper
	    DA_TYPED(str)	scopes_array = DA_INITIALIZER(char *, DC_MEM_CACHE, 1);	// wrapper
		int i;
		for (i = 0; i < info->f_size; i++)
		{
			types_array.tab = info->filter[i].types;
			types_array.nb = types_array.size = info->filter[i].nb_types;
			scopes_array.tab = info->filter[i].scopes;
			scopes_array.nb = scopes_array.size = info->filter[i].nb_scopes;
			// Match one filter is enough to retain the device (once only then)
			if (device_match(&types_array, &scopes_array, info->filter->match_rule, &device->super.types, &device->scopes))
			{
				// copy entry if needed because count_cache_entry() only needs a count...
				if (info->dpws) {
					short * phref;
					if (!(phref = DA_ADD(&info->result)))
						return DC_TRUE;
					*phref = device->super.href;
				}
				info->count++;
				break;
			}
		}
	}
	return (info->count == info->max_result);
}

static short * lookup(struct dpws * dpws, discovery_filter_t * filter, int f_size, int *res_size)
{
	da_allocator_t daa;
	struct lookup_info info = {NULL, 0, NULL, 0, 0, DA_INITIALIZER(short, DC_MEM_API, 5)};

	info.filter = filter;
	info.f_size = f_size;
	info.dpws = dpws;
	info.max_result = (*res_size > 0) ? *res_size : -1;  // To make sure iteration will not stop if first element does not match when *res_size is 0
	info.result.allocator = soap_get_da_allocator(dpws_dpws2soap(dpws), &daa);

	dcpl_mutex_lock(cache_lock);

	if (handlePoolIterate(&cachePool, DEVICE_PROXY_TYPE, (handle_cbk)match_device_by_discovery, (void *)&info)
			&& info.count < info.max_result) {
		if (dpws)
			dpws->err = DPWS_ERR_EOM;
		*res_size = -1;
		return NULL;	// EOM
	}
	dcpl_mutex_unlock(cache_lock);

	*res_size = info.count;
	return (short*)info.result.tab;
}

static struct wsa_endpoint_ref * create_EPR(struct dpws * dpws, char * Xaddr)
{
	struct wsa_endpoint_ref * epr = NULL;
	if (dpws) {
		DC_ERROR_ASSERT_NO_RC(epr = (struct wsa_endpoint_ref *)DC_MSG_MALLOC(DC_MEM_CACHE, dpws, sizeof(struct wsa_endpoint_ref)));
		dpws_default_wsa_endpoint_ref(epr);
		DC_ERROR_ASSERT_NO_RC(epr->address = DC_MSG_STRDUP(DC_MEM_CACHE, dpws, Xaddr));
	}
	else {
		DC_ERROR_ASSERT_NO_RC(epr = (struct wsa_endpoint_ref *)DC_MALLOC(DC_MEM_API, sizeof(struct wsa_endpoint_ref)));
		dpws_default_wsa_endpoint_ref(epr);
		DC_ERROR_ASSERT_NO_RC((epr->address = DC_STRDUP(DC_MEM_API, Xaddr)) || !Xaddr);
	}
	return epr;
DC_FUNC_ERROR
	if (!dpws)
		dpws_endpoint_ref_free(epr);
	return NULL;
}

static DC_BOOL grab_service(short * href, struct grab_service_info * grabInfo)
{
	struct service_proxy * serviceProxy = (struct service_proxy *)getObject(&cachePool, *href);

	if (serviceProxy && is_matching_proxy(&serviceProxy->super.types, &grabInfo->type)) {
		short * phref;
		if (!(phref = DA_ADD(grabInfo->grabbed)))
			return DC_TRUE;
		*phref = *href;
	}
	return DC_FALSE;
}

static DC_BOOL find_service_hook(short * href, char * id)
{
	struct service_proxy * serviceProxy = (struct service_proxy *)getObject(&cachePool, *href);
	return serviceProxy && serviceProxy->id && !strcmp(id, serviceProxy->id);
}

static DC_BOOL compare_proxies_id(short type, void * p_object, char * id)
{
	switch (type) {
		case DEVICE_PROXY_TYPE:
			return !strcasecmp(((struct device_proxy *)p_object)->superEPR.super.address, id);
		case SERVICE_PROXY_TYPE:
			return !strcmp(((struct service_proxy *)p_object)->id, id);
		default:
			return DC_FALSE;
	}
}

static DC_BOOL string_equals(char ** s1, char * s2)
{
	return !strcmp(*s1, s2);
}

static DC_BOOL lookup_proxy_addresses(short type, void * p_object, char * id)
{
	DA_TYPED(str) * addresses;
	switch (type) {
		case DEVICE_PROXY_TYPE:
			addresses = &((struct device_proxy *)p_object)->transport_addresses;
			break;
		case SERVICE_PROXY_TYPE:
			addresses = &((struct service_proxy *)p_object)->endpoint_references;
			break;
		default:
			return DC_FALSE;
	}
	return DA_BROWSE(addresses, string_equals, id) < addresses->nb;
}

/*----------------------------------------------------------------------------*\
 *                              CONFIGURATION API                             *
\*----------------------------------------------------------------------------*/

static void discovery_filter_free_cbk(int mod, const da_allocator_t * p_alloc, discovery_filter_t *filter)
{
	int i;

	for (i = 0; i < filter->nb_types; i++) {
		p_alloc->free_cbk(mod, p_alloc->param, filter->types[i].ns);	// allocator not used for trace
		p_alloc->free_cbk(mod, p_alloc->param, filter->types[i].lname);
	}
	p_alloc->free_cbk(mod, p_alloc->param, filter->types);
	for (i = 0; i < filter->nb_scopes; i++) {
		p_alloc->free_cbk(mod, p_alloc->param, filter->scopes[i]);
	}
	p_alloc->free_cbk(mod, p_default_allocator->param, filter->scopes);
}

static int dup_disc_filter(discovery_filter_t * dest, discovery_filter_t *src)
{
	int i;

	dest->types = DC_MALLOC(DC_MEM_CACHE, src->nb_types * sizeof(qname_t));
	if (!dest->types)
		goto error;
	for (i = 0; i < src->nb_types; i++) {
		dest->types[i].ns = DC_STRDUP(DC_MEM_CACHE, src->types[i].ns);
		dest->types[i].lname = DC_STRDUP(DC_MEM_CACHE, src->types[i].lname);
	}
	dest->nb_types = i;
	if (i < src->nb_types)
		goto error;

	dest->scopes = DC_MALLOC(DC_MEM_CACHE, src->nb_scopes * sizeof(char *));
	if (!dest->scopes)
		goto error;

	for (i = 0; i < src->nb_scopes; i++) {
		dest->scopes[i] = DC_STRDUP(DC_MEM_CACHE, src->scopes[i]);
	}
	dest->nb_scopes = i;
	if (i < src->nb_scopes)
		goto error;

	dest->match_rule = src->match_rule;
	return DPWS_OK;

error:
	discovery_filter_free_cbk(DC_MEM_CACHE, p_default_allocator, dest);
	return DPWS_ERR_EOM;
}

int set_cache_config_ptr_att(int att, const void * value, DC_BOOL reset)
{
	discovery_filter_t * filter;
	int ret = DPWS_OK;
	switch (att)
	{
		case DPWS_PTR_CACHE_FILTER:
			if (reset)
				DA_EMPTY(&cache_cfg.wsd_filters, discovery_filter_free_cbk);
			if ((filter = DA_ADD(&cache_cfg.wsd_filters)))
				ret = dup_disc_filter(filter, (discovery_filter_t *)value);
			else
				ret = DPWS_ERR_EOM;
			break;
		case DPWS_PTR_CALLBACK_HELLO:
			if (reset)
				cache_cfg.hello_hook = (discovery_cbk)value;
			else
				return DPWS_ERR_MONOVALUED_ATT_TYPE;
			break;
		case DPWS_PTR_CALLBACK_BYE:
			if (reset)
				cache_cfg.bye_hook = (discovery_cbk)value;
			else
				return DPWS_ERR_MONOVALUED_ATT_TYPE;
			break;
		case DPWS_INT_MAX_DEVICE_PROXIES:
			if (reset) {
				cache_cfg.max_dev_proxies = *(uint16_t *)value;
				if (setHandleLRU(
							&cachePool,
							DEVICE_PROXY_TYPE,
							cache_cfg.max_dev_proxies
							))
					return DPWS_ERR_EOM;
			}
			else
				return DPWS_ERR_MONOVALUED_ATT_TYPE;
			break;
		default:
			return DPWS_ERR_NO_SUCH_ATT_ON_OBJECT;
	}
	return ret;
}

void * get_cache_config_ptr_att(int att, int index)
{
	switch (att)
	{
		case DPWS_PTR_CACHE_FILTER:
			return GET_ENTRY(&cache_cfg.wsd_filters, index);
		case DPWS_PTR_CALLBACK_HELLO:
			return (void *)cache_cfg.hello_hook;
		case DPWS_PTR_CALLBACK_BYE:
			return (void *)cache_cfg.bye_hook;
		default:
			return NULL;
	}
}

unsigned long get_cache_config_int_att(int att, int index)
{
	switch (att)
	{
		case DPWS_INT_MAX_DEVICE_PROXIES:
			return cache_cfg.max_dev_proxies;
		default:
			return 0xFFFFFFFF;
	}
}

int get_cache_config_att_count(int att)
{
	switch (att)
	{
		case DPWS_PTR_CACHE_FILTER:
			return cache_cfg.wsd_filters.nb;
		case DPWS_PTR_CALLBACK_HELLO:
			return cache_cfg.hello_hook ? 1 : 0;
		case DPWS_PTR_CALLBACK_BYE:
			return cache_cfg.bye_hook ? 1 : 0;
		default:
			return -1;
	}
}

/*----------------------------------------------------------------------------*\
 *                               Discovery Cache                              *
\*----------------------------------------------------------------------------*/

int init_cache()
{
	if (createHandlePool(
					DC_MEM_CACHE,
					&cachePool,
					CACHE_POOL_NB_BUCKETS,
					NULL,	// enables "outside" protection
					(handle_release_cbk)releaseHandleHook
					))
		return DPWS_ERR_EOM;
	// Do not set the LRU (the default in config is no LRU)
	if ((cache_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)	// NOTE: no need for registry
		return DPWS_ERR_CREATING_MUTEX;
	cache_cfg.wsd_filters.allocator = p_default_allocator;	// could not be initialized as a global variable
	return DPWS_OK;
}

void uninit_cache()
{
	if (cache_lock != NULL)
	{
		deleteHandlePool(&cachePool);
		dcpl_mutex_delete(cache_lock);
	  	cache_lock = NULL;
	}
}

void clean_cache()
{
	if (cache_lock != NULL)
	{
		dcpl_mutex_lock(cache_lock);
		DA_FREE(&cache_cfg.wsd_filters, discovery_filter_free_cbk);
		cache_cfg.hello_hook = NULL;
		cache_cfg.bye_hook = NULL;
		cache_cfg.max_dev_proxies = 0xFFFF;
		deleteHandles(&cachePool, ANY_TYPE);
		dcpl_mutex_unlock(cache_lock);
	}
}

static DC_BOOL match_device_by_id(short type, struct device_proxy *device, char *id)
{
	return device->superEPR.status == CACHED && !strcasecmp(id, device->superEPR.super.address);
}

struct device_proxy * find_device_proxy(char * logical_id)
{
	handle_s * h_device = handlePoolIterate(&cachePool, DEVICE_PROXY_TYPE, (handle_cbk)match_device_by_id, logical_id);
	return h_device ? (struct device_proxy *) h_device->pObject : NULL;
}

struct device_proxy * get_device_proxy(short href, DC_BOOL *isDevice)
{
	handle_s *handleInfo = getHandle(&cachePool, href);
	if (!handleInfo)
		return NULL;

	switch (handleInfo->type)
	{
		case DEVICE_PROXY_TYPE:
			*isDevice = DC_TRUE;
			return (struct device_proxy *)handleInfo->pObject;
		case SERVICE_PROXY_TYPE:
			*isDevice = DC_FALSE;
			return ((struct service_proxy *)handleInfo->pObject)->host;
	}
	return NULL;
}

struct wsa_endpoint_ref * get_default_EPR(struct dpws * dpws, short href)
{
	DA_TYPED(str) * Xaddrs;
	struct wsa_endpoint_ref * ret = NULL;
	handle_s *handleInfo = getHandle(&cachePool, href);

	if (!handleInfo) {
		if (dpws)
			dpws->err = DPWS_ERR_NO_HANDLE_FOUND;
		return NULL;
	}

	switch (handleInfo->type)
	{
		case DEVICE_PROXY_TYPE:
			if (dpws)
				ret = endpoint_ref_dup(dpws, &((struct device_proxy *)handleInfo->pObject)->superEPR.super);
			else
				ret = dpws_endpoint_ref_dup(&((struct device_proxy *)handleInfo->pObject)->superEPR.super);
			break;
		case SERVICE_PROXY_TYPE:
			Xaddrs = &((struct service_proxy *)handleInfo->pObject)->endpoint_references;
			if (Xaddrs && Xaddrs->nb > 0) {
				ret = create_EPR(dpws, *DA_GET(Xaddrs, 0));
				ret->dpws_version = ((struct service_proxy *)handleInfo->pObject)->host->superEPR.super.dpws_version;
			}
			break;
	}
	return ret;
}

static struct wsa_endpoint_ref ** get_EPRs(struct dpws * dpws, short href, int * res_size)
{
	struct wsa_endpoint_ref ** ret;
	int i, offset = 0;
	DA_TYPED(str) * Xaddrs = NULL;
	handle_s *handleInfo = getHandle(&cachePool, href);
	struct dpws_protocols * dpws_version = NULL;

	if (!handleInfo) {
		*res_size = -1;
		if (dpws)
			dpws->err = DPWS_ERR_NO_HANDLE_FOUND;
		return NULL;
	}

	switch (handleInfo->type)
	{
		case DEVICE_PROXY_TYPE:
			Xaddrs = &((struct device_proxy *)handleInfo->pObject)->transport_addresses;
			dpws_version = ((struct device_proxy *)handleInfo->pObject)->superEPR.super.dpws_version;
			offset = 1;
			break;
		case SERVICE_PROXY_TYPE:
			Xaddrs = &((struct service_proxy *)handleInfo->pObject)->endpoint_references;
			if (!Xaddrs || Xaddrs->nb == 0)
				return NULL;
			dpws_version = ((struct service_proxy *)handleInfo->pObject)->host->superEPR.super.dpws_version;
			offset = 0;
			break;
	}
	*res_size = offset + Xaddrs->nb;

	if (dpws) {
		ret = (struct wsa_endpoint_ref**)DC_MSG_MALLOC(DC_MEM_API, dpws, *res_size * sizeof(struct wsa_endpoint_ref *));
		if (ret)
		{
			if (handleInfo->type == DEVICE_PROXY_TYPE)
				ret[0] = endpoint_ref_dup(dpws, &((struct device_proxy *)handleInfo->pObject)->superEPR.super);
			for (i = 0; i < Xaddrs->nb; i++) {
				ret[offset + i] = create_EPR(dpws, *(char **)GET_ENTRY(Xaddrs, i));
				ret[offset + i]->dpws_version = dpws_version;
			}
		}
	}
	else {
		ret = (struct wsa_endpoint_ref**)DC_MALLOC(DC_MEM_API, *res_size * sizeof(struct wsa_endpoint_ref *));
		if (ret)
		{
			if (handleInfo->type == DEVICE_PROXY_TYPE) {
				ret[0] = dpws_endpoint_ref_dup(&((struct device_proxy *)handleInfo->pObject)->superEPR.super);
				if (!ret[0]) {
					DC_FREE(DC_MEM_API, ret);
					return NULL;
				}
			}
			for (i = 0; i < Xaddrs->nb; i++) {
				ret[offset + i] = create_EPR(NULL, *(char **)GET_ENTRY(Xaddrs, i));
				if (!ret[offset + i])
					break;
				ret[offset + i]->dpws_version = dpws_version;
			}
			/* roolback */
			if (i < Xaddrs->nb) {
				int j;
				for (j = 0; j < i; j++) {
					DC_FREE(DC_MEM_API, ret[offset + j]);
				}
			}
		}
	}
	return ret;
}

int count_cache_entry(discovery_filter_t * filter, int f_size, int max_res)
{
	lookup(NULL, filter, f_size, &max_res);
	return max_res;
}

short * cache_lookup(struct dpws * dpws, discovery_filter_t * filter, int f_size, int *res_size)
{
	return lookup(dpws, filter, f_size, res_size);
}

DC_BOOL filter_cbk(discovery_filter_t * p_filter, struct wsd__HelloType *hello)
{
	DA_TYPED(qn) 	types = DA_INITIALIZER(qname_t, DC_MEM_CACHE, 1);	// wrapper
	DA_TYPED(str)	scopes = DA_INITIALIZER(char *, DC_MEM_CACHE, 1);	// wrapper

	types.tab = p_filter->types;
	types.nb = types.size = p_filter->nb_types;
	scopes.tab = p_filter->scopes;
	scopes.nb = scopes.size = p_filter->nb_scopes;
	return device_match(&types, &scopes, p_filter->match_rule,
		(DA_TYPED(qn)*)hello->Types, hello->Scopes ? (DA_TYPED(str)*)hello->Scopes->__item : NULL);
}

DC_BOOL dpws_filter_hello(struct wsd__HelloType *hello)
{
	DC_BOOL ret;

	dcpl_mutex_lock(cache_lock);
	ret = (cache_cfg.wsd_filters.nb == 0 || DA_BROWSE(
												&cache_cfg.wsd_filters,
												filter_cbk,
												hello) < cache_cfg.wsd_filters.nb)
												? DC_TRUE : DC_FALSE;
	if (!ret) {
		struct device_proxy * cache_entry = find_device_proxy(hello->wsa__EndpointReference->address);
		if (cache_entry)
			invalidate_device_proxy(cache_entry);
	}
	dcpl_mutex_unlock(cache_lock);

	return ret;
}

struct device_proxy * dpws_register_device(struct dpws* dpws, struct wsd__HelloType *hello, int msg_type)
{
	struct device_proxy * cache_entry = NULL;
	discovery_cbk hellocbk = NULL;

	DPWSLOG1(DC_CACHE, "-> Registering device %s\n", hello->wsa__EndpointReference->address);
	dcpl_mutex_lock(cache_lock);
	cache_entry = find_device_proxy(hello->wsa__EndpointReference->address);

	if (cache_entry)
	{
		if (hello->MetadataVersion > cache_entry->metadata_version ||
			(hello->MetadataVersion == cache_entry->metadata_version && cache_entry->transport_addresses.nb == 0))
		{
			if (!DPWS_IS_HELLO(dpws->status) || is_sequenced_msg(dpws, cache_entry))
			{
				device_content_free(cache_entry);
				if (overwrite_entry(cache_entry, dpws, hello)) {
					cache_entry = NULL;
					goto exit;
				}
				hellocbk = cache_cfg.hello_hook;	// Call callback only if cache was updated (discard especially multi-interface repetitions)
			}
		}
	}
	else	// does not exist create
	{
		cache_entry = (struct device_proxy *)DC_MALLOC(DC_MEM_CACHE, sizeof(struct device_proxy));
		if (cache_entry)
		{
			DA_INIT(qname_t, &cache_entry->super.types, DC_MEM_CACHE, p_default_allocator, 2);
			cache_entry->super.href = createHandle(&cachePool, DEVICE_PROXY_TYPE, cache_entry);
			if (overwrite_entry(cache_entry, dpws, hello))
			{
				DC_FREE(DC_MEM_CACHE, cache_entry);
				cache_entry = NULL;
				goto exit;
			}
			hellocbk = cache_cfg.hello_hook;
			DPWSLOG1(DC_CACHE, "-> Entry added for device %s\n", cache_entry->superEPR.super.address);
		}
		else
			dpws->err = DPWS_ERR_EOM;
	}

exit:
	dcpl_mutex_unlock(cache_lock);

	//  this section must not be protected since a call to the API can be performed
	if (hellocbk && DPWS_IS_HELLO(msg_type))
		hellocbk(dpws, cache_entry->super.href);

	return cache_entry;
}


int is_sequenced_msg(struct dpws* dpws, struct device_proxy * device)
{
   return (!DPWS_IS_UDP(dpws->status) || (dpws->instance_id > device->instance_id || (dpws->instance_id == device->instance_id && dpws->msg_nb > device->message_number)));
}

short * filter_services_by_type(struct dpws * dpws,	struct device_proxy * device_proxy, char * ns, char * port_type, int *res_size)
{
	da_allocator_t daa;
	DA_TYPED(href) result;
	struct grab_service_info grabInfo;

	grabInfo.dpws = dpws;
	grabInfo.grabbed = &result;
	grabInfo.type.ns = ns;
	grabInfo.type.lname = port_type;
	DA_INIT(short, &result, DC_MEM_API, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), 5);

	// loop over device endpoint types
	if (is_matching_proxy(&device_proxy->super.types, &grabInfo.type)) {
		short * phref;
		DC_ERROR_ASSERT_NO_RC(phref = DA_ADD(&result));
		*phref = device_proxy->super.href;
	}

	// loop over services
	DC_ERROR_ASSERT_NO_RC(DA_BROWSE(&device_proxy->hosted_services, grab_service, &grabInfo) == device_proxy->hosted_services.nb);
	*res_size = result.nb;
	return result.tab;

DC_FUNC_ERROR
	*res_size = -1;
	dpws->err = DPWS_ERR_EOM;
	return NULL;
}

short filter_services_by_id(struct device_proxy * device_proxy, char * id)
{
	// loop over services
	int index = DA_BROWSE(&device_proxy->hosted_services, find_service_hook, id);
	return index < device_proxy->hosted_services.nb ? *DA_GET(&device_proxy->hosted_services, index) : -1;
}

struct service_proxy * add_service_proxy(struct device_proxy * device, struct wsa_endpoint_ref **EPRs, int nbEPRs, DA_TYPED(qn)* types, char* service_id)
{
	int i;
	struct service_proxy * service = NULL;

	service = (struct service_proxy*)DC_MALLOC(DC_MEM_CACHE, sizeof(struct service_proxy));
	if (service) {
		short * phref;
		service->super.href = createHandle(&cachePool, SERVICE_PROXY_TYPE, service);
		DC_ERROR_ASSERT_NO_RC(phref = DA_ADD(&device->hosted_services));
		*phref = service->super.href;
		DA_INIT(qname_t, &service->super.types, DC_MEM_CACHE, p_default_allocator, 2);
		DC_ERROR_ASSERT_NO_RC(!DA_COPY(&service->super.types, types, clone_qname, NULL));
		DC_ERROR_ASSERT_NO_RC((service->id = DC_STRDUP(DC_MEM_CACHE, service_id)) || !service_id);
		DA_INIT(char *, &service->endpoint_references, DC_MEM_CACHE, p_default_allocator, nbEPRs);
		for (i = 0; i < nbEPRs; i++) {
			char ** pstr;
			if (url_match_protocol(dc_network_mode, EPRs[i]->address))	// only keep URLs for the toolkit mode
			{
				DC_ERROR_ASSERT_NO_RC(pstr = DA_ADD(&service->endpoint_references));
				DC_ERROR_ASSERT_NO_RC((*pstr = DC_STRDUP(DC_MEM_CACHE, EPRs[i]->address)) || !EPRs[i]->address);
			}
		}
		service->host = device;
	}
	return service;
DC_FUNC_ERROR
	service_free(service);
	return service;
}

int checkout_proxies(struct dpws * dpws, short * handles, int nb_handles)
{
	int i = 0, ret = DPWS_OK;

	if (!dpws->proxyHandles)
	{
		da_allocator_t * pdaa;
		DC_ERROR_ASSERT_ALLOC(dpws->proxyHandles = (dyn_array_t *)DC_MSG_MALLOC(DC_MEM_CACHE, dpws, sizeof(dyn_array_t)));
		DC_ERROR_ASSERT_ALLOC(pdaa = (da_allocator_t *)DC_MSG_MALLOC(DC_MEM_CACHE, dpws, sizeof(da_allocator_t)));
		DA_INIT(short, dpws->proxyHandles, DC_MEM_API, soap_get_da_allocator(dpws_dpws2soap(dpws), pdaa), 5);
	}
	for (; i < nb_handles; i++)
	{
		short * phref;

		DC_ERROR_ASSERT_ALLOC(phref = DA_ADD((DA_TYPED(href) *)dpws->proxyHandles));
		*phref = handles[i];
		DC_ERROR_ASSERT(checkoutHandle(&cachePool, handles[i]), DPWS_ERR_NO_HANDLE_FOUND);
	}
DC_FUNC_ERROR
	return ret;
}

static DC_BOOL release_proxy(short * href, void * dummy)
{
	if (*href >= 0)
		releaseHandle(&cachePool, *href);
	return DC_FALSE;
}

int release_proxies(struct dpws * dpws)
{
	dcpl_mutex_lock(cache_lock);
	if (dpws->proxyHandles) {
		DA_BROWSE(dpws->proxyHandles, release_proxy, NULL);
		dpws->proxyHandles = NULL;	// tab will be freed by dpws_end
	}
	dcpl_mutex_unlock(cache_lock);
	return DPWS_OK;
}

/*----------------------------------------------------------------------------*\
 *                                  USER API                                  *
\*----------------------------------------------------------------------------*/

struct qname ** dpws_cache_get_types(struct dpws* dpws, short href_proxy)
{
	struct qname ** qns = NULL;
	da_allocator_t daa;
	DA_TYPED(qnp) type_infos;
	DA_TYPED(qn) * types;
	int i, ret = DPWS_OK;
	handle_s *hProxy;

	DC_CHECK_PARAM_RETURN(dpws,NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, href_proxy >= 0, NULL);

	DA_INIT(qname_t *, &type_infos, DC_MEM_API, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), 2);

    dcpl_mutex_lock(cache_lock);
	// find proxy types
	DC_ERROR_ASSERT(hProxy = getHandle(&cachePool, href_proxy), DPWS_ERR_NO_HANDLE_FOUND);

	switch (hProxy->type)
	{
		case DEVICE_PROXY_TYPE:
			types = &((struct device_proxy *)hProxy->pObject)->super.types;
			break;
		case SERVICE_PROXY_TYPE:
			types = &((struct service_proxy *)hProxy->pObject)->super.types;
			break;
		default:
			DC_ERROR_THROW(DPWS_ERR_NO_HANDLE_FOUND);
	}

	// duplicate results
	for (i = 0; i < types->nb; i++) {
		struct qname * current = ((struct qname *)types->tab)+i, * copy, **newqn;
		DC_ERROR_ASSERT_ALLOC(copy = (struct qname *)DC_MSG_MALLOC(DC_MEM_CACHE, dpws, sizeof(struct qname)));
		DC_ERROR_ASSERT_ALLOC(copy->ns = DC_MSG_STRDUP(DC_MEM_CACHE, dpws, current->ns));
		DC_ERROR_ASSERT_ALLOC(copy->lname = DC_MSG_STRDUP(DC_MEM_CACHE, dpws, current->lname));
		DC_ERROR_ASSERT_ALLOC(newqn = DA_ADD(&type_infos));
		*newqn = copy;
	}
	qns = type_infos.tab;

DC_FUNC_ERROR
	dpws->err = ret;
	dcpl_mutex_unlock(cache_lock);
	return qns;
}

DC_BOOL dpws_device_proxy_match(short href_device, discovery_filter_t * criterion)
{
	DC_BOOL ret = DC_FALSE;
	struct device_proxy * w_device = NULL;
	DA_TYPED(qn) types = DA_INITIALIZER(qname_t, DC_MEM_CACHE, 1);	// wrapper
	DA_TYPED(str) scopes = DA_INITIALIZER(char *, DC_MEM_CACHE, 1);	// wrapper

	DC_CHECK_PARAM_RETURN(href_device >= 0 && criterion != NULL && check_discovery_filter(criterion), DC_FALSE);

	types.tab = criterion->types;
	types.nb = types.size = criterion->nb_types;
	scopes.tab = criterion->scopes;
	scopes.nb = scopes.size = criterion->nb_scopes;

	dcpl_mutex_lock(cache_lock);
	w_device = (struct device_proxy*)getTypedObject(&cachePool, href_device, DEVICE_PROXY_TYPE);
	if (w_device)
		ret = device_match(&types, &scopes, criterion->match_rule, &w_device->super.types, &w_device->scopes);
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

int dpws_release_proxy(short href_proxy)
{
	int ret = DPWS_OK;

    DC_CHECK_PARAM(href_proxy >= 0);

    dcpl_mutex_lock(cache_lock);
	if (href_proxy >= 0)
		ret = (releaseHandle(&cachePool, href_proxy) == DPWS_OK) ? DPWS_OK : DPWS_ERR_NO_HANDLE_FOUND;
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

int dpws_pin_proxy(short href_proxy)
{
	int ret;

    DC_CHECK_PARAM(href_proxy >= 0);

	dcpl_mutex_lock(cache_lock);
	ret = checkoutHandle(&cachePool, href_proxy) ? DPWS_OK : DPWS_ERR_NO_HANDLE_FOUND;
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

static int invalidate_device_proxy(struct device_proxy * device)
{
	device->superEPR.status = INVALID;
	DPWSLOG1(DC_CACHE, "-> Entry invalidated for device %s\n", device->superEPR.super.address);
	return releaseHandle(&cachePool, device->super.href);
}

int dpws_unregister_device(struct dpws* dpws, char * address)
{
	discovery_cbk byecbk = NULL;
	struct device_proxy * device;

	dcpl_mutex_lock(cache_lock);

	device = find_device_proxy(address);
	if (!device || !is_sequenced_msg(dpws, device))
		goto exit;

	// call the callback before in order that the user can consult the cache
	byecbk = cache_cfg.bye_hook;
	if (byecbk) {
		dcpl_mutex_unlock(cache_lock);
		byecbk(dpws, device->super.href);	//  this section must not be protected since a call to the API can be performed
		dcpl_mutex_lock(cache_lock);
	}

	invalidate_device_proxy(device);

exit:
	dcpl_mutex_unlock(cache_lock);

	return DPWS_OK;
}

int dpws_invalidate_proxy(short href)
{
	struct device_proxy * device;
	handle_s *handleInfo;
	int ret = DPWS_OK;

    DC_CHECK_PARAM(href >= 0);

    dcpl_mutex_lock(cache_lock);

	DC_ERROR_ASSERT(handleInfo = getHandle(&cachePool, href), DPWS_ERR_NO_HANDLE_FOUND);
	switch (handleInfo->type)
	{
		case DEVICE_PROXY_TYPE:
			device = (struct device_proxy *)handleInfo->pObject;
			invalidate_device_proxy(device);
			break;
		case SERVICE_PROXY_TYPE:
			device = ((struct service_proxy *)handleInfo->pObject)->host;
			if (device)	clean_hosted_services(device);
			break;
	}
DC_FUNC_ERROR
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

void dpws_invalidate(struct dpws* dpws, struct wsa_endpoint_ref* device)
{
	short href = dpws_get_proxy_by_address(device->address);
	if (href >= 0)
		dpws_invalidate_proxy(href);
}



short dpws_get_proxy_by_id(char * id)
{
	handle_s * h;
	short ret = -1;

    DC_CHECK_PARAM(id);

    dcpl_mutex_lock(cache_lock);
	h = handlePoolIterate(&cachePool, ANY_TYPE, (handle_cbk)compare_proxies_id, id);
	if (h)
		ret = h->handleRef;
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

short dpws_get_proxy_by_address(char * address)
{
	handle_s * h;
	short ret = -1;

    DC_CHECK_PARAM(address);

	dcpl_mutex_lock(cache_lock);
	h = handlePoolIterate(&cachePool, ANY_TYPE, (handle_cbk)lookup_proxy_addresses, address);
	if (h)
		ret = h->handleRef;
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

DC_BOOL dpws_check_proxy(short href_proxy)
{
	DC_BOOL ret = DC_FALSE;
	handle_s *handleInfo;
	struct device_proxy * dev;

	DC_CHECK_PARAM_RETURN(href_proxy >= 0, DC_FALSE);

    dcpl_mutex_lock(cache_lock);

	DC_ERROR_ASSERT_NO_RC(handleInfo = getHandle(&cachePool, href_proxy));

	switch (handleInfo->type)
	{
		case DEVICE_PROXY_TYPE:
			dev = (struct device_proxy *)handleInfo->pObject;
			ret = dev->superEPR.status != INVALID;
			break;
		case SERVICE_PROXY_TYPE:
			dev = ((struct service_proxy *)handleInfo->pObject)->host;
			ret = (dev != NULL && dev->superEPR.status != INVALID);
			break;
	}
DC_FUNC_ERROR
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

proxy_type_t dpws_get_proxy_type(short href_proxy)
{
	proxy_type_t ret = UNKNOWN_PROXY_TYPE;
	handle_s *handleInfo;

	DC_CHECK_PARAM_RETURN(href_proxy >= 0, UNKNOWN_PROXY_TYPE);

    dcpl_mutex_lock(cache_lock);
	handleInfo = getHandle(&cachePool, href_proxy);
	if (handleInfo)
		ret = (proxy_type_t)handleInfo->type;
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

struct wsa_endpoint_ref * dpws_get_default_endpoint_ref(struct dpws* dpws, short href_proxy)
{
	struct wsa_endpoint_ref * ret = NULL;

	DC_CHECK_PARAM_RETURN(href_proxy >= 0, NULL);

	dcpl_mutex_lock(cache_lock);
	ret = get_default_EPR(dpws, href_proxy);
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

struct wsa_endpoint_ref ** dpws_get_endpoint_refs(struct dpws* dpws, short href_proxy, int * res_size)
{
	struct wsa_endpoint_ref ** ret = NULL;

	DC_CHECK_PARAM_RETURN(href_proxy >= 0 && res_size, NULL);

	dcpl_mutex_lock(cache_lock);
	ret = get_EPRs(dpws, href_proxy, res_size);
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

char ** dpws_cache_get_scopes(struct dpws* dpws, short href_device)
{
	struct device_proxy * device_proxy;
	da_allocator_t daa;
	DA_TYPED(str) scopes;
	int ret = DPWS_OK;

	DC_CHECK_PARAM_RETURN(dpws,NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, href_device >= 0, NULL);

	DA_INIT(char *, &scopes, DC_MEM_API, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), 2);

    dcpl_mutex_lock(cache_lock);
	DC_ERROR_ASSERT(device_proxy = (struct device_proxy*)getTypedObject(&cachePool, href_device, DEVICE_PROXY_TYPE), DPWS_ERR_NO_HANDLE_FOUND);
	DC_ERROR_ASSERT_ALLOC(DA_STRDUP(&scopes, &device_proxy->scopes));

DC_FUNC_ERROR
	dpws->err = ret;
	dcpl_mutex_unlock(cache_lock);
	return (char **)scopes.tab;
}

char * dpws_cache_get_uuid(struct dpws* dpws, short href_device)
{
	struct device_proxy * device_proxy;
	char * uuid = NULL;
	int ret = DPWS_OK;

	DC_CHECK_PARAM_RETURN(dpws != NULL || href_device >= 0, NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, href_device >= 0, NULL);

    dcpl_mutex_lock(cache_lock);
	DC_ERROR_ASSERT(device_proxy = (struct device_proxy*)getTypedObject(&cachePool, href_device, DEVICE_PROXY_TYPE), DPWS_ERR_NO_HANDLE_FOUND);
	DC_ERROR_ASSERT_ALLOC(uuid =	dpws ?
			DC_MSG_STRDUP(DC_MEM_API, dpws, device_proxy->superEPR.super.address):
			DC_STRDUP(DC_MEM_API, device_proxy->superEPR.super.address));

DC_FUNC_ERROR
	if (dpws != NULL)
		dpws->err = ret;
	dcpl_mutex_unlock(cache_lock);
	return uuid;
}

unsigned long dpws_cache_get_device_metadata_version(short href_device)
{
	struct device_proxy * device;
	unsigned long ret = 0;

    DC_CHECK_PARAM_RETURN(href_device >= 0, 0);

    dcpl_mutex_lock(cache_lock);
	// find device entry
	device = (struct device_proxy*)getObject(&cachePool, href_device);
	if (!device) {
		//ret = DPWS_ERR_NO_HANDLE_FOUND;
		goto exit;
	}
	ret = device->metadata_version;
exit:
	dcpl_mutex_unlock(cache_lock);
	return ret;
}

char** dpws_cache_get_device_transport_addresses(struct dpws* dpws, short href_device, int * res_size)
{
	struct device_proxy * device;
	da_allocator_t daa;
	DA_TYPED(str) da;
	int ret = DPWS_OK;

	DC_CHECK_PARAM_RETURN(dpws,NULL);
	DC_CHECK_PARAM_NO_RC(dpws->err, href_device >= 0 && res_size, NULL);

	DA_INIT(char *, &da, DC_MEM_API, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), 1);

	dcpl_mutex_lock(cache_lock);
	// find device entry
	DC_ERROR_ASSERT(device = (struct device_proxy*)getObject(&cachePool, href_device), DPWS_ERR_NO_HANDLE_FOUND);
	DC_ERROR_ASSERT_ALLOC(DA_STRDUP(&da, &device->transport_addresses));
	*res_size = device->transport_addresses.nb;

DC_FUNC_ERROR
	dpws->err = ret;
	dcpl_mutex_unlock(cache_lock);
	return da.tab;
}

short dpws_get_host_device_proxy(short href_service)
{
	struct service_proxy *service = NULL;

    DC_CHECK_PARAM(href_service >= 0);

	service = (struct service_proxy*)getTypedObject(&cachePool, href_service, SERVICE_PROXY_TYPE);
	return service ? service->host->super.href : -1;
}
