/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2091 $
|                     $Date: 2009-02-17 15:38:44 +0100 (mar, 17 fév 2009) $
\*============================================================================*/

#ifndef CACHE_H
# define CACHE_H

# include "dcCOMN_Tools.h"
# include "dcCOMN_DynArray.h"
# include "dcDPWS_Dpws.h"
# include "dcDPWS_Discovery.h"

/*----------------------------------------------------------------------------*\
 *                               Discovery Cache                              *
\*----------------------------------------------------------------------------*/

DA_TYPED_DECL(wsdf, discovery_filter_t);	/**< WS-Discovery filter array type declaration */

/** Configuration structure for the client cache */
struct dpws_cache_config {
	DA_TYPED(wsdf) wsd_filters;	/**< The discovery_filter_t array that will limit the proxy to store in the cache */
	uint16_t max_dev_proxies;	/**< The maximum number of device proxies that the cache can contain after which the LRU algorithm is activated */
	discovery_cbk hello_hook;	/**< The user callback that is called whenever a Hello message is received. */
	discovery_cbk bye_hook;	/**< The user callback that is called whenever a Bye message is received. */
};

/** Base type for cache proxies */
struct proxy
{
	short href; /**< The proxy handle reference. */
	DA_TYPED(qn) types; /**< A array of QName structures containing proxy. */
};

/** Status used to manage endpoint reference life-cycles */
typedef enum {
	CACHED = 0,	/**< The EPR is attached to a proxy in the cache */
	USER = 1,	/**< The EPR was created by the API user and is not known by the cache (not implemented yet) */
	INVALID = 2	/**< The EPR has been invalidated. */
} endpoint_status;

/** A managed EPR. */
struct client_endpoint_ref
{
	struct wsa_endpoint_ref super;	/**< WS-Addressing EPR base structure */
	endpoint_status status;	/**< The status of the EPR that allow to manage its lifecycle. */
};

/** A device proxy */
struct device_proxy
{
	struct proxy super;	/**< Base proxy structure */
	struct client_endpoint_ref superEPR;	/**< Base EPR structure that will contain the device DPWS address (UUID) */
	unsigned long metadata_version;	/**< Device metadata version received in last Hello, Probe Match or Resolve Match, used to establish if metadata must be replaced */
	unsigned short message_number;	/**< Message number received in last Hello, Bye, Probe Match or Resolve Match, used ignore unsequenced messages. */
	unsigned long instance_id;	/**< Instance Id received in last Hello, Bye, Probe Match or Resolve Match, used ignore unsequenced messages. */
	DA_TYPED(str) scopes;	/**< WS-Discovery scopes. */
	DA_TYPED(str) transport_addresses;	/**< Device transport addresses. */
	DC_BOOL hosted_retrieved;	/**< Flag set to true when the get metadata request has been performed to retrieve hosted services */
	DA_TYPED(href) hosted_services;	/**< Service proxies handle reference array for device hosted services. */
};

struct service_proxy
{
	struct proxy super;	/**< Base proxy structure */
	char *id;	/**< Service ID URI [DPWS] */
	DA_TYPED(str) endpoint_references;	/**< Transport addresses array containing strings instead of full EPR for compactness. */
	struct device_proxy * host;	/**< The hosting device. */
};

/*----------------------------------------------------------------------------*\
 *                              CONFIGURATION API                             *
\*----------------------------------------------------------------------------*/

/** Sets an attribute value on the cache.
 * @param att The attribute numeric identifier.
 * @param value The value to set.
 * @param reset If true, the current value will wil erased and replaced. If
 * false and multi-valued, the value is added to the list.
 * @return A DPWS error code.
 */
int set_cache_config_ptr_att(int att, const void * value, DC_BOOL reset);

/** Get the value of a cache attribute of a pointer type.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value or NULL if not available.
 */
void * get_cache_config_ptr_att(int att, int index);

/** Get the value of an integer attribute of the cache.
 * @param att The attribute numeric identifier.
 * @param index The index of the attribute if multivalued or 0 else.
 * @return The retrieved value.
 */
unsigned long get_cache_config_int_att(int att, int index);

/** Get number of values for a cache attribute.
 * @param att The attribute numeric identifier.
 * @return The number of attribute values.
 */
int get_cache_config_att_count(int att);


/*----------------------------------------------------------------------------*\
 *                                 MODULE API                                 *
\*----------------------------------------------------------------------------*/

extern dcpl_mutex_t * cache_lock;	/**< Module lock. */
extern struct dpws_cache_config cache_cfg;	/**< Module configuration. */

/** Initialize cache structures like mutexes.
 * @return A DPWS error code.
 */
int init_cache();

/** Resets the registry.
 * Proxies are destroyed and configuration parameters reset.
 */
void clean_cache();

/** Cleans cache structures like mutexes.
 */
void uninit_cache();

/** Retrieves a device proxy basing on its UUID.
 * @param id The searched UUID.
 * @return A pointer on the device proxy if found. NULL, else.
 */
struct device_proxy * find_device_proxy(char *id);

/** Retrieves the host device.
 * @param href The device or service proxy handle reference.
 * @param[out] isDevice. True if the input proxy was a device, false if is was a
 * hosted service.
 * @return A pointer on the device proxy if found. NULL, else.
 */
struct device_proxy * get_device_proxy(short href, DC_BOOL *isDevice);

/** Perform a local lookup in the cache using the WS-Discovery semantics.
 * @param dpws The runtime structure used for allocating results.
 * @param filter The WS-Discovery filter array used as for probe matches. Each
 * individual filter has the "intersection" semantic (AND) but specifying
 * several filters allows the caller to add a "union" semantic (OR) to the
 * protocol by performing several consecutive probe operations. If NULL is
 * provided, no filtering is performed at all.
 * @param f_size The number of filters provided in the \a filter parameter.
 * @param[in,out] res_size The maximum number of proxies to retrieve. When
 * the function returns, it contains the actual number of retrieved device
 * endpoints.
 * @return An array of device proxy handle references allocated on the DPWS
 * runtime structure. If an error occurs or no device is found, NULL is
 * returned. To distinguish between the two, use the res_size fields that is 0
 * is no device were retrieved and -1 in case of error.
 */
short * cache_lookup(struct dpws * dpws, discovery_filter_t * filter, int f_size, int *res_size);

/** Perform a local lookup in the cache using the WS-Discovery semantics but
 * without retrieving results, just for counting.
 * @param filter The WS-Discovery filter array used as for probe matches. Each
 * individual filter has the "intersection" semantic (AND) but specifying
 * several filters allows the caller to add a "union" semantic (OR) to the
 * protocol by performing several consecutive probe operations. If NULL is
 * provided, no filtering is performed at all.
 * @param f_size The number of filters provided in the \a filter parameter.
 * @param[in,out] max_res The maximum number of proxies to retrieve.
 * @return The number of device proxies found in the cache matching the input
 * criteria.
 */
int count_cache_entry(discovery_filter_t * filter, int f_size, int max_res);

/** Looks if the new device match the cache filter.
 * @param hello gSOAP generated C representation of a Hello, Probe Match or
 * Resolve Match message.
 * @return DC_TRUE if the hello matches the cache filter so it can be
 * added to it.
 */
DC_BOOL dpws_filter_hello(struct wsd__HelloType *hello);

/** Adds a device to the cache.
 * @param dpws The message context that received the WS-Discovery message.
 * @param hello gSOAP generated C representation of a Hello, Probe Match or
 * Resolve Match message.
 * @param msg_type The type of WS-Discovery message that triggers the call.
 * @return A pointer on the created device proxy structure if it could be
 * created.
 */
struct device_proxy * dpws_register_device(struct dpws* dpws, struct wsd__HelloType *hello, int msg_type);

/** Set the device proxy status to "invalid" and destroys cache ownership
 * when a Bye message is received.
 * @param dpws The message context that received the WS-Discovery message.
 * @param address The device logical address (UUID).
 * @return A DPWS error code.
 */
int dpws_unregister_device(struct dpws* dpws, char * address);

/** Test if a received WS-Discovery message must be considered or if it is
 * out of sequence.
 * @param dpws The message context that received the WS-Discovery message and
 * that contains sequence headers for the message.
 * @param device The pointer on the device proxy found in the cache that
 * contains the last received sequence information.
 * @return True if the new message must be considered.
 */
int is_sequenced_msg(struct dpws* dpws, struct device_proxy * device);

/** Retrieves service proxies basing on a single port type.
 * @param dpws The runtime structure used for allocating results.
 * @param device_proxy A pointer on the hosting device proxy structure.
 * @param ns The filtering port type namespace URI.
 * @param port_type The filtering port type local name.
 * @param[in,out] res_size The maximum number of proxies to retrieve. When
 * the function returns, it contains the actual number of retrieved device
 * endpoints.
 * @return An array of service proxy handle references allocated on the DPWS
 * runtime structure. If an error occurs or no device is found, NULL is
 * returned. To distinguish between the two, use the res_size fields that is 0
 * is no device were retrieved and -1 in case of error.
 */
short * filter_services_by_type(struct dpws * dpws,	struct device_proxy * device_proxy, char * ns, char * port_type, int *res_size);

/** Adds a service to the cache.
 * @param device A pointer on the hosting device proxy structure.
 * @param EPRs An array of EPRs containg service transport addresses.
 * @param nbEPRS The number of entries in \a EPRs.
 * @param types A QName structures array containing service port types.
 * @param service_id The service ID defined by [DPWS].
 * @return A pointer on the created service proxy structure if it could be
 * created.
 */
struct service_proxy * add_service_proxy(struct device_proxy * device, struct wsa_endpoint_ref **EPRs, int nbEPRS, DA_TYPED(qn)* types, char* service_id);

/** Retrieves a service proxy basing on its ID (unique for a device).
 * @param device A pointer on the hosting device proxy structure.
 * @param id The service ID defined by [DPWS].
 * @return The service proxy handle reference or -1 if not found.
 */
short filter_services_by_id(struct device_proxy * device, char * id);

/** Builds the default EPR for any kind of proxy.
 * Note that currenly the toolkit is not able to perform a smart choice
 * between the potential multiple EPRs so the first is considered.
 * @param dpws The runtime structure used for allocating results.
 * @param href The device or service proxy handle reference.
 * @return An EPR allocated on the supplied DPWS runtime structure.
 */
struct wsa_endpoint_ref * get_default_EPR(struct dpws * dpws, short href);

/** Used to perform a handle checkout on proxies returned through the lookup
 * API.
 * @param dpws The DPWS runtime structure that will hold the trace of the
 * retrieved proxies.
 * @param handles An array of proxy handle references.
 * @param nb_handles The number of entries of the supplied array.
 * @return A DPWS error code.
 */
int checkout_proxies(struct dpws * dpws, short * handles, int nb_handles);

/** Release the hold on proxies retrieved through the lookup API.
 * @param dpws The DPWS runtime structure that holds the trace of the retrieved
 * proxies.
 * @return A DPWS error code.
 */
int release_proxies(struct dpws * dpws);

/*----------------------------------------------------------------------------*\
 *                                MODULE CONSTANTS                            *
\*----------------------------------------------------------------------------*/

#define CACHE_POOL_NB_BUCKETS	64

#endif
