/*============================================================================*\
|                                                                              |
|                      SOA4D DPWSCore (C DPWS toolkit)                         |
|                                                                              |
|           ->>  Copyright 2004-2009 Schneider Electric SA <<-                 |
|                                                                              |
|   This program is free software; you can redistribute it and/or modify it    |
|   under the terms of the GNU Lesser General Public License as published by   |
|   the Free Software Foundation; either version 2.1 of the License, or (at    |
|   your option) any later version.                                            |
|                                                                              |
|   This program is distributed in the hope that it will be useful, but        |
|   WITHOUT ANY WARRANTY; without even the implied warranty of                 |
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser    |
|   General Public License for more details.                                   |
|                                                                              |
|   You should have received a copy of the GNU Lesser General Public License   |
|   along with this program; if not, write to the Free Software Foundation,    |
|   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307. You can also get  |
|   it at http://www.gnu.org/licenses/lgpl.html                                |
|                                                                              |
|       + File info:                                                           |
|                     $Revision: 2268 $
|                     $Date: 2009-04-16 14:06:01 +0200 (jeu, 16 avr 2009) $
\*============================================================================*/

#include "dcDPWS_Discovery.h"
#include "dcCOMN_Tools.h"
#include "dcCOMN_Uuid.h"
#include "dcCOMN_DynArray.h"
#include "dcDCPL_Os.h"
#include "dcDCPL_Socket.h"
#include "dcDPWS_Cache.h"
#include "dcDPWS_Registry.h"
#include "dc/dc_Plugin.h"
#include "dcGSOAP_Runtime.h"
#include "dcDPWS_Network.h"
#include "dcDPWS_Memory.h"
#include "dcDPWS_Utils.h"
#include "dcDPWS_Reactor.h"
#include "dcDPWS_Udp.h"
#include <stdlib.h>
#include <string.h>

/*----------------------------------- Types ----------------------------------*/

typedef struct discovery_msg {
	short href_device;
	char uuid[SCHEME_UUID_STRING_SIZE];
	char req_uuid[SCHEME_UUID_STRING_SIZE];
	uint32_t instance_id;
	unsigned short msg_nb;
	struct wsa_endpoint_ref reply_to;
	dcpl_ip_addr_t * netif_addr;
	struct transport_data * transport_data;
	uint32_t itf;
	int (*fsend)(struct dpws *, struct wsa_endpoint_ref *, struct wsd__HelloType * body);
	struct dpws_protocols * protocols;
} discovery_msg_t;

struct match_info {
	DA_TYPED(qn) * types;	// contains struct qname
	DA_TYPED(str) * scopes;
	char * rule;
	char * req_id;
	struct wsa_endpoint_ref * reply_to;
	struct dpws * dpws;
	transport_data_t * tdata;
	uint32_t itf;
};

struct XAddrHostInfo {
	struct dpws * dpws;
	DA_TYPED(str) * XAddrs;
	short href;
};

struct XAddrInfo {
	struct XAddrHostInfo info;
	uint32_t itf;
};

#define RANGE_RANDOM(min, max)	((true_random() % (max - min)) + min)

/*----------------------------------- Data -----------------------------------*/

struct Namespace dpws10_discovery_snd_namespaces[] =
{
	{SOAP_ENV_PREFIX, SOAP_ENV_URI, SOAP_ENV_WILDCARD, NULL},
	{WSA_PREFIX, WSA_200408_URI, WSA_WILDCARD, NULL},
	{WSD_PREFIX, DPWS10_WSD_URI, WSD_WILDCARD, NULL},
	{NULL, NULL, NULL, NULL}
};

struct Namespace dpws11_discovery_snd_namespaces[] =
{
	{SOAP_ENV_PREFIX, SOAP_ENV_URI, SOAP_ENV_WILDCARD, NULL},
	{WSA_PREFIX, WSA_200508_URI, WSA_WILDCARD, NULL},
	{WSD_PREFIX, DPWS11_WSD_URI, WSD_WILDCARD, NULL},
	{NULL, NULL, NULL, NULL}
};


handlePool_t discoveryHandlePool;
dcpl_mutex_t * discovery_lock = NULL;

struct device * dp_server = NULL;
struct wsa_endpoint_ref * dp_EPR = NULL;
DC_BOOL mcast_suppress = DC_FALSE;

/*------------------------- Static Functions prototypes ----------------------*/

static int is_new_message(struct dpws * dpws);
static int new_app_seq_msg_nb(uint32_t * instance_id, unsigned short * msg_nb);
static int soap_mcast_recv_request(struct dpws *dpws);
static int soap_ucast_recv_probematch(struct dpws *dpws);
static int soap_ucast_recv_resolvematch(struct dpws *dpws);
static int wsd_recv(struct dpws *dpws, int (*pFunc)(struct dpws *));
static struct wsd__HelloType * build_hello_body(struct dpws * dpws,
												struct device *device,
												struct wsd__HelloType *body,
												struct wsa_endpoint_ref * local,
												DA_TYPED(str) * XAddrs,
												DA_TYPED(pqn) * types,
												struct wsd__ScopesType *scopes
												);
static void init_probe_body(struct dpws * dpws, dyn_array_t * types, dyn_array_t * scopes, scope_match_rule_t match_rule, struct wsd__ProbeType *body, struct wsd__ScopesType *scps);
static int addXAddr(struct dpws * dpws, DA_TYPED(str) * XAddrs, char * ip_address, short href);
static int addNetifXAddr(dcpl_ip_addr_t * netif_addr, struct XAddrInfo * xaddrInfo);
static DC_BOOL addServicePortXAddr4Netif(short *href, struct XAddrInfo * xaddrInfo);
static DC_BOOL addServicePortXAddr4Host(short *href, struct XAddrHostInfo * xaddrInfo);
static int send_hello_callback(struct dpws* dpws, struct reactor_item * item, handle_s * handle);
static int send_bye_callback(struct dpws* dpws, struct reactor_item * item, handle_s * handle);
static int send_probe_match(struct dpws * dpws, struct wsa_endpoint_ref * discovery_endpoint, struct wsd__HelloType * body);
static int send_resolve_match(struct dpws * dpws, struct wsa_endpoint_ref * discovery_endpoint, struct wsd__HelloType * body);
static int send_dp_hello(struct dpws * dpws, struct wsa_endpoint_ref * discovery_endpoint, struct wsd__HelloType * body);
static int send_match_callback(struct dpws * dpws, struct reactor_item * item, handle_s * handle);
static int _schedule_discovery_msg(dcpl_ip_addr_t * netif_addr, struct device *device, short msgType, struct match_info * info, struct dpws_protocols * protocols);
static int qname_struct_equals(prefixed_qname_t *type1, qname_t *type2);
static DC_BOOL device_match_probe(short type, struct device * device, struct match_info * info);
static int dpws_handle_hello(struct dpws* dpws, struct wsd__HelloType * hello);
static int dpws_handle_bye(struct dpws* dpws, struct wsd__ByeType * bye);
static int dpws_handle_probe(struct dpws* dpws, struct wsd__ProbeType * probe);
static int dpws_handle_probe_match(struct dpws* dpws, struct wsd__ProbeMatchesType* probe_matches);
static int dpws_handle_resolve(struct dpws* dpws, struct wsd__ResolveType* resolve);
static int dpws_handle_resolve_match(struct dpws* dpws, struct wsd__ResolveMatchesType* resolve_matches);
static int dpws_handle_directed_probe(struct dpws* dpws, struct wsd__ProbeType * probe, struct wsd__ProbeMatchesType * probeMatches);

/*----------------------------------------------------------------------------*\
 *                           APP sequence management                          *
\*----------------------------------------------------------------------------*/
static unsigned short app_seq_message_nb = 0;

static int new_app_seq_msg_nb(uint32_t * instance_id, unsigned short * msg_nb)
{
	dcpl_mutex_lock(discovery_lock);
	if (app_seq_message_nb == 0xFFFF)
	{
		*instance_id = ++registry_cfg.boot_seq;
		*msg_nb = app_seq_message_nb = 0;
	}
	else
	{
		*instance_id = registry_cfg.boot_seq;
		*msg_nb = ++app_seq_message_nb;
	}
	dcpl_mutex_unlock(discovery_lock);
	return DPWS_OK;
}

int new_instance_id(uint32_t boot_seq)
{
	int ret = DPWS_OK;
	dcpl_mutex_lock(discovery_lock);
	if (boot_seq > registry_cfg.boot_seq)
	{
		registry_cfg.boot_seq = boot_seq;
		app_seq_message_nb = 0;
	}
	else
		ret = DPWS_ERR_INVALID_PARAMETER;
	dcpl_mutex_unlock(discovery_lock);
	return ret;
}

/*----------------------------------------------------------------------------*\
 *                          SOAP over UDP undoubling                          *
\*----------------------------------------------------------------------------*/

static char uuid_cache[UUID_CACHE_SIZE][UUID_STRING_SIZE];
static int next_index = 0;

int is_new_message(struct dpws * dpws)
{
	int ret = 1, i;
	char * uuid = dpws->message_id;

	if (!DPWS_IS_UDP(dpws->status))
		return 1;

	if (!uuid)
		return 0;

	dcpl_mutex_lock(discovery_lock);
	if (next_index == 0)
		i = UUID_CACHE_SIZE - 1;
	else
		i = next_index - 1;
	while (i != next_index)
	{
		if (!strcasecmp(uuid + UUID_SCHEME_SIZE, uuid_cache[i])) {
			ret = 0;
			break;
		}
		if (i == 0)
			i = UUID_CACHE_SIZE - 1;
		else
			i--;
	}

	// store new uuid (truncating if it exceeds UUID length)
	strncpy(uuid_cache[next_index], uuid + UUID_SCHEME_SIZE, UUID_STRING_SIZE);
	uuid_cache[next_index][UUID_STRING_SIZE-1] = 0;
	next_index++;
	if (next_index == UUID_CACHE_SIZE)
		next_index = 0;
	dcpl_mutex_unlock(discovery_lock);

	return ret;
}

/*******************************************************************************
*                                   RUNTIME                                    *
*******************************************************************************/
static int free_discovery_msg(handle_s *handleInfo)
{
	discovery_msg_t * dm = (discovery_msg_t *)handleInfo->pObject;
	DC_FREE(DC_MEM_TRANSIENT, dm->transport_data);	// free should be protected against NULL
	DC_FREE(DC_MEM_TRANSIENT, dm);
	return DPWS_OK;
}

int init_discovery()
{
	if ((discovery_lock = (dcpl_mutex_t*)dcpl_mutex_init()) == NULL)
		return DPWS_ERR_CREATING_MUTEX;

	if (createHandlePool(
					DC_MEM_DISCOVERY,
					&discoveryHandlePool,
					DISCOVERY_HANDLE_POOL_SIZE,
					discovery_lock,
					&free_discovery_msg
					))
		return DPWS_ERR_EOM;
	return DPWS_OK;
}

void uninit_discovery()
{
	if (discovery_lock)
	{
		deleteHandlePool(&discoveryHandlePool);
		dcpl_mutex_delete(discovery_lock);
	  	discovery_lock = NULL;
	}
}

/*******************************************************************************
*                                 MARSHALLING                                  *
*******************************************************************************/
#define WITH_NOGLOBAL
#ifdef SOAP_FMAC3
# undef SOAP_FMAC3
#endif
#define SOAP_FMAC3 static
#include "discoveryC.c"
#undef WITH_NOGLOBAL

/*******************************************************************************
*                                    STUB                                      *
*******************************************************************************/
// NOTE: in order to keep a standard generation (setting is performed by the caller)
#define discovery_namespaces (dpws->protocols->default_namespaces)
#define discovery_snd_namespaces (dpws->namespaces)
#include "discoveryClient.c"
#undef discovery_snd_namespaces
#undef discovery_namespaces

/*******************************************************************************
*                                  SKELETON                                    *
*******************************************************************************/
#define WITH_NOSERVEREQUEST
#define discovery_namespaces (dpws->protocols->default_namespaces)
#define discovery_snd_namespaces (dpws_soap2dpws(soap)->protocols->discovery_snd_namespaces)
#include "discoveryServer.c"
#undef discovery_snd_namespaces
#undef discovery_namespaces
#undef WITH_NOSERVEREQUEST

// NOTE: the 3 next dispactch functions were written splitting the dispatch of discoveryServer.c
static int soap_mcast_recv_request(struct dpws *dpws)
{
	struct soap *soap = dpws_dpws2soap(dpws);
	const char* action = dpws->action ? dpws->action : "";
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;
	soap_clr_imode(soap, SOAP_XML_STRICT);	// Because extensibility has been removed from generation since it adds code but brings nothing since gSOAP cannot control order (equivalent to ignore unknown)
	if (!dpws->to || !is_discovery_urn(dpws, dpws->to))
		return WSA_ERR_DESTINATION_UNREACHABLE;
	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		dpws->protocols = protocols[i];
		soap_peek_element(soap);
		if (!strcmp(action, dpws->protocols->wsd_hello_action)) {
			DPWSLOG(DC_DISCOVERY, "Hello received\n");
			return soap_serve___wsd__HelloOp(soap);
		}
		if (!strcmp(action, dpws->protocols->wsd_bye_action)) {
			DPWSLOG(DC_DISCOVERY, "Bye received\n");
			return soap_serve___wsd__ByeOp(soap);
		}
		if (!strcmp(action, dpws->protocols->wsd_probe_action)) {
			DPWSLOG(DC_DISCOVERY, "Probe received\n");
			return soap_serve___wsd__ProbeOp(soap);
		}
		if (!strcmp(action, dpws->protocols->wsd_resolve_action)) {
			DPWSLOG(DC_DISCOVERY, "Resolve received\n");
			return soap_serve___wsd__ResolveOp(soap);
		}
	}
	return dpws_dpws2soap(dpws)->error = SOAP_NO_METHOD;
}

static int soap_ucast_recv_probematch(struct dpws *dpws)
{
	struct soap *soap = dpws_dpws2soap(dpws);
	const char* action = dpws->action ? dpws->action : "";
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;
	soap_clr_imode(soap, SOAP_XML_STRICT);	// Because extensibility has been removed from generation since it adds code but brings nothing since gSOAP cannot control order (equivalent to ignore unknown)
	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		dpws->protocols = protocols[i];
		if (!strcmp(action, dpws->protocols->wsd_hello_action)) {
			DPWSLOG(DC_DISCOVERY, "Hello received\n");
			return soap_serve___wsd__HelloOp(soap);
		}
		if (!strcmp(action, dpws->protocols->wsd_probematches_action)) {
			DPWSLOG(DC_DISCOVERY, "ProbeMatch received\n");
			return soap_serve___wsd__ProbeMatchOp(soap);
		}
	}
	return dpws_dpws2soap(dpws)->error = SOAP_NO_METHOD;
}

static int soap_ucast_recv_resolvematch(struct dpws *dpws)
{
	struct soap *soap = dpws_dpws2soap(dpws);
	const char* action = dpws->action ? dpws->action : "";
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;
	soap_clr_imode(soap, SOAP_XML_STRICT);	// Because extensibility has been removed from generation since it adds code but brings nothing since gSOAP cannot control order (equivalent to ignore unknown)
	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		dpws->protocols = protocols[i];
		if (!strcmp(action, dpws->protocols->wsd_hello_action)) {
			DPWSLOG(DC_DISCOVERY, "Hello received\n");
			return soap_serve___wsd__HelloOp(soap);
		}
		if (!strcmp(action, dpws->protocols->wsd_resolvematches_action)) {
			DPWSLOG(DC_DISCOVERY, "ResolveMatch received\n");
			return soap_serve___wsd__ResolveMatchOp(soap);
		}
	}
	return dpws_dpws2soap(dpws)->error = SOAP_NO_METHOD;
}

int wsd_serve_directed_probe(struct dpws *dpws)
{
	struct soap *soap = dpws_dpws2soap(dpws);
	const char* action = dpws->action ? dpws->action : "";
	struct dpws_protocols * protocols[N_DPWS_VERSIONS];
	int i, size;
	size = get_protocols_versions(dpws, protocols, N_DPWS_VERSIONS);
	for (i = 0; i < size; i++) {
		dpws->protocols = protocols[i];
		dpws_set_namespaces(dpws, dpws->protocols->default_namespaces);
		if (!strcmp(action, dpws->protocols->wsd_probe_action))
			return soap_serve___wsd__DirectedProbeOp(soap);
	}
	return dpws_dpws2soap(dpws)->error = SOAP_NO_METHOD;
}

static int wsd_recv(struct dpws *dpws, int (*pFunc)(struct dpws *))
{
	struct soap *soap = dpws_dpws2soap(dpws);
	soap_begin(soap);
	dpws_set_namespaces(dpws, default_protocols->default_namespaces);
	if (soap_begin_recv(soap)
	     || soap_envelope_begin_in(soap)
		 || soap_recv_header(soap)
		 || soap_body_begin_in(soap)
		 || pFunc(dpws)) {
		return soap->error;
	}
	return SOAP_OK;
}

int wsd_mcast_recv(struct dpws *dpws)
{
	DPWS_SET_UDP_MULTICAST(dpws->status);
	return wsd_recv(dpws, &soap_mcast_recv_request);
}

int wsd_ucast_recv_probematch(struct dpws *dpws) {
	DPWS_SET_RCV(dpws->status);
	return wsd_recv(dpws, &soap_ucast_recv_probematch);
}

int wsd_ucast_recv_resolvematch(struct dpws *dpws) {
	DPWS_SET_RCV(dpws->status);
	return wsd_recv(dpws, &soap_ucast_recv_resolvematch);
}

/******************************************************************************\
 *                                                                            *
 * Service Operations (Server-side)                                           *
 *                                                                            *
\******************************************************************************/

/*----------------------------------------------------------------------------*\
 *                                Reception only                              *
\*----------------------------------------------------------------------------*/

SOAP_FMAC5 int SOAP_FMAC6 __wsd__HelloOp(struct dpws* dpws, struct wsd__HelloType *wsd__Hello)
{
	int ret = DPWS_OK;
	if (is_new_message(dpws)) {
		ret = dpws_handle_hello(dpws, wsd__Hello);
	}
	return ret;
}

SOAP_FMAC5 int SOAP_FMAC6 __wsd__ByeOp(struct dpws* dpws, struct wsd__ByeType *wsd__Bye)
{
	int ret = DPWS_OK;
	if (is_new_message(dpws)) {
		ret = dpws_handle_bye(dpws, wsd__Bye);
	}
	return ret;
}

SOAP_FMAC5 int SOAP_FMAC6 __wsd__ProbeOp(struct dpws* dpws, struct wsd__ProbeType *wsd__Probe)
{
	int ret = DPWS_OK;
	if (is_new_message(dpws)) {
		ret = dpws_handle_probe(dpws, wsd__Probe);
	}
	return ret;
}

SOAP_FMAC5 int SOAP_FMAC6 __wsd__ProbeMatchOp(struct dpws* dpws, struct wsd__ProbeMatchesType *wsd__ProbeMatches)
{
	int ret = DPWS_OK;
	if (is_new_message(dpws)) {
		ret = dpws_handle_probe_match(dpws, wsd__ProbeMatches);
	}
	return ret;
}

SOAP_FMAC5 int SOAP_FMAC6 __wsd__ResolveOp(struct dpws* dpws, struct wsd__ResolveType *wsd__Resolve)
{
	int ret = DPWS_OK;
	if (is_new_message(dpws)) {
		ret = dpws_handle_resolve(dpws, wsd__Resolve);
	}
	return ret;
}

SOAP_FMAC5 int SOAP_FMAC6 __wsd__ResolveMatchOp(struct dpws* dpws, struct wsd__ResolveMatchesType *wsd__ResolveMatches)
{
	int ret = DPWS_OK;
	if (is_new_message(dpws)) {
		ret = dpws_handle_resolve_match(dpws, wsd__ResolveMatches);
	}
	return ret;
}

SOAP_FMAC5 int SOAP_FMAC6 __wsd__DirectedProbeOp(struct dpws* dpws, struct wsd__ProbeType *wsd__Probe, struct wsd__ProbeMatchesType *wsd__ProbeMatches)
{
	return dpws_handle_directed_probe(dpws, wsd__Probe, wsd__ProbeMatches);
}

/*----------------------------------------------------------------------------*\
 *                              delayed processing                            *
\*----------------------------------------------------------------------------*/
static struct wsd__HelloType * build_hello_body(struct dpws * dpws,
												struct device *device,
												struct wsd__HelloType *body,
												struct wsa_endpoint_ref * local,
												DA_TYPED(str) * XAddrs,
												DA_TYPED(pqn) * types,
												struct wsd__ScopesType *scopes
												)
{
	/* set message data */
	dpws_default_wsa_endpoint_ref(local);
	body->wsa__EndpointReference = local;
	local->address = device->uuid;
	if (!is_physical_address(device->uuid))
		body->XAddrs = (dyn_array_t *)XAddrs;
	else
		body->XAddrs = NULL;
	body->Types = (dyn_array_t *)get_device_types(dpws, device, types, DC_TRUE);

	// No need to lock the registry since hello and mdm modification cannnot occur at the same time
	if (device->scopes.nb == 0)
		body->Scopes = NULL;	// do not send if default value (reduce Hello payload)
	else {
		scopes->__item = (dyn_array_t *)&device->scopes;
		scopes->MatchBy = NULL;	// spec bug
		body->Scopes = scopes;
	}
	body->MetadataVersion = device->metadata_version;
	return body;
}

static void init_probe_body(struct dpws * dpws, dyn_array_t * types, dyn_array_t * scopes, scope_match_rule_t match_rule, struct wsd__ProbeType *body, struct wsd__ScopesType *scps)
{
	if (types->nb > 0)
        body->Types = types;
	else
        body->Types = NULL;

    if (scopes->nb > 0)
    {
        body->Scopes = scps;
        scps->__item = scopes;
        scps->MatchBy = scope_match_string(dpws, match_rule);
    }
    else
        body->Scopes = NULL;
}

static int addXAddr(struct dpws * dpws, DA_TYPED(str) * XAddrs, char * ip_address, short href)
{
	char * XAddr = get_transport_address(dpws, ip_address, href);
	if (XAddr)
	{
		char ** pstr;
		if (!(pstr = DA_ADD(XAddrs)))
			return DC_TRUE;
		*pstr = XAddr;
		return DPWS_OK;
	}
	else
		return dpws->err;
}

static int addNetifXAddr(dcpl_ip_addr_t * netif_addr, struct XAddrInfo * xaddrInfo)
{
	return addXAddr(xaddrInfo->info.dpws, xaddrInfo->info.XAddrs, netif_addr->ip_address, xaddrInfo->info.href);
}

static DC_BOOL addServicePortXAddr4Netif(short *href, struct XAddrInfo * xaddrInfo)
{
	xaddrInfo->info.href = *href;
	return network_browse_interface(
			xaddrInfo->itf,
			DCPL_AF_INET|DCPL_AF_INET6,
			(network_addr_cbk)addNetifXAddr,
			xaddrInfo
		) ? DC_TRUE : DC_FALSE;
}

static DC_BOOL addServicePortXAddr4Host(short *href, struct XAddrHostInfo * xaddrInfo)
{
	xaddrInfo->href = *href;
	return addXAddr(xaddrInfo->dpws, xaddrInfo->XAddrs, xaddrInfo->dpws->soap.host, xaddrInfo->href)
		? DC_TRUE : DC_FALSE;
}

static int send_hello_callback(struct dpws* dpws, struct reactor_item * item, handle_s * handle)
{
	int ret = DPWS_OK;
	struct wsa_endpoint_ref discovery_endpoint;
	struct device *device = NULL;
	struct wsd__HelloType body;
	struct wsd__ScopesType scopes;
	struct wsa_endpoint_ref hello_endpoint;
	DA_TYPED(str) XAddrs;
	DA_TYPED(pqn) types;
	struct XAddrInfo xaddrInfo;
	discovery_msg_t * disc_info = (discovery_msg_t *)handle->pObject;
	da_allocator_t daa;

	device = (struct device *)checkoutObject(&registryPool, disc_info->href_device);
	if (!device)
		return DPWS_ERR_NO_HANDLE_FOUND;

	/* set message data */
	dpws->message_id = disc_info->uuid;
	dpws->instance_id = disc_info->instance_id;
	dpws->msg_nb = disc_info->msg_nb;
	dpws->protocols = disc_info->protocols;
	if (!(dpws->namespaces = get_device_discovery_ns_table(dpws, device)) && dpws->err) {
		ret = dpws->err;
		goto exit;
	}

	DPWS_SET_UDP_MULTICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_HELLO(dpws->status);
	dpws_default_wsa_endpoint_ref(&discovery_endpoint);
	discovery_endpoint.address = dpws->protocols->wsd_urn;

	if ((ret = dc_udp_open_multicast_output_channel(
				dpws,
				NULL,
				DCPL_ADDR_IS_IPV4(disc_info->netif_addr) ? DISCOVERY_MULTICAST_ADDR:DISCOVERY_MULTICAST_ADDRV6,
				DISCOVERY_PORT,
				disc_info->netif_addr
				)
			))
		goto exit;

	xaddrInfo.info.dpws = dpws;
	xaddrInfo.info.XAddrs = &XAddrs;
	xaddrInfo.itf = disc_info->netif_addr->itf_nb;

	DA_INIT(char *, &XAddrs, DC_MEM_DISCOVERY, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), 1);
	if (browse_service_ports(device->href, (service_port_cbk)addServicePortXAddr4Netif, &xaddrInfo)) {
		ret = DPWS_ERR_EOM;
		goto exit;
	}

	ret = dpws_send___wsd__HelloOp(
				dpws,
				&discovery_endpoint,
				build_hello_body(dpws, device, &body,	&hello_endpoint, &XAddrs, &types, &scopes)
				);
	DPWSLOG1(DC_DISCOVERY, "Hello sent with return value %d\n", ret);
	dc_transport_close_channel(dpws);

exit:
	releaseHandle(&discoveryHandlePool, handle->handleRef);	// to trigger free when not used any more
	releaseHandle(&registryPool, device->href);
	return ret;
}

static int send_bye_callback(struct dpws* dpws, struct reactor_item * item, handle_s * handle)
{
	int ret = DPWS_OK;
	struct wsa_endpoint_ref discovery_endpoint;
	struct device *device = NULL;
	struct wsa_endpoint_ref bye_endpoint;
	struct wsd__ByeType body;
	discovery_msg_t * disc_info = (discovery_msg_t *)handle->pObject;

	device = (struct device *)checkoutObject(&registryPool, disc_info->href_device);
	if (!device)
		return DPWS_ERR_NO_HANDLE_FOUND;

	/* set message data */
	dpws->message_id = disc_info->uuid;
	dpws->instance_id = disc_info->instance_id;
	dpws->msg_nb = disc_info->msg_nb;
	dpws->protocols = disc_info->protocols;
	dpws->namespaces = dpws->protocols->discovery_snd_namespaces;

	DPWS_SET_UDP_MULTICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_BYE(dpws->status);
	dpws_default_wsa_endpoint_ref(&discovery_endpoint);
	discovery_endpoint.address = dpws->protocols->wsd_urn;

	if ((ret = dc_udp_open_multicast_output_channel(
				dpws,
				NULL,
				DCPL_ADDR_IS_IPV4(disc_info->netif_addr) ? DISCOVERY_MULTICAST_ADDR:DISCOVERY_MULTICAST_ADDRV6,
				DISCOVERY_PORT,
				disc_info->netif_addr
				)
			))
		goto exit;

	dpws_default_wsa_endpoint_ref(&bye_endpoint);
	body.wsa__EndpointReference = &bye_endpoint;
	bye_endpoint.address = device->uuid;

	ret = dpws_send___wsd__ByeOp(dpws, &discovery_endpoint, &body);
	DPWSLOG1(DC_DISCOVERY, "Bye sent with return value %d\n", ret);
	dc_transport_close_channel(dpws);

exit:
	releaseHandle(&discoveryHandlePool, handle->handleRef);	// to trigger free when not used any more
	releaseHandle(&registryPool, device->href);
	return ret;
}

int send_probe(struct dpws * dpws, discovery_filter_t * filter, const char * message_id)
{
	int ret = DPWS_OK;
	struct wsa_endpoint_ref discovery_endpoint;
    struct wsd__ProbeType body;
    struct wsd__ScopesType scps;
	DA_TYPED(qn) types_array = DA_INITIALIZER(qname_t, DC_MEM_DISCOVERY, 0);	// wrapper
    DA_TYPED(str) scopes_array = DA_INITIALIZER(char *, DC_MEM_DISCOVERY, 0);	// wrapper

	types_array.tab = filter->types;
    types_array.nb = types_array.size = filter->nb_types;
	scopes_array.tab = filter->scopes;
	scopes_array.nb	= scopes_array.size = filter->nb_scopes;
	init_probe_body(dpws, (dyn_array_t *)&types_array, (dyn_array_t *)&scopes_array, filter->match_rule, &body, &scps);

	dpws_default_wsa_endpoint_ref(&discovery_endpoint);
	discovery_endpoint.address = dpws->protocols->wsd_urn;
	DPWS_SET_UDP_MULTICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_PROBE(dpws->status);
	dpws->namespaces = dpws->protocols->discovery_snd_namespaces;
	dpws->message_id = (char *)message_id;
	ret = dpws_send___wsd__ProbeOp(dpws, &discovery_endpoint, &body);
	DPWSLOG1(DC_DISCOVERY, "Probe sent with return value %d\n", ret);
	return ret;
}

int send_resolve(struct dpws * dpws, struct wsa_endpoint_ref *endpoint, const char * message_id)
{
	int ret = DPWS_OK;
	struct wsa_endpoint_ref discovery_endpoint;
	struct wsd__ResolveType resolveBody;

	dpws_default_wsa_endpoint_ref(&discovery_endpoint);
	discovery_endpoint.address = dpws->protocols->wsd_urn;
	DPWS_SET_UDP_MULTICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_RESOLVE(dpws->status);
	dpws->namespaces = dpws->protocols->discovery_snd_namespaces;
	resolveBody.wsa__EndpointReference = endpoint;
	dpws->message_id = (char *)message_id;
	ret = dpws_send___wsd__ResolveOp(dpws, &discovery_endpoint, &resolveBody);
	DPWSLOG1(DC_DISCOVERY, "Resolve sent with return value %d\n", ret);
	return ret;
}

static int send_probe_match(struct dpws * dpws, struct wsa_endpoint_ref * discovery_endpoint, struct wsd__HelloType * body)
{
	int ret = DPWS_OK;
	struct wsd__ProbeMatchesType probeMatchesBody;

	probeMatchesBody.__sizeProbeMatch = 1;
	probeMatchesBody.ProbeMatch = &body;

	DPWS_SET_UNICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_PROBE_MATCH(dpws->status);
	ret = dpws_send___wsd__ProbeMatchOp(
		dpws,
		discovery_endpoint,
		&probeMatchesBody
		);
	DPWSLOG1(DC_DISCOVERY, "ProbeMatch sent with return value %d\n", ret);
	return ret;
}

static int send_resolve_match(struct dpws * dpws, struct wsa_endpoint_ref * discovery_endpoint, struct wsd__HelloType * body)
{
	int ret = DPWS_OK;
	struct wsd__ResolveMatchesType resolveMatchesBody;

	resolveMatchesBody.ResolveMatch = body;

	DPWS_SET_UNICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_RESOLVE_MATCH(dpws->status);
	ret = dpws_send___wsd__ResolveMatchOp(
		dpws,
		discovery_endpoint,
		&resolveMatchesBody
		);
	DPWSLOG1(DC_DISCOVERY, "ResolveMatch sent with return value %d\n", ret);
	return ret;
}

static int send_dp_hello(struct dpws * dpws, struct wsa_endpoint_ref * discovery_endpoint, struct wsd__HelloType * body)
{
	int ret = DPWS_OK;

	DPWS_SET_UNICAST(dpws->status);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_DP_HELLO(dpws->status);
	ret = dpws_send___wsd__HelloOp(
		dpws,
		discovery_endpoint,
		body
		);
	DPWSLOG1(DC_DISCOVERY, "DP Hello sent with return value %d\n", ret);
	return ret;
}

static int send_match_callback(struct dpws * dpws, struct reactor_item * item, handle_s * handle)
{
	int ret = DPWS_OK;
	struct wsd__HelloType body, *pBody;
	struct wsa_endpoint_ref this_endpoint;
	struct wsd__ScopesType scopes;
	DA_TYPED(str) XAddrs;
	DA_TYPED(pqn) types;
	struct XAddrInfo xaddrInfo;
	discovery_msg_t * disc_info = (discovery_msg_t *)handle->pObject;
	struct device *device = NULL;
	da_allocator_t daa;

	device = (struct device *)checkoutObject(&registryPool, disc_info->href_device);
	if (!device)
		return DPWS_ERR_NO_HANDLE_FOUND;

	dpws->instance_id = disc_info->instance_id;
	dpws->msg_nb = disc_info->msg_nb;
	dpws->message_id = disc_info->uuid;
	dpws->relates_to = disc_info->req_uuid;
	dpws->protocols = disc_info->protocols;

	xaddrInfo.info.dpws = dpws;
	xaddrInfo.info.XAddrs = &XAddrs;
	xaddrInfo.itf = disc_info->itf;

	DA_INIT(char *, &XAddrs, DC_MEM_DISCOVERY, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa), 1);
	if (browse_service_ports(device->href, (service_port_cbk)addServicePortXAddr4Netif, &xaddrInfo)) {
		ret = DPWS_ERR_EOM;
		goto exit;
	}

	pBody = build_hello_body(dpws, device, &body, &this_endpoint, &XAddrs, &types, &scopes);
	if (!(dpws->namespaces = get_device_discovery_ns_table(dpws, device)) && dpws->err)
		ret = dpws->err;

	if (!ret) {
		if (disc_info->transport_data)
		{
			if (!(ret = dc_udp_open_response_channel(dpws, NULL, (udp_transport_data_t *)disc_info->transport_data)))
			{
				ret = disc_info->fsend(dpws, &disc_info->reply_to, pBody);
				dc_transport_close_channel(dpws);
			}
		}
		else
			ret = disc_info->fsend(dpws, &disc_info->reply_to, pBody);
	}
exit:
	releaseHandle(&discoveryHandlePool, handle->handleRef);	// to trigger free when not used any more.
	releaseHandle(&registryPool, device->href);
	return ret;
}

uint32_t get_soap_over_udp_schedule(short * send_nb, int nb_msg, int has_initial_delay, uint32_t last_delay)	// ms
{
    uint32_t delay = 0;

    if (++(*send_nb) > nb_msg)
        return 0xFFFFFFFF;

    switch (*send_nb)
    {
    case 1:
        // use initial delay to avoid network storm
        if (has_initial_delay)
            delay = RANGE_RANDOM(0, APP_MAX_DELAY);
        break;
    case 2:
        delay = last_delay + RANGE_RANDOM(UDP_MIN_DELAY, UDP_MAX_DELAY);
        break;
    default:
        delay = 3 * last_delay;	// last_delay + 2*last_delay
        if (delay > UDP_UPPER_DELAY)
            delay = UDP_UPPER_DELAY;
    }
    return delay;
}

static int _schedule_discovery_msg(dcpl_ip_addr_t * netif_addr, struct device *device, short msgType, struct match_info * info, struct dpws_protocols * protocols)
{
	short send_nb = 0, href;
	long delay = 0;
	handle_s * handle = NULL;
	reactor_cbk handler = NULL;
	discovery_msg_t *discMsg = (discovery_msg_t*)DC_MALLOC(DC_MEM_TRANSIENT, sizeof(discovery_msg_t));

	if (!discMsg)
		return DPWS_ERR_EOM;

	discMsg->href_device = device->href;
	new_app_seq_msg_nb(&discMsg->instance_id, &discMsg->msg_nb);
	discMsg->netif_addr = netif_addr;
	discMsg->transport_data = NULL;	// to protect free
	discMsg->protocols = protocols;
	dpws_schemed_uuid_create(discMsg->uuid);

	if (DPWS_IS_DISCOVERY_RESP_MSG(msgType))
	{
		strncpy(discMsg->req_uuid, info->req_id, SCHEME_UUID_STRING_SIZE);
		discMsg->req_uuid[SCHEME_UUID_STRING_SIZE-1] = 0;
		discMsg->itf = info->itf;
		if (info->tdata)
		{
			discMsg->transport_data = (transport_data_t *)DC_MALLOC(DC_MEM_TRANSIENT, sizeof(udp_transport_data_t));
			memcpy(discMsg->transport_data->peer_addr, info->tdata->peer_addr, sizeof(discMsg->transport_data->peer_addr));
			discMsg->transport_data->peer_port = info->tdata->peer_port;
			discMsg->transport_data->itf = info->tdata->itf;
			dpws_set_anonymous_wsa_endpoint_ref(info->dpws, &discMsg->reply_to);
		}
		else
			memcpy(&discMsg->reply_to, info->reply_to, sizeof(struct wsa_endpoint_ref));
	}

	switch (msgType)
	{
		case DPWS_MSG_BYE:
			device->status = OFFLINE;
			handler = (reactor_cbk)send_bye_callback;
			break;
		case DPWS_MSG_HELLO:
			handler = (reactor_cbk)send_hello_callback;
			device->status = ONLINE;
			break;
		case DPWS_MSG_DP_HELLO:
			handler = (reactor_cbk)send_match_callback;
			discMsg->fsend = send_dp_hello;
			break;
		case DPWS_MSG_PROBE_MATCH:
			handler = (reactor_cbk)send_match_callback;
			discMsg->fsend = send_probe_match;
			break;
		case DPWS_MSG_RESOLVE_MATCH:
			handler = (reactor_cbk)send_match_callback;
			discMsg->fsend = send_resolve_match;
			break;
	}

	while ((delay = get_soap_over_udp_schedule(&send_nb,
			(DPWS_MULTICAST_MSG(msgType) ? MULTICAST_UDP_REPEAT : UNICAST_UDP_REPEAT),
			DPWS_APP_DELAY_MSG(msgType), delay)) != 0xFFFFFFFF)
	{
		if (handle == NULL)
		{
			href = createHandle(&discoveryHandlePool, msgType, discMsg);
			if (href >= 0)
				handle = getHandle(&discoveryHandlePool, href);
			else
				return href;
		}
		else
			checkoutHandle(&discoveryHandlePool, handle->handleRef);

		dc_reactor_register_timer(&reactor, DC_FALSE, delay, 0, handler, handle, NULL, NULL);
	}
	return DPWS_OK;
}

int schedule_discovery_msg(dcpl_ip_addr_t * netif_addr, struct device *device, short msgType, struct dpws_protocols * protocols)
{
	return _schedule_discovery_msg(netif_addr, device, msgType, NULL, protocols);
}

static int dpws_handle_hello(struct dpws* dpws, struct wsd__HelloType *hello)
{
	struct device_proxy * device;	// checks for doubles
	DC_BOOL mcast_suppression;

	dcpl_mutex_lock(cache_lock);
	mcast_suppression = mcast_suppress;
	dcpl_mutex_unlock(cache_lock);

	if (dpws_filter_hello(hello) || (mcast_suppression && dpws->relates_to)) // Hello with relatesTo means DP announcement
	{
		device = dpws_register_device(dpws, hello, DPWS_MSG_HELLO);	// checks for doubles
		if (!device)
			return SOAP_EOM;

		if (dpws->relates_to && mcast_suppression) {
			dcpl_mutex_lock(cache_lock);
			dp_EPR = get_default_EPR(NULL, device->super.href); // This disconnects the DP EPR from the cache
			dcpl_mutex_unlock(cache_lock);
		}
	}

	return DPWS_OK;
}

static int dpws_handle_bye(struct dpws* dpws, struct wsd__ByeType * bye)
{
	if (!bye->wsa__EndpointReference->address)	// Defensive code for empty tag that are parsed as NULL string
		return SOAP_OK;

	return dpws_unregister_device(dpws, bye->wsa__EndpointReference->address);
}

static int qname_struct_equals(prefixed_qname_t *type1, qname_t *type2)
{
	return QNAME_STRUCT_EQUALS((&type1->qname), type2);
}

static DC_BOOL device_match_probe(short type, struct device * device, struct match_info * info)
{
	int ret = DPWS_OK;
	DA_TYPED(pqn) types;

	dcpl_mutex_lock(reg_conf_lock);
	if (DEVICE_VISIBLE(device) && probe_match(
			(dyn_array_t *)info->types, info->scopes, parse_match_rule(info->dpws, info->rule),
			(dyn_array_t *)get_device_types(info->dpws, device, &types, DC_FALSE), &device->scopes,
			(da_cmp_cbk)qname_struct_equals
			)
		)
		ret = _schedule_discovery_msg(NULL, device, DPWS_MSG_PROBE_MATCH, info, info->dpws->protocols);
	dcpl_mutex_unlock(reg_conf_lock);

	return ret;	// continue while no error
}

static int dpws_handle_probe(struct dpws* dpws, struct wsd__ProbeType * probe)
{
	int ret = DPWS_OK;
	struct match_info match_info;

	if (probe)	// Lax parsing may have let is go through...
	{
		match_info.types = (DA_TYPED(qn) *)probe->Types;
		if (probe->Scopes) {
			match_info.scopes = (DA_TYPED(str) *)probe->Scopes->__item;
			match_info.rule = probe->Scopes->MatchBy;
		} else {
			match_info.scopes = NULL;
			match_info.rule = NULL;
		}
		match_info.dpws = dpws;
		match_info.req_id = dpws->message_id;
		match_info.itf = ((transport_data_t *)dpws->transport_data)->itf;
		if (dpws_endpoint_ref_is_null(dpws->reply_to) || !strcmp(dpws->wsa_version->wsa_anonymous_uri, dpws->reply_to.address)) {
			match_info.tdata = (transport_data_t *)dpws->transport_data;
			match_info.reply_to = NULL;
		} else {
			match_info.tdata = NULL;
			match_info.reply_to = &dpws->reply_to;
		}

		dcpl_mutex_lock(reg_conf_lock);
		if (dp_server)
			_schedule_discovery_msg(NULL, dp_server, DPWS_MSG_DP_HELLO, &match_info, dpws->protocols);
		dcpl_mutex_unlock(reg_conf_lock);

		handlePoolIterate(&registryPool, DEVICE_INSTANCE_TYPE, (handle_cbk)device_match_probe, &match_info);

		dc_transport_teardown_channel(dpws, NULL); // Cleans up the recv transport before sending
		DPWSLOG(DC_DISCOVERY, "Processing Probe\n");
	}
	return ret;
}

static int dpws_handle_probe_match(struct dpws* dpws, struct wsd__ProbeMatchesType* probe_matches)
{
	int i = 0;

	for (; i < probe_matches->__sizeProbeMatch; i++) {
		if (!dpws_register_device(dpws, probe_matches->ProbeMatch[i], DPWS_MSG_PROBE_MATCH))
			return DPWS_ERR_EOM;	// First inserted !
	}
	return DPWS_OK;
}

static int dpws_handle_resolve(struct dpws* dpws, struct wsd__ResolveType* resolve)
{
	int ret = DPWS_OK;
	struct service_port * port = NULL;
	struct device * device = NULL;
	short msgType = DPWS_MSG_RESOLVE_MATCH;

	if (resolve && resolve->wsa__EndpointReference && resolve->wsa__EndpointReference->address) {
		port = find_endpoint_port(resolve->wsa__EndpointReference->address);
		if (port) {
			if (!port->service->service_id) // a device endpoint
				device = port->service->device;
		} else {
			dcpl_mutex_lock(reg_conf_lock);
			device = dp_server;
			dcpl_mutex_unlock(reg_conf_lock);
			if (device)
				msgType = DPWS_MSG_DP_HELLO;
		}
	}
	if (device)	{
		struct match_info match_info;

		match_info.types = NULL;
		match_info.scopes = NULL;
		match_info.rule = NULL;
		match_info.dpws = dpws;
		match_info.req_id = dpws->message_id;
		match_info.itf = ((transport_data_t *)dpws->transport_data)->itf;
		if (dpws_endpoint_ref_is_null(dpws->reply_to) || !strcmp(dpws->wsa_version->wsa_anonymous_uri, dpws->reply_to.address)) {
			match_info.tdata = (transport_data_t *)dpws->transport_data;
			match_info.reply_to = NULL;
		} else {
			match_info.tdata = NULL;
			match_info.reply_to = &dpws->reply_to;
		}
		_schedule_discovery_msg(NULL, device, msgType, &match_info, dpws->protocols);

		dc_transport_teardown_channel(dpws, NULL); // Cleans up the recv transport before sending
		DPWSLOG(DC_DISCOVERY, "Processing Resolve.");
	}
	if (port) {
		release_endpoint_port(port);
	}
	return ret;
}

static int dpws_handle_resolve_match(struct dpws* dpws, struct wsd__ResolveMatchesType* resolve_matches)
{
	if (!resolve_matches->ResolveMatch || !resolve_matches->ResolveMatch->wsa__EndpointReference)
		return DPWS_OK;
	return dpws_register_device(dpws, resolve_matches->ResolveMatch, DPWS_MSG_RESOLVE_MATCH) ? DPWS_OK : DPWS_ERR_EOM;	// checks for doubles
}

/* Directed probe functions */

char * call_directed_probe(struct dpws * dpws, char * address, discovery_filter_t * filter)
{
	struct wsa_endpoint_ref endpoint;
	struct wsd__ProbeMatchesType probe_matches;
	char *result = NULL;
    struct wsd__ProbeType body;
    struct wsd__ScopesType scps;
	DA_TYPED(qn) types_array = DA_INITIALIZER(qname_t, DC_MEM_DISCOVERY, 0);	// wrapper
    DA_TYPED(str) scopes_array = DA_INITIALIZER(char *, DC_MEM_DISCOVERY, 0);	//wrapper

	types_array.tab = filter->types;
    types_array.nb = types_array.size = filter->nb_types;
	scopes_array.tab = filter->scopes;
	scopes_array.nb	= scopes_array.size = filter->nb_scopes;
	init_probe_body(dpws, (dyn_array_t *)&types_array, (dyn_array_t *)&scopes_array, filter->match_rule, &body, &scps);

	dpws_default_wsa_endpoint_ref(&endpoint);
	soap_default_wsd__ProbeMatchesType(dpws_dpws2soap(dpws), &probe_matches);
	endpoint.address = address;
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_DIRECTED_PROBE(dpws->status);
	dpws->namespaces = dpws->protocols->discovery_snd_namespaces;
	if (dpws_call___wsd__DirectedProbeOp(dpws, &endpoint, NULL, &body, &probe_matches)) {
		dpws_dpws2soap(dpws)->error = SOAP_OK;   // Unsupported directed probe (unknown protocol version)
	} else if (probe_matches.__sizeProbeMatch > 0) {
		result = probe_matches.ProbeMatch[0]->wsa__EndpointReference->address;
		dpws_handle_probe_match(dpws, &probe_matches);
		return result;
	} else if (dpws_dpws2soap(dpws)->status == 202) {
		dpws_dpws2soap(dpws)->error = SOAP_OK;  // Non matching probe
	}
	return result;
}

static int dpws_handle_directed_probe(struct dpws* dpws, struct wsd__ProbeType * probe, struct wsd__ProbeMatchesType * probeMatches)
{
	if (probe)	// Lax parsing may have let is go through...
	{
		/* check that the message must be processed */
		struct device * device = get_endpoint(dpws->href_endpoint)->device;
		if (get_device_mode(device) == DPWS_DEVICE_USE) {
			DA_TYPED(pqn) device_services;
			DA_TYPED(str)* scopes = NULL;
			char * rule = NULL;

			/* check that scope and types matches */
			if (probe->Scopes) {
				scopes = (DA_TYPED(str) *)probe->Scopes->__item;
				rule = probe->Scopes->MatchBy;
			}
			if (DEVICE_VISIBLE(device) && probe_match(
												(dyn_array_t *)probe->Types, scopes, parse_match_rule(dpws, rule),
												(dyn_array_t *)get_device_types(dpws, device, &device_services, DC_TRUE), &device->scopes,
												(da_cmp_cbk)qname_struct_equals
												)
				) {
				struct wsd__HelloType *body, *pBody;
				struct wsd__ScopesType * p_scopes;
				DA_TYPED(str)* XAddrs;
				DA_TYPED(pqn)* types;
				struct XAddrInfo xaddrInfo;
				struct wsa_endpoint_ref * this_endpoint;
				da_allocator_t daa;

				if (!(body = (struct wsd__HelloType*)DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, sizeof(struct wsd__HelloType)))
					|| !(XAddrs = (DA_TYPED(str)*)DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, sizeof(dyn_array_t)))
					|| !(types = (DA_TYPED(pqn)*)DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, sizeof(dyn_array_t)))
					|| !(p_scopes = (struct wsd__ScopesType*)DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, sizeof(struct wsd__ScopesType))))
					return DPWS_ERR_EOM;
				DA_INIT(char *, XAddrs, DC_MEM_DISCOVERY, soap_get_da_allocator(dpws_dpws2soap(dpws), &daa) , 1);
				xaddrInfo.info.dpws = dpws;
				xaddrInfo.info.XAddrs = XAddrs;
				if (browse_service_ports(dpws->href_endpoint, (service_port_cbk)addServicePortXAddr4Host, &xaddrInfo)
					|| !(this_endpoint = (struct wsa_endpoint_ref*)DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, sizeof(struct wsa_endpoint_ref))))
					return DPWS_ERR_EOM;
				probeMatches->__sizeProbeMatch = 1;
				pBody = build_hello_body(dpws, device, body, this_endpoint, XAddrs, types, p_scopes);
				if (!(probeMatches->ProbeMatch = (struct wsd__HelloType**)DC_MSG_MALLOC(DC_MEM_DISCOVERY, dpws, sizeof(struct wsd_HelloType*))))
					return DPWS_ERR_EOM;
				probeMatches->ProbeMatch[0] = pBody;
				if (!(dpws->namespaces = get_device_discovery_ns_table(dpws, device)) && dpws->err)
					return dpws->err;
				dpws_set_namespaces(dpws, dpws->namespaces);
				return DPWS_OK;
			}
		}
	}
	dpws_send_empty_response(dpws);
	soap_closesock(dpws_dpws2soap(dpws));
	return dpws->err = DPWS_ERR_COULD_NOT_MATCH_DIRECTED_PROBE;
}

/* Discovery proxy functions */

int call_discovery_proxy_probe(struct dpws* dpws, struct wsa_endpoint_ref * proxy, discovery_filter_t * filter, int timeout)
{
	struct soap * soap = dpws_dpws2soap(dpws);
	int ret = DPWS_OK;
	struct wsd__ProbeMatchesType probe_matches;
	struct wsd__ProbeType body;
	struct wsd__ScopesType scps;
	DA_TYPED(qn) types_array = DA_INITIALIZER(qname_t, DC_MEM_DISCOVERY, 0);	// wrapper
	DA_TYPED(str) scopes_array = DA_INITIALIZER(char *, DC_MEM_DISCOVERY, 0);	// wrapper

	timeout = MIN(timeout, DP_MAX_TIMEOUT);

	types_array.tab = filter->types;
	types_array.nb = types_array.size = filter->nb_types;
	scopes_array.tab = filter->scopes;
	scopes_array.nb	= scopes_array.size = filter->nb_scopes;
	init_probe_body(dpws, (dyn_array_t *)&types_array, (dyn_array_t *)&scopes_array, filter->match_rule, &body, &scps);

	soap_default_wsd__ProbeMatchesType(soap, &probe_matches);
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_PROBE(dpws->status);
	dpws->namespaces = dpws->protocols->discovery_snd_namespaces;
	soap->connect_timeout = -timeout * 1000;
	soap->send_timeout = -timeout * 1000;
	soap->recv_timeout = -timeout * 1000;
	if ((ret = dpws_call___wsd__DiscoveryProxyProbeOp(dpws, proxy, NULL, &body, &probe_matches))) {
		dcpl_mutex_lock(cache_lock);
		dpws_endpoint_ref_free(dp_EPR);
		dp_EPR = NULL;
		dcpl_mutex_unlock(cache_lock);
		return ret;
	}
	dpws_handle_probe_match(dpws, &probe_matches);
	return DPWS_OK;
}

struct device_proxy * call_discovery_proxy_resolve(struct dpws* dpws, struct wsa_endpoint_ref * proxy, char * id, int timeout)
{
	struct wsa_endpoint_ref endpoint;
	struct wsd__ResolveType resolveBody;
	struct wsd__ResolveMatchesType resolve_matches;
	struct soap * soap = dpws_dpws2soap(dpws);
	int ret = DPWS_OK;

	timeout = MIN(timeout, DP_MAX_TIMEOUT);
	dpws_default_wsa_endpoint_ref(&endpoint);
	endpoint.address = id;
	soap_default_wsd__ResolveType(soap, &resolveBody);
	soap_default_wsd__ResolveMatchesType(soap, &resolve_matches);
	resolveBody.wsa__EndpointReference = &endpoint;
	DPWS_SET_SEND(dpws->status);
	DPWS_SET_RESOLVE(dpws->status);
	dpws->namespaces = dpws->protocols->discovery_snd_namespaces;
	soap->connect_timeout = -timeout * 1000;
	soap->send_timeout = -timeout * 1000;
	soap->recv_timeout = -timeout * 1000;
	if ((ret = dpws_call___wsd__DiscoveryProxyResolveOp(dpws, proxy, NULL, &resolveBody, &resolve_matches))) {
		dcpl_mutex_lock(cache_lock);
		dpws_endpoint_ref_free(dp_EPR);
		dp_EPR = NULL;
		dcpl_mutex_unlock(cache_lock);
		return NULL;
	}
	dpws_handle_resolve_match(dpws, &resolve_matches);
	return find_device_proxy(id);
}

